/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hdfs.qjournal.client;

import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.TimeoutException;

import org.apache.hadoop.util.FakeTimer;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.Timeout;

import org.apache.hadoop.thirdparty.com.google.common.base.Joiner;
import org.apache.hadoop.thirdparty.com.google.common.collect.ImmutableMap;
import org.apache.hadoop.thirdparty.com.google.common.util.concurrent.SettableFuture;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.fail;

public class TestQuorumCall {
  @Test
  @Timeout(value = 10)
  public void testQuorums() throws Exception {
    Map<String, SettableFuture<String>> futures = ImmutableMap.of(
        "f1", SettableFuture.<String>create(),
        "f2", SettableFuture.<String>create(),
        "f3", SettableFuture.<String>create());
    
    QuorumCall<String, String> q = QuorumCall.create(futures);
    assertEquals(0, q.countResponses());
    
    futures.get("f1").set("first future");
    q.waitFor(1, 0, 0, 100000, "test"); // wait for 1 response
    q.waitFor(0, 1, 0, 100000, "test"); // wait for 1 success
    assertEquals(1, q.countResponses());
    
    
    futures.get("f2").setException(new Exception("error"));
    assertEquals(2, q.countResponses());
    
    futures.get("f3").set("second future");
    q.waitFor(3, 0, 100, 100000, "test"); // wait for 3 responses
    q.waitFor(0, 2, 100, 100000, "test"); // 2 successes

    assertEquals(3, q.countResponses());
    assertEquals("f1=first future,f3=second future",
        Joiner.on(",").withKeyValueSeparator("=").join(
            new TreeMap<String, String>(q.getResults())));
    
    try {
      q.waitFor(0, 4, 100, 10, "test");
      fail("Didn't time out waiting for more responses than came back");
    } catch (TimeoutException te) {
      // expected
    }
  }
  @Test
  @Timeout(value = 10)
  public void testQuorumFailsWithoutResponse() throws Exception {
    Map<String, SettableFuture<String>> futures = ImmutableMap.of(
        "f1", SettableFuture.<String>create());

    QuorumCall<String, String> q = QuorumCall.create(futures);
    assertEquals(0, q.countResponses(), "The number of quorum calls for which a response has been"
            + " received should be 0");

    try {
      q.waitFor(0, 1, 100, 10, "test");
      fail("Didn't time out waiting for more responses than came back");
    } catch (TimeoutException te) {
      // expected
    }
  }

  @Test
  @Timeout(value = 10)
  public void testQuorumSucceedsWithLongPause() throws Exception {
    final Map<String, SettableFuture<String>> futures = ImmutableMap.of(
        "f1", SettableFuture.<String>create());

    FakeTimer timer = new FakeTimer() {
      private int callCount = 0;
      @Override
      public long monotonicNowNanos() {
        callCount++;
        if (callCount == 1) {
          long old = super.monotonicNowNanos();
          advance(1000000);
          return old;
        } else if (callCount == 10) {
          futures.get("f1").set("first future");
          return super.monotonicNowNanos();
        } else {
          return super.monotonicNowNanos();
        }
      }
    };

    QuorumCall<String, String> q = QuorumCall.create(futures, timer);
    assertEquals(0, q.countResponses());

    q.waitFor(1, 0, 0, 3000, "test"); // wait for 1 response
  }

}
