/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.qjournal;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.concurrent.TimeoutException;
import java.util.function.Supplier;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.qjournal.QJMTestUtil;
import org.apache.hadoop.hdfs.qjournal.client.QuorumJournalManager;
import org.apache.hadoop.hdfs.qjournal.server.JournalNode;
import org.apache.hadoop.metrics2.lib.DefaultMetricsSystem;
import org.apache.hadoop.net.NetUtils;
import org.apache.hadoop.test.GenericTestUtils;
import org.apache.hadoop.thirdparty.com.google.common.base.Joiner;
import org.apache.hadoop.util.Lists;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class MiniJournalCluster
implements Closeable {
    public static final String CLUSTER_WAITACTIVE_URI = "waitactive";
    private static final Logger LOG = LoggerFactory.getLogger(MiniJournalCluster.class);
    private final File baseDir;
    private final JNInfo[] nodes;

    private MiniJournalCluster(Builder b) throws IOException {
        if (b.httpPorts != null && b.httpPorts.length != b.numJournalNodes) {
            throw new IllegalArgumentException("Num of http ports (" + b.httpPorts.length + ") should match num of JournalNodes (" + b.numJournalNodes + ")");
        }
        if (b.rpcPorts != null && b.rpcPorts.length != b.numJournalNodes) {
            throw new IllegalArgumentException("Num of rpc ports (" + b.rpcPorts.length + ") should match num of JournalNodes (" + b.numJournalNodes + ")");
        }
        LOG.info("Starting MiniJournalCluster with " + b.numJournalNodes + " journal nodes");
        this.baseDir = b.baseDir != null ? new File(b.baseDir) : new File(MiniDFSCluster.getBaseDirectory());
        this.nodes = new JNInfo[b.numJournalNodes];
        for (int i = 0; i < b.numJournalNodes; ++i) {
            if (b.format) {
                File dir = this.getStorageDir(i);
                LOG.debug("Fully deleting JN directory " + dir);
                FileUtil.fullyDelete((File)dir);
            }
            JournalNode jn = new JournalNode();
            jn.setConf(this.createConfForNode(b, i));
            jn.start();
            this.nodes[i] = new JNInfo(jn);
        }
    }

    public URI getQuorumJournalURI(String jid) {
        ArrayList addrs = Lists.newArrayList();
        for (JNInfo info : this.nodes) {
            addrs.add("127.0.0.1:" + info.ipcAddr.getPort());
        }
        String addrsVal = Joiner.on((String)";").join((Iterable)addrs);
        LOG.debug("Setting logger addresses to: " + addrsVal);
        try {
            return new URI("qjournal://" + addrsVal + "/" + jid);
        }
        catch (URISyntaxException e) {
            throw new AssertionError((Object)e);
        }
    }

    public void start() throws IOException {
        for (JNInfo info : this.nodes) {
            info.node.start();
        }
    }

    public void shutdown() throws IOException {
        boolean failed = false;
        for (JNInfo info : this.nodes) {
            try {
                info.node.stopAndJoin(0);
            }
            catch (Exception e) {
                failed = true;
                LOG.warn("Unable to stop journal node " + info.node, (Throwable)e);
            }
        }
        if (failed) {
            throw new IOException("Unable to shut down. Check log for details");
        }
    }

    private Configuration createConfForNode(Builder b, int idx) {
        Configuration conf = new Configuration(b.conf);
        File logDir = this.getStorageDir(idx);
        conf.set("dfs.journalnode.edits.dir", logDir.toString());
        int httpPort = b.httpPorts != null ? b.httpPorts[idx] : 0;
        int rpcPort = b.rpcPorts != null ? b.rpcPorts[idx] : 0;
        conf.set("dfs.journalnode.rpc-address", "localhost:" + rpcPort);
        conf.set("dfs.journalnode.http-address", "localhost:" + httpPort);
        return conf;
    }

    public File getStorageDir(int idx) {
        return new File(this.baseDir, "journalnode-" + idx).getAbsoluteFile();
    }

    public File getJournalDir(int idx, String jid) {
        return new File(this.getStorageDir(idx), jid);
    }

    public File getCurrentDir(int idx, String jid) {
        return new File(this.getJournalDir(idx, jid), "current");
    }

    public File getPreviousDir(int idx, String jid) {
        return new File(this.getJournalDir(idx, jid), "previous");
    }

    public JournalNode getJournalNode(int i) {
        return this.nodes[i].node;
    }

    public String getJournalNodeIpcAddress(int i) {
        return this.nodes[i].ipcAddr.toString();
    }

    public void restartJournalNode(int i) throws InterruptedException, IOException {
        JNInfo info = this.nodes[i];
        JournalNode jn = info.node;
        Configuration conf = new Configuration(jn.getConf());
        if (jn.isStarted()) {
            jn.stopAndJoin(0);
        }
        conf.set("dfs.journalnode.rpc-address", NetUtils.getHostPortString((InetSocketAddress)info.ipcAddr));
        String uri = info.httpServerURI;
        if (uri.startsWith("http://")) {
            conf.set("dfs.journalnode.http-address", uri.substring("http://".length()));
        } else if (info.httpServerURI.startsWith("https://")) {
            conf.set("dfs.journalnode.https-address", uri.substring("https://".length()));
        }
        JournalNode newJN = new JournalNode();
        newJN.setConf(conf);
        newJN.start();
        info.node = newJN;
    }

    public int getQuorumSize() {
        return this.nodes.length / 2 + 1;
    }

    public int getNumNodes() {
        return this.nodes.length;
    }

    public void waitActive() throws IOException {
        for (int i = 0; i < this.nodes.length; ++i) {
            final int index = i;
            try {
                GenericTestUtils.waitFor((Supplier)new Supplier<Boolean>(){

                    @Override
                    public Boolean get() {
                        try {
                            QuorumJournalManager qjm = new QuorumJournalManager(MiniJournalCluster.this.nodes[index].node.getConf(), MiniJournalCluster.this.getQuorumJournalURI(MiniJournalCluster.CLUSTER_WAITACTIVE_URI), QJMTestUtil.FAKE_NSINFO);
                            qjm.hasSomeData();
                            qjm.close();
                        }
                        catch (IOException e) {
                            return false;
                        }
                        return true;
                    }
                }, (long)50L, (long)3000L);
                continue;
            }
            catch (TimeoutException e) {
                throw new AssertionError((Object)("Time out while waiting for journal node " + index + " to start."));
            }
            catch (InterruptedException ite) {
                LOG.warn("Thread interrupted when waiting for node start", (Throwable)ite);
            }
        }
    }

    public void setNamenodeSharedEditsConf(String jid) {
        URI quorumJournalURI = this.getQuorumJournalURI(jid);
        for (int i = 0; i < this.nodes.length; ++i) {
            this.nodes[i].node.getConf().set("dfs.namenode.shared.edits.dir", quorumJournalURI.toString());
        }
    }

    @Override
    public void close() throws IOException {
        this.shutdown();
    }

    public static class Builder {
        private String baseDir;
        private int numJournalNodes = 3;
        private boolean format = true;
        private final Configuration conf;
        private int[] httpPorts = null;
        private int[] rpcPorts = null;

        public Builder(Configuration conf) {
            this.conf = conf;
        }

        public Builder(Configuration conf, File baseDir) {
            this.conf = conf;
            this.baseDir(baseDir.toString());
        }

        public Builder baseDir(String d) {
            this.baseDir = d;
            return this;
        }

        public Builder numJournalNodes(int n) {
            this.numJournalNodes = n;
            return this;
        }

        public Builder format(boolean f) {
            this.format = f;
            return this;
        }

        public Builder setHttpPorts(int ... ports) {
            this.httpPorts = ports;
            return this;
        }

        public Builder setRpcPorts(int ... ports) {
            this.rpcPorts = ports;
            return this;
        }

        public MiniJournalCluster build() throws IOException {
            return new MiniJournalCluster(this);
        }

        static {
            DefaultMetricsSystem.setMiniClusterMode((boolean)true);
        }
    }

    private static final class JNInfo {
        private JournalNode node;
        private final InetSocketAddress ipcAddr;
        private final String httpServerURI;

        private JNInfo(JournalNode node) {
            this.node = node;
            this.ipcAddr = node.getBoundIpcAddress();
            this.httpServerURI = node.getHttpServerURI();
        }
    }
}

