/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.concurrent.TimeoutException;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CreateFlag;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.StreamCapabilities;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.hdfs.DFSClient;
import org.apache.hadoop.hdfs.DFSOutputStream;
import org.apache.hadoop.hdfs.DFSStripedOutputStream;
import org.apache.hadoop.hdfs.DFSTestUtil;
import org.apache.hadoop.hdfs.DataStreamer;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.StripedFileTestUtil;
import org.apache.hadoop.hdfs.client.HdfsDataOutputStream;
import org.apache.hadoop.hdfs.protocol.DatanodeInfo;
import org.apache.hadoop.hdfs.protocol.ErasureCodingPolicy;
import org.apache.hadoop.hdfs.protocol.ExtendedBlock;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.io.erasurecode.ErasureCodeNative;
import org.apache.hadoop.test.GenericTestUtils;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.Timeout;
import org.mockito.Mockito;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.event.Level;

public class TestDFSStripedOutputStream {
    public static final Logger LOG = LoggerFactory.getLogger(TestDFSStripedOutputStream.class);
    private ErasureCodingPolicy ecPolicy;
    private int dataBlocks;
    private int parityBlocks;
    private MiniDFSCluster cluster;
    private DistributedFileSystem fs;
    private Configuration conf;
    private int cellSize;
    private final int stripesPerBlock = 4;
    private int blockSize;
    @Rule
    public Timeout globalTimeout = new Timeout(300000);

    public ErasureCodingPolicy getEcPolicy() {
        return StripedFileTestUtil.getDefaultECPolicy();
    }

    @Before
    public void setup() throws IOException {
        this.ecPolicy = this.getEcPolicy();
        this.dataBlocks = (short)this.ecPolicy.getNumDataUnits();
        this.parityBlocks = (short)this.ecPolicy.getNumParityUnits();
        this.cellSize = this.ecPolicy.getCellSize();
        this.blockSize = 4 * this.cellSize;
        System.out.println("EC policy = " + this.ecPolicy);
        int numDNs = this.dataBlocks + this.parityBlocks + 2;
        this.conf = new Configuration();
        this.conf.setLong("dfs.blocksize", (long)this.blockSize);
        this.conf.setBoolean("dfs.namenode.redundancy.considerLoad", false);
        this.conf.setInt("dfs.namenode.replication.max-streams", 0);
        if (ErasureCodeNative.isNativeCodeLoaded()) {
            this.conf.set("io.erasurecode.codec.rs.rawcoders", "rs_native");
        }
        this.cluster = new MiniDFSCluster.Builder(this.conf).numDataNodes(numDNs).build();
        this.fs = this.cluster.getFileSystem();
        DFSTestUtil.enableAllECPolicies(this.fs);
        this.fs.getClient().setErasureCodingPolicy("/", this.ecPolicy.getName());
    }

    @After
    public void tearDown() {
        if (this.cluster != null) {
            this.cluster.shutdown();
            this.cluster = null;
        }
    }

    @Test
    public void testFileEmpty() throws Exception {
        this.testOneFile("/EmptyFile", 0);
    }

    @Test
    public void testFileSmallerThanOneCell1() throws Exception {
        this.testOneFile("/SmallerThanOneCell", 1);
    }

    @Test
    public void testFileSmallerThanOneCell2() throws Exception {
        this.testOneFile("/SmallerThanOneCell", this.cellSize - 1);
    }

    @Test
    public void testFileEqualsWithOneCell() throws Exception {
        this.testOneFile("/EqualsWithOneCell", this.cellSize);
    }

    @Test
    public void testFileSmallerThanOneStripe1() throws Exception {
        this.testOneFile("/SmallerThanOneStripe", this.cellSize * this.dataBlocks - 1);
    }

    @Test
    public void testFileSmallerThanOneStripe2() throws Exception {
        this.testOneFile("/SmallerThanOneStripe", this.cellSize + 123);
    }

    @Test
    public void testFileEqualsWithOneStripe() throws Exception {
        this.testOneFile("/EqualsWithOneStripe", this.cellSize * this.dataBlocks);
    }

    @Test
    public void testFileMoreThanOneStripe1() throws Exception {
        this.testOneFile("/MoreThanOneStripe1", this.cellSize * this.dataBlocks + 123);
    }

    @Test
    public void testFileMoreThanOneStripe2() throws Exception {
        this.testOneFile("/MoreThanOneStripe2", this.cellSize * this.dataBlocks + this.cellSize * this.dataBlocks + 123);
    }

    @Test
    public void testFileLessThanFullBlockGroup() throws Exception {
        this.testOneFile("/LessThanFullBlockGroup", this.cellSize * this.dataBlocks * 3 + this.cellSize);
    }

    @Test
    public void testFileFullBlockGroup() throws Exception {
        this.testOneFile("/FullBlockGroup", this.blockSize * this.dataBlocks);
    }

    @Test
    public void testFileMoreThanABlockGroup1() throws Exception {
        this.testOneFile("/MoreThanABlockGroup1", this.blockSize * this.dataBlocks + 123);
    }

    @Test
    public void testFileMoreThanABlockGroup2() throws Exception {
        this.testOneFile("/MoreThanABlockGroup2", this.blockSize * this.dataBlocks + this.cellSize + 123);
    }

    @Test
    public void testFileMoreThanABlockGroup3() throws Exception {
        this.testOneFile("/MoreThanABlockGroup3", this.blockSize * this.dataBlocks * 3 + this.cellSize * this.dataBlocks + this.cellSize + 123);
    }

    @Test
    public void testStreamFlush() throws Exception {
        byte[] bytes = StripedFileTestUtil.generateBytes(this.blockSize * this.dataBlocks * 3 + this.cellSize * this.dataBlocks + this.cellSize + 123);
        try (FSDataOutputStream os = this.fs.create(new Path("/ec-file-1"));){
            Assert.assertFalse((String)"DFSStripedOutputStream should not have hflush() capability yet!", (boolean)os.hasCapability(StreamCapabilities.StreamCapability.HFLUSH.getValue()));
            Assert.assertFalse((String)"DFSStripedOutputStream should not have hsync() capability yet!", (boolean)os.hasCapability(StreamCapabilities.StreamCapability.HSYNC.getValue()));
            try (ByteArrayInputStream is = new ByteArrayInputStream(bytes);){
                IOUtils.copyBytes((InputStream)is, (OutputStream)os, (int)bytes.length);
                os.hflush();
                IOUtils.copyBytes((InputStream)is, (OutputStream)os, (int)bytes.length);
                os.hsync();
                IOUtils.copyBytes((InputStream)is, (OutputStream)os, (int)bytes.length);
            }
            Assert.assertTrue((String)"stream is not a DFSStripedOutputStream", (boolean)(os.getWrappedStream() instanceof DFSStripedOutputStream));
            DFSStripedOutputStream dfssos = (DFSStripedOutputStream)os.getWrappedStream();
            dfssos.hsync(EnumSet.of(HdfsDataOutputStream.SyncFlag.UPDATE_LENGTH));
        }
    }

    private void testOneFile(String src, int writeBytes) throws Exception {
        src = (String)src + "_" + writeBytes;
        Path testPath = new Path((String)src);
        byte[] bytes = StripedFileTestUtil.generateBytes(writeBytes);
        DFSTestUtil.writeFile((FileSystem)this.fs, testPath, new String(bytes));
        StripedFileTestUtil.waitBlockGroupsReported(this.fs, (String)src);
        StripedFileTestUtil.checkData(this.fs, testPath, writeBytes, new ArrayList<DatanodeInfo>(), null, this.blockSize * this.dataBlocks);
    }

    @Test
    public void testFileBlockSizeSmallerThanCellSize() throws Exception {
        Path path = new Path("testFileBlockSizeSmallerThanCellSize");
        byte[] bytes = StripedFileTestUtil.generateBytes(this.cellSize * 2);
        try {
            DFSTestUtil.writeFile((FileSystem)this.fs, path, bytes, this.cellSize / 2);
            Assert.fail((String)"Creating a file with block size smaller than ec policy's cell size should fail");
        }
        catch (IOException expected) {
            LOG.info("Caught expected exception", (Throwable)expected);
            GenericTestUtils.assertExceptionContains((String)"less than the cell size", (Throwable)expected);
        }
    }

    @Test
    public void testExceptionInCloseECFileWithRecoverLease() throws Exception {
        Configuration config = new Configuration();
        config.setBoolean("dfs.client.write.recover.lease.on.close.exception", true);
        DFSClient client = new DFSClient(this.cluster.getNameNode(0).getNameNodeAddress(), config);
        DFSClient spyClient = (DFSClient)Mockito.spy((Object)client);
        DFSOutputStream dfsOutputStream = spyClient.create("/testExceptionInCloseECFileWithRecoverLease", FsPermission.getFileDefault(), EnumSet.of(CreateFlag.CREATE), (short)3, 0x100000L, null, 1024, null);
        Assert.assertTrue((String)"stream should be a DFSStripedOutputStream", (boolean)(dfsOutputStream instanceof DFSStripedOutputStream));
        DFSOutputStream spyDFSOutputStream = (DFSOutputStream)Mockito.spy((Object)dfsOutputStream);
        ((DFSOutputStream)Mockito.doThrow((Throwable[])new Throwable[]{new IOException("Emulated IOException in close")}).when((Object)spyDFSOutputStream)).completeFile((ExtendedBlock)Mockito.any());
        try {
            spyDFSOutputStream.close();
            Assert.fail();
        }
        catch (IOException ioe) {
            Assert.assertTrue((boolean)spyDFSOutputStream.isLeaseRecovered());
            this.waitForFileClosed("/testExceptionInCloseECFileWithRecoverLease");
            Assert.assertTrue((boolean)this.isFileClosed("/testExceptionInCloseECFileWithRecoverLease"));
        }
    }

    @Test
    public void testExceptionInCloseECFileWithoutRecoverLease() throws Exception {
        Configuration config = new Configuration();
        DFSClient client = new DFSClient(this.cluster.getNameNode(0).getNameNodeAddress(), config);
        DFSClient spyClient = (DFSClient)Mockito.spy((Object)client);
        DFSOutputStream dfsOutputStream = spyClient.create("/testExceptionInCloseECFileWithoutRecoverLease", FsPermission.getFileDefault(), EnumSet.of(CreateFlag.CREATE), (short)3, 0x100000L, null, 1024, null);
        Assert.assertTrue((String)"stream should be a DFSStripedOutputStream", (boolean)(dfsOutputStream instanceof DFSStripedOutputStream));
        DFSOutputStream spyDFSOutputStream = (DFSOutputStream)Mockito.spy((Object)dfsOutputStream);
        ((DFSOutputStream)Mockito.doThrow((Throwable[])new Throwable[]{new IOException("Emulated IOException in close")}).when((Object)spyDFSOutputStream)).completeFile((ExtendedBlock)Mockito.any());
        try {
            spyDFSOutputStream.close();
            Assert.fail();
        }
        catch (IOException ioe) {
            Assert.assertFalse((boolean)spyDFSOutputStream.isLeaseRecovered());
            try {
                this.waitForFileClosed("/testExceptionInCloseECFileWithoutRecoverLease");
            }
            catch (TimeoutException e) {
                Assert.assertFalse((boolean)this.isFileClosed("/testExceptionInCloseECFileWithoutRecoverLease"));
            }
        }
    }

    private boolean isFileClosed(String path) throws IOException {
        return this.cluster.getFileSystem().isFileClosed(new Path(path));
    }

    private void waitForFileClosed(String path) throws Exception {
        GenericTestUtils.waitFor(() -> {
            boolean closed;
            try {
                closed = this.isFileClosed(path);
            }
            catch (IOException e) {
                return false;
            }
            return closed;
        }, (long)1000L, (long)5000L);
    }

    static {
        GenericTestUtils.setLogLevel((Logger)DFSOutputStream.LOG, (Level)Level.TRACE);
        GenericTestUtils.setLogLevel((Logger)DataStreamer.LOG, (Level)Level.TRACE);
    }
}

