/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.federation.router;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.UUID;
import java.util.function.Predicate;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CreateFlag;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.QuotaUsage;
import org.apache.hadoop.fs.StorageType;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.hdfs.DFSClient;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.client.HdfsDataOutputStream;
import org.apache.hadoop.hdfs.protocol.ClientProtocol;
import org.apache.hadoop.hdfs.protocol.DSQuotaExceededException;
import org.apache.hadoop.hdfs.protocol.NSQuotaExceededException;
import org.apache.hadoop.hdfs.protocol.QuotaByStorageTypeExceededException;
import org.apache.hadoop.hdfs.server.federation.MiniRouterDFSCluster;
import org.apache.hadoop.hdfs.server.federation.RouterConfigBuilder;
import org.apache.hadoop.hdfs.server.federation.StateStoreDFSCluster;
import org.apache.hadoop.hdfs.server.federation.resolver.MountTableManager;
import org.apache.hadoop.hdfs.server.federation.resolver.MountTableResolver;
import org.apache.hadoop.hdfs.server.federation.resolver.RemoteLocation;
import org.apache.hadoop.hdfs.server.federation.router.Quota;
import org.apache.hadoop.hdfs.server.federation.router.Router;
import org.apache.hadoop.hdfs.server.federation.router.RouterClient;
import org.apache.hadoop.hdfs.server.federation.router.RouterQuotaManager;
import org.apache.hadoop.hdfs.server.federation.router.RouterQuotaUpdateService;
import org.apache.hadoop.hdfs.server.federation.router.RouterQuotaUsage;
import org.apache.hadoop.hdfs.server.federation.store.driver.StateStoreDriver;
import org.apache.hadoop.hdfs.server.federation.store.protocol.AddMountTableEntryRequest;
import org.apache.hadoop.hdfs.server.federation.store.protocol.AddMountTableEntryResponse;
import org.apache.hadoop.hdfs.server.federation.store.protocol.GetMountTableEntriesRequest;
import org.apache.hadoop.hdfs.server.federation.store.protocol.GetMountTableEntriesResponse;
import org.apache.hadoop.hdfs.server.federation.store.protocol.RemoveMountTableEntryRequest;
import org.apache.hadoop.hdfs.server.federation.store.protocol.RemoveMountTableEntryResponse;
import org.apache.hadoop.hdfs.server.federation.store.protocol.UpdateMountTableEntryRequest;
import org.apache.hadoop.hdfs.server.federation.store.protocol.UpdateMountTableEntryResponse;
import org.apache.hadoop.hdfs.server.federation.store.records.MountTable;
import org.apache.hadoop.hdfs.server.federation.store.records.QueryResult;
import org.apache.hadoop.security.AccessControlException;
import org.apache.hadoop.test.GenericTestUtils;
import org.apache.hadoop.test.LambdaTestUtils;
import org.apache.hadoop.util.Time;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

public class TestRouterQuota {
    private static StateStoreDFSCluster cluster;
    private static MiniRouterDFSCluster.NamenodeContext nnContext1;
    private static MiniRouterDFSCluster.NamenodeContext nnContext2;
    private static MiniRouterDFSCluster.RouterContext routerContext;
    private static MountTableResolver resolver;
    private static final int BLOCK_SIZE = 512;

    @BeforeEach
    public void setUp() throws Exception {
        cluster = new StateStoreDFSCluster(false, 2);
        Configuration routerConf = new RouterConfigBuilder().stateStore().admin().quota().rpc().build();
        routerConf.set("dfs.federation.router.quota-cache.update.interval", "2s");
        Configuration hdfsConf = new Configuration(false);
        hdfsConf.setInt("dfs.blocksize", 512);
        hdfsConf.setInt("dfs.replication", 1);
        cluster.addRouterOverrides(routerConf);
        cluster.addNamenodeOverrides(hdfsConf);
        cluster.startCluster();
        cluster.startRouters();
        cluster.waitClusterUp();
        nnContext1 = cluster.getNamenode(cluster.getNameservices().get(0), null);
        nnContext2 = cluster.getNamenode(cluster.getNameservices().get(1), null);
        routerContext = cluster.getRandomRouter();
        Router router = routerContext.getRouter();
        resolver = (MountTableResolver)router.getSubclusterResolver();
    }

    @AfterEach
    public void tearDown() {
        if (cluster != null) {
            cluster.stopRouter(routerContext);
            cluster.shutdown();
            cluster = null;
        }
    }

    @Test
    public void testNamespaceQuotaExceed() throws Exception {
        long nsQuota = 3L;
        FileSystem nnFs1 = nnContext1.getFileSystem();
        FileSystem nnFs2 = nnContext2.getFileSystem();
        nnFs1.mkdirs(new Path("/testdir1"));
        nnFs2.mkdirs(new Path("/testdir2"));
        MountTable mountTable1 = MountTable.newInstance((String)"/nsquota", Collections.singletonMap("ns0", "/testdir1"));
        mountTable1.setQuota(new RouterQuotaUsage.Builder().quota(nsQuota).build());
        this.addMountTable(mountTable1);
        MountTable mountTable2 = MountTable.newInstance((String)"/nsquota/subdir", Collections.singletonMap("ns1", "/testdir2"));
        mountTable2.setQuota(new RouterQuotaUsage.Builder().quota(nsQuota).build());
        this.addMountTable(mountTable2);
        FileSystem routerFs = routerContext.getFileSystem();
        ArrayList created = new ArrayList();
        GenericTestUtils.waitFor(() -> {
            boolean isNsQuotaViolated = false;
            try {
                Path p = new Path("/nsquota/" + UUID.randomUUID());
                routerFs.mkdirs(p);
                created.add(p);
                p = new Path("/nsquota/subdir/" + UUID.randomUUID());
                routerFs.mkdirs(p);
                created.add(p);
            }
            catch (NSQuotaExceededException e) {
                isNsQuotaViolated = true;
            }
            catch (IOException iOException) {
                // empty catch block
            }
            return isNsQuotaViolated;
        }, (long)5000L, (long)60000L);
        nnFs1.mkdirs(new Path("/testdir1/" + UUID.randomUUID()));
        nnFs2.mkdirs(new Path("/testdir2/" + UUID.randomUUID()));
        Assertions.assertFalse((boolean)created.isEmpty());
        for (Path src : created) {
            Path dst = new Path(src.toString() + "-renamed");
            routerFs.rename(src, dst);
            routerFs.delete(dst, true);
        }
    }

    @Test
    public void testStorageSpaceQuotaExceed() throws Exception {
        long ssQuota = 3071L;
        FileSystem nnFs1 = nnContext1.getFileSystem();
        FileSystem nnFs2 = nnContext2.getFileSystem();
        nnFs1.mkdirs(new Path("/testdir3"));
        nnFs2.mkdirs(new Path("/testdir4"));
        MountTable mountTable1 = MountTable.newInstance((String)"/ssquota", Collections.singletonMap("ns0", "/testdir3"));
        mountTable1.setQuota(new RouterQuotaUsage.Builder().spaceQuota(ssQuota).build());
        this.addMountTable(mountTable1);
        MountTable mountTable2 = MountTable.newInstance((String)"/ssquota/subdir", Collections.singletonMap("ns1", "/testdir4"));
        mountTable2.setQuota(new RouterQuotaUsage.Builder().spaceQuota(ssQuota).build());
        this.addMountTable(mountTable2);
        DFSClient routerClient = routerContext.getClient();
        routerClient.create("/ssquota/file", true).close();
        routerClient.create("/ssquota/subdir/file", true).close();
        GenericTestUtils.waitFor(() -> {
            boolean isDsQuotaViolated = false;
            try {
                this.appendData("/ssquota/file", routerClient, 512);
                this.appendData("/ssquota/subdir/file", routerClient, 512);
            }
            catch (DSQuotaExceededException e) {
                isDsQuotaViolated = true;
            }
            catch (IOException iOException) {
                // empty catch block
            }
            return isDsQuotaViolated;
        }, (long)5000L, (long)60000L);
        this.appendData("/testdir3/file", nnContext1.getClient(), 512);
        this.appendData("/testdir4/file", nnContext2.getClient(), 512);
    }

    @Test
    public void testStorageTypeQuotaExceed() throws Exception {
        long ssQuota = 1536L;
        DFSClient routerClient = routerContext.getClient();
        this.prepareStorageTypeQuotaTestMountTable(StorageType.DISK, 512L, ssQuota * 2L, ssQuota, 513, 513);
        LambdaTestUtils.intercept(DSQuotaExceededException.class, (String)"The DiskSpace quota is exceeded", (String)"Expect quota exceed exception.", () -> this.appendData("/type0/file", routerClient, 512));
        LambdaTestUtils.intercept(DSQuotaExceededException.class, (String)"The DiskSpace quota is exceeded", (String)"Expect quota exceed exception.", () -> this.appendData("/type0/type1/file", routerClient, 512));
        LambdaTestUtils.intercept(QuotaByStorageTypeExceededException.class, (String)"Quota by storage type", (String)"Expect quota exceed exception.", () -> this.appendData("/type0/file", nnContext1.getClient(), 512));
        LambdaTestUtils.intercept(QuotaByStorageTypeExceededException.class, (String)"Quota by storage type", (String)"Expect quota exceed exception.", () -> this.appendData("/type1/file", nnContext1.getClient(), 512));
    }

    private boolean addMountTable(MountTable entry) throws IOException {
        RouterClient client = routerContext.getAdminClient();
        MountTableManager mountTableManager = client.getMountTableManager();
        AddMountTableEntryRequest addRequest = AddMountTableEntryRequest.newInstance((MountTable)entry);
        AddMountTableEntryResponse addResponse = mountTableManager.addMountTableEntry(addRequest);
        resolver.loadCache(true);
        return addResponse.getStatus();
    }

    private boolean updateMountTable(MountTable entry) throws IOException {
        RouterClient client = routerContext.getAdminClient();
        MountTableManager mountTableManager = client.getMountTableManager();
        UpdateMountTableEntryRequest updateRequest = UpdateMountTableEntryRequest.newInstance((MountTable)entry);
        UpdateMountTableEntryResponse updateResponse = mountTableManager.updateMountTableEntry(updateRequest);
        resolver.loadCache(true);
        return updateResponse.getStatus();
    }

    private void appendData(String path, DFSClient client, int dataLen) throws IOException {
        EnumSet<CreateFlag> createFlag = EnumSet.of(CreateFlag.APPEND);
        HdfsDataOutputStream stream = client.append(path, 1024, createFlag, null, null);
        byte[] data = new byte[dataLen];
        stream.write(data);
        stream.close();
    }

    @Test
    public void testSetQuotaToMountTableEntry() throws Exception {
        long nsQuota = 10L;
        long ssQuota = 10240L;
        long diskQuota = 1024L;
        FileSystem nnFs1 = nnContext1.getFileSystem();
        nnFs1.mkdirs(new Path("/testSetQuotaToFederationPath"));
        nnFs1.mkdirs(new Path("/testSetQuotaToFederationPath/dir0"));
        MountTable mountTable = MountTable.newInstance((String)"/setquota", Collections.singletonMap("ns0", "/testSetQuotaToFederationPath"));
        this.addMountTable(mountTable);
        RouterQuotaUpdateService updateService = routerContext.getRouter().getQuotaCacheUpdateService();
        updateService.periodicInvoke();
        FileSystem routerFs = routerContext.getFileSystem();
        LambdaTestUtils.intercept(AccessControlException.class, (String)"is not allowed to change quota of", (String)"Expect an AccessControlException.", () -> routerFs.setQuota(new Path("/setquota"), nsQuota, ssQuota));
        LambdaTestUtils.intercept(AccessControlException.class, (String)"is not allowed to change quota of", (String)"Expect an AccessControlException.", () -> routerFs.setQuotaByStorageType(new Path("/setquota"), StorageType.DISK, diskQuota));
        QuotaUsage quota = nnFs1.getQuotaUsage(new Path("/testSetQuotaToFederationPath"));
        Assertions.assertEquals((long)-1L, (long)quota.getQuota());
        Assertions.assertEquals((long)-1L, (long)quota.getSpaceQuota());
        routerFs.setQuota(new Path("/setquota/dir0"), nsQuota, ssQuota);
        routerFs.setQuotaByStorageType(new Path("/setquota/dir0"), StorageType.DISK, diskQuota);
        quota = nnFs1.getQuotaUsage(new Path("/testSetQuotaToFederationPath/dir0"));
        Assertions.assertEquals((long)nsQuota, (long)quota.getQuota());
        Assertions.assertEquals((long)ssQuota, (long)quota.getSpaceQuota());
        Assertions.assertEquals((long)diskQuota, (long)quota.getTypeQuota(StorageType.DISK));
    }

    @Test
    public void testSetQuota() throws Exception {
        long nsQuota = 5L;
        long ssQuota = 100L;
        FileSystem nnFs1 = nnContext1.getFileSystem();
        FileSystem nnFs2 = nnContext2.getFileSystem();
        nnFs1.mkdirs(new Path("/testdir5"));
        nnFs2.mkdirs(new Path("/testdir6"));
        MountTable mountTable1 = MountTable.newInstance((String)"/setquota", Collections.singletonMap("ns0", "/testdir5"));
        mountTable1.setQuota(new RouterQuotaUsage.Builder().quota(nsQuota).spaceQuota(ssQuota).build());
        this.addMountTable(mountTable1);
        MountTable mountTable2 = MountTable.newInstance((String)"/setquota/subdir", Collections.singletonMap("ns1", "/testdir6"));
        this.addMountTable(mountTable2);
        RouterQuotaUpdateService updateService = routerContext.getRouter().getQuotaCacheUpdateService();
        updateService.periodicInvoke();
        ClientProtocol client1 = nnContext1.getClient().getNamenode();
        ClientProtocol client2 = nnContext2.getClient().getNamenode();
        QuotaUsage quota1 = client1.getQuotaUsage("/testdir5");
        QuotaUsage quota2 = client2.getQuotaUsage("/testdir6");
        Assertions.assertEquals((long)nsQuota, (long)quota1.getQuota());
        Assertions.assertEquals((long)ssQuota, (long)quota1.getSpaceQuota());
        Assertions.assertEquals((long)nsQuota, (long)quota2.getQuota());
        Assertions.assertEquals((long)ssQuota, (long)quota2.getSpaceQuota());
    }

    @Test
    public void testStorageTypeQuota() throws Exception {
        long ssQuota = 1536L;
        int fileSize = 512;
        this.prepareStorageTypeQuotaTestMountTable(StorageType.DISK, 512L, ssQuota * 2L, ssQuota, fileSize, fileSize);
        ClientProtocol client = nnContext1.getClient().getNamenode();
        QuotaUsage usage = client.getQuotaUsage("/type0");
        Assertions.assertEquals((long)-1L, (long)usage.getQuota());
        Assertions.assertEquals((long)-1L, (long)usage.getSpaceQuota());
        this.verifyTypeQuotaAndConsume(new long[]{-1L, -1L, ssQuota * 2L, -1L, -1L, -1L}, null, usage);
        usage = client.getQuotaUsage("/type1");
        Assertions.assertEquals((long)-1L, (long)usage.getQuota());
        Assertions.assertEquals((long)-1L, (long)usage.getSpaceQuota());
        this.verifyTypeQuotaAndConsume(new long[]{-1L, -1L, ssQuota, -1L, -1L, -1L}, null, usage);
        FileSystem routerFs = routerContext.getFileSystem();
        QuotaUsage u0 = routerFs.getQuotaUsage(new Path("/type0"));
        QuotaUsage u1 = routerFs.getQuotaUsage(new Path("/type0/type1"));
        Assertions.assertEquals((long)-1L, (long)u1.getQuota());
        Assertions.assertEquals((long)2L, (long)u1.getFileAndDirectoryCount());
        Assertions.assertEquals((long)-1L, (long)u1.getSpaceQuota());
        Assertions.assertEquals((long)(fileSize * 3), (long)u1.getSpaceConsumed());
        this.verifyTypeQuotaAndConsume(new long[]{-1L, -1L, ssQuota, -1L, -1L, -1L}, new long[]{0L, 0L, fileSize * 3, 0L, 0L, 0L}, u1);
        Assertions.assertEquals((long)-1L, (long)u0.getQuota());
        Assertions.assertEquals((long)4L, (long)u0.getFileAndDirectoryCount());
        Assertions.assertEquals((long)-1L, (long)u0.getSpaceQuota());
        Assertions.assertEquals((long)(fileSize * 3 * 2), (long)u0.getSpaceConsumed());
        this.verifyTypeQuotaAndConsume(new long[]{-1L, -1L, ssQuota * 2L, -1L, -1L, -1L}, new long[]{0L, 0L, fileSize * 3 * 2, 0L, 0L, 0L}, u0);
    }

    @Test
    public void testGetQuota() throws Exception {
        long nsQuota = 10L;
        long ssQuota = 100L;
        FileSystem nnFs1 = nnContext1.getFileSystem();
        FileSystem nnFs2 = nnContext2.getFileSystem();
        nnFs1.mkdirs(new Path("/testdir7"));
        nnFs1.mkdirs(new Path("/testdir7/subdir"));
        nnFs2.mkdirs(new Path("/testdir8"));
        nnFs2.mkdirs(new Path("/testdir8-ext"));
        MountTable mountTable1 = MountTable.newInstance((String)"/getquota", Collections.singletonMap("ns0", "/testdir7"));
        mountTable1.setQuota(new RouterQuotaUsage.Builder().quota(nsQuota).spaceQuota(ssQuota).build());
        this.addMountTable(mountTable1);
        MountTable mountTable2 = MountTable.newInstance((String)"/getquota/subdir1", Collections.singletonMap("ns0", "/testdir7/subdir"));
        this.addMountTable(mountTable2);
        MountTable mountTable3 = MountTable.newInstance((String)"/getquota/subdir2", Collections.singletonMap("ns1", "/testdir8"));
        this.addMountTable(mountTable3);
        MountTable mountTable4 = MountTable.newInstance((String)"/getquota/subdir3", Collections.singletonMap("ns1", "/testdir8-ext"));
        this.addMountTable(mountTable4);
        DFSClient routerClient = routerContext.getClient();
        routerClient.create("/getquota/file", true).close();
        routerClient.create("/getquota/subdir1/file", true).close();
        routerClient.create("/getquota/subdir2/file", true).close();
        routerClient.create("/getquota/subdir3/file", true).close();
        ClientProtocol clientProtocol = routerContext.getClient().getNamenode();
        RouterQuotaUpdateService updateService = routerContext.getRouter().getQuotaCacheUpdateService();
        updateService.periodicInvoke();
        QuotaUsage quota = clientProtocol.getQuotaUsage("/getquota");
        Assertions.assertEquals((long)8L, (long)quota.getFileAndDirectoryCount());
    }

    @Test
    public void testStaleQuotaRemoving() throws Exception {
        long nsQuota = 20L;
        long ssQuota = 200L;
        String stalePath = "/stalequota";
        FileSystem nnFs1 = nnContext1.getFileSystem();
        nnFs1.mkdirs(new Path("/testdir9"));
        MountTable mountTable = MountTable.newInstance((String)stalePath, Collections.singletonMap("ns0", "/testdir9"));
        mountTable.setQuota(new RouterQuotaUsage.Builder().quota(nsQuota).spaceQuota(ssQuota).build());
        this.addMountTable(mountTable);
        RouterQuotaUpdateService updateService = routerContext.getRouter().getQuotaCacheUpdateService();
        updateService.periodicInvoke();
        RouterQuotaManager quotaManager = routerContext.getRouter().getQuotaManager();
        RouterQuotaUsage quota = quotaManager.getQuotaUsage(stalePath);
        Assertions.assertEquals((long)nsQuota, (long)quota.getQuota());
        Assertions.assertEquals((long)ssQuota, (long)quota.getSpaceQuota());
        this.removeMountTable(stalePath);
        updateService.periodicInvoke();
        quota = quotaManager.getQuotaUsage(stalePath);
        Assertions.assertNull((Object)quota);
    }

    private boolean removeMountTable(String path) throws IOException {
        RouterClient client = routerContext.getAdminClient();
        MountTableManager mountTableManager = client.getMountTableManager();
        RemoveMountTableEntryRequest removeRequest = RemoveMountTableEntryRequest.newInstance((String)path);
        RemoveMountTableEntryResponse removeResponse = mountTableManager.removeMountTableEntry(removeRequest);
        resolver.loadCache(true);
        return removeResponse.getStatus();
    }

    @Test
    public void testQuotaUpdating() throws Exception {
        long nsQuota = 30L;
        long ssQuota = 1024L;
        String path = "/updatequota";
        FileSystem nnFs1 = nnContext1.getFileSystem();
        nnFs1.mkdirs(new Path("/testdir10"));
        MountTable mountTable = MountTable.newInstance((String)path, Collections.singletonMap("ns0", "/testdir10"));
        mountTable.setQuota(new RouterQuotaUsage.Builder().quota(nsQuota).spaceQuota(ssQuota).build());
        this.addMountTable(mountTable);
        RouterQuotaUpdateService updateService = routerContext.getRouter().getQuotaCacheUpdateService();
        updateService.periodicInvoke();
        MountTable updatedMountTable = this.getMountTable(path);
        RouterQuotaUsage quota = updatedMountTable.getQuota();
        Assertions.assertEquals((long)nsQuota, (long)quota.getQuota());
        Assertions.assertEquals((long)ssQuota, (long)quota.getSpaceQuota());
        Assertions.assertEquals((long)1L, (long)quota.getFileAndDirectoryCount());
        Assertions.assertEquals((long)0L, (long)quota.getSpaceConsumed());
        FileSystem routerFs = routerContext.getFileSystem();
        routerFs.mkdirs(new Path(path + "/" + UUID.randomUUID()));
        DFSClient routerClient = routerContext.getClient();
        routerClient.create(path + "/file", true).close();
        this.appendData(path + "/file", routerClient, 512);
        updateService.periodicInvoke();
        updatedMountTable = this.getMountTable(path);
        quota = updatedMountTable.getQuota();
        Assertions.assertEquals((long)nsQuota, (long)quota.getQuota());
        Assertions.assertEquals((long)ssQuota, (long)quota.getSpaceQuota());
        Assertions.assertEquals((long)3L, (long)quota.getFileAndDirectoryCount());
        Assertions.assertEquals((long)512L, (long)quota.getSpaceConsumed());
        updatedMountTable = this.getMountTable(path);
        nnFs1.mkdirs(new Path("/newPath"));
        updatedMountTable.setDestinations(Collections.singletonList(new RemoteLocation("ns0", "/newPath", path)));
        this.updateMountTable(updatedMountTable);
        Assertions.assertEquals((long)nsQuota, (long)nnFs1.getQuotaUsage(new Path("/newPath")).getQuota());
    }

    private MountTable getMountTable(String path) throws IOException {
        resolver.loadCache(true);
        RouterClient client = routerContext.getAdminClient();
        MountTableManager mountTableManager = client.getMountTableManager();
        GetMountTableEntriesRequest getRequest = GetMountTableEntriesRequest.newInstance((String)path);
        GetMountTableEntriesResponse response = mountTableManager.getMountTableEntries(getRequest);
        List results = response.getEntries();
        return !results.isEmpty() ? (MountTable)results.get(0) : null;
    }

    @Test
    public void testQuotaSynchronization() throws IOException {
        long updateNsQuota = 3L;
        long updateSsQuota = 4L;
        FileSystem nnFs = nnContext1.getFileSystem();
        nnFs.mkdirs(new Path("/testsync"));
        MountTable mountTable = MountTable.newInstance((String)"/quotaSync", Collections.singletonMap("ns0", "/testsync"), (long)Time.now(), (long)Time.now());
        mountTable.setQuota(new RouterQuotaUsage.Builder().quota(1L).spaceQuota(2L).build());
        this.addMountTable(mountTable);
        QuotaUsage realQuota = nnContext1.getFileSystem().getQuotaUsage(new Path("/testsync"));
        Assertions.assertNotEquals((long)updateNsQuota, (long)realQuota.getQuota());
        Assertions.assertNotEquals((long)updateSsQuota, (long)realQuota.getSpaceQuota());
        RouterQuotaUpdateService updateService = routerContext.getRouter().getQuotaCacheUpdateService();
        updateService.periodicInvoke();
        mountTable.setQuota(new RouterQuotaUsage.Builder().quota(updateNsQuota).spaceQuota(updateSsQuota).build());
        this.updateMountTable(mountTable);
        realQuota = nnContext1.getFileSystem().getQuotaUsage(new Path("/testsync"));
        Assertions.assertEquals((long)updateNsQuota, (long)realQuota.getQuota());
        Assertions.assertEquals((long)updateSsQuota, (long)realQuota.getSpaceQuota());
        mountTable.setQuota(new RouterQuotaUsage.Builder().quota(-1L).spaceQuota(-1L).build());
        this.updateMountTable(mountTable);
        realQuota = nnContext1.getFileSystem().getQuotaUsage(new Path("/testsync"));
        Assertions.assertEquals((long)-1L, (long)realQuota.getQuota());
        Assertions.assertEquals((long)-1L, (long)realQuota.getSpaceQuota());
        mountTable = MountTable.newInstance((String)"/testupdate", Collections.singletonMap("ns0", "/testupdate"), (long)Time.now(), (long)Time.now());
        this.addMountTable(mountTable);
        mountTable.setQuota(new RouterQuotaUsage.Builder().quota(1L).spaceQuota(2L).build());
        Assertions.assertTrue((boolean)this.updateMountTable(mountTable));
    }

    @Test
    public void testQuotaRefreshAfterQuotaExceed() throws Exception {
        long nsQuota = 3L;
        long ssQuota = 100L;
        FileSystem nnFs1 = nnContext1.getFileSystem();
        FileSystem nnFs2 = nnContext2.getFileSystem();
        nnFs1.mkdirs(new Path("/testdir11"));
        nnFs2.mkdirs(new Path("/testdir12"));
        MountTable mountTable1 = MountTable.newInstance((String)"/setquota1", Collections.singletonMap("ns0", "/testdir11"));
        mountTable1.setQuota(new RouterQuotaUsage.Builder().quota(nsQuota).spaceQuota(ssQuota).build());
        this.addMountTable(mountTable1);
        MountTable mountTable2 = MountTable.newInstance((String)"/setquota2", Collections.singletonMap("ns1", "/testdir12"));
        mountTable2.setQuota(new RouterQuotaUsage.Builder().quota(nsQuota).spaceQuota(ssQuota).build());
        this.addMountTable(mountTable2);
        FileSystem routerFs = routerContext.getFileSystem();
        routerFs.mkdirs(new Path("/setquota1/" + UUID.randomUUID()));
        routerFs.mkdirs(new Path("/setquota1/" + UUID.randomUUID()));
        routerFs.mkdirs(new Path("/setquota1/" + UUID.randomUUID()));
        RouterQuotaUpdateService updateService = routerContext.getRouter().getQuotaCacheUpdateService();
        updateService.periodicInvoke();
        resolver.loadCache(true);
        RouterQuotaManager quotaManager = routerContext.getRouter().getQuotaManager();
        ClientProtocol client1 = nnContext1.getClient().getNamenode();
        ClientProtocol client2 = nnContext2.getClient().getNamenode();
        QuotaUsage quota1 = client1.getQuotaUsage("/testdir11");
        QuotaUsage quota2 = client2.getQuotaUsage("/testdir12");
        RouterQuotaUsage cacheQuota1 = quotaManager.getQuotaUsage("/setquota1");
        RouterQuotaUsage cacheQuota2 = quotaManager.getQuotaUsage("/setquota2");
        Assertions.assertEquals((long)4L, (long)quota1.getFileAndDirectoryCount());
        Assertions.assertEquals((long)4L, (long)cacheQuota1.getFileAndDirectoryCount());
        Assertions.assertEquals((long)1L, (long)quota2.getFileAndDirectoryCount());
        Assertions.assertEquals((long)1L, (long)cacheQuota2.getFileAndDirectoryCount());
        try {
            routerFs.mkdirs(new Path("/testdir11/" + UUID.randomUUID()));
            Assertions.fail((String)"Mkdir should be failed under dir /testdir11.");
        }
        catch (NSQuotaExceededException nSQuotaExceededException) {
            // empty catch block
        }
        routerFs.mkdirs(new Path("/setquota2/" + UUID.randomUUID()));
        routerFs.mkdirs(new Path("/setquota2/" + UUID.randomUUID()));
        updateService.periodicInvoke();
        quota1 = client1.getQuotaUsage("/testdir11");
        cacheQuota1 = quotaManager.getQuotaUsage("/setquota1");
        quota2 = client2.getQuotaUsage("/testdir12");
        cacheQuota2 = quotaManager.getQuotaUsage("/setquota2");
        Assertions.assertEquals((long)4L, (long)quota1.getFileAndDirectoryCount());
        Assertions.assertEquals((long)4L, (long)cacheQuota1.getFileAndDirectoryCount());
        Assertions.assertEquals((long)3L, (long)quota2.getFileAndDirectoryCount());
        Assertions.assertEquals((long)3L, (long)cacheQuota2.getFileAndDirectoryCount());
    }

    @Test
    public void testQuotaRefreshWhenDestinationNotPresent() throws Exception {
        long nsQuota = 5L;
        long ssQuota = 1536L;
        FileSystem nnFs = nnContext1.getFileSystem();
        nnFs.mkdirs(new Path("/testdir13"));
        nnFs.mkdirs(new Path("/testdir14"));
        MountTable mountTable = MountTable.newInstance((String)"/setdir1", Collections.singletonMap("ns0", "/testdir13"));
        mountTable.setQuota(new RouterQuotaUsage.Builder().quota(nsQuota).spaceQuota(ssQuota).build());
        this.addMountTable(mountTable);
        mountTable = MountTable.newInstance((String)"/setdir2", Collections.singletonMap("ns0", "/testdir14"));
        mountTable.setQuota(new RouterQuotaUsage.Builder().quota(nsQuota).spaceQuota(ssQuota).build());
        this.addMountTable(mountTable);
        DFSClient routerClient = routerContext.getClient();
        routerClient.create("/setdir1/file1", true).close();
        routerClient.create("/setdir2/file2", true).close();
        this.appendData("/setdir1/file1", routerClient, 512);
        this.appendData("/setdir2/file2", routerClient, 512);
        RouterQuotaUpdateService updateService = routerContext.getRouter().getQuotaCacheUpdateService();
        updateService.periodicInvoke();
        resolver.loadCache(true);
        ClientProtocol client1 = nnContext1.getClient().getNamenode();
        RouterQuotaManager quotaManager = routerContext.getRouter().getQuotaManager();
        QuotaUsage quota1 = client1.getQuotaUsage("/testdir13");
        QuotaUsage quota2 = client1.getQuotaUsage("/testdir14");
        RouterQuotaUsage cacheQuota1 = quotaManager.getQuotaUsage("/setdir1");
        RouterQuotaUsage cacheQuota2 = quotaManager.getQuotaUsage("/setdir2");
        MountTable updatedMountTable = this.getMountTable("/setdir1");
        RouterQuotaUsage mountQuota1 = updatedMountTable.getQuota();
        updatedMountTable = this.getMountTable("/setdir2");
        RouterQuotaUsage mountQuota2 = updatedMountTable.getQuota();
        Assertions.assertEquals((long)2L, (long)quota1.getFileAndDirectoryCount());
        Assertions.assertEquals((long)2L, (long)cacheQuota1.getFileAndDirectoryCount());
        Assertions.assertEquals((long)2L, (long)mountQuota1.getFileAndDirectoryCount());
        Assertions.assertEquals((long)2L, (long)quota2.getFileAndDirectoryCount());
        Assertions.assertEquals((long)2L, (long)cacheQuota2.getFileAndDirectoryCount());
        Assertions.assertEquals((long)2L, (long)mountQuota2.getFileAndDirectoryCount());
        Assertions.assertEquals((long)512L, (long)quota1.getSpaceConsumed());
        Assertions.assertEquals((long)512L, (long)cacheQuota1.getSpaceConsumed());
        Assertions.assertEquals((long)512L, (long)mountQuota1.getSpaceConsumed());
        Assertions.assertEquals((long)512L, (long)quota2.getSpaceConsumed());
        Assertions.assertEquals((long)512L, (long)cacheQuota2.getSpaceConsumed());
        Assertions.assertEquals((long)512L, (long)mountQuota2.getSpaceConsumed());
        FileSystem routerFs = routerContext.getFileSystem();
        routerFs.delete(new Path("/setdir1/file1"), true);
        routerClient.create("/setdir2/file3", true).close();
        this.appendData("/setdir2/file3", routerClient, 512);
        int updatedSpace = 1024;
        updateService.periodicInvoke();
        quota2 = client1.getQuotaUsage("/testdir14");
        cacheQuota1 = quotaManager.getQuotaUsage("/setdir1");
        cacheQuota2 = quotaManager.getQuotaUsage("/setdir2");
        updatedMountTable = this.getMountTable("/setdir1");
        mountQuota1 = updatedMountTable.getQuota();
        updatedMountTable = this.getMountTable("/setdir2");
        mountQuota2 = updatedMountTable.getQuota();
        Assertions.assertEquals((long)1L, (long)cacheQuota1.getFileAndDirectoryCount());
        Assertions.assertEquals((long)1L, (long)mountQuota1.getFileAndDirectoryCount());
        Assertions.assertEquals((long)0L, (long)cacheQuota1.getSpaceConsumed());
        Assertions.assertEquals((long)0L, (long)mountQuota1.getSpaceConsumed());
        Assertions.assertEquals((long)3L, (long)quota2.getFileAndDirectoryCount());
        Assertions.assertEquals((long)3L, (long)cacheQuota2.getFileAndDirectoryCount());
        Assertions.assertEquals((long)3L, (long)mountQuota2.getFileAndDirectoryCount());
        Assertions.assertEquals((long)updatedSpace, (long)quota2.getSpaceConsumed());
        Assertions.assertEquals((long)updatedSpace, (long)cacheQuota2.getSpaceConsumed());
        Assertions.assertEquals((long)updatedSpace, (long)mountQuota2.getSpaceConsumed());
    }

    @Test
    public void testClearQuotaDefAfterRemovingMountTable() throws Exception {
        long nsQuota = 5L;
        long ssQuota = 1536L;
        FileSystem nnFs = nnContext1.getFileSystem();
        nnFs.mkdirs(new Path("/testdir15"));
        MountTable mountTable = MountTable.newInstance((String)"/setdir", Collections.singletonMap("ns0", "/testdir15"));
        mountTable.setQuota(new RouterQuotaUsage.Builder().quota(nsQuota).spaceQuota(ssQuota).build());
        this.addMountTable(mountTable);
        RouterQuotaUpdateService updateService = routerContext.getRouter().getQuotaCacheUpdateService();
        updateService.periodicInvoke();
        RouterQuotaManager quotaManager = routerContext.getRouter().getQuotaManager();
        ClientProtocol client = nnContext1.getClient().getNamenode();
        RouterQuotaUsage routerQuota = quotaManager.getQuotaUsage("/setdir");
        QuotaUsage subClusterQuota = client.getQuotaUsage("/testdir15");
        Assertions.assertEquals((long)nsQuota, (long)routerQuota.getQuota());
        Assertions.assertEquals((long)ssQuota, (long)routerQuota.getSpaceQuota());
        Assertions.assertEquals((long)nsQuota, (long)subClusterQuota.getQuota());
        Assertions.assertEquals((long)ssQuota, (long)subClusterQuota.getSpaceQuota());
        this.removeMountTable("/setdir");
        updateService.periodicInvoke();
        routerQuota = quotaManager.getQuotaUsage("/setdir");
        subClusterQuota = client.getQuotaUsage("/testdir15");
        Assertions.assertNull((Object)routerQuota);
        Assertions.assertEquals((long)-1L, (long)subClusterQuota.getQuota());
        Assertions.assertEquals((long)-1L, (long)subClusterQuota.getSpaceQuota());
        mountTable = MountTable.newInstance((String)"/mount", Collections.singletonMap("ns0", "/testdir16"));
        this.addMountTable(mountTable);
        Assertions.assertTrue((boolean)this.removeMountTable("/mount"));
    }

    @Test
    public void testSetQuotaNotMountTable() throws Exception {
        long nsQuota = 5L;
        long ssQuota = 100L;
        FileSystem nnFs1 = nnContext1.getFileSystem();
        MountTable mountTable1 = MountTable.newInstance((String)"/setquotanmt", Collections.singletonMap("ns0", "/testdir16"));
        this.addMountTable(mountTable1);
        nnFs1.mkdirs(new Path("/testdir16/testdir17"));
        routerContext.getRouter().getRpcServer().setQuota("/setquotanmt/testdir17", nsQuota, ssQuota, null);
        RouterQuotaUpdateService updateService = routerContext.getRouter().getQuotaCacheUpdateService();
        updateService.periodicInvoke();
        ClientProtocol client1 = nnContext1.getClient().getNamenode();
        QuotaUsage quota1 = client1.getQuotaUsage("/testdir16/testdir17");
        Assertions.assertEquals((long)nsQuota, (long)quota1.getQuota());
        Assertions.assertEquals((long)ssQuota, (long)quota1.getSpaceQuota());
    }

    @Test
    public void testNoQuotaaExceptionForUnrelatedOperations() throws Exception {
        FileSystem nnFs = nnContext1.getFileSystem();
        DistributedFileSystem routerFs = (DistributedFileSystem)routerContext.getFileSystem();
        Path path = new Path("/quota");
        nnFs.mkdirs(new Path("/dir"));
        MountTable mountTable1 = MountTable.newInstance((String)"/quota", Collections.singletonMap("ns0", "/dir"));
        mountTable1.setQuota(new RouterQuotaUsage.Builder().quota(0L).build());
        this.addMountTable(mountTable1);
        routerFs.mkdirs(new Path("/quota/1"));
        routerContext.getRouter().getQuotaCacheUpdateService().periodicInvoke();
        LambdaTestUtils.intercept(NSQuotaExceededException.class, (String)"The NameSpace quota (directories and files) is exceeded", () -> routerFs.mkdirs(new Path("/quota/2")));
        routerFs.setStoragePolicy(path, "COLD");
        routerFs.setErasureCodingPolicy(path, "RS-6-3-1024k");
        routerFs.unsetErasureCodingPolicy(path);
        routerFs.setPermission(path, new FsPermission(1023));
        routerFs.setOwner(path, "user", "group");
        routerFs.setTimes(path, 1L, 1L);
        routerFs.listStatus(path);
        routerFs.getContentSummary(path);
    }

    @Test
    public void testGetGlobalQuota() throws Exception {
        long nsQuota = 5L;
        long ssQuota = 1536L;
        this.prepareGlobalQuotaTestMountTable(nsQuota, ssQuota);
        Quota qModule = routerContext.getRouter().getRpcServer().getQuotaModule();
        QuotaUsage qu = qModule.getGlobalQuota("/dir-1");
        Assertions.assertEquals((long)nsQuota, (long)qu.getQuota());
        Assertions.assertEquals((long)ssQuota, (long)qu.getSpaceQuota());
        qu = qModule.getGlobalQuota("/dir-1/dir-2");
        Assertions.assertEquals((long)nsQuota, (long)qu.getQuota());
        Assertions.assertEquals((long)(ssQuota * 2L), (long)qu.getSpaceQuota());
        qu = qModule.getGlobalQuota("/dir-1/dir-2/dir-3");
        Assertions.assertEquals((long)nsQuota, (long)qu.getQuota());
        Assertions.assertEquals((long)(ssQuota * 2L), (long)qu.getSpaceQuota());
        qu = qModule.getGlobalQuota("/dir-4");
        Assertions.assertEquals((long)-1L, (long)qu.getQuota());
        Assertions.assertEquals((long)-1L, (long)qu.getSpaceQuota());
    }

    @Test
    public void testFixGlobalQuota() throws Exception {
        long nsQuota = 5L;
        long ssQuota = 1536L;
        FileSystem nnFs = nnContext1.getFileSystem();
        this.prepareGlobalQuotaTestMountTable(nsQuota, ssQuota);
        QuotaUsage qu = nnFs.getQuotaUsage(new Path("/dir-1"));
        Assertions.assertEquals((long)nsQuota, (long)qu.getQuota());
        Assertions.assertEquals((long)ssQuota, (long)qu.getSpaceQuota());
        qu = nnFs.getQuotaUsage(new Path("/dir-2"));
        Assertions.assertEquals((long)nsQuota, (long)qu.getQuota());
        Assertions.assertEquals((long)(ssQuota * 2L), (long)qu.getSpaceQuota());
        qu = nnFs.getQuotaUsage(new Path("/dir-3"));
        Assertions.assertEquals((long)nsQuota, (long)qu.getQuota());
        Assertions.assertEquals((long)(ssQuota * 2L), (long)qu.getSpaceQuota());
        qu = nnFs.getQuotaUsage(new Path("/dir-4"));
        Assertions.assertEquals((long)-1L, (long)qu.getQuota());
        Assertions.assertEquals((long)-1L, (long)qu.getSpaceQuota());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testGetQuotaUsageOnMountPoint() throws Exception {
        long nsQuota = 5L;
        long ssQuota = 1536L;
        this.prepareGlobalQuotaTestMountTable(nsQuota, ssQuota);
        FileSystem routerFs = routerContext.getFileSystem();
        QuotaUsage quotaUsage = routerFs.getQuotaUsage(new Path("/dir-1"));
        Assertions.assertEquals((long)nsQuota, (long)quotaUsage.getQuota());
        Assertions.assertEquals((long)ssQuota, (long)quotaUsage.getSpaceQuota());
        quotaUsage = routerFs.getQuotaUsage(new Path("/dir-1/dir-2"));
        Assertions.assertEquals((long)nsQuota, (long)quotaUsage.getQuota());
        Assertions.assertEquals((long)(ssQuota * 2L), (long)quotaUsage.getSpaceQuota());
        quotaUsage = routerFs.getQuotaUsage(new Path("/dir-1/dir-2/dir-3"));
        Assertions.assertEquals((long)nsQuota, (long)quotaUsage.getQuota());
        Assertions.assertEquals((long)(ssQuota * 2L), (long)quotaUsage.getSpaceQuota());
        quotaUsage = routerFs.getQuotaUsage(new Path("/dir-4"));
        Assertions.assertEquals((long)-1L, (long)quotaUsage.getQuota());
        Assertions.assertEquals((long)-1L, (long)quotaUsage.getSpaceQuota());
        routerFs.mkdirs(new Path("/dir-1/dir-normal"));
        try {
            quotaUsage = routerFs.getQuotaUsage(new Path("/dir-1/dir-normal"));
            Assertions.assertEquals((long)-1L, (long)quotaUsage.getQuota());
            Assertions.assertEquals((long)-1L, (long)quotaUsage.getSpaceQuota());
            routerFs.setQuota(new Path("/dir-1/dir-normal"), 100L, 200L);
            quotaUsage = routerFs.getQuotaUsage(new Path("/dir-1/dir-normal"));
            Assertions.assertEquals((long)100L, (long)quotaUsage.getQuota());
            Assertions.assertEquals((long)200L, (long)quotaUsage.getSpaceQuota());
        }
        finally {
            routerFs.delete(new Path("/dir-1/dir-normal"), true);
        }
    }

    @Test
    public void testRouterQuotaUpdateService() throws Exception {
        Router router = routerContext.getRouter();
        StateStoreDriver driver = router.getStateStore().getDriver();
        RouterQuotaUpdateService updateService = router.getQuotaCacheUpdateService();
        RouterQuotaManager quotaManager = router.getQuotaManager();
        long nsQuota = 5L;
        long ssQuota = 1536L;
        FileSystem nnFs = nnContext1.getFileSystem();
        nnFs.mkdirs(new Path("/dir-1"));
        MountTable mountTable = MountTable.newInstance((String)"/dir-1", Collections.singletonMap("ns0", "/dir-1"));
        mountTable.setQuota(new RouterQuotaUsage.Builder().quota(nsQuota).spaceQuota(ssQuota).build());
        this.addMountTable(mountTable);
        QueryResult result = driver.get(MountTable.class);
        RouterQuotaUsage quotaOnStorage = ((MountTable)result.getRecords().get(0)).getQuota();
        Assertions.assertEquals((long)nsQuota, (long)quotaOnStorage.getQuota());
        Assertions.assertEquals((long)ssQuota, (long)quotaOnStorage.getSpaceQuota());
        Assertions.assertEquals((long)0L, (long)quotaOnStorage.getFileAndDirectoryCount());
        Assertions.assertEquals((long)0L, (long)quotaOnStorage.getSpaceConsumed());
        updateService.periodicInvoke();
        RouterQuotaUsage quotaUsage = quotaManager.getQuotaUsage("/dir-1");
        Assertions.assertEquals((long)nsQuota, (long)quotaUsage.getQuota());
        Assertions.assertEquals((long)ssQuota, (long)quotaUsage.getSpaceQuota());
        Assertions.assertEquals((long)1L, (long)quotaUsage.getFileAndDirectoryCount());
        Assertions.assertEquals((long)0L, (long)quotaUsage.getSpaceConsumed());
        result = driver.get(MountTable.class);
        quotaOnStorage = ((MountTable)result.getRecords().get(0)).getQuota();
        Assertions.assertEquals((long)nsQuota, (long)quotaOnStorage.getQuota());
        Assertions.assertEquals((long)ssQuota, (long)quotaOnStorage.getSpaceQuota());
        Assertions.assertEquals((long)0L, (long)quotaOnStorage.getFileAndDirectoryCount());
        Assertions.assertEquals((long)0L, (long)quotaOnStorage.getSpaceConsumed());
    }

    @Test
    public void testQuotaUpdateWhenDestinationNotPresent() throws Exception {
        long nsQuota = 5L;
        long ssQuota = 1536L;
        String path = "/dst-not-present";
        FileSystem nnFs = nnContext1.getFileSystem();
        nnFs.mkdirs(new Path(path));
        MountTable mountTable = MountTable.newInstance((String)path, Collections.singletonMap("ns0", path));
        mountTable.setQuota(new RouterQuotaUsage.Builder().quota(nsQuota).spaceQuota(ssQuota).build());
        this.addMountTable(mountTable);
        Router router = routerContext.getRouter();
        RouterQuotaManager quotaManager = router.getQuotaManager();
        RouterQuotaUpdateService updateService = router.getQuotaCacheUpdateService();
        updateService.periodicInvoke();
        RouterQuotaUsage quotaUsage = quotaManager.getQuotaUsage(path);
        Assertions.assertEquals((long)nsQuota, (long)quotaUsage.getQuota());
        Assertions.assertEquals((long)ssQuota, (long)quotaUsage.getSpaceQuota());
        Assertions.assertEquals((long)1L, (long)quotaUsage.getFileAndDirectoryCount());
        Assertions.assertEquals((long)0L, (long)quotaUsage.getSpaceConsumed());
        mountTable.setQuota(new RouterQuotaUsage.Builder().quota(nsQuota * 2L).spaceQuota(ssQuota * 2L).build());
        this.updateMountTable(mountTable);
        nnFs.delete(new Path(path), true);
        updateService.periodicInvoke();
        quotaUsage = quotaManager.getQuotaUsage(path);
        Assertions.assertEquals((long)(nsQuota * 2L), (long)quotaUsage.getQuota());
        Assertions.assertEquals((long)(ssQuota * 2L), (long)quotaUsage.getSpaceQuota());
        Assertions.assertEquals((long)0L, (long)quotaUsage.getFileAndDirectoryCount());
        Assertions.assertEquals((long)0L, (long)quotaUsage.getSpaceConsumed());
    }

    @Test
    public void testAndByStorageType() {
        final long[] typeQuota = new long[StorageType.values().length];
        Arrays.fill(typeQuota, Long.MAX_VALUE);
        Predicate<StorageType> predicate = new Predicate<StorageType>(){

            @Override
            public boolean test(StorageType storageType) {
                return typeQuota[storageType.ordinal()] == Long.MAX_VALUE;
            }
        };
        Assertions.assertTrue((boolean)Quota.andByStorageType((Predicate)predicate));
        typeQuota[0] = -1L;
        Assertions.assertFalse((boolean)Quota.andByStorageType((Predicate)predicate));
        Arrays.fill(typeQuota, Long.MAX_VALUE);
        typeQuota[1] = -1L;
        Assertions.assertFalse((boolean)Quota.andByStorageType((Predicate)predicate));
        Arrays.fill(typeQuota, Long.MAX_VALUE);
        typeQuota[typeQuota.length - 1] = -1L;
        Assertions.assertFalse((boolean)Quota.andByStorageType((Predicate)predicate));
    }

    private void prepareGlobalQuotaTestMountTable(long nsQuota, long ssQuota) throws IOException {
        FileSystem nnFs = nnContext1.getFileSystem();
        nnFs.mkdirs(new Path("/dir-1"));
        nnFs.mkdirs(new Path("/dir-2"));
        nnFs.mkdirs(new Path("/dir-3"));
        nnFs.mkdirs(new Path("/dir-4"));
        MountTable mountTable = MountTable.newInstance((String)"/dir-1", Collections.singletonMap("ns0", "/dir-1"));
        mountTable.setQuota(new RouterQuotaUsage.Builder().quota(nsQuota).spaceQuota(ssQuota).build());
        this.addMountTable(mountTable);
        mountTable = MountTable.newInstance((String)"/dir-1/dir-2", Collections.singletonMap("ns0", "/dir-2"));
        mountTable.setQuota(new RouterQuotaUsage.Builder().spaceQuota(ssQuota * 2L).build());
        this.addMountTable(mountTable);
        mountTable = MountTable.newInstance((String)"/dir-1/dir-2/dir-3", Collections.singletonMap("ns0", "/dir-3"));
        this.addMountTable(mountTable);
        mountTable = MountTable.newInstance((String)"/dir-4", Collections.singletonMap("ns0", "/dir-4"));
        this.addMountTable(mountTable);
        RouterQuotaUpdateService updateService = routerContext.getRouter().getQuotaCacheUpdateService();
        updateService.periodicInvoke();
    }

    private void prepareStorageTypeQuotaTestMountTable(StorageType type, long blkSize, long quota0, long quota1, int len0, int len1) throws Exception {
        FileSystem nnFs1 = nnContext1.getFileSystem();
        nnFs1.mkdirs(new Path("/type0"));
        nnFs1.mkdirs(new Path("/type1"));
        ((DistributedFileSystem.HdfsDataOutputStreamBuilder)((DistributedFileSystem)nnContext1.getFileSystem()).createFile(new Path("/type0/file")).storagePolicyName("HOT").blockSize(blkSize)).build().close();
        ((DistributedFileSystem.HdfsDataOutputStreamBuilder)((DistributedFileSystem)nnContext1.getFileSystem()).createFile(new Path("/type1/file")).storagePolicyName("HOT").blockSize(blkSize)).build().close();
        DFSClient client = nnContext1.getClient();
        this.appendData("/type0/file", client, len0);
        this.appendData("/type1/file", client, len1);
        MountTable mountTable = MountTable.newInstance((String)"/type0", Collections.singletonMap("ns0", "/type0"));
        mountTable.setQuota(new RouterQuotaUsage.Builder().typeQuota(type, quota0).build());
        this.addMountTable(mountTable);
        mountTable = MountTable.newInstance((String)"/type0/type1", Collections.singletonMap("ns0", "/type1"));
        mountTable.setQuota(new RouterQuotaUsage.Builder().typeQuota(type, quota1).build());
        this.addMountTable(mountTable);
        RouterQuotaUpdateService updateService = routerContext.getRouter().getQuotaCacheUpdateService();
        updateService.periodicInvoke();
    }

    private void verifyTypeQuotaAndConsume(long[] quota, long[] consume, QuotaUsage usage) {
        for (StorageType t : StorageType.values()) {
            if (quota != null) {
                Assertions.assertEquals((long)quota[t.ordinal()], (long)usage.getTypeQuota(t));
            }
            if (consume == null) continue;
            Assertions.assertEquals((long)consume[t.ordinal()], (long)usage.getTypeConsumed(t));
        }
    }
}

