/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.lib.server;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.ConfigRedactor;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.lib.server.ServerException;
import org.apache.hadoop.lib.server.Service;
import org.apache.hadoop.lib.util.Check;
import org.apache.hadoop.lib.util.ConfigurationUtils;
import org.apache.hadoop.util.StringUtils;
import org.apache.log4j.LogManager;
import org.apache.log4j.PropertyConfigurator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class Server {
    private Logger log;
    public static final String CONF_SERVICES = "services";
    public static final String CONF_SERVICES_EXT = "services.ext";
    public static final String CONF_STARTUP_STATUS = "startup.status";
    public static final String DEFAULT_LOG4J_PROPERTIES = "default-log4j.properties";
    private Status status;
    private String name;
    private String homeDir;
    private String configDir;
    private String logDir;
    private String tempDir;
    private Configuration config;
    private Map<Class, Service> services = new LinkedHashMap<Class, Service>();

    public Server(String name, String homeDir) {
        this(name, homeDir, null);
    }

    public Server(String name, String homeDir, String configDir, String logDir, String tempDir) {
        this(name, homeDir, configDir, logDir, tempDir, null);
    }

    public Server(String name, String homeDir, Configuration config) {
        this(name, homeDir, homeDir + "/conf", homeDir + "/log", homeDir + "/temp", config);
    }

    public Server(String name, String homeDir, String configDir, String logDir, String tempDir, Configuration config) {
        this.name = StringUtils.toLowerCase((String)Check.notEmpty(name, "name").trim());
        this.homeDir = Check.notEmpty(homeDir, "homeDir");
        this.configDir = Check.notEmpty(configDir, "configDir");
        this.logDir = Check.notEmpty(logDir, "logDir");
        this.tempDir = Check.notEmpty(tempDir, "tempDir");
        this.checkAbsolutePath(homeDir, "homeDir");
        this.checkAbsolutePath(configDir, "configDir");
        this.checkAbsolutePath(logDir, "logDir");
        this.checkAbsolutePath(tempDir, "tempDir");
        if (config != null) {
            this.config = new Configuration(false);
            ConfigurationUtils.copy(config, this.config);
        }
        this.status = Status.UNDEF;
    }

    private String checkAbsolutePath(String value, String name) {
        if (!new File(value).isAbsolute()) {
            throw new IllegalArgumentException(MessageFormat.format("[{0}] must be an absolute path [{1}]", name, value));
        }
        return value;
    }

    public Status getStatus() {
        return this.status;
    }

    public void setStatus(Status status) throws ServerException {
        Check.notNull(status, "status");
        if (status.settable) {
            if (status != this.status) {
                Status oldStatus = this.status;
                this.status = status;
                for (Service service : this.services.values()) {
                    try {
                        service.serverStatusChange(oldStatus, status);
                    }
                    catch (Exception ex) {
                        this.log.error("Service [{}] exception during status change to [{}] -server shutting down-,  {}", new Object[]{service.getInterface().getSimpleName(), status, ex.getMessage(), ex});
                        this.destroy();
                        throw new ServerException(ServerException.ERROR.S11, new Object[]{service.getInterface().getSimpleName(), status, ex.getMessage(), ex});
                    }
                }
            }
        } else {
            throw new IllegalArgumentException("Status [" + status + " is not settable");
        }
    }

    protected void ensureOperational() {
        if (!this.getStatus().isOperational()) {
            throw new IllegalStateException("Server is not running");
        }
    }

    static InputStream getResource(String name) {
        Check.notEmpty(name, "name");
        ClassLoader cl = Thread.currentThread().getContextClassLoader();
        if (cl == null) {
            cl = Server.class.getClassLoader();
        }
        return cl.getResourceAsStream(name);
    }

    public void init() throws ServerException {
        if (this.status != Status.UNDEF) {
            throw new IllegalStateException("Server already initialized");
        }
        this.status = Status.BOOTING;
        this.verifyDir(this.homeDir);
        this.verifyDir(this.tempDir);
        Properties serverInfo = new Properties();
        try {
            InputStream is = Server.getResource(this.name + ".properties");
            serverInfo.load(is);
            is.close();
        }
        catch (IOException ex) {
            throw new RuntimeException("Could not load server information file: " + this.name + ".properties");
        }
        this.initLog();
        this.log.info("++++++++++++++++++++++++++++++++++++++++++++++++++++++");
        this.log.info("Server [{}] starting", (Object)this.name);
        this.log.info("  Built information:");
        this.log.info("    Version           : {}", (Object)serverInfo.getProperty(this.name + ".version", "undef"));
        this.log.info("    Source Repository : {}", (Object)serverInfo.getProperty(this.name + ".source.repository", "undef"));
        this.log.info("    Source Revision   : {}", (Object)serverInfo.getProperty(this.name + ".source.revision", "undef"));
        this.log.info("    Built by          : {}", (Object)serverInfo.getProperty(this.name + ".build.username", "undef"));
        this.log.info("    Built timestamp   : {}", (Object)serverInfo.getProperty(this.name + ".build.timestamp", "undef"));
        this.log.info("  Runtime information:");
        this.log.info("    Home   dir: {}", (Object)this.homeDir);
        this.log.info("    Config dir: {}", (Object)(this.config == null ? this.configDir : "-"));
        this.log.info("    Log    dir: {}", (Object)this.logDir);
        this.log.info("    Temp   dir: {}", (Object)this.tempDir);
        this.initConfig();
        this.log.debug("Loading services");
        List<Service> list = this.loadServices();
        try {
            this.log.debug("Initializing services");
            this.initServices(list);
            this.log.info("Services initialized");
        }
        catch (ServerException ex) {
            this.log.error("Services initialization failure, destroying initialized services");
            this.destroyServices();
            throw ex;
        }
        Status status = Status.valueOf(this.getConfig().get(this.getPrefixedName(CONF_STARTUP_STATUS), Status.NORMAL.toString()));
        this.setStatus(status);
        this.log.info("Server [{}] started!, status [{}]", (Object)this.name, (Object)status);
    }

    private void verifyDir(String dir) throws ServerException {
        File file = new File(dir);
        if (!file.exists()) {
            throw new ServerException(ServerException.ERROR.S01, dir);
        }
        if (!file.isDirectory()) {
            throw new ServerException(ServerException.ERROR.S02, dir);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void initLog() throws ServerException {
        this.verifyDir(this.logDir);
        LogManager.resetConfiguration();
        File log4jFile = new File(this.configDir, this.name + "-log4j.properties");
        if (log4jFile.exists()) {
            PropertyConfigurator.configureAndWatch((String)log4jFile.toString(), (long)10000L);
            this.log = LoggerFactory.getLogger(Server.class);
        } else {
            Properties props = new Properties();
            try (InputStream is = Server.getResource(DEFAULT_LOG4J_PROPERTIES);){
                props.load(is);
            }
            catch (IOException ex) {
                throw new ServerException(ServerException.ERROR.S03, DEFAULT_LOG4J_PROPERTIES, ex.getMessage(), ex);
            }
            PropertyConfigurator.configure((Properties)props);
            this.log = LoggerFactory.getLogger(Server.class);
            this.log.warn("Log4j [{}] configuration file not found, using default configuration from classpath", (Object)log4jFile);
        }
    }

    protected void initConfig() throws ServerException {
        Configuration defaultConf;
        this.verifyDir(this.configDir);
        File file = new File(this.configDir);
        String defaultConfig = this.name + "-default.xml";
        ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
        InputStream inputStream = classLoader.getResourceAsStream(defaultConfig);
        if (inputStream == null) {
            this.log.warn("Default configuration file not available in classpath [{}]", (Object)defaultConfig);
            defaultConf = new Configuration(false);
        } else {
            try {
                defaultConf = new Configuration(false);
                ConfigurationUtils.load(defaultConf, inputStream);
            }
            catch (Exception ex) {
                throw new ServerException(ServerException.ERROR.S03, defaultConfig, ex.getMessage(), ex);
            }
        }
        if (this.config == null) {
            Configuration siteConf;
            File siteFile = new File(file, this.name + "-site.xml");
            if (!siteFile.exists()) {
                this.log.warn("Site configuration file [{}] not found in config directory", (Object)siteFile);
                siteConf = new Configuration(false);
            } else {
                if (!siteFile.isFile()) {
                    throw new ServerException(ServerException.ERROR.S05, siteFile.getAbsolutePath());
                }
                try {
                    this.log.debug("Loading site configuration from [{}]", (Object)siteFile);
                    inputStream = Files.newInputStream(siteFile.toPath(), new OpenOption[0]);
                    siteConf = new Configuration(false);
                    ConfigurationUtils.load(siteConf, inputStream);
                }
                catch (IOException ex) {
                    throw new ServerException(ServerException.ERROR.S06, siteFile, ex.getMessage(), ex);
                }
            }
            this.config = new Configuration(false);
            ConfigurationUtils.copy(siteConf, this.config);
        }
        ConfigurationUtils.injectDefaults(defaultConf, this.config);
        ConfigRedactor redactor = new ConfigRedactor(this.config);
        for (String name : System.getProperties().stringPropertyNames()) {
            String value = System.getProperty(name);
            if (!name.startsWith(this.getPrefix() + ".")) continue;
            this.config.set(name, value);
            String redacted = redactor.redact(name, value);
            this.log.info("System property sets  {}: {}", (Object)name, (Object)redacted);
        }
        this.log.debug("Loaded Configuration:");
        this.log.debug("------------------------------------------------------");
        for (Map.Entry entry : this.config) {
            String name = (String)entry.getKey();
            String value = this.config.get((String)entry.getKey());
            String redacted = redactor.redact(name, value);
            this.log.debug("  {}: {}", entry.getKey(), (Object)redacted);
        }
        this.log.debug("------------------------------------------------------");
    }

    private void loadServices(Class[] classes, List<Service> list) throws ServerException {
        for (Class klass : classes) {
            try {
                Service service = (Service)klass.newInstance();
                this.log.debug("Loading service [{}] implementation [{}]", (Object)service.getInterface(), service.getClass());
                if (!service.getInterface().isInstance(service)) {
                    throw new ServerException(ServerException.ERROR.S04, klass, service.getInterface().getName());
                }
                list.add(service);
            }
            catch (ServerException ex) {
                throw ex;
            }
            catch (Exception ex) {
                throw new ServerException(ServerException.ERROR.S07, klass, ex.getMessage(), ex);
            }
        }
    }

    protected List<Service> loadServices() throws ServerException {
        try {
            LinkedHashMap<Class, Service> map = new LinkedHashMap<Class, Service>();
            Class[] classes = this.getConfig().getClasses(this.getPrefixedName(CONF_SERVICES), new Class[0]);
            Class[] classesExt = this.getConfig().getClasses(this.getPrefixedName(CONF_SERVICES_EXT), new Class[0]);
            ArrayList<Service> list = new ArrayList<Service>();
            this.loadServices(classes, list);
            this.loadServices(classesExt, list);
            for (Service service : list) {
                if (map.containsKey(service.getInterface())) {
                    this.log.debug("Replacing service [{}] implementation [{}]", (Object)service.getInterface(), service.getClass());
                }
                map.put(service.getInterface(), service);
            }
            list = new ArrayList();
            for (Map.Entry entry : map.entrySet()) {
                list.add((Service)entry.getValue());
            }
            return list;
        }
        catch (RuntimeException ex) {
            throw new ServerException(ServerException.ERROR.S08, ex.getMessage(), ex);
        }
    }

    protected void initServices(List<Service> services) throws ServerException {
        for (Service service : services) {
            this.log.debug("Initializing service [{}]", (Object)service.getInterface());
            this.checkServiceDependencies(service);
            service.init(this);
            this.services.put(service.getInterface(), service);
        }
        for (Service service : services) {
            service.postInit();
        }
    }

    protected void checkServiceDependencies(Service service) throws ServerException {
        if (service.getServiceDependencies() != null) {
            for (Class dependency : service.getServiceDependencies()) {
                if (this.services.get(dependency) != null) continue;
                throw new ServerException(ServerException.ERROR.S10, service.getClass(), dependency);
            }
        }
    }

    protected void destroyServices() {
        ArrayList<Service> list = new ArrayList<Service>(this.services.values());
        Collections.reverse(list);
        for (Service service : list) {
            try {
                this.log.debug("Destroying service [{}]", (Object)service.getInterface());
                service.destroy();
            }
            catch (Throwable ex) {
                this.log.error("Could not destroy service [{}], {}", new Object[]{service.getInterface(), ex.getMessage(), ex});
            }
        }
        this.log.info("Services destroyed");
    }

    public void destroy() {
        this.ensureOperational();
        this.destroyServices();
        this.log.info("Server [{}] shutdown!", (Object)this.name);
        this.log.info("======================================================");
        if (!Boolean.getBoolean("test.circus")) {
            LogManager.shutdown();
        }
        this.status = Status.SHUTDOWN;
    }

    public String getName() {
        return this.name;
    }

    public String getPrefix() {
        return this.getName();
    }

    public String getPrefixedName(String name) {
        return this.getPrefix() + "." + Check.notEmpty(name, "name");
    }

    public String getHomeDir() {
        return this.homeDir;
    }

    public String getConfigDir() {
        return this.configDir;
    }

    public String getLogDir() {
        return this.logDir;
    }

    public String getTempDir() {
        return this.tempDir;
    }

    public Configuration getConfig() {
        return this.config;
    }

    public <T> T get(Class<T> serviceKlass) {
        this.ensureOperational();
        Check.notNull(serviceKlass, "serviceKlass");
        return (T)this.services.get(serviceKlass);
    }

    public void setService(Class<? extends Service> klass) throws ServerException {
        this.ensureOperational();
        Check.notNull(klass, "serviceKlass");
        if (this.getStatus() == Status.SHUTTING_DOWN) {
            throw new IllegalStateException("Server shutting down");
        }
        try {
            Service newService = klass.newInstance();
            Service oldService = this.services.get(newService.getInterface());
            if (oldService != null) {
                try {
                    oldService.destroy();
                }
                catch (Throwable ex) {
                    this.log.error("Could not destroy service [{}], {}", new Object[]{oldService.getInterface(), ex.getMessage(), ex});
                }
            }
            newService.init(this);
            this.services.put(newService.getInterface(), newService);
        }
        catch (Exception ex) {
            this.log.error("Could not set service [{}] programmatically -server shutting down-, {}", klass, (Object)ex);
            this.destroy();
            throw new ServerException(ServerException.ERROR.S09, klass, ex.getMessage(), ex);
        }
    }

    @InterfaceAudience.Private
    public static enum Status {
        UNDEF(false, false),
        BOOTING(false, true),
        HALTED(true, true),
        ADMIN(true, true),
        NORMAL(true, true),
        SHUTTING_DOWN(false, true),
        SHUTDOWN(false, false);

        private boolean settable;
        private boolean operational;

        private Status(boolean settable, boolean operational) {
            this.settable = settable;
            this.operational = operational;
        }

        public boolean isOperational() {
            return this.operational;
        }
    }
}

