/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs;

import java.io.Closeable;
import java.io.DataOutput;
import java.io.DataOutputStream;
import java.io.IOException;
import java.util.List;
import org.apache.hadoop.fs.CompositeCrcFileChecksum;
import org.apache.hadoop.fs.FileChecksum;
import org.apache.hadoop.fs.MD5MD5CRC32CastagnoliFileChecksum;
import org.apache.hadoop.fs.MD5MD5CRC32GzipFileChecksum;
import org.apache.hadoop.fs.Options;
import org.apache.hadoop.fs.PathIOException;
import org.apache.hadoop.hdfs.DFSClient;
import org.apache.hadoop.hdfs.protocol.BlockChecksumOptions;
import org.apache.hadoop.hdfs.protocol.BlockChecksumType;
import org.apache.hadoop.hdfs.protocol.ClientProtocol;
import org.apache.hadoop.hdfs.protocol.DatanodeInfo;
import org.apache.hadoop.hdfs.protocol.DatanodeInfoWithStorage;
import org.apache.hadoop.hdfs.protocol.ErasureCodingPolicy;
import org.apache.hadoop.hdfs.protocol.ExtendedBlock;
import org.apache.hadoop.hdfs.protocol.LocatedBlock;
import org.apache.hadoop.hdfs.protocol.LocatedBlocks;
import org.apache.hadoop.hdfs.protocol.LocatedStripedBlock;
import org.apache.hadoop.hdfs.protocol.StripedBlockInfo;
import org.apache.hadoop.hdfs.protocol.datatransfer.DataTransferProtoUtil;
import org.apache.hadoop.hdfs.protocol.datatransfer.IOStreamPair;
import org.apache.hadoop.hdfs.protocol.datatransfer.InvalidEncryptionKeyException;
import org.apache.hadoop.hdfs.protocol.datatransfer.Op;
import org.apache.hadoop.hdfs.protocol.datatransfer.Sender;
import org.apache.hadoop.hdfs.protocol.proto.DataTransferProtos;
import org.apache.hadoop.hdfs.protocolPB.PBHelperClient;
import org.apache.hadoop.hdfs.security.token.block.InvalidBlockTokenException;
import org.apache.hadoop.io.DataOutputBuffer;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.io.MD5Hash;
import org.apache.hadoop.util.CrcComposer;
import org.apache.hadoop.util.CrcUtil;
import org.apache.hadoop.util.DataChecksum;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class FileChecksumHelper {
    static final Logger LOG = LoggerFactory.getLogger(FileChecksumHelper.class);

    private FileChecksumHelper() {
    }

    static class StripedFileNonStripedChecksumComputer
    extends FileChecksumComputer {
        private final ErasureCodingPolicy ecPolicy;
        private int bgIdx;

        StripedFileNonStripedChecksumComputer(String src, long length, LocatedBlocks blockLocations, ClientProtocol namenode, DFSClient client, ErasureCodingPolicy ecPolicy, Options.ChecksumCombineMode combineMode) throws IOException {
            super(src, length, blockLocations, namenode, client, combineMode);
            this.ecPolicy = ecPolicy;
        }

        @Override
        void checksumBlocks() throws IOException {
            int tmpTimeout = this.getClient().getConf().getChecksumEcSocketTimeout() * 1 + this.getClient().getConf().getSocketTimeout();
            this.setTimeout(tmpTimeout);
            this.bgIdx = 0;
            while (this.bgIdx < this.getLocatedBlocks().size() && this.getRemaining() >= 0L) {
                LocatedBlock locatedBlock;
                LocatedStripedBlock blockGroup;
                if (this.isRefetchBlocks()) {
                    this.refetchBlocks();
                }
                if (!this.checksumBlockGroup(blockGroup = (LocatedStripedBlock)(locatedBlock = this.getLocatedBlocks().get(this.bgIdx)))) {
                    throw new PathIOException(this.getSrc(), "Fail to get block checksum for " + locatedBlock);
                }
                ++this.bgIdx;
            }
        }

        private boolean checksumBlockGroup(LocatedStripedBlock blockGroup) throws IOException {
            ExtendedBlock block = blockGroup.getBlock();
            long requestedNumBytes = block.getNumBytes();
            if (this.getRemaining() < block.getNumBytes()) {
                requestedNumBytes = this.getRemaining();
            }
            this.setRemaining(this.getRemaining() - requestedNumBytes);
            StripedBlockInfo stripedBlockInfo = new StripedBlockInfo(block, blockGroup.getLocations(), blockGroup.getBlockTokens(), blockGroup.getBlockIndices(), this.ecPolicy);
            DatanodeInfoWithStorage[] datanodes = blockGroup.getLocations();
            boolean done = false;
            for (int j = 0; !done && j < datanodes.length; ++j) {
                try {
                    this.tryDatanode(blockGroup, stripedBlockInfo, datanodes[j], requestedNumBytes);
                    done = true;
                    continue;
                }
                catch (InvalidBlockTokenException ibte) {
                    if (this.bgIdx <= this.getLastRetriedIndex()) continue;
                    LOG.debug("Got access token error in response to OP_BLOCK_CHECKSUM for file {} for block {} from datanode {}. Will retry the block once.", new Object[]{this.getSrc(), block, datanodes[j]});
                    this.setLastRetriedIndex(this.bgIdx);
                    done = true;
                    --this.bgIdx;
                    this.setRefetchBlocks(true);
                    continue;
                }
                catch (IOException ie) {
                    LOG.warn("src={}, datanodes[{}]={}", new Object[]{this.getSrc(), j, datanodes[j], ie});
                }
            }
            return done;
        }

        private void tryDatanode(LocatedStripedBlock blockGroup, StripedBlockInfo stripedBlockInfo, DatanodeInfo datanode, long requestedNumBytes) throws IOException {
            try (IOStreamPair pair = this.getClient().connectToDN(datanode, this.getTimeout(), blockGroup.getBlockToken());){
                LOG.debug("write to {}: {}, blockGroup={}", new Object[]{datanode, Op.BLOCK_GROUP_CHECKSUM, blockGroup});
                this.createSender(pair).blockGroupChecksum(stripedBlockInfo, blockGroup.getBlockToken(), requestedNumBytes, new BlockChecksumOptions(this.getBlockChecksumType()));
                DataTransferProtos.BlockOpResponseProto reply = DataTransferProtos.BlockOpResponseProto.parseFrom(PBHelperClient.vintPrefixed(pair.in));
                String logInfo = "for blockGroup " + blockGroup + " from datanode " + datanode;
                DataTransferProtoUtil.checkBlockOpStatus(reply, logInfo);
                DataTransferProtos.OpBlockChecksumResponseProto checksumData = reply.getChecksumResponse();
                this.extractChecksumProperties(checksumData, blockGroup, datanode, this.bgIdx);
                String blockChecksumForDebug = this.populateBlockChecksumBuf(checksumData);
                LOG.debug("got reply from {}: blockChecksum={}, blockChecksumType={}", new Object[]{datanode, blockChecksumForDebug, this.getBlockChecksumType()});
            }
        }
    }

    static class ReplicatedFileChecksumComputer
    extends FileChecksumComputer {
        private int blockIdx;

        ReplicatedFileChecksumComputer(String src, long length, LocatedBlocks blockLocations, ClientProtocol namenode, DFSClient client, Options.ChecksumCombineMode combineMode) throws IOException {
            super(src, length, blockLocations, namenode, client, combineMode);
        }

        @Override
        void checksumBlocks() throws IOException {
            this.blockIdx = 0;
            while (this.blockIdx < this.getLocatedBlocks().size() && this.getRemaining() >= 0L) {
                LocatedBlock locatedBlock;
                if (this.isRefetchBlocks()) {
                    this.refetchBlocks();
                }
                if (!this.checksumBlock(locatedBlock = this.getLocatedBlocks().get(this.blockIdx))) {
                    throw new PathIOException(this.getSrc(), "Fail to get block MD5 for " + locatedBlock);
                }
                ++this.blockIdx;
            }
        }

        private boolean checksumBlock(LocatedBlock locatedBlock) {
            ExtendedBlock block = locatedBlock.getBlock();
            if (this.getRemaining() < block.getNumBytes()) {
                block.setNumBytes(this.getRemaining());
            }
            this.setRemaining(this.getRemaining() - block.getNumBytes());
            DatanodeInfoWithStorage[] datanodes = locatedBlock.getLocations();
            int tmpTimeout = 3000 * datanodes.length + this.getClient().getConf().getSocketTimeout();
            this.setTimeout(tmpTimeout);
            boolean done = false;
            for (int j = 0; !done && j < datanodes.length; ++j) {
                try {
                    this.tryDatanode(locatedBlock, datanodes[j]);
                    done = true;
                    continue;
                }
                catch (InvalidBlockTokenException ibte) {
                    if (this.blockIdx <= this.getLastRetriedIndex()) continue;
                    LOG.debug("Got access token error in response to OP_BLOCK_CHECKSUM for file {} for block {} from datanode {}. Will retry the block once.", new Object[]{this.getSrc(), block, datanodes[j]});
                    this.setLastRetriedIndex(this.blockIdx);
                    done = true;
                    --this.blockIdx;
                    this.setRefetchBlocks(true);
                    continue;
                }
                catch (InvalidEncryptionKeyException iee) {
                    if (this.blockIdx <= this.getLastRetriedIndex()) continue;
                    LOG.debug("Got invalid encryption key error in response to OP_BLOCK_CHECKSUM for file {} for block {} from datanode {}. Will retry the block once.", new Object[]{this.getSrc(), block, datanodes[j]});
                    this.setLastRetriedIndex(this.blockIdx);
                    done = true;
                    --this.blockIdx;
                    this.getClient().clearDataEncryptionKey();
                    continue;
                }
                catch (IOException ie) {
                    LOG.warn("src={}, datanodes[{}]={}", new Object[]{this.getSrc(), j, datanodes[j], ie});
                }
            }
            return done;
        }

        private void tryDatanode(LocatedBlock locatedBlock, DatanodeInfo datanode) throws IOException {
            ExtendedBlock block = locatedBlock.getBlock();
            try (IOStreamPair pair = this.getClient().connectToDN(datanode, this.getTimeout(), locatedBlock.getBlockToken());){
                LOG.debug("write to {}: {}, block={}", new Object[]{datanode, Op.BLOCK_CHECKSUM, block});
                this.createSender(pair).blockChecksum(block, locatedBlock.getBlockToken(), new BlockChecksumOptions(this.getBlockChecksumType()));
                DataTransferProtos.BlockOpResponseProto reply = DataTransferProtos.BlockOpResponseProto.parseFrom(PBHelperClient.vintPrefixed(pair.in));
                String logInfo = "for block " + block + " from datanode " + datanode;
                DataTransferProtoUtil.checkBlockOpStatus(reply, logInfo);
                DataTransferProtos.OpBlockChecksumResponseProto checksumData = reply.getChecksumResponse();
                this.extractChecksumProperties(checksumData, locatedBlock, datanode, this.blockIdx);
                String blockChecksumForDebug = this.populateBlockChecksumBuf(checksumData);
                LOG.debug("got reply from {}: blockChecksum={}, blockChecksumType={}", new Object[]{datanode, blockChecksumForDebug, this.getBlockChecksumType()});
            }
        }
    }

    static abstract class FileChecksumComputer {
        private final String src;
        private final long length;
        private final DFSClient client;
        private final ClientProtocol namenode;
        private final Options.ChecksumCombineMode combineMode;
        private final BlockChecksumType blockChecksumType;
        private final DataOutputBuffer blockChecksumBuf = new DataOutputBuffer();
        private FileChecksum fileChecksum;
        private LocatedBlocks blockLocations;
        private int timeout;
        private List<LocatedBlock> locatedBlocks;
        private long remaining = 0L;
        private int bytesPerCRC = -1;
        private DataChecksum.Type crcType = DataChecksum.Type.DEFAULT;
        private long crcPerBlock = 0L;
        private boolean isRefetchBlocks = false;
        private int lastRetriedIndex = -1;

        FileChecksumComputer(String src, long length, LocatedBlocks blockLocations, ClientProtocol namenode, DFSClient client, Options.ChecksumCombineMode combineMode) throws IOException {
            this.src = src;
            this.length = length;
            this.blockLocations = blockLocations;
            this.namenode = namenode;
            this.client = client;
            this.combineMode = combineMode;
            switch (combineMode) {
                case MD5MD5CRC: {
                    this.blockChecksumType = BlockChecksumType.MD5CRC;
                    break;
                }
                case COMPOSITE_CRC: {
                    this.blockChecksumType = BlockChecksumType.COMPOSITE_CRC;
                    break;
                }
                default: {
                    throw new IOException("Unknown ChecksumCombineMode: " + combineMode);
                }
            }
            this.remaining = length;
            if (blockLocations != null) {
                if (src.contains("/.snapshot/")) {
                    this.remaining = Math.min(length, blockLocations.getFileLength());
                }
                this.locatedBlocks = blockLocations.getLocatedBlocks();
            }
        }

        String getSrc() {
            return this.src;
        }

        long getLength() {
            return this.length;
        }

        DFSClient getClient() {
            return this.client;
        }

        ClientProtocol getNamenode() {
            return this.namenode;
        }

        Options.ChecksumCombineMode getCombineMode() {
            return this.combineMode;
        }

        BlockChecksumType getBlockChecksumType() {
            return this.blockChecksumType;
        }

        DataOutputBuffer getBlockChecksumBuf() {
            return this.blockChecksumBuf;
        }

        FileChecksum getFileChecksum() {
            return this.fileChecksum;
        }

        LocatedBlocks getBlockLocations() {
            return this.blockLocations;
        }

        void refetchBlocks() throws IOException {
            this.blockLocations = this.getClient().getBlockLocations(this.getSrc(), this.getLength());
            this.locatedBlocks = this.getBlockLocations().getLocatedBlocks();
            this.isRefetchBlocks = false;
        }

        int getTimeout() {
            return this.timeout;
        }

        void setTimeout(int timeout) {
            this.timeout = timeout;
        }

        List<LocatedBlock> getLocatedBlocks() {
            return this.locatedBlocks;
        }

        long getRemaining() {
            return this.remaining;
        }

        void setRemaining(long remaining) {
            this.remaining = remaining;
        }

        int getBytesPerCRC() {
            return this.bytesPerCRC;
        }

        void setBytesPerCRC(int bytesPerCRC) {
            this.bytesPerCRC = bytesPerCRC;
        }

        DataChecksum.Type getCrcType() {
            return this.crcType;
        }

        void setCrcType(DataChecksum.Type crcType) {
            this.crcType = crcType;
        }

        long getCrcPerBlock() {
            return this.crcPerBlock;
        }

        void setCrcPerBlock(long crcPerBlock) {
            this.crcPerBlock = crcPerBlock;
        }

        boolean isRefetchBlocks() {
            return this.isRefetchBlocks;
        }

        void setRefetchBlocks(boolean refetchBlocks) {
            this.isRefetchBlocks = refetchBlocks;
        }

        int getLastRetriedIndex() {
            return this.lastRetriedIndex;
        }

        void setLastRetriedIndex(int lastRetriedIndex) {
            this.lastRetriedIndex = lastRetriedIndex;
        }

        void compute() throws IOException {
            if (this.locatedBlocks == null || this.locatedBlocks.isEmpty()) {
                int lenOfZeroBytes = 32;
                byte[] emptyBlockMd5 = new byte[32];
                MD5Hash fileMD5 = MD5Hash.digest((byte[])emptyBlockMd5);
                this.fileChecksum = new MD5MD5CRC32GzipFileChecksum(0, 0L, fileMD5);
            } else {
                this.checksumBlocks();
                this.fileChecksum = this.makeFinalResult();
            }
        }

        abstract void checksumBlocks() throws IOException;

        FileChecksum makeFinalResult() throws IOException {
            switch (this.combineMode) {
                case MD5MD5CRC: {
                    return this.makeMd5CrcResult();
                }
                case COMPOSITE_CRC: {
                    return this.makeCompositeCrcResult();
                }
            }
            throw new IOException("Unknown ChecksumCombineMode: " + this.combineMode);
        }

        FileChecksum makeMd5CrcResult() {
            MD5Hash fileMD5 = MD5Hash.digest((byte[])this.blockChecksumBuf.getData());
            switch (this.crcType) {
                case CRC32: {
                    return new MD5MD5CRC32GzipFileChecksum(this.bytesPerCRC, this.crcPerBlock, fileMD5);
                }
                case CRC32C: {
                    return new MD5MD5CRC32CastagnoliFileChecksum(this.bytesPerCRC, this.crcPerBlock, fileMD5);
                }
            }
            return null;
        }

        FileChecksum makeCompositeCrcResult() throws IOException {
            int i;
            long blockSizeHint = 0L;
            if (this.locatedBlocks.size() > 0) {
                blockSizeHint = this.locatedBlocks.get(0).getBlockSize();
            }
            CrcComposer crcComposer = CrcComposer.newCrcComposer((DataChecksum.Type)this.getCrcType(), (long)blockSizeHint);
            byte[] blockChecksumBytes = this.blockChecksumBuf.getData();
            long sumBlockLengths = 0L;
            for (i = 0; i < this.locatedBlocks.size() - 1; ++i) {
                LocatedBlock block = this.locatedBlocks.get(i);
                sumBlockLengths += block.getBlockSize();
                int blockCrc = CrcUtil.readInt((byte[])blockChecksumBytes, (int)(i * 4));
                crcComposer.update(blockCrc, block.getBlockSize());
                LOG.debug("Added blockCrc 0x{} for block index {} of size {}", new Object[]{Integer.toString(blockCrc, 16), i, block.getBlockSize()});
            }
            LocatedBlock nextBlock = this.locatedBlocks.get(i);
            long consumedLastBlockLength = Math.min(this.length - sumBlockLengths, nextBlock.getBlockSize());
            int lastBlockCrc = CrcUtil.readInt((byte[])blockChecksumBytes, (int)(4 * (this.locatedBlocks.size() - 1)));
            crcComposer.update(lastBlockCrc, consumedLastBlockLength);
            LOG.debug("Added lastBlockCrc 0x{} for block index {} of size {}", new Object[]{Integer.toString(lastBlockCrc, 16), this.locatedBlocks.size() - 1, consumedLastBlockLength});
            int compositeCrc = CrcUtil.readInt((byte[])crcComposer.digest(), (int)0);
            return new CompositeCrcFileChecksum(compositeCrc, this.getCrcType(), this.bytesPerCRC);
        }

        Sender createSender(IOStreamPair pair) {
            DataOutputStream out = (DataOutputStream)pair.out;
            return new Sender(out);
        }

        void close(IOStreamPair pair) {
            if (pair != null) {
                IOUtils.closeStream((Closeable)pair.in);
                IOUtils.closeStream((Closeable)pair.out);
            }
        }

        void extractChecksumProperties(DataTransferProtos.OpBlockChecksumResponseProto checksumData, LocatedBlock locatedBlock, DatanodeInfo datanode, int blockIdx) throws IOException {
            DataChecksum.Type ct;
            int bpc = checksumData.getBytesPerCrc();
            if (blockIdx == 0) {
                this.setBytesPerCRC(bpc);
            } else if (bpc != this.getBytesPerCRC()) {
                if (this.getBlockChecksumType() == BlockChecksumType.COMPOSITE_CRC) {
                    LOG.warn("Current bytesPerCRC={} doesn't match next bpc={}, but continuing anyway because we're using COMPOSITE_CRC. If trying to preserve CHECKSUMTYPE, only the current bytesPerCRC will be preserved.", (Object)this.getBytesPerCRC(), (Object)bpc);
                } else {
                    throw new IOException("Byte-per-checksum not matched: bpc=" + bpc + " but bytesPerCRC=" + this.getBytesPerCRC());
                }
            }
            long cpb = checksumData.getCrcPerBlock();
            if (this.getLocatedBlocks().size() > 1 && blockIdx == 0) {
                this.setCrcPerBlock(cpb);
            }
            if (checksumData.hasCrcType()) {
                ct = PBHelperClient.convert(checksumData.getCrcType());
            } else {
                LOG.debug("Retrieving checksum from an earlier-version DataNode: inferring checksum by reading first byte");
                ct = this.getClient().inferChecksumTypeByReading(locatedBlock, datanode);
            }
            if (blockIdx == 0) {
                this.setCrcType(ct);
            } else if (this.getCrcType() != DataChecksum.Type.MIXED && this.getCrcType() != ct) {
                if (this.getBlockChecksumType() == BlockChecksumType.COMPOSITE_CRC) {
                    throw new IOException("DataChecksum.Type.MIXED is not supported for COMPOSITE_CRC");
                }
                this.setCrcType(DataChecksum.Type.MIXED);
            }
            if (blockIdx == 0) {
                LOG.debug("set bytesPerCRC={}, crcPerBlock={}", (Object)this.getBytesPerCRC(), (Object)this.getCrcPerBlock());
            }
        }

        String populateBlockChecksumBuf(DataTransferProtos.OpBlockChecksumResponseProto checksumData) throws IOException {
            String blockChecksumForDebug = null;
            switch (this.getBlockChecksumType()) {
                case MD5CRC: {
                    MD5Hash md5 = new MD5Hash(checksumData.getBlockChecksum().toByteArray());
                    md5.write((DataOutput)this.getBlockChecksumBuf());
                    if (!LOG.isDebugEnabled()) break;
                    blockChecksumForDebug = md5.toString();
                    break;
                }
                case COMPOSITE_CRC: {
                    BlockChecksumType returnedType = PBHelperClient.convert(checksumData.getBlockChecksumOptions().getBlockChecksumType());
                    if (returnedType != BlockChecksumType.COMPOSITE_CRC) {
                        throw new IOException(String.format("Unexpected blockChecksumType '%s', expecting COMPOSITE_CRC", new Object[]{returnedType}));
                    }
                    byte[] crcBytes = checksumData.getBlockChecksum().toByteArray();
                    if (LOG.isDebugEnabled()) {
                        blockChecksumForDebug = CrcUtil.toSingleCrcString((byte[])crcBytes);
                    }
                    this.getBlockChecksumBuf().write(crcBytes);
                    break;
                }
                default: {
                    throw new IOException("Unknown BlockChecksumType: " + this.getBlockChecksumType());
                }
            }
            return blockChecksumForDebug;
        }
    }
}

