/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.gs;

import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.net.URI;
import java.nio.channels.Channels;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.WritableByteChannel;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import javax.annotation.Nonnull;
import org.apache.hadoop.fs.FileAlreadyExistsException;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.StreamCapabilities;
import org.apache.hadoop.fs.Syncable;
import org.apache.hadoop.fs.gs.CreateFileOptions;
import org.apache.hadoop.fs.gs.CreateObjectOptions;
import org.apache.hadoop.fs.gs.GoogleCloudStorageFileSystem;
import org.apache.hadoop.fs.gs.GoogleCloudStorageItemInfo;
import org.apache.hadoop.fs.gs.GoogleHadoopFileSystem;
import org.apache.hadoop.fs.gs.StorageResourceId;
import org.apache.hadoop.thirdparty.com.google.common.base.Ascii;
import org.apache.hadoop.thirdparty.com.google.common.base.Preconditions;
import org.apache.hadoop.thirdparty.com.google.common.base.Strings;
import org.apache.hadoop.thirdparty.com.google.common.collect.ImmutableList;
import org.apache.hadoop.thirdparty.com.google.common.util.concurrent.RateLimiter;
import org.apache.hadoop.thirdparty.com.google.common.util.concurrent.ThreadFactoryBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class GoogleHadoopOutputStream
extends OutputStream
implements StreamCapabilities,
Syncable {
    private static final Logger LOG = LoggerFactory.getLogger(GoogleHadoopOutputStream.class);
    private static final String TMP_FILE_PREFIX = "_GHFS_SYNC_TMP_FILE_";
    private static final CreateFileOptions TMP_FILE_CREATE_OPTIONS = CreateFileOptions.builder().setEnsureNoDirectoryConflict(false).build();
    private static final ExecutorService TMP_FILE_CLEANUP_THREADPOOL = Executors.newCachedThreadPool(new ThreadFactoryBuilder().setNameFormat("ghfs-output-stream-sync-cleanup-%d").setDaemon(true).build());
    private final GoogleHadoopFileSystem ghfs;
    private final CreateObjectOptions composeObjectOptions;
    private final URI dstGcsPath;
    private long dstGenerationId;
    private URI tmpGcsPath;
    private int tmpIndex;
    private OutputStream tmpOut;
    private final RateLimiter syncRateLimiter;
    private final List<Future<Void>> tmpDeletionFutures = new ArrayList<Future<Void>>();
    private final FileSystem.Statistics statistics;

    GoogleHadoopOutputStream(GoogleHadoopFileSystem ghfs, URI dstGcsPath, CreateFileOptions createFileOptions, FileSystem.Statistics statistics) throws IOException {
        LOG.trace("GoogleHadoopOutputStream(gcsPath: {}, createFileOptions: {})", (Object)dstGcsPath, (Object)createFileOptions);
        this.ghfs = ghfs;
        this.dstGcsPath = dstGcsPath;
        this.statistics = statistics;
        Duration minSyncInterval = ghfs.getFileSystemConfiguration().getMinSyncInterval();
        this.syncRateLimiter = minSyncInterval.isNegative() || minSyncInterval.isZero() ? null : RateLimiter.create((double)(1000.0 / (double)minSyncInterval.toMillis()));
        this.composeObjectOptions = GoogleCloudStorageFileSystem.objectOptionsFromFileOptions(createFileOptions.toBuilder().setWriteMode(CreateFileOptions.WriteMode.OVERWRITE).build());
        if (createFileOptions.getWriteMode() == CreateFileOptions.WriteMode.APPEND) {
            this.tmpGcsPath = this.getNextTmpPath();
            this.tmpIndex = 1;
        } else {
            this.tmpGcsPath = dstGcsPath;
            this.tmpIndex = 0;
        }
        this.tmpOut = this.createOutputStream(ghfs.getGcsFs(), this.tmpGcsPath, this.tmpIndex == 0 ? createFileOptions : TMP_FILE_CREATE_OPTIONS);
        this.dstGenerationId = -1L;
    }

    private OutputStream createOutputStream(GoogleCloudStorageFileSystem gcsfs, URI gcsPath, CreateFileOptions options) throws IOException {
        WritableByteChannel channel;
        try {
            channel = gcsfs.create(gcsPath, options);
        }
        catch (java.nio.file.FileAlreadyExistsException e) {
            throw (FileAlreadyExistsException)new FileAlreadyExistsException(String.format("'%s' already exists", gcsPath)).initCause((Throwable)e);
        }
        OutputStream outputStream = Channels.newOutputStream(channel);
        int bufferSize = gcsfs.getConfiguration().getOutStreamBufferSize();
        return bufferSize > 0 ? new BufferedOutputStream(outputStream, bufferSize) : outputStream;
    }

    @Override
    public void write(int b) throws IOException {
        this.throwIfNotOpen();
        this.tmpOut.write(b);
        this.statistics.incrementBytesWritten(1L);
        this.statistics.incrementWriteOps(1);
    }

    @Override
    public void write(@Nonnull byte[] b, int offset, int len) throws IOException {
        this.throwIfNotOpen();
        this.tmpOut.write(b, offset, len);
        this.statistics.incrementBytesWritten((long)len);
        this.statistics.incrementWriteOps(1);
    }

    private void commitTempFile() throws IOException {
        this.tmpOut.close();
        long tmpGenerationId = -1L;
        LOG.trace("tmpOut is an instance of {}; expected generationId {}.", this.tmpOut.getClass(), (Object)tmpGenerationId);
        if (this.dstGcsPath.equals(this.tmpGcsPath)) {
            this.dstGenerationId = tmpGenerationId;
        } else {
            StorageResourceId dstId = StorageResourceId.fromUriPath(this.dstGcsPath, false, this.dstGenerationId);
            StorageResourceId tmpId = StorageResourceId.fromUriPath(this.tmpGcsPath, false, tmpGenerationId);
            Preconditions.checkState((boolean)dstId.getBucketName().equals(tmpId.getBucketName()), (String)"Destination bucket in path '%s' doesn't match temp file bucket in path '%s'", (Object)this.dstGcsPath, (Object)this.tmpGcsPath);
            GoogleCloudStorageFileSystem gcs = this.ghfs.getGcsFs();
            GoogleCloudStorageItemInfo composedObject = gcs.composeObjects((ImmutableList<StorageResourceId>)ImmutableList.of((Object)dstId, (Object)tmpId), dstId, this.composeObjectOptions);
            this.dstGenerationId = composedObject.getContentGeneration();
            this.tmpDeletionFutures.add(TMP_FILE_CLEANUP_THREADPOOL.submit(() -> {
                gcs.delete((List<StorageResourceId>)ImmutableList.of((Object)tmpId));
                return null;
            }));
        }
    }

    @Override
    public void close() throws IOException {
        LOG.trace("close(): temp tail file: %s final destination: {}", (Object)this.tmpGcsPath, (Object)this.dstGcsPath);
        if (this.tmpOut == null) {
            LOG.trace("close(): Ignoring; stream already closed.");
            return;
        }
        this.commitTempFile();
        try {
            this.tmpOut.close();
        }
        finally {
            this.tmpOut = null;
        }
        this.tmpGcsPath = null;
        this.tmpIndex = -1;
        LOG.trace("close(): Awaiting {} deletionFutures", (Object)this.tmpDeletionFutures.size());
        for (Future<Void> deletion : this.tmpDeletionFutures) {
            try {
                deletion.get();
            }
            catch (InterruptedException | ExecutionException e) {
                if (e instanceof InterruptedException) {
                    Thread.currentThread().interrupt();
                }
                throw new IOException(String.format("Failed to delete temporary files while closing stream: '%s'", this.dstGcsPath), e);
            }
        }
    }

    private void throwIfNotOpen() throws IOException {
        if (this.tmpOut == null) {
            throw new ClosedChannelException();
        }
    }

    public boolean hasCapability(String capability) {
        Preconditions.checkArgument((!Strings.isNullOrEmpty((String)capability) ? 1 : 0) != 0, (Object)"capability must not be null or empty string");
        switch (Ascii.toLowerCase((String)capability)) {
            case "hflush": 
            case "hsync": {
                return this.syncRateLimiter != null;
            }
            case "iostatistics": {
                return false;
            }
        }
        return false;
    }

    public void hflush() throws IOException {
        LOG.trace("hflush(): {}", (Object)this.dstGcsPath);
        long startMs = System.currentTimeMillis();
        this.throwIfNotOpen();
        if (this.syncRateLimiter == null || this.syncRateLimiter.tryAcquire()) {
            LOG.trace("hflush() uses hsyncInternal() for {}", (Object)this.dstGcsPath);
            this.hsyncInternal(startMs);
            return;
        }
        LOG.trace("hflush(): No-op due to rate limit ({}): readers will *not* yet see flushed data for {}", (Object)this.syncRateLimiter, (Object)this.dstGcsPath);
    }

    public void hsync() throws IOException {
        LOG.trace("hsync(): {}", (Object)this.dstGcsPath);
        long startMs = System.currentTimeMillis();
        this.throwIfNotOpen();
        if (this.syncRateLimiter != null) {
            LOG.trace("hsync(): Rate limited ({}) with blocking permit acquisition for {}", (Object)this.syncRateLimiter, (Object)this.dstGcsPath);
            this.syncRateLimiter.acquire();
        }
        this.hsyncInternal(startMs);
    }

    private void hsyncInternal(long startMs) throws IOException {
        LOG.trace("hsyncInternal(): Committing tail file {} to final destination {}", (Object)this.tmpGcsPath, (Object)this.dstGcsPath);
        this.commitTempFile();
        ++this.tmpIndex;
        this.tmpGcsPath = this.getNextTmpPath();
        LOG.trace("hsync(): Opening next temporary tail file {} at {} index", (Object)this.tmpGcsPath, (Object)this.tmpIndex);
        this.tmpOut = this.createOutputStream(this.ghfs.getGcsFs(), this.tmpGcsPath, TMP_FILE_CREATE_OPTIONS);
        long finishMs = System.currentTimeMillis();
        LOG.trace("Took {}ms to sync() for {}", (Object)(finishMs - startMs), (Object)this.dstGcsPath);
    }

    private URI getNextTmpPath() {
        Path basePath = this.ghfs.getHadoopPath(this.dstGcsPath);
        Path tempPath = new Path(basePath.getParent(), String.format("%s%s.%d.%s", TMP_FILE_PREFIX, basePath.getName(), this.tmpIndex, UUID.randomUUID()));
        return this.ghfs.getGcsPath(tempPath);
    }
}

