/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.gs;

import com.google.cloud.hadoop.repackaged.ossgcs.com.google.auth.Credentials;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URI;
import java.nio.channels.SeekableByteChannel;
import java.nio.channels.WritableByteChannel;
import java.nio.file.DirectoryNotEmptyException;
import java.nio.file.FileAlreadyExistsException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.TreeMap;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import org.apache.hadoop.fs.gs.CreateBucketOptions;
import org.apache.hadoop.fs.gs.CreateFileOptions;
import org.apache.hadoop.fs.gs.CreateObjectOptions;
import org.apache.hadoop.fs.gs.FileInfo;
import org.apache.hadoop.fs.gs.GoogleCloudStorage;
import org.apache.hadoop.fs.gs.GoogleCloudStorageItemInfo;
import org.apache.hadoop.fs.gs.GoogleHadoopFileSystemConfiguration;
import org.apache.hadoop.fs.gs.ListFileOptions;
import org.apache.hadoop.fs.gs.ListObjectOptions;
import org.apache.hadoop.fs.gs.StorageResourceId;
import org.apache.hadoop.fs.gs.StringPaths;
import org.apache.hadoop.fs.gs.UriPaths;
import org.apache.hadoop.thirdparty.com.google.common.annotations.VisibleForTesting;
import org.apache.hadoop.thirdparty.com.google.common.base.Preconditions;
import org.apache.hadoop.thirdparty.com.google.common.base.Strings;
import org.apache.hadoop.thirdparty.com.google.common.collect.ImmutableList;
import org.apache.hadoop.thirdparty.com.google.common.collect.ImmutableMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class GoogleCloudStorageFileSystem {
    private static final Logger LOG = LoggerFactory.getLogger(GoogleCloudStorageFileSystem.class);
    @VisibleForTesting
    static final Comparator<URI> PATH_COMPARATOR = Comparator.comparing(URI::toString, (as, bs) -> as.length() == bs.length() ? as.compareTo((String)bs) : Integer.compare(as.length(), bs.length()));
    static final Comparator<FileInfo> FILE_INFO_PATH_COMPARATOR = Comparator.comparing(FileInfo::getPath, PATH_COMPARATOR);
    private static final ListObjectOptions GET_FILE_INFO_LIST_OPTIONS = ListObjectOptions.DEFAULT.builder().setIncludePrefix(true).setMaxResults(1L).build();
    private static final ListObjectOptions LIST_FILE_INFO_LIST_OPTIONS = ListObjectOptions.DEFAULT.builder().setIncludePrefix(true).build();
    static final URI GCSROOT = URI.create("gs:/");
    private final GoogleHadoopFileSystemConfiguration configuration;
    private GoogleCloudStorage gcs;

    private static GoogleCloudStorage createCloudStorage(GoogleHadoopFileSystemConfiguration configuration, Credentials credentials) throws IOException {
        Preconditions.checkNotNull((Object)configuration, (Object)"configuration must not be null");
        return new GoogleCloudStorage(configuration, credentials);
    }

    GoogleCloudStorageFileSystem(GoogleHadoopFileSystemConfiguration configuration, Credentials credentials) throws IOException {
        this.configuration = configuration;
        this.gcs = GoogleCloudStorageFileSystem.createCloudStorage(configuration, credentials);
    }

    WritableByteChannel create(URI path, CreateFileOptions createOptions) throws IOException {
        LOG.trace("create(path: {}, createOptions: {})", (Object)path, (Object)createOptions);
        Preconditions.checkNotNull((Object)path, (Object)"path could not be null");
        StorageResourceId resourceId = StorageResourceId.fromUriPath(path, true);
        if (resourceId.isDirectory()) {
            throw new IOException(String.format("Cannot create a file whose name looks like a directory: '%s'", resourceId));
        }
        if (this.configuration.isEnsureNoConflictingItems()) {
            StorageResourceId dirId = resourceId.toDirectoryId();
            Boolean conflictingDirExist = false;
            if (createOptions.isEnsureNoDirectoryConflict()) {
                conflictingDirExist = this.getFileInfoInternal(dirId, true).exists();
            }
            this.checkNoFilesConflictingWithDirs(resourceId);
            if (conflictingDirExist.booleanValue()) {
                throw new FileAlreadyExistsException("A directory with that name exists: " + path);
            }
        }
        if (createOptions.getOverwriteGenerationId() != -1L) {
            resourceId = new StorageResourceId(resourceId.getBucketName(), resourceId.getObjectName(), createOptions.getOverwriteGenerationId());
        }
        return this.gcs.create(resourceId, createOptions);
    }

    void close() {
        if (this.gcs == null) {
            return;
        }
        LOG.trace("close()");
        try {
            this.gcs.close();
        }
        finally {
            this.gcs = null;
        }
    }

    FileInfo getFileInfo(URI path) throws IOException {
        Preconditions.checkArgument((path != null ? 1 : 0) != 0, (Object)"path must not be null");
        StorageResourceId resourceId = StorageResourceId.fromUriPath(path, true);
        FileInfo fileInfo = FileInfo.fromItemInfo(this.getFileInfoInternal(resourceId, true));
        LOG.trace("getFileInfo(path: {}): {}", (Object)path, (Object)fileInfo);
        return fileInfo;
    }

    private GoogleCloudStorageItemInfo getFileInfoInternal(StorageResourceId resourceId, boolean inferImplicitDirectories) throws IOException {
        if (resourceId.isRoot() || resourceId.isBucket()) {
            return this.gcs.getItemInfo(resourceId);
        }
        GoogleCloudStorageItemInfo dirOrObject = this.gcs.getFileOrDirectoryInfo(resourceId);
        if (dirOrObject.exists() || !inferImplicitDirectories) {
            return dirOrObject;
        }
        return this.gcs.getImplicitDirectory(resourceId);
    }

    void mkdirs(URI path) throws IOException {
        LOG.trace("mkdirs(path: {})", (Object)path);
        Preconditions.checkNotNull((Object)path, (Object)"path should not be null");
        StorageResourceId resourceId = StorageResourceId.fromUriPath(path, true);
        if (resourceId.isRoot()) {
            return;
        }
        if (resourceId.isBucket()) {
            try {
                this.gcs.createBucket(resourceId.getBucketName(), CreateBucketOptions.DEFAULT);
            }
            catch (FileAlreadyExistsException e) {
                LOG.trace("mkdirs: {} already exists, ignoring creation failure", (Object)resourceId, (Object)e);
            }
            return;
        }
        resourceId = resourceId.toDirectoryId();
        if (this.configuration.isEnsureNoConflictingItems()) {
            this.checkNoFilesConflictingWithDirs(resourceId);
        }
        try {
            this.gcs.createEmptyObject(resourceId);
        }
        catch (FileAlreadyExistsException e) {
            LOG.trace("mkdirs: {} already exists, ignoring creation failure", (Object)resourceId, (Object)e);
        }
    }

    void delete(URI path, boolean recursive) throws IOException {
        List<Object> itemsToDelete;
        Preconditions.checkNotNull((Object)path, (Object)"path should not be null");
        Preconditions.checkArgument((!path.equals(GCSROOT) ? 1 : 0) != 0, (String)"Cannot delete root path (%s)", (Object)path);
        FileInfo fileInfo = this.getFileInfo(path);
        if (!fileInfo.exists()) {
            throw new FileNotFoundException("Item not found: " + path);
        }
        if (fileInfo.isDirectory()) {
            List<FileInfo> list = itemsToDelete = recursive ? this.listRecursive(fileInfo.getPath()) : this.listDirectory(fileInfo.getPath());
            if (!itemsToDelete.isEmpty() && !recursive) {
                throw new DirectoryNotEmptyException("Cannot delete a non-empty directory. : " + path);
            }
        } else {
            itemsToDelete = new ArrayList();
        }
        ArrayList<FileInfo> bucketsToDelete = new ArrayList<FileInfo>();
        (fileInfo.getItemInfo().isBucket() ? bucketsToDelete : itemsToDelete).add(fileInfo);
        this.deleteObjects(itemsToDelete, bucketsToDelete);
        StorageResourceId parentId = StorageResourceId.fromUriPath(UriPaths.getParentPath(path), true);
        GoogleCloudStorageItemInfo parentInfo = this.getFileInfoInternal(parentId, false);
        StorageResourceId resourceId = parentInfo.getResourceId();
        if (parentInfo.exists() || resourceId.isRoot() || resourceId.isBucket() || "/".equals(resourceId.getObjectName())) {
            return;
        }
        this.gcs.createEmptyObject(parentId);
    }

    private List<FileInfo> listRecursive(URI prefix) throws IOException {
        StorageResourceId prefixId = this.getPrefixId(prefix);
        List<GoogleCloudStorageItemInfo> itemInfos = this.gcs.listDirectoryRecursive(prefixId.getBucketName(), prefixId.getObjectName());
        List<FileInfo> fileInfos = FileInfo.fromItemInfos(itemInfos);
        fileInfos.sort(FILE_INFO_PATH_COMPARATOR);
        return fileInfos;
    }

    private StorageResourceId getPrefixId(URI prefix) {
        Preconditions.checkNotNull((Object)prefix, (Object)"prefix could not be null");
        StorageResourceId prefixId = StorageResourceId.fromUriPath(prefix, true);
        Preconditions.checkArgument((!prefixId.isRoot() ? 1 : 0) != 0, (String)"prefix must not be global root, got '%s'", (Object)prefix);
        return prefixId;
    }

    private void deleteObjects(List<FileInfo> itemsToDelete, List<FileInfo> bucketsToDelete) throws IOException {
        LOG.trace("deleteInternalWithFolders; fileSize={} bucketSize={}", (Object)itemsToDelete.size(), (Object)bucketsToDelete.size());
        this.deleteObjects(itemsToDelete);
        this.deleteBucket(bucketsToDelete);
    }

    private void deleteObjects(List<FileInfo> itemsToDelete) throws IOException {
        itemsToDelete.sort(FILE_INFO_PATH_COMPARATOR.reversed());
        if (!itemsToDelete.isEmpty()) {
            ArrayList<StorageResourceId> objectsToDelete = new ArrayList<StorageResourceId>(itemsToDelete.size());
            for (FileInfo fileInfo : itemsToDelete) {
                if (fileInfo.isInferredDirectory()) continue;
                objectsToDelete.add(new StorageResourceId(fileInfo.getItemInfo().getBucketName(), fileInfo.getItemInfo().getObjectName(), fileInfo.getItemInfo().getContentGeneration()));
            }
            this.gcs.deleteObjects(objectsToDelete);
        }
    }

    private void deleteBucket(List<FileInfo> bucketsToDelete) throws IOException {
        if (!bucketsToDelete.isEmpty()) {
            ArrayList<String> bucketNames = new ArrayList<String>(bucketsToDelete.size());
            for (FileInfo bucketInfo : bucketsToDelete) {
                bucketNames.add(bucketInfo.getItemInfo().getResourceId().getBucketName());
            }
            if (this.configuration.isBucketDeleteEnabled()) {
                this.gcs.deleteBuckets(bucketNames);
            } else {
                LOG.info("Skipping deletion of buckets because enableBucketDelete is false: {}", bucketNames);
            }
        }
    }

    FileInfo getFileInfoObject(URI path) throws IOException {
        Preconditions.checkArgument((path != null ? 1 : 0) != 0, (Object)"path must not be null");
        StorageResourceId resourceId = StorageResourceId.fromUriPath(path, true);
        Preconditions.checkArgument((!resourceId.isDirectory() ? 1 : 0) != 0, (Object)String.format("path must be an object and not a directory, path: %s, resourceId: %s", path, resourceId));
        FileInfo fileInfo = FileInfo.fromItemInfo(this.gcs.getItemInfo(resourceId));
        LOG.trace("getFileInfoObject(path: {}): {}", (Object)path, (Object)fileInfo);
        return fileInfo;
    }

    SeekableByteChannel open(FileInfo fileInfo, GoogleHadoopFileSystemConfiguration config) throws IOException {
        Preconditions.checkNotNull((Object)fileInfo, (Object)"fileInfo should not be null");
        Preconditions.checkArgument((!fileInfo.isDirectory() ? 1 : 0) != 0, (String)"Cannot open a directory for reading: %s", (Object)fileInfo.getPath());
        return this.gcs.open(fileInfo.getItemInfo(), config);
    }

    void rename(URI src, URI dst) throws IOException {
        FileInfo dstParentInfo;
        LOG.trace("rename(src: {}, dst: {})", (Object)src, (Object)dst);
        Preconditions.checkNotNull((Object)src);
        Preconditions.checkNotNull((Object)dst);
        Preconditions.checkArgument((!src.equals(GCSROOT) ? 1 : 0) != 0, (Object)"Root path cannot be renamed.");
        URI dstParent = UriPaths.getParentPath(dst);
        ArrayList<URI> paths = new ArrayList<URI>();
        paths.add(src);
        paths.add(dst);
        if (dstParent != null) {
            paths.add(dstParent);
        }
        List<FileInfo> fileInfos = this.getFileInfos(paths);
        FileInfo srcInfo = fileInfos.get(0);
        FileInfo dstInfo = fileInfos.get(1);
        FileInfo fileInfo = dstParentInfo = dstParent == null ? null : fileInfos.get(2);
        if (!srcInfo.exists()) {
            throw new FileNotFoundException("Item not found: " + src);
        }
        src = srcInfo.getPath();
        if (src.equals(dst = this.getDstUri(srcInfo, dstInfo, dstParentInfo))) {
            return;
        }
        if (srcInfo.isDirectory()) {
            this.renameDirectoryInternal(srcInfo, dst);
        } else {
            this.renameObject(src, dst, srcInfo);
        }
    }

    private void renameObject(URI src, URI dst, FileInfo srcInfo) throws IOException {
        StorageResourceId srcResourceId = StorageResourceId.fromUriPath(src, true);
        StorageResourceId dstResourceId = StorageResourceId.fromUriPath(dst, true, 0L);
        if (srcResourceId.getBucketName().equals(dstResourceId.getBucketName())) {
            this.gcs.move((Map<StorageResourceId, StorageResourceId>)ImmutableMap.of((Object)new StorageResourceId(srcInfo.getItemInfo().getBucketName(), srcInfo.getItemInfo().getObjectName(), srcInfo.getItemInfo().getContentGeneration()), (Object)dstResourceId));
        } else {
            this.gcs.copy((Map<StorageResourceId, StorageResourceId>)ImmutableMap.of((Object)srcResourceId, (Object)dstResourceId));
            this.gcs.deleteObjects((List<StorageResourceId>)ImmutableList.of((Object)new StorageResourceId(srcInfo.getItemInfo().getBucketName(), srcInfo.getItemInfo().getObjectName(), srcInfo.getItemInfo().getContentGeneration())));
        }
    }

    private void renameDirectoryInternal(FileInfo srcInfo, URI dst) throws IOException {
        Preconditions.checkArgument((boolean)srcInfo.isDirectory(), (String)"'%s' should be a directory", (Object)srcInfo);
        Preconditions.checkArgument((boolean)dst.toString().endsWith("/"), (String)"'%s' should be a directory", (Object)dst);
        URI src = srcInfo.getPath();
        TreeMap<FileInfo, URI> srcToDstItemNames = new TreeMap<FileInfo, URI>(FILE_INFO_PATH_COMPARATOR);
        TreeMap<FileInfo, URI> srcToDstMarkerItemNames = new TreeMap<FileInfo, URI>(FILE_INFO_PATH_COMPARATOR);
        List<FileInfo> srcItemInfos = this.listFileInfoForPrefix(src, ListFileOptions.DELETE_RENAME_LIST_OPTIONS);
        Pattern markerFilePattern = this.configuration.getMarkerFilePattern();
        String prefix = src.toString();
        for (FileInfo srcItemInfo : srcItemInfos) {
            String relativeItemName = srcItemInfo.getPath().toString().substring(prefix.length());
            URI dstItemName = dst.resolve(relativeItemName);
            if (markerFilePattern != null && markerFilePattern.matcher(relativeItemName).matches()) {
                srcToDstMarkerItemNames.put(srcItemInfo, dstItemName);
                continue;
            }
            srcToDstItemNames.put(srcItemInfo, dstItemName);
        }
        StorageResourceId srcResourceId = StorageResourceId.fromUriPath(src, true);
        StorageResourceId dstResourceId = StorageResourceId.fromUriPath(dst, true, 0L);
        if (srcResourceId.getBucketName().equals(dstResourceId.getBucketName())) {
            this.moveInternal(srcToDstItemNames);
            this.moveInternal(srcToDstMarkerItemNames);
            if (srcInfo.getItemInfo().isBucket()) {
                this.deleteBucket(Collections.singletonList(srcInfo));
            } else {
                this.deleteObjects(Collections.singletonList(srcInfo));
            }
            return;
        }
        throw new UnsupportedOperationException(String.format("Moving object from bucket '%s' to '%s' is not supported", srcResourceId.getBucketName(), dstResourceId.getBucketName()));
    }

    List<FileInfo> listFileInfoForPrefix(URI prefix, ListFileOptions listOptions) throws IOException {
        LOG.trace("listAllFileInfoForPrefix(prefix: {})", (Object)prefix);
        StorageResourceId prefixId = this.getPrefixId(prefix);
        List<GoogleCloudStorageItemInfo> itemInfos = this.gcs.listDirectoryRecursive(prefixId.getBucketName(), prefixId.getObjectName());
        List<FileInfo> fileInfos = FileInfo.fromItemInfos(itemInfos);
        fileInfos.sort(FILE_INFO_PATH_COMPARATOR);
        return fileInfos;
    }

    private void moveInternal(Map<FileInfo, URI> srcToDstItemNames) throws IOException {
        if (srcToDstItemNames.isEmpty()) {
            return;
        }
        HashMap<StorageResourceId, StorageResourceId> sourceToDestinationObjectsMap = new HashMap<StorageResourceId, StorageResourceId>();
        for (Map.Entry<FileInfo, URI> srcToDstItemName : srcToDstItemNames.entrySet()) {
            StorageResourceId srcResourceId = srcToDstItemName.getKey().getItemInfo().getResourceId();
            StorageResourceId dstResourceId = StorageResourceId.fromUriPath(srcToDstItemName.getValue(), true);
            sourceToDestinationObjectsMap.put(srcResourceId, dstResourceId);
        }
        this.gcs.move(sourceToDestinationObjectsMap);
    }

    private List<FileInfo> getFileInfos(List<URI> paths) throws IOException {
        ArrayList<FileInfo> result = new ArrayList<FileInfo>(paths.size());
        for (URI path : paths) {
            result.add(this.getFileInfo(path));
        }
        return result;
    }

    private URI getDstUri(FileInfo srcInfo, FileInfo dstInfo, @Nullable FileInfo dstParentInfo) throws IOException {
        URI src = srcInfo.getPath();
        URI dst = dstInfo.getPath();
        if (!srcInfo.isDirectory() && dst.equals(GCSROOT)) {
            throw new IOException("A file cannot be created in root.");
        }
        if (dstInfo.exists() && !dstInfo.isDirectory() && (srcInfo.isDirectory() || !dst.equals(src))) {
            throw new IOException("Cannot overwrite an existing file: " + dst);
        }
        if (dstParentInfo != null && !dstParentInfo.exists()) {
            throw new IOException("Cannot rename because path does not exist: " + dstParentInfo.getPath());
        }
        String srcItemName = GoogleCloudStorageFileSystem.getItemName(src);
        if (srcInfo.isDirectory()) {
            if (!dstInfo.isDirectory()) {
                dst = UriPaths.toDirectory(dst);
            }
            if (src.equals(dst)) {
                throw new IOException("Rename dir to self is forbidden");
            }
            URI dstRelativeToSrc = src.relativize(dst);
            if (!dstRelativeToSrc.equals(dst)) {
                throw new IOException("Rename to subdir is forbidden");
            }
            if (dstInfo.exists()) {
                dst = dst.equals(GCSROOT) ? UriPaths.fromStringPathComponents(srcItemName, null, true) : UriPaths.toDirectory(dst.resolve(srcItemName));
            }
        } else if (dstInfo.isDirectory()) {
            if (!dstInfo.exists()) {
                throw new IOException("Cannot rename because path does not exist: " + dstInfo.getPath());
            }
            dst = dst.resolve(srcItemName);
        }
        return dst;
    }

    @Nullable
    static String getItemName(URI path) {
        Preconditions.checkNotNull((Object)path, (Object)"path can not be null");
        if (path.equals(GCSROOT)) {
            return null;
        }
        StorageResourceId resourceId = StorageResourceId.fromUriPath(path, true);
        if (resourceId.isBucket()) {
            return resourceId.getBucketName();
        }
        String objectName = resourceId.getObjectName();
        int index = StringPaths.isDirectoryPath(objectName) ? objectName.lastIndexOf("/", objectName.length() - 2) : objectName.lastIndexOf("/");
        return index < 0 ? objectName : objectName.substring(index + 1);
    }

    static CreateObjectOptions objectOptionsFromFileOptions(CreateFileOptions options) {
        Preconditions.checkArgument((options.getWriteMode() == CreateFileOptions.WriteMode.CREATE_NEW || options.getWriteMode() == CreateFileOptions.WriteMode.OVERWRITE ? 1 : 0) != 0, (String)"unsupported write mode: %s", (Object)((Object)options.getWriteMode()));
        return CreateObjectOptions.builder().setContentType(options.getContentType()).setMetadata((Map<String, byte[]>)options.getAttributes()).setOverwriteExisting(options.getWriteMode() == CreateFileOptions.WriteMode.OVERWRITE).build();
    }

    GoogleHadoopFileSystemConfiguration getConfiguration() {
        return this.configuration;
    }

    GoogleCloudStorageItemInfo composeObjects(ImmutableList<StorageResourceId> sources, StorageResourceId dstId, CreateObjectOptions composeObjectOptions) throws IOException {
        return this.gcs.composeObjects((List<StorageResourceId>)sources, dstId, composeObjectOptions);
    }

    void delete(List<StorageResourceId> items) throws IOException {
        this.gcs.deleteObjects(items);
    }

    private void checkNoFilesConflictingWithDirs(StorageResourceId resourceId) throws IOException {
        List fileIds = (List)GoogleCloudStorageFileSystem.getDirs(resourceId.getObjectName()).stream().filter(subdir -> !Strings.isNullOrEmpty((String)subdir)).map(subdir -> new StorageResourceId(resourceId.getBucketName(), StringPaths.toFilePath(subdir))).collect(ImmutableList.toImmutableList());
        for (GoogleCloudStorageItemInfo fileInfo : this.gcs.getItemInfos(fileIds)) {
            if (!fileInfo.exists()) continue;
            throw new FileAlreadyExistsException("Cannot create directories because of existing file: " + fileInfo.getResourceId());
        }
    }

    static List<String> getDirs(String objectName) {
        if (Strings.isNullOrEmpty((String)objectName)) {
            return ImmutableList.of();
        }
        ArrayList<String> dirs = new ArrayList<String>();
        int index = 0;
        while ((index = objectName.indexOf("/", index)) >= 0) {
            dirs.add(objectName.substring(0, index += "/".length()));
        }
        return dirs;
    }

    List<FileInfo> listDirectory(URI path) throws IOException {
        List<GoogleCloudStorageItemInfo> dirItemInfos;
        GoogleCloudStorageItemInfo pathInfo;
        Preconditions.checkNotNull((Object)path, (Object)"path can not be null");
        LOG.trace("listStatus(path: {})", (Object)path);
        StorageResourceId pathId = StorageResourceId.fromUriPath(path, true);
        if (!pathId.isDirectory() && (pathInfo = this.gcs.getItemInfo(pathId)).exists()) {
            ArrayList<FileInfo> listedInfo = new ArrayList<FileInfo>();
            listedInfo.add(FileInfo.fromItemInfo(pathInfo));
            return listedInfo;
        }
        StorageResourceId dirId = pathId.toDirectoryId();
        List<GoogleCloudStorageItemInfo> list = dirItemInfos = dirId.isRoot() ? this.gcs.listBucketInfo() : this.gcs.listDirectory(dirId.getBucketName(), dirId.getObjectName());
        if (pathId.isStorageObject() && dirItemInfos.isEmpty()) {
            throw new FileNotFoundException("Item not found: " + path);
        }
        if (!dirItemInfos.isEmpty() && Objects.equals(dirItemInfos.get(0).getResourceId(), dirId)) {
            dirItemInfos.remove(0);
        }
        List<FileInfo> fileInfos = FileInfo.fromItemInfos(dirItemInfos);
        fileInfos.sort(FILE_INFO_PATH_COMPARATOR);
        return fileInfos;
    }

    void compose(List<URI> sources, URI destination, String contentType) throws IOException {
        StorageResourceId destResource = StorageResourceId.fromStringPath(destination.toString());
        List<String> sourceObjects = sources.stream().map(uri -> StorageResourceId.fromStringPath(uri.toString()).getObjectName()).collect(Collectors.toList());
        this.gcs.compose(destResource.getBucketName(), sourceObjects, destResource.getObjectName(), contentType);
    }
}

