/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.gs;

import com.google.cloud.hadoop.repackaged.ossgcs.com.google.cloud.ReadChannel;
import com.google.cloud.hadoop.repackaged.ossgcs.com.google.cloud.storage.BlobId;
import com.google.cloud.hadoop.repackaged.ossgcs.com.google.cloud.storage.Storage;
import java.io.ByteArrayInputStream;
import java.io.EOFException;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.SeekableByteChannel;
import java.util.ArrayList;
import javax.annotation.Nullable;
import org.apache.hadoop.fs.gs.ErrorTypeExtractor;
import org.apache.hadoop.fs.gs.Fadvise;
import org.apache.hadoop.fs.gs.FileAccessPatternManager;
import org.apache.hadoop.fs.gs.GoogleCloudStorageExceptions;
import org.apache.hadoop.fs.gs.GoogleCloudStorageItemInfo;
import org.apache.hadoop.fs.gs.GoogleHadoopFileSystemConfiguration;
import org.apache.hadoop.fs.gs.StorageResourceId;
import org.apache.hadoop.thirdparty.com.google.common.base.Preconditions;
import org.apache.hadoop.thirdparty.com.google.common.base.Strings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class GoogleCloudStorageClientReadChannel
implements SeekableByteChannel {
    private static final Logger LOG = LoggerFactory.getLogger(GoogleCloudStorageClientReadChannel.class);
    private static final String GZIP_ENCODING = "gzip";
    private final StorageResourceId resourceId;
    private final Storage storage;
    private final GoogleHadoopFileSystemConfiguration config;
    private long objectSize;
    private ContentReadChannel contentReadChannel;
    private boolean gzipEncoded = false;
    private boolean open = true;
    private long currentPosition = 0L;

    GoogleCloudStorageClientReadChannel(Storage storage, GoogleCloudStorageItemInfo itemInfo, GoogleHadoopFileSystemConfiguration config) throws IOException {
        GoogleCloudStorageClientReadChannel.validate(itemInfo);
        this.storage = storage;
        this.resourceId = new StorageResourceId(itemInfo.getBucketName(), itemInfo.getObjectName(), itemInfo.getContentGeneration());
        this.contentReadChannel = new ContentReadChannel(config, this.resourceId);
        this.initMetadata(itemInfo.getContentEncoding(), itemInfo.getSize());
        this.config = config;
    }

    protected void initMetadata(@Nullable String encoding, long sizeFromMetadata) throws IOException {
        this.gzipEncoded = Strings.nullToEmpty((String)encoding).contains(GZIP_ENCODING);
        if (this.gzipEncoded && !this.config.isGzipEncodingSupportEnabled()) {
            throw new IOException("Cannot read GZIP encoded files - content encoding support is disabled.");
        }
        this.objectSize = this.gzipEncoded ? Long.MAX_VALUE : sizeFromMetadata;
    }

    @Override
    public int read(ByteBuffer dst) throws IOException {
        this.throwIfNotOpen();
        if (dst.remaining() == 0) {
            return 0;
        }
        LOG.trace("Reading {} bytes at {} position from '{}'", new Object[]{dst.remaining(), this.currentPosition, this.resourceId});
        if (this.currentPosition == this.objectSize) {
            return -1;
        }
        return this.contentReadChannel.readContent(dst);
    }

    @Override
    public int write(ByteBuffer src) throws IOException {
        throw new UnsupportedOperationException("Cannot mutate read-only channel");
    }

    @Override
    public long position() throws IOException {
        return this.currentPosition;
    }

    @Override
    public SeekableByteChannel position(long newPosition) throws IOException {
        this.throwIfNotOpen();
        if (newPosition == this.currentPosition) {
            return this;
        }
        this.validatePosition(newPosition);
        LOG.trace("Seek from {} to {} position for '{}'", new Object[]{this.currentPosition, newPosition, this.resourceId});
        this.currentPosition = newPosition;
        return this;
    }

    @Override
    public long size() throws IOException {
        return this.objectSize;
    }

    @Override
    public SeekableByteChannel truncate(long size) throws IOException {
        throw new UnsupportedOperationException("Cannot mutate read-only channel");
    }

    @Override
    public boolean isOpen() {
        return this.open;
    }

    @Override
    public void close() throws IOException {
        if (this.open) {
            try {
                LOG.trace("Closing channel for '{}'", (Object)this.resourceId);
                this.contentReadChannel.closeContentChannel();
            }
            catch (Exception e) {
                throw new IOException(String.format("Exception occurred while closing channel '%s'", this.resourceId), e);
            }
            finally {
                this.contentReadChannel = null;
                this.open = false;
            }
        }
    }

    private static void validate(GoogleCloudStorageItemInfo itemInfo) throws IOException {
        Preconditions.checkNotNull((Object)itemInfo, (Object)"itemInfo cannot be null");
        StorageResourceId resourceId = itemInfo.getResourceId();
        Preconditions.checkArgument((boolean)resourceId.isStorageObject(), (String)"Can not open a non-file object for read: %s", (Object)resourceId);
        if (!itemInfo.exists()) {
            throw new FileNotFoundException(String.format("Item not found: %s", resourceId));
        }
    }

    private IOException convertError(Exception error) {
        String msg = String.format("Error reading '%s'", this.resourceId);
        switch (ErrorTypeExtractor.getErrorType(error)) {
            case NOT_FOUND: {
                return GoogleCloudStorageExceptions.createFileNotFoundException(this.resourceId.getBucketName(), this.resourceId.getObjectName(), new IOException(msg, error));
            }
            case OUT_OF_RANGE: {
                return (IOException)new EOFException(msg).initCause(error);
            }
        }
        return new IOException(msg, error);
    }

    private void validatePosition(long position) throws IOException {
        if (position < 0L) {
            throw new EOFException(String.format("Invalid seek offset: position value (%d) must be >= 0 for '%s'", position, this.resourceId));
        }
        if (this.objectSize >= 0L && position >= this.objectSize) {
            throw new EOFException(String.format("Invalid seek offset: position value (%d) must be between 0 and %d for '%s'", position, this.objectSize, this.resourceId));
        }
    }

    private void throwIfNotOpen() throws IOException {
        if (!this.isOpen()) {
            throw new ClosedChannelException();
        }
    }

    private class ContentReadChannel {
        private static final int SKIP_BUFFER_SIZE = 8192;
        private final BlobId blobId;
        private long contentChannelCurrentPosition = -1L;
        private long contentChannelEnd = -1L;
        private byte[] footerContent;
        private byte[] skipBuffer = null;
        private ReadableByteChannel byteChannel = null;
        private final FileAccessPatternManager fileAccessManager;

        ContentReadChannel(GoogleHadoopFileSystemConfiguration config, StorageResourceId resourceId) {
            this.blobId = BlobId.of(resourceId.getBucketName(), resourceId.getObjectName(), resourceId.getGenerationId());
            this.fileAccessManager = new FileAccessPatternManager(resourceId, config);
            if (GoogleCloudStorageClientReadChannel.this.gzipEncoded) {
                this.fileAccessManager.overrideAccessPattern(false);
            }
        }

        int readContent(ByteBuffer dst) throws IOException {
            this.performPendingSeeks();
            Preconditions.checkState((this.contentChannelCurrentPosition == GoogleCloudStorageClientReadChannel.this.currentPosition || this.byteChannel == null ? 1 : 0) != 0, (String)"contentChannelCurrentPosition (%s) should be equal to currentPosition (%s) after lazy seek, if channel is open", (long)this.contentChannelCurrentPosition, (long)GoogleCloudStorageClientReadChannel.this.currentPosition);
            int totalBytesRead = 0;
            while (dst.hasRemaining()) {
                int remainingBeforeRead = dst.remaining();
                try {
                    int bytesRead;
                    if (this.byteChannel == null) {
                        this.byteChannel = this.openByteChannel(dst.remaining());
                        if (GoogleCloudStorageClientReadChannel.this.currentPosition > this.contentChannelCurrentPosition) {
                            this.skipInPlace();
                        }
                        Preconditions.checkState((this.contentChannelCurrentPosition == GoogleCloudStorageClientReadChannel.this.currentPosition ? 1 : 0) != 0, (Object)"position of read offset isn't in alignment with channel's read offset");
                    }
                    if ((bytesRead = this.byteChannel.read(dst)) == 0) {
                        LOG.trace("Read {} from storage-client's byte channel at position: {} with channel ending at: {} for resourceId: {} of size: {}", new Object[]{bytesRead, GoogleCloudStorageClientReadChannel.this.currentPosition, this.contentChannelEnd, GoogleCloudStorageClientReadChannel.this.resourceId, GoogleCloudStorageClientReadChannel.this.objectSize});
                    }
                    if (bytesRead < 0) {
                        if (GoogleCloudStorageClientReadChannel.this.gzipEncoded) {
                            GoogleCloudStorageClientReadChannel.this.objectSize = GoogleCloudStorageClientReadChannel.this.currentPosition;
                            this.contentChannelEnd = GoogleCloudStorageClientReadChannel.this.currentPosition;
                        }
                        if (GoogleCloudStorageClientReadChannel.this.currentPosition != this.contentChannelEnd && GoogleCloudStorageClientReadChannel.this.currentPosition != GoogleCloudStorageClientReadChannel.this.objectSize) {
                            throw new IOException(String.format("Received end of stream result before all requestedBytes were received;EndOf stream signal received at offset: %d where as stream was suppose to end at: %d for resource: %s of size: %d", GoogleCloudStorageClientReadChannel.this.currentPosition, this.contentChannelEnd, GoogleCloudStorageClientReadChannel.this.resourceId, GoogleCloudStorageClientReadChannel.this.objectSize));
                        }
                        if (this.contentChannelEnd == GoogleCloudStorageClientReadChannel.this.objectSize || GoogleCloudStorageClientReadChannel.this.currentPosition != this.contentChannelEnd) break;
                        this.closeContentChannel();
                        continue;
                    }
                    totalBytesRead += bytesRead;
                    GoogleCloudStorageClientReadChannel.this.currentPosition += (long)bytesRead;
                    this.contentChannelCurrentPosition += (long)bytesRead;
                    Preconditions.checkState((this.contentChannelCurrentPosition == GoogleCloudStorageClientReadChannel.this.currentPosition ? 1 : 0) != 0, (String)"contentChannelPosition (%s) should be equal to currentPosition (%s) after successful read", (long)this.contentChannelCurrentPosition, (long)GoogleCloudStorageClientReadChannel.this.currentPosition);
                }
                catch (Exception e) {
                    int partialBytes = this.partiallyReadBytes(remainingBeforeRead, dst);
                    GoogleCloudStorageClientReadChannel.this.currentPosition += (long)partialBytes;
                    this.contentChannelCurrentPosition += (long)partialBytes;
                    LOG.trace("Closing contentChannel after {} exception for '{}'.", (Object)e.getMessage(), (Object)GoogleCloudStorageClientReadChannel.this.resourceId);
                    this.closeContentChannel();
                    throw GoogleCloudStorageClientReadChannel.this.convertError(e);
                }
            }
            return totalBytesRead;
        }

        private int partiallyReadBytes(int remainingBeforeRead, ByteBuffer dst) {
            int partialReadBytes = 0;
            if (remainingBeforeRead != dst.remaining()) {
                partialReadBytes = remainingBeforeRead - dst.remaining();
            }
            return partialReadBytes;
        }

        private ReadableByteChannel openByteChannel(long bytesToRead) throws IOException {
            Preconditions.checkArgument((bytesToRead > 0L ? 1 : 0) != 0, (String)"bytesToRead should be greater than 0, but was %s", (long)bytesToRead);
            Preconditions.checkState((this.byteChannel == null && this.contentChannelEnd < 0L ? 1 : 0) != 0, (String)"contentChannel and contentChannelEnd should be not initialized yet for '%s'", (Object)GoogleCloudStorageClientReadChannel.this.resourceId);
            if (this.footerContent != null && GoogleCloudStorageClientReadChannel.this.currentPosition >= GoogleCloudStorageClientReadChannel.this.objectSize - (long)this.footerContent.length) {
                return this.serveFooterContent();
            }
            this.fileAccessManager.updateAccessPattern(GoogleCloudStorageClientReadChannel.this.currentPosition);
            this.setChannelBoundaries(bytesToRead);
            ReadableByteChannel readableByteChannel = this.getStorageReadChannel(this.contentChannelCurrentPosition, this.contentChannelEnd);
            if (this.contentChannelEnd == GoogleCloudStorageClientReadChannel.this.objectSize && this.contentChannelEnd - this.contentChannelCurrentPosition <= GoogleCloudStorageClientReadChannel.this.config.getMinRangeRequestSize()) {
                if (this.footerContent == null) {
                    this.cacheFooter(readableByteChannel);
                }
                return this.serveFooterContent();
            }
            return readableByteChannel;
        }

        private void setChannelBoundaries(long bytesToRead) {
            this.contentChannelCurrentPosition = this.getRangeRequestStart();
            this.contentChannelEnd = this.getRangeRequestEnd(this.contentChannelCurrentPosition, bytesToRead);
            Preconditions.checkState((this.contentChannelEnd >= this.contentChannelCurrentPosition ? 1 : 0) != 0, (Object)String.format("Start position should be <= endPosition startPosition:%d, endPosition: %d", this.contentChannelCurrentPosition, this.contentChannelEnd));
        }

        private void cacheFooter(ReadableByteChannel readableByteChannel) throws IOException {
            int footerSize = Math.toIntExact(GoogleCloudStorageClientReadChannel.this.objectSize - this.contentChannelCurrentPosition);
            this.footerContent = new byte[footerSize];
            try (InputStream footerStream = Channels.newInputStream(readableByteChannel);){
                int bytesRead;
                int totalBytesRead = 0;
                do {
                    if ((bytesRead = footerStream.read(this.footerContent, totalBytesRead, footerSize - totalBytesRead)) < 0) continue;
                    totalBytesRead += bytesRead;
                } while (bytesRead >= 0 && totalBytesRead < footerSize);
                Preconditions.checkState((bytesRead >= 0 ? 1 : 0) != 0, (String)"footerStream shouldn't be empty before reading the footer of size %s, totalBytesRead %s, read via last call %s, for '%s'", (Object)footerSize, (Object)totalBytesRead, (Object)bytesRead, (Object)GoogleCloudStorageClientReadChannel.this.resourceId);
                Preconditions.checkState((totalBytesRead == footerSize ? 1 : 0) != 0, (String)"totalBytesRead (%s) should equal footerSize (%s) for '%s'", (Object)totalBytesRead, (Object)footerSize, (Object)GoogleCloudStorageClientReadChannel.this.resourceId);
            }
            catch (Exception e) {
                this.footerContent = null;
                throw e;
            }
            LOG.trace("Prefetched {} bytes footer for '{}'", (Object)this.footerContent.length, (Object)GoogleCloudStorageClientReadChannel.this.resourceId);
        }

        private ReadableByteChannel serveFooterContent() {
            this.contentChannelCurrentPosition = GoogleCloudStorageClientReadChannel.this.currentPosition;
            int offset = Math.toIntExact(GoogleCloudStorageClientReadChannel.this.currentPosition - (GoogleCloudStorageClientReadChannel.this.objectSize - (long)this.footerContent.length));
            int length = this.footerContent.length - offset;
            LOG.trace("Opened channel (prefetched footer) from {} position for '{}'", (Object)GoogleCloudStorageClientReadChannel.this.currentPosition, (Object)GoogleCloudStorageClientReadChannel.this.resourceId);
            return Channels.newChannel(new ByteArrayInputStream(this.footerContent, offset, length));
        }

        private long getRangeRequestStart() {
            if (GoogleCloudStorageClientReadChannel.this.gzipEncoded) {
                return 0L;
            }
            if (GoogleCloudStorageClientReadChannel.this.config.getFadvise() != Fadvise.SEQUENTIAL && this.isFooterRead() && !GoogleCloudStorageClientReadChannel.this.config.isReadExactRequestedBytesEnabled()) {
                return Math.max(0L, GoogleCloudStorageClientReadChannel.this.objectSize - GoogleCloudStorageClientReadChannel.this.config.getMinRangeRequestSize());
            }
            return GoogleCloudStorageClientReadChannel.this.currentPosition;
        }

        private long getRangeRequestEnd(long startPosition, long bytesToRead) {
            if (GoogleCloudStorageClientReadChannel.this.gzipEncoded) {
                return GoogleCloudStorageClientReadChannel.this.objectSize;
            }
            long endPosition = GoogleCloudStorageClientReadChannel.this.objectSize;
            if (this.fileAccessManager.shouldAdaptToRandomAccess()) {
                endPosition = startPosition + Math.max(bytesToRead, GoogleCloudStorageClientReadChannel.this.config.getMinRangeRequestSize());
            } else if (GoogleCloudStorageClientReadChannel.this.config.getFadvise() == Fadvise.AUTO_RANDOM) {
                endPosition = Math.min(startPosition + GoogleCloudStorageClientReadChannel.this.config.getBlockSize(), GoogleCloudStorageClientReadChannel.this.objectSize);
            }
            if (this.footerContent != null) {
                endPosition = Math.min(endPosition, GoogleCloudStorageClientReadChannel.this.objectSize - (long)this.footerContent.length);
            }
            return endPosition;
        }

        void closeContentChannel() {
            if (this.byteChannel != null) {
                LOG.trace("Closing internal contentChannel for '{}'", (Object)GoogleCloudStorageClientReadChannel.this.resourceId);
                try {
                    this.byteChannel.close();
                }
                catch (Exception e) {
                    LOG.trace("Got an exception on contentChannel.close() for '{}'; ignoring it.", (Object)GoogleCloudStorageClientReadChannel.this.resourceId, (Object)e);
                }
                finally {
                    this.byteChannel = null;
                    this.fileAccessManager.updateLastServedIndex(this.contentChannelCurrentPosition);
                    this.reset();
                }
            }
        }

        private void reset() {
            Preconditions.checkState((this.byteChannel == null ? 1 : 0) != 0, (String)"contentChannel should be null for '%s'", (Object)GoogleCloudStorageClientReadChannel.this.resourceId);
            this.contentChannelCurrentPosition = -1L;
            this.contentChannelEnd = -1L;
        }

        private boolean isInRangeSeek() {
            long seekDistance = GoogleCloudStorageClientReadChannel.this.currentPosition - this.contentChannelCurrentPosition;
            return this.byteChannel != null && seekDistance > 0L && (GoogleCloudStorageClientReadChannel.this.gzipEncoded || seekDistance <= GoogleCloudStorageClientReadChannel.this.config.getInplaceSeekLimit()) && GoogleCloudStorageClientReadChannel.this.currentPosition < this.contentChannelEnd;
        }

        private void skipInPlace() {
            if (this.skipBuffer == null) {
                this.skipBuffer = new byte[8192];
            }
            long seekDistance = GoogleCloudStorageClientReadChannel.this.currentPosition - this.contentChannelCurrentPosition;
            while (seekDistance > 0L && this.byteChannel != null) {
                try {
                    int bufferSize = Math.toIntExact(Math.min((long)this.skipBuffer.length, seekDistance));
                    int bytesRead = this.byteChannel.read(ByteBuffer.wrap(this.skipBuffer, 0, bufferSize));
                    if (bytesRead < 0) {
                        LOG.info("Somehow read {} bytes trying to skip {} bytes to seek to position {}, size: {}", new Object[]{bytesRead, seekDistance, GoogleCloudStorageClientReadChannel.this.currentPosition, GoogleCloudStorageClientReadChannel.this.objectSize});
                        this.closeContentChannel();
                        continue;
                    }
                    seekDistance -= (long)bytesRead;
                    this.contentChannelCurrentPosition += (long)bytesRead;
                }
                catch (Exception e) {
                    LOG.info("Got an IO exception on contentChannel.read(), a lazy-seek will be pending for '{}'", (Object)GoogleCloudStorageClientReadChannel.this.resourceId, (Object)e);
                    this.closeContentChannel();
                }
            }
            Preconditions.checkState((this.byteChannel == null || this.contentChannelCurrentPosition == GoogleCloudStorageClientReadChannel.this.currentPosition ? 1 : 0) != 0, (String)"contentChannelPosition (%s) should be equal to currentPosition (%s) after successful in-place skip", (long)this.contentChannelCurrentPosition, (long)GoogleCloudStorageClientReadChannel.this.currentPosition);
        }

        private void performPendingSeeks() {
            if (GoogleCloudStorageClientReadChannel.this.currentPosition == this.contentChannelCurrentPosition && this.byteChannel != null) {
                return;
            }
            LOG.trace("Performing lazySeek from {} to {} position '{}'", new Object[]{this.contentChannelCurrentPosition, GoogleCloudStorageClientReadChannel.this.currentPosition, GoogleCloudStorageClientReadChannel.this.resourceId});
            if (this.isInRangeSeek()) {
                this.skipInPlace();
            } else {
                this.closeContentChannel();
            }
        }

        private ReadableByteChannel getStorageReadChannel(long seek, long limit) throws IOException {
            ReadChannel readChannel = GoogleCloudStorageClientReadChannel.this.storage.reader(this.blobId, this.generateReadOptions());
            try {
                readChannel.seek(seek);
                readChannel.limit(limit);
                readChannel.setChunkSize(0);
                return readChannel;
            }
            catch (Exception e) {
                throw new IOException(String.format("Unable to update the boundaries/Range of contentChannel %s", GoogleCloudStorageClientReadChannel.this.resourceId.toString()), e);
            }
        }

        private Storage.BlobSourceOption[] generateReadOptions() {
            ArrayList<Storage.BlobSourceOption> blobReadOptions = new ArrayList<Storage.BlobSourceOption>();
            blobReadOptions.add(Storage.BlobSourceOption.shouldReturnRawInputStream(false));
            if (this.blobId.getGeneration() != null) {
                blobReadOptions.add(Storage.BlobSourceOption.generationMatch(this.blobId.getGeneration()));
            }
            return blobReadOptions.toArray(new Storage.BlobSourceOption[blobReadOptions.size()]);
        }

        private boolean isFooterRead() {
            return GoogleCloudStorageClientReadChannel.this.objectSize - GoogleCloudStorageClientReadChannel.this.currentPosition <= GoogleCloudStorageClientReadChannel.this.config.getMinRangeRequestSize();
        }
    }
}

