/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.gs;

import com.google.cloud.hadoop.repackaged.ossgcs.com.google.api.client.util.BackOff;
import com.google.cloud.hadoop.repackaged.ossgcs.com.google.api.client.util.ExponentialBackOff;
import com.google.cloud.hadoop.repackaged.ossgcs.com.google.api.client.util.Sleeper;
import com.google.cloud.hadoop.repackaged.ossgcs.com.google.api.gax.paging.Page;
import com.google.cloud.hadoop.repackaged.ossgcs.com.google.auth.Credentials;
import com.google.cloud.hadoop.repackaged.ossgcs.com.google.cloud.storage.Blob;
import com.google.cloud.hadoop.repackaged.ossgcs.com.google.cloud.storage.BlobId;
import com.google.cloud.hadoop.repackaged.ossgcs.com.google.cloud.storage.BlobInfo;
import com.google.cloud.hadoop.repackaged.ossgcs.com.google.cloud.storage.Bucket;
import com.google.cloud.hadoop.repackaged.ossgcs.com.google.cloud.storage.BucketInfo;
import com.google.cloud.hadoop.repackaged.ossgcs.com.google.cloud.storage.CopyWriter;
import com.google.cloud.hadoop.repackaged.ossgcs.com.google.cloud.storage.Storage;
import com.google.cloud.hadoop.repackaged.ossgcs.com.google.cloud.storage.StorageClass;
import com.google.cloud.hadoop.repackaged.ossgcs.com.google.cloud.storage.StorageException;
import com.google.cloud.hadoop.repackaged.ossgcs.com.google.cloud.storage.StorageOptions;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.channels.SeekableByteChannel;
import java.nio.channels.WritableByteChannel;
import java.nio.file.FileAlreadyExistsException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import org.apache.hadoop.fs.gs.CreateBucketOptions;
import org.apache.hadoop.fs.gs.CreateFileOptions;
import org.apache.hadoop.fs.gs.CreateObjectOptions;
import org.apache.hadoop.fs.gs.ErrorTypeExtractor;
import org.apache.hadoop.fs.gs.GcsListOperation;
import org.apache.hadoop.fs.gs.GoogleCloudStorageClientReadChannel;
import org.apache.hadoop.fs.gs.GoogleCloudStorageClientWriteChannel;
import org.apache.hadoop.fs.gs.GoogleCloudStorageExceptions;
import org.apache.hadoop.fs.gs.GoogleCloudStorageItemInfo;
import org.apache.hadoop.fs.gs.GoogleHadoopFileSystemConfiguration;
import org.apache.hadoop.fs.gs.StorageResourceId;
import org.apache.hadoop.fs.gs.StringPaths;
import org.apache.hadoop.fs.gs.VerificationAttributes;
import org.apache.hadoop.thirdparty.com.google.common.base.Preconditions;
import org.apache.hadoop.thirdparty.com.google.common.base.Strings;
import org.apache.hadoop.thirdparty.com.google.common.collect.ImmutableList;
import org.apache.hadoop.thirdparty.com.google.common.collect.Maps;
import org.apache.hadoop.thirdparty.com.google.common.io.BaseEncoding;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class GoogleCloudStorage {
    static final Logger LOG = LoggerFactory.getLogger(GoogleCloudStorage.class);
    static final List<Storage.BlobField> BLOB_FIELDS = ImmutableList.of((Object)Storage.BlobField.BUCKET, (Object)Storage.BlobField.CONTENT_ENCODING, (Object)Storage.BlobField.CONTENT_TYPE, (Object)Storage.BlobField.CRC32C, (Object)Storage.BlobField.GENERATION, (Object)Storage.BlobField.METADATA, (Object)Storage.BlobField.MD5HASH, (Object)Storage.BlobField.METAGENERATION, (Object)Storage.BlobField.NAME, (Object)Storage.BlobField.SIZE, (Object)Storage.BlobField.TIME_CREATED, (Object)Storage.BlobField.UPDATED, (Object[])new Storage.BlobField[0]);
    static final CreateObjectOptions EMPTY_OBJECT_CREATE_OPTIONS = CreateObjectOptions.DEFAULT_OVERWRITE.toBuilder().setEnsureEmptyObjectsMetadataMatch(false).build();
    private final Storage storage;
    private final GoogleHadoopFileSystemConfiguration configuration;

    GoogleCloudStorage(GoogleHadoopFileSystemConfiguration configuration, Credentials credentials) throws IOException {
        this.storage = GoogleCloudStorage.createStorage(configuration.getProjectId(), credentials);
        this.configuration = configuration;
    }

    private static Storage createStorage(String projectId, Credentials credentials) {
        StorageOptions.Builder builder = StorageOptions.newBuilder();
        if (projectId != null) {
            builder.setProjectId(projectId);
        }
        return (Storage)((StorageOptions.Builder)builder.setCredentials(credentials)).build().getService();
    }

    WritableByteChannel create(StorageResourceId resourceId, CreateFileOptions options) throws IOException {
        LOG.trace("create({})", (Object)resourceId);
        Preconditions.checkArgument((boolean)resourceId.isStorageObject(), (String)"Expected full StorageObject id, got %s", (Object)resourceId);
        StorageResourceId resourceIdWithGeneration = resourceId;
        if (!resourceId.hasGenerationId()) {
            resourceIdWithGeneration = new StorageResourceId(resourceId.getBucketName(), resourceId.getObjectName(), this.getWriteGeneration(resourceId, options.isOverwriteExisting()));
        }
        return new GoogleCloudStorageClientWriteChannel(this.storage, resourceIdWithGeneration, options);
    }

    private long getWriteGeneration(StorageResourceId resourceId, boolean overwrite) throws IOException {
        LOG.trace("getWriteGeneration({}, {})", (Object)resourceId, (Object)overwrite);
        GoogleCloudStorageItemInfo info = this.getItemInfo(resourceId);
        if (!info.exists()) {
            return 0L;
        }
        if (info.exists() && overwrite) {
            long generation = info.getContentGeneration();
            Preconditions.checkState((generation != 0L ? 1 : 0) != 0, (Object)"Generation should not be 0 for an existing item");
            return generation;
        }
        throw new FileAlreadyExistsException(String.format("Object %s already exists.", resourceId));
    }

    void close() {
        try {
            this.storage.close();
        }
        catch (Exception e) {
            LOG.warn("Error occurred while closing the storage client", (Throwable)e);
        }
    }

    GoogleCloudStorageItemInfo getItemInfo(StorageResourceId resourceId) throws IOException {
        LOG.trace("getItemInfo({})", (Object)resourceId);
        if (resourceId.isRoot()) {
            return GoogleCloudStorageItemInfo.ROOT_INFO;
        }
        GoogleCloudStorageItemInfo itemInfo = null;
        if (resourceId.isBucket()) {
            Bucket bucket = this.getBucket(resourceId.getBucketName());
            if (bucket != null) {
                itemInfo = GoogleCloudStorage.createItemInfoForBucket(resourceId, bucket);
            } else {
                LOG.debug("getBucket({}): not found", (Object)resourceId.getBucketName());
            }
        } else {
            Blob blob = this.getBlob(resourceId);
            if (blob != null) {
                itemInfo = GoogleCloudStorage.createItemInfoForBlob(resourceId, blob);
            } else {
                LOG.debug("getObject({}): not found", (Object)resourceId);
            }
        }
        if (itemInfo == null) {
            itemInfo = GoogleCloudStorageItemInfo.createNotFound(resourceId);
        }
        LOG.debug("getItemInfo: {}", (Object)itemInfo);
        return itemInfo;
    }

    @Nullable
    private Bucket getBucket(String bucketName) throws IOException {
        LOG.debug("getBucket({})", (Object)bucketName);
        Preconditions.checkArgument((!Strings.isNullOrEmpty((String)bucketName) ? 1 : 0) != 0, (Object)"bucketName must not be null or empty");
        try {
            return this.storage.get(bucketName, new Storage.BucketGetOption[0]);
        }
        catch (StorageException e) {
            if (ErrorTypeExtractor.getErrorType(e) == ErrorTypeExtractor.ErrorType.NOT_FOUND) {
                return null;
            }
            throw new IOException("Error accessing Bucket " + bucketName, e);
        }
    }

    private static GoogleCloudStorageItemInfo createItemInfoForBlob(StorageResourceId resourceId, Blob blob) {
        Preconditions.checkArgument((resourceId != null ? 1 : 0) != 0, (Object)"resourceId must not be null");
        Preconditions.checkArgument((blob != null ? 1 : 0) != 0, (Object)"object must not be null");
        Preconditions.checkArgument((boolean)resourceId.isStorageObject(), (String)"resourceId must be a StorageObject. resourceId: %s", (Object)resourceId);
        Preconditions.checkArgument((boolean)resourceId.getBucketName().equals(blob.getBucket()), (String)"resourceId.getBucketName() must equal object.getBucket(): '%s' vs '%s'", (Object)resourceId.getBucketName(), (Object)blob.getBucket());
        Preconditions.checkArgument((boolean)resourceId.getObjectName().equals(blob.getName()), (String)"resourceId.getObjectName() must equal object.getName(): '%s' vs '%s'", (Object)resourceId.getObjectName(), (Object)blob.getName());
        Map<String, byte[]> decodedMetadata = blob.getMetadata() == null ? null : GoogleCloudStorage.decodeMetadata(blob.getMetadata());
        byte[] md5Hash = null;
        byte[] crc32c = null;
        if (!Strings.isNullOrEmpty((String)blob.getCrc32c())) {
            crc32c = BaseEncoding.base64().decode((CharSequence)blob.getCrc32c());
        }
        if (!Strings.isNullOrEmpty((String)blob.getMd5())) {
            md5Hash = BaseEncoding.base64().decode((CharSequence)blob.getMd5());
        }
        return GoogleCloudStorageItemInfo.createObject(resourceId, blob.getCreateTimeOffsetDateTime() == null ? 0L : blob.getCreateTimeOffsetDateTime().toInstant().toEpochMilli(), blob.getUpdateTimeOffsetDateTime() == null ? 0L : blob.getUpdateTimeOffsetDateTime().toInstant().toEpochMilli(), blob.getSize() == null ? 0L : blob.getSize(), blob.getContentType(), blob.getContentEncoding(), decodedMetadata, blob.getGeneration() == null ? 0L : blob.getGeneration(), blob.getMetageneration() == null ? 0L : blob.getMetageneration(), new VerificationAttributes(md5Hash, crc32c));
    }

    static Map<String, byte[]> decodeMetadata(Map<String, String> metadata) {
        return Maps.transformValues(metadata, GoogleCloudStorage::decodeMetadataValues);
    }

    @Nullable
    private static byte[] decodeMetadataValues(String value) {
        try {
            return BaseEncoding.base64().decode((CharSequence)value);
        }
        catch (IllegalArgumentException iae) {
            LOG.error("Failed to parse base64 encoded attribute value {}", (Object)value, (Object)iae);
            return null;
        }
    }

    @Nullable
    Blob getBlob(StorageResourceId resourceId) throws IOException {
        Blob blob;
        Preconditions.checkArgument((boolean)resourceId.isStorageObject(), (String)"Expected full StorageObject id, got %s", (Object)resourceId);
        String bucketName = resourceId.getBucketName();
        String objectName = resourceId.getObjectName();
        try {
            blob = this.storage.get(BlobId.of(bucketName, objectName), Storage.BlobGetOption.fields(BLOB_FIELDS.toArray(new Storage.BlobField[0])));
        }
        catch (StorageException e) {
            throw new IOException("Error accessing " + resourceId, e);
        }
        return blob;
    }

    private static GoogleCloudStorageItemInfo createItemInfoForBucket(StorageResourceId resourceId, Bucket bucket) {
        Preconditions.checkArgument((resourceId != null ? 1 : 0) != 0, (Object)"resourceId must not be null");
        Preconditions.checkArgument((bucket != null ? 1 : 0) != 0, (Object)"bucket must not be null");
        Preconditions.checkArgument((boolean)resourceId.isBucket(), (String)"resourceId must be a Bucket. resourceId: %s", (Object)resourceId);
        Preconditions.checkArgument((boolean)resourceId.getBucketName().equals(bucket.getName()), (String)"resourceId.getBucketName() must equal bucket.getName(): '%s' vs '%s'", (Object)resourceId.getBucketName(), (Object)bucket.getName());
        return GoogleCloudStorageItemInfo.createBucket(resourceId, bucket.asBucketInfo().getCreateTimeOffsetDateTime().toInstant().toEpochMilli(), bucket.asBucketInfo().getUpdateTimeOffsetDateTime().toInstant().toEpochMilli(), bucket.getLocation(), bucket.getStorageClass() == null ? null : bucket.getStorageClass().name());
    }

    private GoogleCloudStorageItemInfo createItemInfoForBlob(Blob blob) {
        long generationId = blob.getGeneration() == null ? 0L : blob.getGeneration();
        StorageResourceId resourceId = new StorageResourceId(blob.getBucket(), blob.getName(), generationId);
        return GoogleCloudStorage.createItemInfoForBlob(resourceId, blob);
    }

    void createBucket(String bucketName, CreateBucketOptions options) throws IOException {
        LOG.trace("createBucket({})", (Object)bucketName);
        Preconditions.checkArgument((!Strings.isNullOrEmpty((String)bucketName) ? 1 : 0) != 0, (Object)"bucketName must not be null or empty");
        Preconditions.checkNotNull((Object)options, (Object)"options must not be null");
        BucketInfo.Builder bucketInfoBuilder = BucketInfo.newBuilder(bucketName).setLocation(options.getLocation());
        if (options.getStorageClass() != null) {
            bucketInfoBuilder.setStorageClass(StorageClass.valueOfStrict(options.getStorageClass().toUpperCase()));
        }
        if (options.getTtl() != null) {
            bucketInfoBuilder.setLifecycleRules(Collections.singletonList(new BucketInfo.LifecycleRule(BucketInfo.LifecycleRule.LifecycleAction.newDeleteAction(), BucketInfo.LifecycleRule.LifecycleCondition.newBuilder().setAge(Math.toIntExact(options.getTtl().toDays())).build())));
        }
        try {
            this.storage.create(bucketInfoBuilder.build(), new Storage.BucketTargetOption[0]);
        }
        catch (StorageException e) {
            if (ErrorTypeExtractor.bucketAlreadyExists(e)) {
                throw (FileAlreadyExistsException)new FileAlreadyExistsException(String.format("Bucket '%s' already exists.", bucketName)).initCause(e);
            }
            throw new IOException(e);
        }
    }

    void createEmptyObject(StorageResourceId resourceId) throws IOException {
        LOG.trace("createEmptyObject({})", (Object)resourceId);
        Preconditions.checkArgument((boolean)resourceId.isStorageObject(), (String)"Expected full StorageObject id, got %s", (Object)resourceId);
        this.createEmptyObject(resourceId, EMPTY_OBJECT_CREATE_OPTIONS);
    }

    void createEmptyObject(StorageResourceId resourceId, CreateObjectOptions options) throws IOException {
        Preconditions.checkArgument((boolean)resourceId.isStorageObject(), (String)"Expected full StorageObject id, got %s", (Object)resourceId);
        try {
            this.createEmptyObjectInternal(resourceId, options);
        }
        catch (StorageException e) {
            if (this.canIgnoreExceptionForEmptyObject(e, resourceId, options)) {
                LOG.info("Ignoring exception of type {}; verified object already exists with desired state.", (Object)e.getClass().getSimpleName());
                LOG.trace("Ignored exception while creating empty object: {}", (Object)resourceId, (Object)e);
            }
            if (ErrorTypeExtractor.getErrorType(e) == ErrorTypeExtractor.ErrorType.ALREADY_EXISTS) {
                throw (FileAlreadyExistsException)new FileAlreadyExistsException(String.format("Object '%s' already exists.", resourceId)).initCause(e);
            }
            throw new IOException(e);
        }
    }

    GoogleCloudStorageItemInfo composeObjects(List<StorageResourceId> sources, StorageResourceId destination, CreateObjectOptions options) throws IOException {
        Blob composedBlob;
        LOG.trace("composeObjects({}, {}, {})", new Object[]{sources, destination, options});
        for (StorageResourceId inputId : sources) {
            if (destination.getBucketName().equals(inputId.getBucketName())) continue;
            throw new IOException(String.format("Bucket doesn't match for source '%s' and destination '%s'!", inputId, destination));
        }
        Storage.ComposeRequest request = Storage.ComposeRequest.newBuilder().addSource(sources.stream().map(StorageResourceId::getObjectName).collect(Collectors.toList())).setTarget(BlobInfo.newBuilder(destination.getBucketName(), destination.getObjectName()).setContentType(options.getContentType()).setContentEncoding(options.getContentEncoding()).setMetadata(GoogleCloudStorage.encodeMetadata(options.getMetadata())).build()).setTargetOptions(Storage.BlobTargetOption.generationMatch(destination.hasGenerationId() ? destination.getGenerationId() : this.getWriteGeneration(destination, true))).build();
        try {
            composedBlob = this.storage.compose(request);
        }
        catch (StorageException e) {
            throw new IOException(e);
        }
        GoogleCloudStorageItemInfo compositeInfo = GoogleCloudStorage.createItemInfoForBlob(destination, composedBlob);
        LOG.trace("composeObjects() done, returning: {}", (Object)compositeInfo);
        return compositeInfo;
    }

    private boolean canIgnoreExceptionForEmptyObject(StorageException exceptionOnCreate, StorageResourceId resourceId, CreateObjectOptions options) throws IOException {
        ErrorTypeExtractor.ErrorType errorType = ErrorTypeExtractor.getErrorType(exceptionOnCreate);
        if (GoogleCloudStorage.shouldBackoff(resourceId, errorType)) {
            GoogleCloudStorageItemInfo existingInfo;
            Duration maxWaitTime = Duration.ofSeconds(3L);
            BackOff backOff = !maxWaitTime.isZero() && !maxWaitTime.isNegative() ? new ExponentialBackOff.Builder().setMaxElapsedTimeMillis(Math.toIntExact(maxWaitTime.toMillis())).setMaxIntervalMillis(500).setInitialIntervalMillis(100).setMultiplier(1.5).setRandomizationFactor(0.15).build() : BackOff.STOP_BACKOFF;
            long nextSleep = 0L;
            do {
                if (nextSleep > 0L) {
                    try {
                        Sleeper.DEFAULT.sleep(nextSleep);
                    }
                    catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                        nextSleep = -1L;
                    }
                }
                existingInfo = this.getItemInfo(resourceId);
                long l = nextSleep = nextSleep == -1L ? -1L : backOff.nextBackOffMillis();
            } while (!existingInfo.exists() && nextSleep != -1L);
            if (existingInfo.exists() && existingInfo.getSize() == 0L) {
                if (options.isEnsureEmptyObjectsMetadataMatch()) {
                    return existingInfo.metadataEquals(options.getMetadata());
                }
                return true;
            }
        }
        return false;
    }

    private static boolean shouldBackoff(StorageResourceId resourceId, ErrorTypeExtractor.ErrorType errorType) {
        return errorType == ErrorTypeExtractor.ErrorType.RESOURCE_EXHAUSTED || errorType == ErrorTypeExtractor.ErrorType.INTERNAL || resourceId.isDirectory() && errorType == ErrorTypeExtractor.ErrorType.FAILED_PRECONDITION;
    }

    private void createEmptyObjectInternal(StorageResourceId resourceId, CreateObjectOptions createObjectOptions) throws IOException {
        Map<String, String> rewrittenMetadata = GoogleCloudStorage.encodeMetadata(createObjectOptions.getMetadata());
        ArrayList<Storage.BlobTargetOption> blobTargetOptions = new ArrayList<Storage.BlobTargetOption>();
        blobTargetOptions.add(Storage.BlobTargetOption.disableGzipContent());
        if (resourceId.hasGenerationId()) {
            blobTargetOptions.add(Storage.BlobTargetOption.generationMatch(resourceId.getGenerationId()));
        } else if (resourceId.isDirectory() || !createObjectOptions.isOverwriteExisting()) {
            blobTargetOptions.add(Storage.BlobTargetOption.doesNotExist());
        }
        this.storage.create(BlobInfo.newBuilder(BlobId.of(resourceId.getBucketName(), resourceId.getObjectName())).setMetadata(rewrittenMetadata).setContentEncoding(createObjectOptions.getContentEncoding()).setContentType(createObjectOptions.getContentType()).build(), blobTargetOptions.toArray(new Storage.BlobTargetOption[0]));
    }

    private static Map<String, String> encodeMetadata(Map<String, byte[]> metadata) {
        return Maps.transformValues(metadata, GoogleCloudStorage::encodeMetadataValues);
    }

    private static String encodeMetadataValues(byte[] bytes) {
        return bytes == null ? null : BaseEncoding.base64().encode(bytes);
    }

    List<GoogleCloudStorageItemInfo> listDirectoryRecursive(String bucketName, String objectName) throws IOException {
        Preconditions.checkArgument((objectName == null || objectName.endsWith("/") ? 1 : 0) != 0, (Object)String.format("%s should end with /", objectName));
        try {
            List<Blob> blobs = new GcsListOperation.Builder(bucketName, objectName, this.storage).forRecursiveListing().build().execute();
            ArrayList<GoogleCloudStorageItemInfo> result = new ArrayList<GoogleCloudStorageItemInfo>();
            for (Blob blob : blobs) {
                result.add(this.createItemInfoForBlob(blob));
            }
            return result;
        }
        catch (StorageException e) {
            throw new IOException(String.format("Listing '%s' failed", BlobId.of(bucketName, objectName)), e);
        }
    }

    void deleteObjects(List<StorageResourceId> fullObjectNames) throws IOException {
        LOG.trace("deleteObjects({})", fullObjectNames);
        if (fullObjectNames.isEmpty()) {
            return;
        }
        for (StorageResourceId toDelete : fullObjectNames) {
            Preconditions.checkArgument((boolean)toDelete.isStorageObject(), (String)"Expected full StorageObject names only, got: %s", (Object)toDelete);
        }
        for (StorageResourceId toDelete : fullObjectNames) {
            try {
                LOG.trace("Deleting Object ({})", (Object)toDelete);
                if (toDelete.hasGenerationId() && toDelete.getGenerationId() != 0L) {
                    this.storage.delete(BlobId.of(toDelete.getBucketName(), toDelete.getObjectName()), Storage.BlobSourceOption.generationMatch(toDelete.getGenerationId()));
                    continue;
                }
                this.storage.delete(BlobId.of(toDelete.getBucketName(), toDelete.getObjectName()));
                LOG.trace("Deleting Object without generationId ({})", (Object)toDelete);
            }
            catch (StorageException e) {
                throw new IOException(String.format("Deleting resource %s failed.", toDelete), e);
            }
        }
    }

    List<GoogleCloudStorageItemInfo> listBucketInfo() throws IOException {
        List<Bucket> allBuckets = this.listBucketsInternal();
        ArrayList<GoogleCloudStorageItemInfo> bucketInfos = new ArrayList<GoogleCloudStorageItemInfo>(allBuckets.size());
        for (Bucket bucket : allBuckets) {
            bucketInfos.add(GoogleCloudStorage.createItemInfoForBucket(new StorageResourceId(bucket.getName()), bucket));
        }
        return bucketInfos;
    }

    private List<Bucket> listBucketsInternal() throws IOException {
        Preconditions.checkNotNull((Object)this.configuration.getProjectId(), (Object)"projectId must not be null");
        ArrayList<Bucket> allBuckets = new ArrayList<Bucket>();
        try {
            Page<Bucket> buckets = this.storage.list(Storage.BucketListOption.pageSize(this.configuration.getMaxListItemsPerCall()), Storage.BucketListOption.fields(Storage.BucketField.LOCATION, Storage.BucketField.STORAGE_CLASS, Storage.BucketField.TIME_CREATED, Storage.BucketField.UPDATED));
            for (Bucket bucket : buckets.iterateAll()) {
                allBuckets.add(bucket);
            }
        }
        catch (StorageException e) {
            throw new IOException(e);
        }
        return allBuckets;
    }

    SeekableByteChannel open(GoogleCloudStorageItemInfo itemInfo, GoogleHadoopFileSystemConfiguration config) throws IOException {
        LOG.trace("open({})", (Object)itemInfo);
        Preconditions.checkNotNull((Object)itemInfo, (Object)"itemInfo should not be null");
        StorageResourceId resourceId = itemInfo.getResourceId();
        Preconditions.checkArgument((boolean)resourceId.isStorageObject(), (String)"Expected full StorageObject id, got %s", (Object)resourceId);
        return this.open(resourceId, itemInfo, config);
    }

    private SeekableByteChannel open(StorageResourceId resourceId, GoogleCloudStorageItemInfo itemInfo, GoogleHadoopFileSystemConfiguration config) throws IOException {
        return new GoogleCloudStorageClientReadChannel(this.storage, itemInfo == null ? this.getItemInfo(resourceId) : itemInfo, config);
    }

    void move(Map<StorageResourceId, StorageResourceId> sourceToDestinationObjectsMap) throws IOException {
        GoogleCloudStorage.validateMoveArguments(sourceToDestinationObjectsMap);
        if (sourceToDestinationObjectsMap.isEmpty()) {
            return;
        }
        for (Map.Entry<StorageResourceId, StorageResourceId> entry : sourceToDestinationObjectsMap.entrySet()) {
            StorageResourceId srcObject = entry.getKey();
            StorageResourceId dstObject = entry.getValue();
            this.moveInternal(srcObject.getBucketName(), srcObject.getGenerationId(), srcObject.getObjectName(), dstObject.getGenerationId(), dstObject.getObjectName());
        }
    }

    private void moveInternal(String srcBucketName, long srcContentGeneration, String srcObjectName, long dstContentGeneration, String dstObjectName) throws IOException {
        Storage.MoveBlobRequest.Builder moveRequestBuilder = this.createMoveRequestBuilder(srcBucketName, srcObjectName, dstObjectName, srcContentGeneration, dstContentGeneration);
        try {
            String srcString = StringPaths.fromComponents(srcBucketName, srcObjectName);
            String dstString = StringPaths.fromComponents(srcBucketName, dstObjectName);
            Blob movedBlob = this.storage.moveBlob(moveRequestBuilder.build());
            if (movedBlob != null) {
                LOG.trace("Successfully moved {} to {}", (Object)srcString, (Object)dstString);
            }
        }
        catch (StorageException e) {
            if (ErrorTypeExtractor.getErrorType(e) == ErrorTypeExtractor.ErrorType.NOT_FOUND) {
                throw GoogleCloudStorageExceptions.createFileNotFoundException(srcBucketName, srcObjectName, new IOException(e));
            }
            throw new IOException(String.format("Error moving '%s'", StringPaths.fromComponents(srcBucketName, srcObjectName)), e);
        }
    }

    private Storage.MoveBlobRequest.Builder createMoveRequestBuilder(String srcBucketName, String srcObjectName, String dstObjectName, long srcContentGeneration, long dstContentGeneration) {
        Storage.MoveBlobRequest.Builder moveRequestBuilder = Storage.MoveBlobRequest.newBuilder().setSource(BlobId.of(srcBucketName, srcObjectName));
        moveRequestBuilder.setTarget(BlobId.of(srcBucketName, dstObjectName));
        ArrayList<Storage.BlobTargetOption> blobTargetOptions = new ArrayList<Storage.BlobTargetOption>();
        ArrayList<Storage.BlobSourceOption> blobSourceOptions = new ArrayList<Storage.BlobSourceOption>();
        if (srcContentGeneration != -1L) {
            blobSourceOptions.add(Storage.BlobSourceOption.generationMatch(srcContentGeneration));
        }
        if (dstContentGeneration != -1L) {
            blobTargetOptions.add(Storage.BlobTargetOption.generationMatch(dstContentGeneration));
        }
        moveRequestBuilder.setSourceOptions(blobSourceOptions);
        moveRequestBuilder.setTargetOptions(blobTargetOptions);
        return moveRequestBuilder;
    }

    static void validateMoveArguments(Map<StorageResourceId, StorageResourceId> sourceToDestinationObjectsMap) throws IOException {
        Preconditions.checkNotNull(sourceToDestinationObjectsMap, (Object)"srcObjects must not be null");
        if (sourceToDestinationObjectsMap.isEmpty()) {
            return;
        }
        for (Map.Entry<StorageResourceId, StorageResourceId> entry : sourceToDestinationObjectsMap.entrySet()) {
            String dstBucketName;
            StorageResourceId source = entry.getKey();
            StorageResourceId destination = entry.getValue();
            String srcBucketName = source.getBucketName();
            if (!srcBucketName.equals(dstBucketName = destination.getBucketName())) {
                throw new UnsupportedOperationException("This operation is not supported across two different buckets.");
            }
            Preconditions.checkArgument((!Strings.isNullOrEmpty((String)source.getObjectName()) ? 1 : 0) != 0, (Object)"srcObjectName must not be null or empty");
            Preconditions.checkArgument((!Strings.isNullOrEmpty((String)destination.getObjectName()) ? 1 : 0) != 0, (Object)"dstObjectName must not be null or empty");
            if (!srcBucketName.equals(dstBucketName) || !source.getObjectName().equals(destination.getObjectName())) continue;
            throw new IllegalArgumentException(String.format("Move destination must be different from source for %s.", StringPaths.fromComponents(srcBucketName, source.getObjectName())));
        }
    }

    void copy(Map<StorageResourceId, StorageResourceId> sourceToDestinationObjectsMap) throws IOException {
        GoogleCloudStorage.validateCopyArguments(sourceToDestinationObjectsMap, this);
        if (sourceToDestinationObjectsMap.isEmpty()) {
            return;
        }
        for (Map.Entry<StorageResourceId, StorageResourceId> entry : sourceToDestinationObjectsMap.entrySet()) {
            StorageResourceId srcObject = entry.getKey();
            StorageResourceId dstObject = entry.getValue();
            this.copyInternal(srcObject.getBucketName(), srcObject.getObjectName(), dstObject.getGenerationId(), dstObject.getBucketName(), dstObject.getObjectName());
        }
    }

    private void copyInternal(String srcBucketName, String srcObjectName, long dstContentGeneration, String dstBucketName, String dstObjectName) throws IOException {
        Storage.CopyRequest.Builder copyRequestBuilder = Storage.CopyRequest.newBuilder().setSource(BlobId.of(srcBucketName, srcObjectName));
        if (dstContentGeneration != -1L) {
            copyRequestBuilder.setTarget(BlobId.of(dstBucketName, dstObjectName), Storage.BlobTargetOption.generationMatch(dstContentGeneration));
        } else {
            copyRequestBuilder.setTarget(BlobId.of(dstBucketName, dstObjectName));
        }
        if (this.configuration.getMaxRewriteChunkSize() > 0L) {
            copyRequestBuilder.setMegabytesCopiedPerChunk(this.configuration.getMaxRewriteChunkSize() / 0x100000L);
        }
        String srcString = StringPaths.fromComponents(srcBucketName, srcObjectName);
        String dstString = StringPaths.fromComponents(dstBucketName, dstObjectName);
        try {
            CopyWriter copyWriter = this.storage.copy(copyRequestBuilder.build());
            while (!copyWriter.isDone()) {
                copyWriter.copyChunk();
                LOG.trace("Copy ({} to {}) did not complete. Resuming...", (Object)srcString, (Object)dstString);
            }
            LOG.trace("Successfully copied {} to {}", (Object)srcString, (Object)dstString);
        }
        catch (StorageException e) {
            if (ErrorTypeExtractor.getErrorType(e) == ErrorTypeExtractor.ErrorType.NOT_FOUND) {
                throw GoogleCloudStorageExceptions.createFileNotFoundException(srcBucketName, srcObjectName, new IOException(e));
            }
            throw new IOException(String.format("copy(%s->%s) failed.", srcString, dstString), e);
        }
    }

    static void validateCopyArguments(Map<StorageResourceId, StorageResourceId> sourceToDestinationObjectsMap, GoogleCloudStorage gcsImpl) throws IOException {
        Preconditions.checkNotNull(sourceToDestinationObjectsMap, (Object)"srcObjects must not be null");
        if (sourceToDestinationObjectsMap.isEmpty()) {
            return;
        }
        HashMap<StorageResourceId, GoogleCloudStorageItemInfo> bucketInfoCache = new HashMap<StorageResourceId, GoogleCloudStorageItemInfo>();
        for (Map.Entry<StorageResourceId, StorageResourceId> entry : sourceToDestinationObjectsMap.entrySet()) {
            String dstBucketName;
            StorageResourceId source = entry.getKey();
            StorageResourceId destination = entry.getValue();
            String srcBucketName = source.getBucketName();
            if (!srcBucketName.equals(dstBucketName = destination.getBucketName())) {
                StorageResourceId srcBucketResourceId = new StorageResourceId(srcBucketName);
                GoogleCloudStorageItemInfo srcBucketInfo = GoogleCloudStorage.getGoogleCloudStorageItemInfo(gcsImpl, bucketInfoCache, srcBucketResourceId);
                if (!srcBucketInfo.exists()) {
                    throw new FileNotFoundException("Bucket not found: " + srcBucketName);
                }
                StorageResourceId dstBucketResourceId = new StorageResourceId(dstBucketName);
                GoogleCloudStorageItemInfo dstBucketInfo = GoogleCloudStorage.getGoogleCloudStorageItemInfo(gcsImpl, bucketInfoCache, dstBucketResourceId);
                if (!dstBucketInfo.exists()) {
                    throw new FileNotFoundException("Bucket not found: " + dstBucketName);
                }
                if (!srcBucketInfo.getLocation().equals(dstBucketInfo.getLocation())) {
                    throw new UnsupportedOperationException("This operation is not supported across two different storage locations.");
                }
                if (!srcBucketInfo.getStorageClass().equals(dstBucketInfo.getStorageClass())) {
                    throw new UnsupportedOperationException("This operation is not supported across two different storage classes.");
                }
            }
            Preconditions.checkArgument((!Strings.isNullOrEmpty((String)source.getObjectName()) ? 1 : 0) != 0, (Object)"srcObjectName must not be null or empty");
            Preconditions.checkArgument((!Strings.isNullOrEmpty((String)destination.getObjectName()) ? 1 : 0) != 0, (Object)"dstObjectName must not be null or empty");
            if (!srcBucketName.equals(dstBucketName) || !source.getObjectName().equals(destination.getObjectName())) continue;
            throw new IllegalArgumentException(String.format("Copy destination must be different from source for %s.", StringPaths.fromComponents(srcBucketName, source.getObjectName())));
        }
    }

    private static GoogleCloudStorageItemInfo getGoogleCloudStorageItemInfo(GoogleCloudStorage gcsImpl, Map<StorageResourceId, GoogleCloudStorageItemInfo> bucketInfoCache, StorageResourceId resourceId) throws IOException {
        GoogleCloudStorageItemInfo storageItemInfo = bucketInfoCache.get(resourceId);
        if (storageItemInfo != null) {
            return storageItemInfo;
        }
        storageItemInfo = gcsImpl.getItemInfo(resourceId);
        bucketInfoCache.put(resourceId, storageItemInfo);
        return storageItemInfo;
    }

    List<GoogleCloudStorageItemInfo> getItemInfos(List<StorageResourceId> resourceIds) throws IOException {
        LOG.trace("getItemInfos({})", resourceIds);
        if (resourceIds.isEmpty()) {
            return new ArrayList<GoogleCloudStorageItemInfo>();
        }
        ArrayList<GoogleCloudStorageItemInfo> result = new ArrayList<GoogleCloudStorageItemInfo>(resourceIds.size());
        for (StorageResourceId resourceId : resourceIds) {
            result.add(this.getItemInfo(resourceId));
        }
        return result;
    }

    List<GoogleCloudStorageItemInfo> listDirectory(String bucketName, String objectNamePrefix) throws IOException {
        Preconditions.checkArgument((objectNamePrefix == null || objectNamePrefix.endsWith("/") ? 1 : 0) != 0, (Object)String.format("%s should end with /", objectNamePrefix));
        try {
            List<Blob> blobs = new GcsListOperation.Builder(bucketName, objectNamePrefix, this.storage).forCurrentDirectoryListing().build().execute();
            ListOperationResult result = new ListOperationResult();
            for (Blob blob : blobs) {
                result.add(blob);
            }
            return result.getItems();
        }
        catch (StorageException e) {
            throw new IOException(String.format("listing object '%s' failed.", BlobId.of(bucketName, objectNamePrefix)), e);
        }
    }

    void compose(String bucketName, List<String> sources, String destination, String contentType) throws IOException {
        LOG.trace("compose({}, {}, {}, {})", new Object[]{bucketName, sources, destination, contentType});
        List<StorageResourceId> sourceIds = sources.stream().map(objectName -> new StorageResourceId(bucketName, (String)objectName)).collect(Collectors.toList());
        StorageResourceId destinationId = new StorageResourceId(bucketName, destination);
        CreateObjectOptions options = CreateObjectOptions.DEFAULT_OVERWRITE.toBuilder().setContentType(contentType).setEnsureEmptyObjectsMetadataMatch(false).build();
        this.composeObjects(sourceIds, destinationId, options);
    }

    GoogleCloudStorageItemInfo getFileOrDirectoryInfo(StorageResourceId resourceId) {
        BlobId blobId = resourceId.toBlobId();
        if (resourceId.isDirectory()) {
            Blob blob = this.storage.get(blobId);
            if (blob != null) {
                return this.createItemInfoForBlob(blob);
            }
        } else {
            BlobId dirId = resourceId.toDirectoryId().toBlobId();
            List<Blob> blobs = this.storage.get(blobId, dirId);
            for (Blob blob : blobs) {
                if (blob == null) continue;
                return this.createItemInfoForBlob(blob);
            }
        }
        return GoogleCloudStorageItemInfo.createNotFound(resourceId);
    }

    GoogleCloudStorageItemInfo getImplicitDirectory(StorageResourceId resourceId) {
        List<Blob> blobs = new GcsListOperation.Builder(resourceId.getBucketName(), resourceId.getObjectName(), this.storage).forImplicitDirectoryCheck().build().execute();
        if (blobs.isEmpty()) {
            return GoogleCloudStorageItemInfo.createNotFound(resourceId);
        }
        return GoogleCloudStorageItemInfo.createInferredDirectory(resourceId.toDirectoryId());
    }

    public void deleteBuckets(List<String> bucketNames) throws IOException {
        LOG.trace("deleteBuckets({})", bucketNames);
        for (String bucketName : bucketNames) {
            Preconditions.checkArgument((!Strings.isNullOrEmpty((String)bucketName) ? 1 : 0) != 0, (Object)"bucketName must not be null or empty");
        }
        ArrayList<IOException> innerExceptions = new ArrayList<IOException>();
        for (String bucketName : bucketNames) {
            try {
                boolean isDeleted = this.storage.delete(bucketName, new Storage.BucketSourceOption[0]);
                if (isDeleted) continue;
                innerExceptions.add(GoogleCloudStorageExceptions.createFileNotFoundException(bucketName, null, null));
            }
            catch (StorageException e) {
                innerExceptions.add(new IOException(String.format("Error deleting '%s' bucket", bucketName), e));
            }
        }
        if (!innerExceptions.isEmpty()) {
            throw GoogleCloudStorageExceptions.createCompositeException(innerExceptions);
        }
    }

    private class ListOperationResult {
        private final Map<String, Blob> prefixes = new HashMap<String, Blob>();
        private final List<Blob> objects = new ArrayList<Blob>();
        private final Set<String> objectsSet = new HashSet<String>();

        private ListOperationResult() {
        }

        void add(Blob blob) {
            String path = blob.getBlobId().toGsUtilUri();
            if (blob.getGeneration() != null) {
                this.prefixes.remove(path);
                this.objects.add(blob);
                this.objectsSet.add(path);
            } else if (!this.objectsSet.contains(path)) {
                this.prefixes.put(path, blob);
            }
        }

        List<GoogleCloudStorageItemInfo> getItems() {
            ArrayList<GoogleCloudStorageItemInfo> result = new ArrayList<GoogleCloudStorageItemInfo>(this.prefixes.size() + this.objects.size());
            for (Blob blob : this.objects) {
                result.add(GoogleCloudStorage.this.createItemInfoForBlob(blob));
            }
            for (Blob blob : this.prefixes.values()) {
                result.add(GoogleCloudStorage.this.createItemInfoForBlob(blob));
            }
            return result;
        }
    }
}

