/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.tools.mapred;

import java.io.Closeable;
import java.io.IOException;
import java.security.PrivilegedAction;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.List;
import java.util.Random;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CreateFlag;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Options;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.permission.FsAction;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.server.datanode.DataNode;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.mapreduce.Mapper;
import org.apache.hadoop.metrics2.MetricsRecordBuilder;
import org.apache.hadoop.security.AccessControlException;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.test.GenericTestUtils;
import org.apache.hadoop.test.MetricsAsserts;
import org.apache.hadoop.tools.CopyListingFileStatus;
import org.apache.hadoop.tools.DistCpOptionSwitch;
import org.apache.hadoop.tools.DistCpOptions;
import org.apache.hadoop.tools.StubContext;
import org.apache.hadoop.tools.mapred.CopyMapper;
import org.apache.hadoop.tools.util.DistCpUtils;
import org.apache.hadoop.util.DataChecksum;
import org.apache.hadoop.util.StringUtils;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.Timeout;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TestCopyMapper {
    private static final Logger LOG = LoggerFactory.getLogger(TestCopyMapper.class);
    private static List<Path> pathList = new ArrayList<Path>();
    private static int nFiles = 0;
    private static final int DEFAULT_FILE_SIZE = 1024;
    private static final long NON_DEFAULT_BLOCK_SIZE = 4096L;
    private static MiniDFSCluster cluster;
    private static final String SOURCE_PATH = "/tmp/source";
    private static final String TARGET_PATH = "/tmp/target";

    @BeforeAll
    public static void setup() throws Exception {
        Configuration configuration = TestCopyMapper.getConfigurationForCluster();
        TestCopyMapper.setCluster(new MiniDFSCluster.Builder(configuration).numDataNodes(1).format(true).build());
    }

    protected boolean expectDifferentBlockSizesMultipleBlocksToSucceed() {
        return false;
    }

    protected boolean expectDifferentBytesPerCrcToSucceed() {
        return false;
    }

    protected static void setCluster(MiniDFSCluster c) {
        cluster = c;
    }

    protected static Configuration getConfigurationForCluster() throws IOException {
        Configuration configuration = new Configuration();
        System.setProperty("test.build.data", "target/tmp/build/TEST_COPY_MAPPER/data");
        configuration.set("hadoop.log.dir", "target/tmp");
        configuration.set("dfs.namenode.fs-limits.min-block-size", "0");
        LOG.debug("fs.default.name  == " + configuration.get("fs.default.name"));
        LOG.debug("dfs.http.address == " + configuration.get("dfs.http.address"));
        return configuration;
    }

    private static Configuration getConfiguration() throws IOException {
        Configuration configuration = TestCopyMapper.getConfigurationForCluster();
        DistributedFileSystem fs = cluster.getFileSystem();
        Path workPath = new Path(TARGET_PATH).makeQualified(fs.getUri(), fs.getWorkingDirectory());
        configuration.set("distcp.target.work.path", workPath.toString());
        configuration.set("distcp.target.final.path", workPath.toString());
        configuration.setBoolean(DistCpOptionSwitch.OVERWRITE.getConfigLabel(), false);
        configuration.setBoolean(DistCpOptionSwitch.SKIP_CRC.getConfigLabel(), false);
        configuration.setBoolean(DistCpOptionSwitch.SYNC_FOLDERS.getConfigLabel(), true);
        configuration.set(DistCpOptionSwitch.PRESERVE_STATUS.getConfigLabel(), "br");
        return configuration;
    }

    private static void createSourceData() throws Exception {
        TestCopyMapper.mkdirs("/tmp/source/1");
        TestCopyMapper.mkdirs("/tmp/source/2");
        TestCopyMapper.mkdirs("/tmp/source/2/3/4");
        TestCopyMapper.mkdirs("/tmp/source/2/3");
        TestCopyMapper.mkdirs("/tmp/source/5");
        TestCopyMapper.touchFile("/tmp/source/5/6");
        TestCopyMapper.mkdirs("/tmp/source/7");
        TestCopyMapper.mkdirs("/tmp/source/7/8");
        TestCopyMapper.touchFile("/tmp/source/7/8/9");
    }

    private static void appendSourceData() throws Exception {
        DistributedFileSystem fs = cluster.getFileSystem();
        for (Path source : pathList) {
            if (!fs.getFileStatus(source).isFile()) continue;
            TestCopyMapper.appendFile(source, 2048);
        }
    }

    private static void createSourceDataWithDifferentBlockSize() throws Exception {
        TestCopyMapper.mkdirs("/tmp/source/1");
        TestCopyMapper.mkdirs("/tmp/source/2");
        TestCopyMapper.mkdirs("/tmp/source/2/3/4");
        TestCopyMapper.mkdirs("/tmp/source/2/3");
        TestCopyMapper.mkdirs("/tmp/source/5");
        TestCopyMapper.touchFile("/tmp/source/5/6", true, null);
        TestCopyMapper.mkdirs("/tmp/source/7");
        TestCopyMapper.mkdirs("/tmp/source/7/8");
        TestCopyMapper.touchFile("/tmp/source/7/8/9");
    }

    private static void createSourceDataWithDifferentChecksumType() throws Exception {
        TestCopyMapper.mkdirs("/tmp/source/1");
        TestCopyMapper.mkdirs("/tmp/source/2");
        TestCopyMapper.mkdirs("/tmp/source/2/3/4");
        TestCopyMapper.mkdirs("/tmp/source/2/3");
        TestCopyMapper.mkdirs("/tmp/source/5");
        TestCopyMapper.touchFile("/tmp/source/5/6", new Options.ChecksumOpt(DataChecksum.Type.CRC32, 512));
        TestCopyMapper.mkdirs("/tmp/source/7");
        TestCopyMapper.mkdirs("/tmp/source/7/8");
        TestCopyMapper.touchFile("/tmp/source/7/8/9", new Options.ChecksumOpt(DataChecksum.Type.CRC32C, 512));
    }

    private static void createSourceDataWithDifferentBytesPerCrc() throws Exception {
        TestCopyMapper.mkdirs("/tmp/source/1");
        TestCopyMapper.mkdirs("/tmp/source/2");
        TestCopyMapper.mkdirs("/tmp/source/2/3/4");
        TestCopyMapper.mkdirs("/tmp/source/2/3");
        TestCopyMapper.mkdirs("/tmp/source/5");
        TestCopyMapper.touchFile("/tmp/source/5/6", false, new Options.ChecksumOpt(DataChecksum.Type.CRC32C, 32));
        TestCopyMapper.mkdirs("/tmp/source/7");
        TestCopyMapper.mkdirs("/tmp/source/7/8");
        TestCopyMapper.touchFile("/tmp/source/7/8/9", false, new Options.ChecksumOpt(DataChecksum.Type.CRC32C, 64));
    }

    private static void mkdirs(String path) throws Exception {
        DistributedFileSystem fileSystem = cluster.getFileSystem();
        Path qualifiedPath = new Path(path).makeQualified(fileSystem.getUri(), fileSystem.getWorkingDirectory());
        pathList.add(qualifiedPath);
        fileSystem.mkdirs(qualifiedPath);
    }

    private static void touchFile(String path) throws Exception {
        TestCopyMapper.touchFile(path, false, null);
    }

    private static void touchFile(String path, Options.ChecksumOpt checksumOpt) throws Exception {
        TestCopyMapper.touchFile(path, true, checksumOpt);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void touchFile(String path, boolean createMultipleBlocks, Options.ChecksumOpt checksumOpt) throws Exception {
        FSDataOutputStream outputStream = null;
        try {
            DistributedFileSystem fs = cluster.getFileSystem();
            Path qualifiedPath = new Path(path).makeQualified(fs.getUri(), fs.getWorkingDirectory());
            long blockSize = createMultipleBlocks ? 4096L : fs.getDefaultBlockSize(qualifiedPath) * 2L;
            FsPermission permission = FsPermission.getFileDefault().applyUMask(FsPermission.getUMask((Configuration)fs.getConf()));
            outputStream = fs.create(qualifiedPath, permission, EnumSet.of(CreateFlag.CREATE, CreateFlag.OVERWRITE), 0, (short)(fs.getDefaultReplication(qualifiedPath) * 2), blockSize, null, checksumOpt);
            byte[] bytes = new byte[1024];
            outputStream.write(bytes);
            if (createMultipleBlocks) {
                for (long fileSize = 1024L; fileSize < 2L * blockSize; fileSize += 1024L) {
                    outputStream.write(bytes);
                    outputStream.flush();
                }
            }
            pathList.add(qualifiedPath);
            ++nFiles;
            FileStatus fileStatus = fs.getFileStatus(qualifiedPath);
            System.out.println(fileStatus.getBlockSize());
            System.out.println(fileStatus.getReplication());
        }
        catch (Throwable throwable) {
            IOUtils.cleanupWithLogger(null, (Closeable[])new Closeable[]{outputStream});
            throw throwable;
        }
        IOUtils.cleanupWithLogger(null, (Closeable[])new Closeable[]{outputStream});
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void appendFile(Path p, int length) throws IOException {
        byte[] toAppend = new byte[length];
        Random random = new Random();
        random.nextBytes(toAppend);
        FSDataOutputStream out = cluster.getFileSystem().append(p);
        try {
            out.write(toAppend);
        }
        finally {
            IOUtils.closeStream((Closeable)out);
        }
    }

    @Test
    public void testCopyWithDifferentChecksumType() throws Exception {
        this.testCopy(true);
    }

    @Test
    @Timeout(value=40L)
    public void testRun() throws Exception {
        this.testCopy(false);
    }

    @Test
    public void testCopyWithAppend() throws Exception {
        DistributedFileSystem fs = cluster.getFileSystem();
        this.testCopy(false);
        TestCopyMapper.appendSourceData();
        CopyMapper copyMapper = new CopyMapper();
        Configuration conf = TestCopyMapper.getConfiguration();
        conf.setInt(DistCpOptionSwitch.COPY_BUFFER_SIZE.getConfigLabel(), 102);
        StubContext stubContext = new StubContext(conf, null, 0);
        Mapper.Context context = stubContext.getContext();
        context.getConfiguration().setBoolean(DistCpOptionSwitch.APPEND.getConfigLabel(), true);
        copyMapper.setup(context);
        int numFiles = 0;
        MetricsRecordBuilder rb = MetricsAsserts.getMetrics((String)((DataNode)cluster.getDataNodes().get(0)).getMetrics().name());
        String readCounter = "ReadsFromLocalClient";
        long readsFromClient = MetricsAsserts.getLongCounter((String)readCounter, (MetricsRecordBuilder)rb);
        for (Path path : pathList) {
            if (fs.getFileStatus(path).isFile()) {
                ++numFiles;
            }
            copyMapper.map(new Text(DistCpUtils.getRelativePath((Path)new Path(SOURCE_PATH), (Path)path)), new CopyListingFileStatus(cluster.getFileSystem().getFileStatus(path)), context);
        }
        this.verifyCopy((FileSystem)fs, false, true);
        org.junit.jupiter.api.Assertions.assertEquals((long)(nFiles * 1024 * 2), (long)stubContext.getReporter().getCounter((Enum)CopyMapper.Counter.BYTESCOPIED).getValue());
        org.junit.jupiter.api.Assertions.assertEquals((long)numFiles, (long)stubContext.getReporter().getCounter((Enum)CopyMapper.Counter.COPY).getValue());
        rb = MetricsAsserts.getMetrics((String)((DataNode)cluster.getDataNodes().get(0)).getMetrics().name());
        MetricsAsserts.assertCounter((String)readCounter, (long)(readsFromClient + (long)numFiles), (MetricsRecordBuilder)rb);
    }

    private void testCopy(boolean preserveChecksum) throws Exception {
        TestCopyMapper.deleteState();
        if (preserveChecksum) {
            TestCopyMapper.createSourceDataWithDifferentChecksumType();
        } else {
            TestCopyMapper.createSourceData();
        }
        DistributedFileSystem fs = cluster.getFileSystem();
        CopyMapper copyMapper = new CopyMapper();
        StubContext stubContext = new StubContext(TestCopyMapper.getConfiguration(), null, 0);
        Mapper.Context context = stubContext.getContext();
        Configuration configuration = context.getConfiguration();
        EnumSet<DistCpOptions.FileAttribute> fileAttributes = EnumSet.of(DistCpOptions.FileAttribute.REPLICATION);
        if (preserveChecksum) {
            fileAttributes.add(DistCpOptions.FileAttribute.CHECKSUMTYPE);
        }
        configuration.set(DistCpOptionSwitch.PRESERVE_STATUS.getConfigLabel(), DistCpUtils.packAttributes(fileAttributes));
        copyMapper.setup(context);
        int numFiles = 0;
        int numDirs = 0;
        for (Path path : pathList) {
            if (fs.getFileStatus(path).isDirectory()) {
                ++numDirs;
            } else {
                ++numFiles;
            }
            copyMapper.map(new Text(DistCpUtils.getRelativePath((Path)new Path(SOURCE_PATH), (Path)path)), new CopyListingFileStatus(fs.getFileStatus(path)), context);
        }
        this.verifyCopy((FileSystem)fs, preserveChecksum, true);
        org.junit.jupiter.api.Assertions.assertEquals((long)numFiles, (long)stubContext.getReporter().getCounter((Enum)CopyMapper.Counter.COPY).getValue());
        org.junit.jupiter.api.Assertions.assertEquals((long)numDirs, (long)stubContext.getReporter().getCounter((Enum)CopyMapper.Counter.DIR_COPY).getValue());
        if (!preserveChecksum) {
            org.junit.jupiter.api.Assertions.assertEquals((long)(nFiles * 1024), (long)stubContext.getReporter().getCounter((Enum)CopyMapper.Counter.BYTESCOPIED).getValue());
        } else {
            org.junit.jupiter.api.Assertions.assertEquals((long)((long)nFiles * 4096L * 2L), (long)stubContext.getReporter().getCounter((Enum)CopyMapper.Counter.BYTESCOPIED).getValue());
        }
        this.testCopyingExistingFiles((FileSystem)fs, copyMapper, context);
        for (Text value : stubContext.getWriter().values()) {
            org.junit.jupiter.api.Assertions.assertTrue((boolean)value.toString().startsWith("SKIP:"), (String)(value.toString() + " is not skipped"));
        }
    }

    private void verifyCopy(FileSystem fs, boolean preserveChecksum, boolean preserveReplication) throws Exception {
        for (Path path : pathList) {
            Path targetPath = new Path(path.toString().replaceAll(SOURCE_PATH, TARGET_PATH));
            org.junit.jupiter.api.Assertions.assertTrue((boolean)fs.exists(targetPath));
            org.junit.jupiter.api.Assertions.assertTrue((fs.isFile(targetPath) == fs.isFile(path) ? 1 : 0) != 0);
            FileStatus sourceStatus = fs.getFileStatus(path);
            FileStatus targetStatus = fs.getFileStatus(targetPath);
            if (preserveReplication) {
                org.junit.jupiter.api.Assertions.assertEquals((short)sourceStatus.getReplication(), (short)targetStatus.getReplication());
            }
            if (preserveChecksum) {
                org.junit.jupiter.api.Assertions.assertEquals((long)sourceStatus.getBlockSize(), (long)targetStatus.getBlockSize());
            }
            org.junit.jupiter.api.Assertions.assertTrue((!fs.isFile(targetPath) || fs.getFileChecksum(targetPath).equals((Object)fs.getFileChecksum(path)) ? 1 : 0) != 0);
        }
    }

    private void testCopyingExistingFiles(FileSystem fs, CopyMapper copyMapper, Mapper.Context context) {
        try {
            for (Path path : pathList) {
                copyMapper.map(new Text(DistCpUtils.getRelativePath((Path)new Path(SOURCE_PATH), (Path)path)), new CopyListingFileStatus(fs.getFileStatus(path)), context);
            }
            org.junit.jupiter.api.Assertions.assertEquals((long)nFiles, (long)context.getCounter((Enum)CopyMapper.Counter.SKIP).getValue());
        }
        catch (Exception exception) {
            org.junit.jupiter.api.Assertions.assertTrue((boolean)false, (String)("Caught unexpected exception:" + exception.getMessage()));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    @Timeout(value=40L)
    public void testCopyWhileAppend() throws Exception {
        TestCopyMapper.deleteState();
        TestCopyMapper.mkdirs("/tmp/source/1");
        TestCopyMapper.touchFile("/tmp/source/1/3");
        CopyMapper copyMapper = new CopyMapper();
        StubContext stubContext = new StubContext(TestCopyMapper.getConfiguration(), null, 0);
        Mapper.Context context = stubContext.getContext();
        copyMapper.setup(context);
        final Path path = new Path("/tmp/source/1/3");
        int manyBytes = 100000000;
        TestCopyMapper.appendFile(path, manyBytes);
        ScheduledExecutorService scheduledExecutorService = Executors.newSingleThreadScheduledExecutor();
        Runnable task = new Runnable(){

            @Override
            public void run() {
                try {
                    int maxAppendAttempts = 20;
                    for (int appendCount = 0; appendCount < maxAppendAttempts; ++appendCount) {
                        TestCopyMapper.appendFile(path, 1000);
                        Thread.sleep(200L);
                    }
                }
                catch (IOException | InterruptedException e) {
                    LOG.error("Exception encountered ", (Throwable)e);
                    org.junit.jupiter.api.Assertions.fail((String)("Test failed: " + e.getMessage()));
                }
            }
        };
        scheduledExecutorService.schedule(task, 10L, TimeUnit.MILLISECONDS);
        try {
            copyMapper.map(new Text(DistCpUtils.getRelativePath((Path)new Path(SOURCE_PATH), (Path)path)), new CopyListingFileStatus(cluster.getFileSystem().getFileStatus(path)), context);
        }
        catch (Exception ex) {
            LOG.error("Exception encountered ", (Throwable)ex);
            String exceptionAsString = StringUtils.stringifyException((Throwable)ex);
            if (exceptionAsString.contains("Mismatch in length of source:") || exceptionAsString.contains("Checksum mismatch between ")) {
                org.junit.jupiter.api.Assertions.fail((String)("Test failed: " + exceptionAsString));
            }
        }
        finally {
            scheduledExecutorService.shutdown();
        }
    }

    @Test
    @Timeout(value=40L)
    public void testMakeDirFailure() {
        try {
            TestCopyMapper.deleteState();
            TestCopyMapper.createSourceData();
            DistributedFileSystem fs = cluster.getFileSystem();
            CopyMapper copyMapper = new CopyMapper();
            StubContext stubContext = new StubContext(TestCopyMapper.getConfiguration(), null, 0);
            Mapper.Context context = stubContext.getContext();
            Configuration configuration = context.getConfiguration();
            String workPath = new Path("webhdfs://localhost:1234/*/*/*/?/").makeQualified(fs.getUri(), fs.getWorkingDirectory()).toString();
            configuration.set("distcp.target.work.path", workPath);
            copyMapper.setup(context);
            copyMapper.map(new Text(DistCpUtils.getRelativePath((Path)new Path(SOURCE_PATH), (Path)pathList.get(0))), new CopyListingFileStatus(fs.getFileStatus(pathList.get(0))), context);
            org.junit.jupiter.api.Assertions.assertTrue((boolean)false, (String)"There should have been an exception.");
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    @Test
    @Timeout(value=40L)
    public void testIgnoreFailures() {
        this.doTestIgnoreFailures(true);
        this.doTestIgnoreFailures(false);
        this.doTestIgnoreFailuresDoubleWrapped(true);
        this.doTestIgnoreFailuresDoubleWrapped(false);
    }

    @Test
    @Timeout(value=40L)
    public void testDirToFile() {
        try {
            TestCopyMapper.deleteState();
            TestCopyMapper.createSourceData();
            DistributedFileSystem fs = cluster.getFileSystem();
            CopyMapper copyMapper = new CopyMapper();
            StubContext stubContext = new StubContext(TestCopyMapper.getConfiguration(), null, 0);
            Mapper.Context context = stubContext.getContext();
            TestCopyMapper.mkdirs("/tmp/source/src/file");
            TestCopyMapper.touchFile("/tmp/target/src/file");
            try {
                copyMapper.setup(context);
                copyMapper.map(new Text("/src/file"), new CopyListingFileStatus(fs.getFileStatus(new Path("/tmp/source/src/file"))), context);
            }
            catch (IOException e) {
                org.junit.jupiter.api.Assertions.assertTrue((boolean)e.getMessage().startsWith("Can't replace"));
            }
        }
        catch (Exception e) {
            LOG.error("Exception encountered ", (Throwable)e);
            org.junit.jupiter.api.Assertions.fail((String)("Test failed: " + e.getMessage()));
        }
    }

    @Test
    @Timeout(value=40L)
    public void testPreserve() {
        try {
            TestCopyMapper.deleteState();
            TestCopyMapper.createSourceData();
            UserGroupInformation tmpUser = UserGroupInformation.createRemoteUser((String)"guest");
            final CopyMapper copyMapper = new CopyMapper();
            final Mapper.Context context = (Mapper.Context)tmpUser.doAs((PrivilegedAction)new PrivilegedAction<Mapper.Context>(){

                @Override
                public Mapper.Context run() {
                    try {
                        StubContext stubContext = new StubContext(TestCopyMapper.getConfiguration(), null, 0);
                        return stubContext.getContext();
                    }
                    catch (Exception e) {
                        LOG.error("Exception encountered ", (Throwable)e);
                        throw new RuntimeException(e);
                    }
                }
            });
            EnumSet<DistCpOptions.FileAttribute> preserveStatus = EnumSet.allOf(DistCpOptions.FileAttribute.class);
            preserveStatus.remove(DistCpOptions.FileAttribute.ACL);
            preserveStatus.remove(DistCpOptions.FileAttribute.XATTR);
            context.getConfiguration().set("distcp.preserve.status", DistCpUtils.packAttributes(preserveStatus));
            TestCopyMapper.touchFile("/tmp/source/src/file");
            TestCopyMapper.mkdirs(TARGET_PATH);
            cluster.getFileSystem().setPermission(new Path(TARGET_PATH), new FsPermission(511));
            final FileSystem tmpFS = (FileSystem)tmpUser.doAs((PrivilegedAction)new PrivilegedAction<FileSystem>(){

                @Override
                public FileSystem run() {
                    try {
                        return FileSystem.get((Configuration)cluster.getConfiguration(0));
                    }
                    catch (IOException e) {
                        LOG.error("Exception encountered ", (Throwable)e);
                        org.junit.jupiter.api.Assertions.fail((String)("Test failed: " + e.getMessage()));
                        throw new RuntimeException("Test ought to fail here");
                    }
                }
            });
            tmpUser.doAs((PrivilegedAction)new PrivilegedAction<Integer>(){

                @Override
                public Integer run() {
                    try {
                        copyMapper.setup(context);
                        copyMapper.map(new Text("/src/file"), new CopyListingFileStatus(tmpFS.getFileStatus(new Path("/tmp/source/src/file"))), context);
                        org.junit.jupiter.api.Assertions.fail((String)"Expected copy to fail");
                    }
                    catch (AccessControlException e) {
                        org.junit.jupiter.api.Assertions.assertTrue((boolean)true, (String)("Got exception: " + e.getMessage()));
                    }
                    catch (Exception e) {
                        throw new RuntimeException(e);
                    }
                    return null;
                }
            });
        }
        catch (Exception e) {
            LOG.error("Exception encountered ", (Throwable)e);
            org.junit.jupiter.api.Assertions.fail((String)("Test failed: " + e.getMessage()));
        }
    }

    @Test
    @Timeout(value=40L)
    public void testCopyReadableFiles() {
        try {
            TestCopyMapper.deleteState();
            TestCopyMapper.createSourceData();
            UserGroupInformation tmpUser = UserGroupInformation.createRemoteUser((String)"guest");
            final CopyMapper copyMapper = new CopyMapper();
            final Mapper.Context context = (Mapper.Context)tmpUser.doAs((PrivilegedAction)new PrivilegedAction<Mapper.Context>(){

                @Override
                public Mapper.Context run() {
                    try {
                        StubContext stubContext = new StubContext(TestCopyMapper.getConfiguration(), null, 0);
                        return stubContext.getContext();
                    }
                    catch (Exception e) {
                        LOG.error("Exception encountered ", (Throwable)e);
                        throw new RuntimeException(e);
                    }
                }
            });
            TestCopyMapper.touchFile("/tmp/source/src/file");
            TestCopyMapper.mkdirs(TARGET_PATH);
            cluster.getFileSystem().setPermission(new Path("/tmp/source/src/file"), new FsPermission(FsAction.READ, FsAction.READ, FsAction.READ));
            cluster.getFileSystem().setPermission(new Path(TARGET_PATH), new FsPermission(511));
            final FileSystem tmpFS = (FileSystem)tmpUser.doAs((PrivilegedAction)new PrivilegedAction<FileSystem>(){

                @Override
                public FileSystem run() {
                    try {
                        return FileSystem.get((Configuration)cluster.getConfiguration(0));
                    }
                    catch (IOException e) {
                        LOG.error("Exception encountered ", (Throwable)e);
                        org.junit.jupiter.api.Assertions.fail((String)("Test failed: " + e.getMessage()));
                        throw new RuntimeException("Test ought to fail here");
                    }
                }
            });
            tmpUser.doAs((PrivilegedAction)new PrivilegedAction<Integer>(){

                @Override
                public Integer run() {
                    try {
                        copyMapper.setup(context);
                        copyMapper.map(new Text("/src/file"), new CopyListingFileStatus(tmpFS.getFileStatus(new Path("/tmp/source/src/file"))), context);
                    }
                    catch (Exception e) {
                        throw new RuntimeException(e);
                    }
                    return null;
                }
            });
        }
        catch (Exception e) {
            LOG.error("Exception encountered ", (Throwable)e);
            org.junit.jupiter.api.Assertions.fail((String)("Test failed: " + e.getMessage()));
        }
    }

    @Test
    @Timeout(value=40L)
    public void testSkipCopyNoPerms() {
        try {
            TestCopyMapper.deleteState();
            TestCopyMapper.createSourceData();
            UserGroupInformation tmpUser = UserGroupInformation.createRemoteUser((String)"guest");
            final CopyMapper copyMapper = new CopyMapper();
            final StubContext stubContext = (StubContext)tmpUser.doAs((PrivilegedAction)new PrivilegedAction<StubContext>(){

                @Override
                public StubContext run() {
                    try {
                        return new StubContext(TestCopyMapper.getConfiguration(), null, 0);
                    }
                    catch (Exception e) {
                        LOG.error("Exception encountered ", (Throwable)e);
                        throw new RuntimeException(e);
                    }
                }
            });
            final Mapper.Context context = stubContext.getContext();
            EnumSet<DistCpOptions.FileAttribute> preserveStatus = EnumSet.allOf(DistCpOptions.FileAttribute.class);
            preserveStatus.remove(DistCpOptions.FileAttribute.ACL);
            preserveStatus.remove(DistCpOptions.FileAttribute.XATTR);
            preserveStatus.remove(DistCpOptions.FileAttribute.TIMES);
            context.getConfiguration().set("distcp.preserve.status", DistCpUtils.packAttributes(preserveStatus));
            TestCopyMapper.touchFile("/tmp/source/src/file");
            TestCopyMapper.touchFile("/tmp/target/src/file");
            cluster.getFileSystem().setPermission(new Path("/tmp/source/src/file"), new FsPermission(FsAction.READ, FsAction.READ, FsAction.READ));
            cluster.getFileSystem().setPermission(new Path("/tmp/target/src/file"), new FsPermission(FsAction.READ, FsAction.READ, FsAction.READ));
            final FileSystem tmpFS = (FileSystem)tmpUser.doAs((PrivilegedAction)new PrivilegedAction<FileSystem>(){

                @Override
                public FileSystem run() {
                    try {
                        return FileSystem.get((Configuration)cluster.getConfiguration(0));
                    }
                    catch (IOException e) {
                        LOG.error("Exception encountered ", (Throwable)e);
                        org.junit.jupiter.api.Assertions.fail((String)("Test failed: " + e.getMessage()));
                        throw new RuntimeException("Test ought to fail here");
                    }
                }
            });
            tmpUser.doAs((PrivilegedAction)new PrivilegedAction<Integer>(){

                @Override
                public Integer run() {
                    try {
                        copyMapper.setup(context);
                        copyMapper.map(new Text("/src/file"), new CopyListingFileStatus(tmpFS.getFileStatus(new Path("/tmp/source/src/file"))), context);
                        Assertions.assertThat((int)stubContext.getWriter().values().size()).isEqualTo(1);
                        org.junit.jupiter.api.Assertions.assertTrue((boolean)stubContext.getWriter().values().get(0).toString().startsWith("SKIP"));
                        org.junit.jupiter.api.Assertions.assertTrue((boolean)stubContext.getWriter().values().get(0).toString().contains("/tmp/source/src/file"));
                    }
                    catch (Exception e) {
                        throw new RuntimeException(e);
                    }
                    return null;
                }
            });
        }
        catch (Exception e) {
            LOG.error("Exception encountered ", (Throwable)e);
            org.junit.jupiter.api.Assertions.fail((String)("Test failed: " + e.getMessage()));
        }
    }

    @Test
    @Timeout(value=40L)
    public void testFailCopyWithAccessControlException() {
        try {
            TestCopyMapper.deleteState();
            TestCopyMapper.createSourceData();
            UserGroupInformation tmpUser = UserGroupInformation.createRemoteUser((String)"guest");
            final CopyMapper copyMapper = new CopyMapper();
            StubContext stubContext = (StubContext)tmpUser.doAs((PrivilegedAction)new PrivilegedAction<StubContext>(){

                @Override
                public StubContext run() {
                    try {
                        return new StubContext(TestCopyMapper.getConfiguration(), null, 0);
                    }
                    catch (Exception e) {
                        LOG.error("Exception encountered ", (Throwable)e);
                        throw new RuntimeException(e);
                    }
                }
            });
            EnumSet<DistCpOptions.FileAttribute> preserveStatus = EnumSet.allOf(DistCpOptions.FileAttribute.class);
            preserveStatus.remove(DistCpOptions.FileAttribute.ACL);
            preserveStatus.remove(DistCpOptions.FileAttribute.XATTR);
            final Mapper.Context context = stubContext.getContext();
            context.getConfiguration().set("distcp.preserve.status", DistCpUtils.packAttributes(preserveStatus));
            TestCopyMapper.touchFile("/tmp/source/src/file");
            FSDataOutputStream out = cluster.getFileSystem().create(new Path("/tmp/target/src/file"));
            out.write("hello world".getBytes());
            out.close();
            cluster.getFileSystem().setPermission(new Path("/tmp/source/src/file"), new FsPermission(FsAction.READ, FsAction.READ, FsAction.READ));
            cluster.getFileSystem().setPermission(new Path("/tmp/target/src/file"), new FsPermission(FsAction.READ, FsAction.READ, FsAction.READ));
            final FileSystem tmpFS = (FileSystem)tmpUser.doAs((PrivilegedAction)new PrivilegedAction<FileSystem>(){

                @Override
                public FileSystem run() {
                    try {
                        return FileSystem.get((Configuration)cluster.getConfiguration(0));
                    }
                    catch (IOException e) {
                        LOG.error("Exception encountered ", (Throwable)e);
                        org.junit.jupiter.api.Assertions.fail((String)("Test failed: " + e.getMessage()));
                        throw new RuntimeException("Test ought to fail here");
                    }
                }
            });
            tmpUser.doAs((PrivilegedAction)new PrivilegedAction<Integer>(){

                @Override
                public Integer run() {
                    block3: {
                        try {
                            copyMapper.setup(context);
                            copyMapper.map(new Text("/src/file"), new CopyListingFileStatus(tmpFS.getFileStatus(new Path("/tmp/source/src/file"))), context);
                            org.junit.jupiter.api.Assertions.fail((String)"Didn't expect the file to be copied");
                        }
                        catch (AccessControlException accessControlException) {
                        }
                        catch (Exception e) {
                            if (e.getCause() != null && e.getCause().getCause() != null && e.getCause().getCause() instanceof AccessControlException) break block3;
                            throw new RuntimeException(e);
                        }
                    }
                    return null;
                }
            });
        }
        catch (Exception e) {
            LOG.error("Exception encountered ", (Throwable)e);
            org.junit.jupiter.api.Assertions.fail((String)("Test failed: " + e.getMessage()));
        }
    }

    @Test
    @Timeout(value=40L)
    public void testFileToDir() {
        try {
            TestCopyMapper.deleteState();
            TestCopyMapper.createSourceData();
            DistributedFileSystem fs = cluster.getFileSystem();
            CopyMapper copyMapper = new CopyMapper();
            StubContext stubContext = new StubContext(TestCopyMapper.getConfiguration(), null, 0);
            Mapper.Context context = stubContext.getContext();
            TestCopyMapper.touchFile("/tmp/source/src/file");
            TestCopyMapper.mkdirs("/tmp/target/src/file");
            try {
                copyMapper.setup(context);
                copyMapper.map(new Text("/src/file"), new CopyListingFileStatus(fs.getFileStatus(new Path("/tmp/source/src/file"))), context);
            }
            catch (IOException e) {
                org.junit.jupiter.api.Assertions.assertTrue((boolean)e.getMessage().startsWith("Can't replace"));
            }
        }
        catch (Exception e) {
            LOG.error("Exception encountered ", (Throwable)e);
            org.junit.jupiter.api.Assertions.fail((String)("Test failed: " + e.getMessage()));
        }
    }

    private void doTestIgnoreFailures(boolean ignoreFailures) {
        try {
            TestCopyMapper.deleteState();
            TestCopyMapper.createSourceData();
            DistributedFileSystem fs = cluster.getFileSystem();
            CopyMapper copyMapper = new CopyMapper();
            StubContext stubContext = new StubContext(TestCopyMapper.getConfiguration(), null, 0);
            Mapper.Context context = stubContext.getContext();
            Configuration configuration = context.getConfiguration();
            configuration.setBoolean(DistCpOptionSwitch.IGNORE_FAILURES.getConfigLabel(), ignoreFailures);
            configuration.setBoolean(DistCpOptionSwitch.OVERWRITE.getConfigLabel(), true);
            configuration.setBoolean(DistCpOptionSwitch.SKIP_CRC.getConfigLabel(), true);
            copyMapper.setup(context);
            for (Path path : pathList) {
                FileStatus fileStatus = fs.getFileStatus(path);
                if (fileStatus.isDirectory()) continue;
                fs.delete(path, true);
                copyMapper.map(new Text(DistCpUtils.getRelativePath((Path)new Path(SOURCE_PATH), (Path)path)), new CopyListingFileStatus(fileStatus), context);
            }
            if (ignoreFailures) {
                for (Text value : stubContext.getWriter().values()) {
                    org.junit.jupiter.api.Assertions.assertTrue((boolean)value.toString().startsWith("FAIL:"), (String)(value.toString() + " is not skipped"));
                }
            }
            org.junit.jupiter.api.Assertions.assertTrue((boolean)ignoreFailures, (String)"There should have been an exception.");
        }
        catch (Exception e) {
            org.junit.jupiter.api.Assertions.assertTrue((!ignoreFailures ? 1 : 0) != 0, (String)("Unexpected exception: " + e.getMessage()));
            e.printStackTrace();
        }
    }

    private void doTestIgnoreFailuresDoubleWrapped(final boolean ignoreFailures) {
        try {
            TestCopyMapper.deleteState();
            TestCopyMapper.createSourceData();
            UserGroupInformation tmpUser = UserGroupInformation.createRemoteUser((String)"guest");
            final CopyMapper copyMapper = new CopyMapper();
            final Mapper.Context context = (Mapper.Context)tmpUser.doAs((PrivilegedAction)new PrivilegedAction<Mapper.Context>(){

                @Override
                public Mapper.Context run() {
                    try {
                        StubContext stubContext = new StubContext(TestCopyMapper.getConfiguration(), null, 0);
                        return stubContext.getContext();
                    }
                    catch (Exception e) {
                        LOG.error("Exception encountered when get stub context", (Throwable)e);
                        throw new RuntimeException(e);
                    }
                }
            });
            TestCopyMapper.touchFile("/tmp/source/src/file");
            TestCopyMapper.mkdirs(TARGET_PATH);
            cluster.getFileSystem().setPermission(new Path("/tmp/source/src/file"), new FsPermission(FsAction.NONE, FsAction.NONE, FsAction.NONE));
            cluster.getFileSystem().setPermission(new Path(TARGET_PATH), new FsPermission(511));
            context.getConfiguration().setBoolean(DistCpOptionSwitch.IGNORE_FAILURES.getConfigLabel(), ignoreFailures);
            final FileSystem tmpFS = (FileSystem)tmpUser.doAs((PrivilegedAction)new PrivilegedAction<FileSystem>(){

                @Override
                public FileSystem run() {
                    try {
                        return FileSystem.get((Configuration)cluster.getConfiguration(0));
                    }
                    catch (IOException e) {
                        LOG.error("Exception encountered when get FileSystem.", (Throwable)e);
                        throw new RuntimeException(e);
                    }
                }
            });
            tmpUser.doAs((PrivilegedAction)new PrivilegedAction<Integer>(){

                @Override
                public Integer run() {
                    try {
                        copyMapper.setup(context);
                        copyMapper.map(new Text("/src/file"), new CopyListingFileStatus(tmpFS.getFileStatus(new Path("/tmp/source/src/file"))), context);
                        org.junit.jupiter.api.Assertions.assertTrue((boolean)ignoreFailures, (String)"Should have thrown an IOException if not ignoring failures");
                    }
                    catch (IOException e) {
                        LOG.error("Unexpected exception encountered. ", (Throwable)e);
                        org.junit.jupiter.api.Assertions.assertFalse((boolean)ignoreFailures, (String)"Should not have thrown an IOException if ignoring failures");
                    }
                    catch (Exception e) {
                        LOG.error("Exception encountered when the mapper copies file.", (Throwable)e);
                        throw new RuntimeException(e);
                    }
                    return null;
                }
            });
        }
        catch (Exception e) {
            LOG.error("Unexpected exception encountered. ", (Throwable)e);
            org.junit.jupiter.api.Assertions.fail((String)("Test failed: " + e.getMessage()));
        }
    }

    private static void deleteState() throws IOException {
        pathList.clear();
        nFiles = 0;
        cluster.getFileSystem().delete(new Path(SOURCE_PATH), true);
        cluster.getFileSystem().delete(new Path(TARGET_PATH), true);
    }

    @Test
    @Timeout(value=40L)
    public void testPreserveBlockSizeAndReplication() {
        this.testPreserveBlockSizeAndReplicationImpl(true);
        this.testPreserveBlockSizeAndReplicationImpl(false);
    }

    @Test
    @Timeout(value=40L)
    public void testCopyWithDifferentBlockSizes() throws Exception {
        try {
            TestCopyMapper.deleteState();
            TestCopyMapper.createSourceDataWithDifferentBlockSize();
            DistributedFileSystem fs = cluster.getFileSystem();
            CopyMapper copyMapper = new CopyMapper();
            StubContext stubContext = new StubContext(TestCopyMapper.getConfiguration(), null, 0);
            Mapper.Context context = stubContext.getContext();
            Configuration configuration = context.getConfiguration();
            EnumSet<DistCpOptions.FileAttribute> fileAttributes = EnumSet.noneOf(DistCpOptions.FileAttribute.class);
            configuration.set(DistCpOptionSwitch.PRESERVE_STATUS.getConfigLabel(), DistCpUtils.packAttributes(fileAttributes));
            copyMapper.setup(context);
            for (Path path : pathList) {
                FileStatus fileStatus = fs.getFileStatus(path);
                copyMapper.map(new Text(DistCpUtils.getRelativePath((Path)new Path(SOURCE_PATH), (Path)path)), new CopyListingFileStatus(fileStatus), context);
            }
            if (this.expectDifferentBlockSizesMultipleBlocksToSucceed()) {
                this.verifyCopy((FileSystem)fs, false, false);
            } else {
                org.junit.jupiter.api.Assertions.fail((String)"Copy should have failed because of block-size difference.");
            }
        }
        catch (Exception exception) {
            if (this.expectDifferentBlockSizesMultipleBlocksToSucceed()) {
                throw exception;
            }
            Throwable cause = exception.getCause().getCause();
            GenericTestUtils.assertExceptionContains((String)"-pb", (Throwable)cause);
            GenericTestUtils.assertExceptionContains((String)"-skipcrccheck", (Throwable)cause);
        }
    }

    @Test
    @Timeout(value=40L)
    public void testCopyWithDifferentBytesPerCrc() throws Exception {
        try {
            TestCopyMapper.deleteState();
            TestCopyMapper.createSourceDataWithDifferentBytesPerCrc();
            DistributedFileSystem fs = cluster.getFileSystem();
            CopyMapper copyMapper = new CopyMapper();
            StubContext stubContext = new StubContext(TestCopyMapper.getConfiguration(), null, 0);
            Mapper.Context context = stubContext.getContext();
            Configuration configuration = context.getConfiguration();
            EnumSet<DistCpOptions.FileAttribute> fileAttributes = EnumSet.noneOf(DistCpOptions.FileAttribute.class);
            configuration.set(DistCpOptionSwitch.PRESERVE_STATUS.getConfigLabel(), DistCpUtils.packAttributes(fileAttributes));
            copyMapper.setup(context);
            for (Path path : pathList) {
                FileStatus fileStatus = fs.getFileStatus(path);
                copyMapper.map(new Text(DistCpUtils.getRelativePath((Path)new Path(SOURCE_PATH), (Path)path)), new CopyListingFileStatus(fileStatus), context);
            }
            if (this.expectDifferentBytesPerCrcToSucceed()) {
                this.verifyCopy((FileSystem)fs, false, false);
            } else {
                org.junit.jupiter.api.Assertions.fail((String)"Copy should have failed because of bytes-per-crc difference.");
            }
        }
        catch (Exception exception) {
            if (this.expectDifferentBytesPerCrcToSucceed()) {
                throw exception;
            }
            Throwable cause = exception.getCause().getCause();
            GenericTestUtils.assertExceptionContains((String)"mismatch", (Throwable)cause);
        }
    }

    private void testPreserveBlockSizeAndReplicationImpl(boolean preserve) {
        try {
            TestCopyMapper.deleteState();
            TestCopyMapper.createSourceData();
            DistributedFileSystem fs = cluster.getFileSystem();
            CopyMapper copyMapper = new CopyMapper();
            StubContext stubContext = new StubContext(TestCopyMapper.getConfiguration(), null, 0);
            Mapper.Context context = stubContext.getContext();
            Configuration configuration = context.getConfiguration();
            EnumSet<DistCpOptions.FileAttribute> fileAttributes = EnumSet.noneOf(DistCpOptions.FileAttribute.class);
            if (preserve) {
                fileAttributes.add(DistCpOptions.FileAttribute.BLOCKSIZE);
                fileAttributes.add(DistCpOptions.FileAttribute.REPLICATION);
            }
            configuration.set(DistCpOptionSwitch.PRESERVE_STATUS.getConfigLabel(), DistCpUtils.packAttributes(fileAttributes));
            copyMapper.setup(context);
            for (Path path : pathList) {
                FileStatus fileStatus = fs.getFileStatus(path);
                copyMapper.map(new Text(DistCpUtils.getRelativePath((Path)new Path(SOURCE_PATH), (Path)path)), new CopyListingFileStatus(fileStatus), context);
            }
            for (Path path : pathList) {
                Path targetPath = new Path(path.toString().replaceAll(SOURCE_PATH, TARGET_PATH));
                FileStatus source = fs.getFileStatus(path);
                FileStatus target = fs.getFileStatus(targetPath);
                if (source.isDirectory()) continue;
                org.junit.jupiter.api.Assertions.assertTrue((preserve || source.getBlockSize() != target.getBlockSize() ? 1 : 0) != 0);
                org.junit.jupiter.api.Assertions.assertTrue((preserve || source.getReplication() != target.getReplication() ? 1 : 0) != 0);
                org.junit.jupiter.api.Assertions.assertTrue((!preserve || source.getBlockSize() == target.getBlockSize() ? 1 : 0) != 0);
                org.junit.jupiter.api.Assertions.assertTrue((!preserve || source.getReplication() == target.getReplication() ? 1 : 0) != 0);
            }
        }
        catch (Exception e) {
            org.junit.jupiter.api.Assertions.assertTrue((boolean)false, (String)("Unexpected exception: " + e.getMessage()));
            e.printStackTrace();
        }
    }

    private static void changeUserGroup(String user, String group) throws IOException {
        DistributedFileSystem fs = cluster.getFileSystem();
        FsPermission changedPermission = new FsPermission(FsAction.ALL, FsAction.ALL, FsAction.ALL);
        for (Path path : pathList) {
            if (!fs.isFile(path)) continue;
            fs.setOwner(path, user, group);
            fs.setPermission(path, changedPermission);
        }
    }

    @Test
    @Timeout(value=40L)
    public void testSingleFileCopy() {
        try {
            TestCopyMapper.deleteState();
            TestCopyMapper.touchFile("/tmp/source/1");
            Path sourceFilePath = pathList.get(0);
            Path targetFilePath = new Path(sourceFilePath.toString().replaceAll(SOURCE_PATH, TARGET_PATH));
            TestCopyMapper.touchFile(targetFilePath.toString());
            DistributedFileSystem fs = cluster.getFileSystem();
            CopyMapper copyMapper = new CopyMapper();
            StubContext stubContext = new StubContext(TestCopyMapper.getConfiguration(), null, 0);
            Mapper.Context context = stubContext.getContext();
            context.getConfiguration().set("distcp.target.final.path", targetFilePath.getParent().toString());
            copyMapper.setup(context);
            CopyListingFileStatus sourceFileStatus = new CopyListingFileStatus(fs.getFileStatus(sourceFilePath));
            long before = fs.getFileStatus(targetFilePath).getModificationTime();
            copyMapper.map(new Text(DistCpUtils.getRelativePath((Path)new Path(SOURCE_PATH), (Path)sourceFilePath)), sourceFileStatus, context);
            long after = fs.getFileStatus(targetFilePath).getModificationTime();
            org.junit.jupiter.api.Assertions.assertTrue((before == after ? 1 : 0) != 0, (String)"File should have been skipped");
            context.getConfiguration().set("distcp.target.final.path", targetFilePath.toString());
            copyMapper.setup(context);
            before = fs.getFileStatus(targetFilePath).getModificationTime();
            try {
                Thread.sleep(2L);
            }
            catch (Throwable throwable) {
                // empty catch block
            }
            copyMapper.map(new Text(DistCpUtils.getRelativePath((Path)new Path(SOURCE_PATH), (Path)sourceFilePath)), sourceFileStatus, context);
            after = fs.getFileStatus(targetFilePath).getModificationTime();
            org.junit.jupiter.api.Assertions.assertTrue((before < after ? 1 : 0) != 0, (String)"File should have been overwritten.");
        }
        catch (Exception exception) {
            org.junit.jupiter.api.Assertions.fail((String)("Unexpected exception: " + exception.getMessage()));
            exception.printStackTrace();
        }
    }

    @Test
    @Timeout(value=40L)
    public void testPreserveUserGroup() {
        this.testPreserveUserGroupImpl(true);
        this.testPreserveUserGroupImpl(false);
    }

    private void testPreserveUserGroupImpl(boolean preserve) {
        try {
            TestCopyMapper.deleteState();
            TestCopyMapper.createSourceData();
            TestCopyMapper.changeUserGroup("Michael", "Corleone");
            DistributedFileSystem fs = cluster.getFileSystem();
            CopyMapper copyMapper = new CopyMapper();
            StubContext stubContext = new StubContext(TestCopyMapper.getConfiguration(), null, 0);
            Mapper.Context context = stubContext.getContext();
            Configuration configuration = context.getConfiguration();
            EnumSet<DistCpOptions.FileAttribute> fileAttributes = EnumSet.noneOf(DistCpOptions.FileAttribute.class);
            if (preserve) {
                fileAttributes.add(DistCpOptions.FileAttribute.USER);
                fileAttributes.add(DistCpOptions.FileAttribute.GROUP);
                fileAttributes.add(DistCpOptions.FileAttribute.PERMISSION);
            }
            configuration.set(DistCpOptionSwitch.PRESERVE_STATUS.getConfigLabel(), DistCpUtils.packAttributes(fileAttributes));
            copyMapper.setup(context);
            for (Path path : pathList) {
                FileStatus fileStatus = fs.getFileStatus(path);
                copyMapper.map(new Text(DistCpUtils.getRelativePath((Path)new Path(SOURCE_PATH), (Path)path)), new CopyListingFileStatus(fileStatus), context);
            }
            for (Path path : pathList) {
                Path targetPath = new Path(path.toString().replaceAll(SOURCE_PATH, TARGET_PATH));
                FileStatus source = fs.getFileStatus(path);
                FileStatus target = fs.getFileStatus(targetPath);
                if (source.isDirectory()) continue;
                org.junit.jupiter.api.Assertions.assertTrue((!preserve || source.getOwner().equals(target.getOwner()) ? 1 : 0) != 0);
                org.junit.jupiter.api.Assertions.assertTrue((!preserve || source.getGroup().equals(target.getGroup()) ? 1 : 0) != 0);
                org.junit.jupiter.api.Assertions.assertTrue((!preserve || source.getPermission().equals((Object)target.getPermission()) ? 1 : 0) != 0);
                org.junit.jupiter.api.Assertions.assertTrue((preserve || !source.getOwner().equals(target.getOwner()) ? 1 : 0) != 0);
                org.junit.jupiter.api.Assertions.assertTrue((preserve || !source.getGroup().equals(target.getGroup()) ? 1 : 0) != 0);
                org.junit.jupiter.api.Assertions.assertTrue((preserve || !source.getPermission().equals((Object)target.getPermission()) ? 1 : 0) != 0);
                org.junit.jupiter.api.Assertions.assertTrue((source.isDirectory() || source.getReplication() != target.getReplication() ? 1 : 0) != 0);
            }
        }
        catch (Exception e) {
            org.junit.jupiter.api.Assertions.assertTrue((boolean)false, (String)("Unexpected exception: " + e.getMessage()));
            e.printStackTrace();
        }
    }

    /*
     * WARNING - void declaration
     */
    @Test
    public void testVerboseLogging() throws Exception {
        void var7_15;
        TestCopyMapper.deleteState();
        TestCopyMapper.createSourceData();
        DistributedFileSystem fs = cluster.getFileSystem();
        CopyMapper copyMapper = new CopyMapper();
        StubContext stubContext = new StubContext(TestCopyMapper.getConfiguration(), null, 0);
        Mapper.Context context = stubContext.getContext();
        copyMapper.setup(context);
        int numFiles = 0;
        for (Path path : pathList) {
            if (fs.getFileStatus(path).isFile()) {
                ++numFiles;
            }
            copyMapper.map(new Text(DistCpUtils.getRelativePath((Path)new Path(SOURCE_PATH), (Path)path)), new CopyListingFileStatus(fs.getFileStatus(path)), context);
        }
        org.junit.jupiter.api.Assertions.assertEquals((long)numFiles, (long)stubContext.getReporter().getCounter((Enum)CopyMapper.Counter.COPY).getValue());
        this.testCopyingExistingFiles((FileSystem)fs, copyMapper, context);
        for (Text text : stubContext.getWriter().values()) {
            org.junit.jupiter.api.Assertions.assertTrue((!text.toString().startsWith("FILE_COPIED:") ? 1 : 0) != 0);
            org.junit.jupiter.api.Assertions.assertTrue((!text.toString().startsWith("FILE_SKIPPED:") ? 1 : 0) != 0);
        }
        TestCopyMapper.deleteState();
        TestCopyMapper.createSourceData();
        stubContext = new StubContext(TestCopyMapper.getConfiguration(), null, 0);
        context = stubContext.getContext();
        copyMapper.setup(context);
        context.getConfiguration().setBoolean(DistCpOptionSwitch.VERBOSE_LOG.getConfigLabel(), true);
        copyMapper.setup(context);
        for (Path path : pathList) {
            copyMapper.map(new Text(DistCpUtils.getRelativePath((Path)new Path(SOURCE_PATH), (Path)path)), new CopyListingFileStatus(fs.getFileStatus(path)), context);
        }
        org.junit.jupiter.api.Assertions.assertEquals((long)numFiles, (long)stubContext.getReporter().getCounter((Enum)CopyMapper.Counter.COPY).getValue());
        int numFileCopied = 0;
        for (Text value : stubContext.getWriter().values()) {
            if (!value.toString().startsWith("FILE_COPIED:")) continue;
            ++numFileCopied;
        }
        org.junit.jupiter.api.Assertions.assertEquals((int)numFiles, (int)numFileCopied);
        boolean bl = false;
        this.testCopyingExistingFiles((FileSystem)fs, copyMapper, context);
        for (Text value : stubContext.getWriter().values()) {
            if (!value.toString().startsWith("FILE_SKIPPED:")) continue;
            ++var7_15;
        }
        org.junit.jupiter.api.Assertions.assertEquals((int)numFiles, (int)var7_15);
    }
}

