/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.tools;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.EnumMap;
import java.util.HashSet;
import java.util.List;
import java.util.Random;
import org.apache.hadoop.HadoopIllegalArgumentException;
import org.apache.hadoop.classification.VisibleForTesting;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.DFSUtilClient;
import org.apache.hadoop.hdfs.protocol.SnapshotDiffReport;
import org.apache.hadoop.tools.CopyFilter;
import org.apache.hadoop.tools.CopyListing;
import org.apache.hadoop.tools.DiffInfo;
import org.apache.hadoop.tools.DistCp;
import org.apache.hadoop.tools.DistCpContext;

class DistCpSync {
    private DistCpContext context;
    private Configuration conf;
    private EnumMap<SnapshotDiffReport.DiffType, List<DiffInfo>> diffMap;
    private DiffInfo[] renameDiffs;
    private List<DiffInfo> deletedByExclusionDiffs;
    private CopyFilter copyFilter;

    DistCpSync(DistCpContext context, Configuration conf) {
        this.context = context;
        this.conf = conf;
        this.copyFilter = CopyFilter.getCopyFilter(conf);
        this.copyFilter.initialize();
    }

    @VisibleForTesting
    public void setCopyFilter(CopyFilter copyFilter) {
        this.copyFilter = copyFilter;
    }

    private boolean isRdiff() {
        return this.context.shouldUseRdiff();
    }

    private boolean preSyncCheck() throws IOException {
        List<Path> sourcePaths = this.context.getSourcePaths();
        if (sourcePaths.size() != 1) {
            throw new IllegalArgumentException(sourcePaths.size() + " source paths are provided");
        }
        Path sourceDir = sourcePaths.get(0);
        Path targetDir = this.context.getTargetPath();
        FileSystem srcFs = sourceDir.getFileSystem(this.conf);
        FileSystem tgtFs = targetDir.getFileSystem(this.conf);
        FileSystem snapshotDiffFs = this.isRdiff() ? tgtFs : srcFs;
        Path snapshotDiffDir = this.isRdiff() ? targetDir : sourceDir;
        this.checkFilesystemSupport(sourceDir, targetDir, srcFs, tgtFs);
        if (!this.checkNoChange(tgtFs, targetDir)) {
            this.context.setSourcePaths(Arrays.asList(this.getSnapshotPath(sourceDir, this.context.getToSnapshot())));
            return false;
        }
        String from = this.getSnapshotName(this.context.getFromSnapshot());
        String to = this.getSnapshotName(this.context.getToSnapshot());
        try {
            FileStatus fromSnapshotStat = snapshotDiffFs.getFileStatus(this.getSnapshotPath(snapshotDiffDir, from));
            FileStatus toSnapshotStat = snapshotDiffFs.getFileStatus(this.getSnapshotPath(snapshotDiffDir, to));
            if (this.isRdiff()) {
                if (!from.equals("") && fromSnapshotStat.getModificationTime() < toSnapshotStat.getModificationTime()) {
                    throw new HadoopIllegalArgumentException("Snapshot " + from + " should be newer than " + to);
                }
            } else if (!to.equals("") && fromSnapshotStat.getModificationTime() > toSnapshotStat.getModificationTime()) {
                throw new HadoopIllegalArgumentException("Snapshot " + to + " should be newer than " + from);
            }
        }
        catch (FileNotFoundException nfe) {
            throw new CopyListing.InvalidInputException("Input snapshot is not found", nfe);
        }
        return true;
    }

    private void checkFilesystemSupport(Path sourceDir, Path targetDir, FileSystem srcFs, FileSystem tgtFs) throws IOException {
        if (!srcFs.hasPathCapability(sourceDir, "fs.capability.paths.snapshots")) {
            throw new UnsupportedOperationException("The source file system " + srcFs.getScheme() + " does not support snapshot.");
        }
        if (!tgtFs.hasPathCapability(targetDir, "fs.capability.paths.snapshots")) {
            throw new UnsupportedOperationException("The target file system " + tgtFs.getScheme() + " does not support snapshot.");
        }
        try {
            DistCpSync.getSnapshotDiffReportMethod(srcFs);
        }
        catch (NoSuchMethodException e) {
            throw new UnsupportedOperationException("The source file system " + srcFs.getScheme() + " does not support getSnapshotDiffReport", e);
        }
        try {
            DistCpSync.getSnapshotDiffReportMethod(tgtFs);
        }
        catch (NoSuchMethodException e) {
            throw new UnsupportedOperationException("The target file system " + tgtFs.getScheme() + " does not support getSnapshotDiffReport", e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean sync() throws IOException {
        boolean bl;
        if (!this.preSyncCheck()) {
            return false;
        }
        if (!this.getAllDiffs()) {
            return false;
        }
        List<Path> sourcePaths = this.context.getSourcePaths();
        Path sourceDir = sourcePaths.get(0);
        Path targetDir = this.context.getTargetPath();
        FileSystem tfs = targetDir.getFileSystem(this.conf);
        Path tmpDir = null;
        try {
            tmpDir = this.createTargetTmpDir(tfs, targetDir);
            DiffInfo[] renameAndDeleteDiffs = this.getRenameAndDeleteDiffsForSync(targetDir);
            if (renameAndDeleteDiffs.length > 0) {
                this.syncDiff(renameAndDeleteDiffs, tfs, tmpDir);
            }
            bl = true;
            this.deleteTargetTmpDir(tfs, tmpDir);
        }
        catch (Exception e) {
            boolean bl2;
            try {
                DistCp.LOG.warn("Failed to use snapshot diff for distcp", (Throwable)e);
                bl2 = false;
                this.deleteTargetTmpDir(tfs, tmpDir);
            }
            catch (Throwable throwable) {
                this.deleteTargetTmpDir(tfs, tmpDir);
                this.context.setSourcePaths(Arrays.asList(this.getSnapshotPath(sourceDir, this.context.getToSnapshot())));
                throw throwable;
            }
            this.context.setSourcePaths(Arrays.asList(this.getSnapshotPath(sourceDir, this.context.getToSnapshot())));
            return bl2;
        }
        this.context.setSourcePaths(Arrays.asList(this.getSnapshotPath(sourceDir, this.context.getToSnapshot())));
        return bl;
    }

    private boolean getAllDiffs() throws IOException {
        Path ssDir = this.isRdiff() ? this.context.getTargetPath() : this.context.getSourcePaths().get(0);
        try {
            String from = this.getSnapshotName(this.context.getFromSnapshot());
            String to = this.getSnapshotName(this.context.getToSnapshot());
            SnapshotDiffReport report = DistCpSync.getSnapshotDiffReport(ssDir.getFileSystem(this.conf), ssDir, from, to);
            this.diffMap = new EnumMap(SnapshotDiffReport.DiffType.class);
            for (SnapshotDiffReport.DiffType type : SnapshotDiffReport.DiffType.values()) {
                this.diffMap.put(type, new ArrayList());
            }
            this.deletedByExclusionDiffs = null;
            for (SnapshotDiffReport.DiffReportEntry entry : report.getDiffList()) {
                if (entry.getSourcePath().length <= 0) continue;
                SnapshotDiffReport.DiffType dt = entry.getType();
                List<DiffInfo> list = this.diffMap.get(dt);
                Path source = new Path(DFSUtilClient.bytes2String((byte[])entry.getSourcePath()));
                Path relativeSource = new Path("/" + source);
                if (dt == SnapshotDiffReport.DiffType.MODIFY || dt == SnapshotDiffReport.DiffType.CREATE || dt == SnapshotDiffReport.DiffType.DELETE) {
                    if (!this.copyFilter.shouldCopy(relativeSource)) continue;
                    list.add(new DiffInfo(source, null, dt));
                    continue;
                }
                if (dt != SnapshotDiffReport.DiffType.RENAME) continue;
                Path target = new Path(DFSUtilClient.bytes2String((byte[])entry.getTargetPath()));
                Path relativeTarget = new Path("/" + target);
                if (this.copyFilter.shouldCopy(relativeSource)) {
                    if (this.copyFilter.shouldCopy(relativeTarget)) {
                        list.add(new DiffInfo(source, target, dt));
                        continue;
                    }
                    list = this.diffMap.get(SnapshotDiffReport.DiffType.DELETE);
                    DiffInfo info = new DiffInfo(source, null, SnapshotDiffReport.DiffType.DELETE);
                    list.add(info);
                    if (this.deletedByExclusionDiffs == null) {
                        this.deletedByExclusionDiffs = new ArrayList<DiffInfo>();
                    }
                    this.deletedByExclusionDiffs.add(info);
                    continue;
                }
                if (!this.copyFilter.shouldCopy(relativeTarget)) continue;
                list = this.diffMap.get(SnapshotDiffReport.DiffType.CREATE);
                list.add(new DiffInfo(target, null, SnapshotDiffReport.DiffType.CREATE));
            }
            if (this.deletedByExclusionDiffs != null) {
                Collections.sort(this.deletedByExclusionDiffs, DiffInfo.sourceComparator);
            }
            return true;
        }
        catch (IOException e) {
            DistCp.LOG.warn("Failed to compute snapshot diff on " + ssDir, (Throwable)e);
            this.diffMap = null;
            return false;
        }
    }

    private static Method getSnapshotDiffReportMethod(FileSystem fs) throws NoSuchMethodException {
        return fs.getClass().getMethod("getSnapshotDiffReport", Path.class, String.class, String.class);
    }

    private static SnapshotDiffReport getSnapshotDiffReport(FileSystem fs, Path snapshotDir, String fromSnapshot, String toSnapshot) throws IOException {
        try {
            return (SnapshotDiffReport)DistCpSync.getSnapshotDiffReportMethod(fs).invoke((Object)fs, snapshotDir, fromSnapshot, toSnapshot);
        }
        catch (InvocationTargetException e) {
            throw new IOException(e.getCause());
        }
        catch (IllegalAccessException | NoSuchMethodException e) {
            throw new IllegalArgumentException("Failed to invoke getSnapshotDiffReport.", e);
        }
    }

    private String getSnapshotName(String name) {
        return ".".equals(name) ? "" : name;
    }

    private Path getSnapshotPath(Path inputDir, String snapshotName) {
        if (".".equals(snapshotName)) {
            return inputDir;
        }
        return new Path(inputDir, ".snapshot/" + snapshotName);
    }

    private Path createTargetTmpDir(FileSystem targetFs, Path targetDir) throws IOException {
        Path tmp = new Path(targetDir, ".distcp.diff.tmp" + DistCp.rand.nextInt());
        if (!targetFs.mkdirs(tmp)) {
            throw new IOException("The tmp directory " + tmp + " already exists");
        }
        return tmp;
    }

    private void deleteTargetTmpDir(FileSystem targetFs, Path tmpDir) {
        try {
            if (tmpDir != null) {
                targetFs.delete(tmpDir, true);
            }
        }
        catch (IOException e) {
            DistCp.LOG.error("Unable to cleanup tmp dir: " + tmpDir, (Throwable)e);
        }
    }

    private boolean checkNoChange(FileSystem fs, Path path) {
        try {
            String from = this.getSnapshotName(this.context.getFromSnapshot());
            SnapshotDiffReport targetDiff = DistCpSync.getSnapshotDiffReport(fs, path, from, "");
            if (!targetDiff.getDiffList().isEmpty()) {
                DistCp.LOG.warn("The target has been modified since snapshot " + this.context.getFromSnapshot());
                return false;
            }
            return true;
        }
        catch (IOException e) {
            DistCp.LOG.warn("Failed to compute snapshot diff on " + path + " at snapshot " + this.context.getFromSnapshot(), (Throwable)e);
            return false;
        }
    }

    private void syncDiff(DiffInfo[] diffs, FileSystem targetFs, Path tmpDir) throws IOException {
        this.moveToTmpDir(diffs, targetFs, tmpDir);
        this.moveToTarget(diffs, targetFs);
    }

    private void moveToTmpDir(DiffInfo[] diffs, FileSystem targetFs, Path tmpDir) throws IOException {
        Arrays.sort(diffs, DiffInfo.sourceComparator);
        Random random = new Random();
        for (DiffInfo diff : diffs) {
            Path tmpTarget = new Path(tmpDir, diff.getSource().getName());
            while (targetFs.exists(tmpTarget)) {
                tmpTarget = new Path(tmpDir, diff.getSource().getName() + random.nextInt());
            }
            diff.setTmp(tmpTarget);
            targetFs.rename(diff.getSource(), tmpTarget);
        }
    }

    private void moveToTarget(DiffInfo[] diffs, FileSystem targetFs) throws IOException {
        Arrays.sort(diffs, DiffInfo.targetComparator);
        for (DiffInfo diff : diffs) {
            if (diff.getTarget() == null) continue;
            targetFs.mkdirs(diff.getTarget().getParent());
            targetFs.rename(diff.getTmp(), diff.getTarget());
        }
    }

    private DiffInfo[] getRenameAndDeleteDiffsForSync(Path targetDir) {
        if (this.isRdiff()) {
            return this.getRenameAndDeleteDiffsRdiff(targetDir);
        }
        return this.getRenameAndDeleteDiffsFdiff(targetDir);
    }

    private DiffInfo[] getRenameAndDeleteDiffsRdiff(Path targetDir) {
        List<DiffInfo> renameDiffsList = this.diffMap.get(SnapshotDiffReport.DiffType.RENAME);
        ArrayList<DiffInfo> renameDiffsListReversed = new ArrayList<DiffInfo>(renameDiffsList.size());
        for (DiffInfo diff : renameDiffsList) {
            renameDiffsListReversed.add(new DiffInfo(diff.getTarget(), diff.getSource(), diff.getType()));
        }
        DiffInfo[] renameDiffArray = renameDiffsListReversed.toArray(new DiffInfo[renameDiffsList.size()]);
        Arrays.sort(renameDiffArray, DiffInfo.sourceComparator);
        ArrayList<DiffInfo> renameAndDeleteDiff = new ArrayList<DiffInfo>();
        for (DiffInfo diff : this.diffMap.get(SnapshotDiffReport.DiffType.DELETE)) {
            DiffInfo renameItem = this.getRenameItem(diff, renameDiffArray);
            Path source = renameItem != null ? new Path(targetDir, this.translateRenamedPath(diff.getSource(), renameItem)) : new Path(targetDir, diff.getSource());
            renameAndDeleteDiff.add(new DiffInfo(source, null, SnapshotDiffReport.DiffType.DELETE));
        }
        for (DiffInfo diff : this.diffMap.get(SnapshotDiffReport.DiffType.RENAME)) {
            Path source = new Path(targetDir, diff.getSource());
            Path target = new Path(targetDir, diff.getTarget());
            renameAndDeleteDiff.add(new DiffInfo(source, target, diff.getType()));
        }
        return renameAndDeleteDiff.toArray(new DiffInfo[renameAndDeleteDiff.size()]);
    }

    private DiffInfo[] getRenameAndDeleteDiffsFdiff(Path targetDir) {
        Path source;
        ArrayList<DiffInfo> renameAndDeleteDiff = new ArrayList<DiffInfo>();
        for (DiffInfo diff : this.diffMap.get(SnapshotDiffReport.DiffType.DELETE)) {
            source = new Path(targetDir, diff.getSource());
            renameAndDeleteDiff.add(new DiffInfo(source, diff.getTarget(), diff.getType()));
        }
        for (DiffInfo diff : this.diffMap.get(SnapshotDiffReport.DiffType.RENAME)) {
            source = new Path(targetDir, diff.getSource());
            Path target = new Path(targetDir, diff.getTarget());
            renameAndDeleteDiff.add(new DiffInfo(source, target, diff.getType()));
        }
        return renameAndDeleteDiff.toArray(new DiffInfo[renameAndDeleteDiff.size()]);
    }

    private DiffInfo[] getCreateAndModifyDiffs() {
        List<DiffInfo> createDiff = this.diffMap.get(SnapshotDiffReport.DiffType.CREATE);
        List<DiffInfo> modifyDiff = this.diffMap.get(SnapshotDiffReport.DiffType.MODIFY);
        ArrayList<DiffInfo> diffs = new ArrayList<DiffInfo>(createDiff.size() + modifyDiff.size());
        diffs.addAll(createDiff);
        diffs.addAll(modifyDiff);
        return diffs.toArray(new DiffInfo[diffs.size()]);
    }

    private boolean isParentOf(Path parent, Path child) {
        Object parentPath = parent.toString();
        String childPath = child.toString();
        if (!((String)parentPath).endsWith("/")) {
            parentPath = (String)parentPath + "/";
        }
        return childPath.length() > ((String)parentPath).length() && childPath.startsWith((String)parentPath);
    }

    private DiffInfo getRenameItem(DiffInfo diff, DiffInfo[] renameDiffArray) {
        for (DiffInfo renameItem : renameDiffArray) {
            if (!(diff.getSource().equals((Object)renameItem.getSource()) ? diff.getType() == SnapshotDiffReport.DiffType.MODIFY : this.isParentOf(renameItem.getSource(), diff.getSource()))) continue;
            return renameItem;
        }
        return null;
    }

    private boolean isParentOrSelfMarkedDeleted(DiffInfo diff, List<DiffInfo> deletedDirDiffArray) {
        for (DiffInfo item : deletedDirDiffArray) {
            if (!(item.getSource().equals((Object)diff.getSource()) ? diff.getType() == SnapshotDiffReport.DiffType.MODIFY : this.isParentOf(item.getSource(), diff.getSource()))) continue;
            return true;
        }
        return false;
    }

    private Path translateRenamedPath(Path sourcePath, DiffInfo renameItem) {
        if (sourcePath.equals((Object)renameItem.getSource())) {
            return renameItem.getTarget();
        }
        StringBuffer sb = new StringBuffer(sourcePath.toString());
        String remain = sb.substring(renameItem.getSource().toString().length() + 1);
        return new Path(renameItem.getTarget(), remain);
    }

    public ArrayList<DiffInfo> prepareDiffListForCopyListing() {
        DiffInfo[] modifyAndCreateDiffs = this.getCreateAndModifyDiffs();
        ArrayList<DiffInfo> finalListWithTarget = new ArrayList<DiffInfo>();
        if (this.isRdiff()) {
            for (DiffInfo diff : modifyAndCreateDiffs) {
                diff.setTarget(diff.getSource());
                finalListWithTarget.add(diff);
            }
        } else {
            List<DiffInfo> renameDiffsList = this.diffMap.get(SnapshotDiffReport.DiffType.RENAME);
            DiffInfo[] renameDiffArray = renameDiffsList.toArray(new DiffInfo[renameDiffsList.size()]);
            Arrays.sort(renameDiffArray, DiffInfo.sourceComparator);
            for (DiffInfo diff : modifyAndCreateDiffs) {
                if (this.deletedByExclusionDiffs != null && this.isParentOrSelfMarkedDeleted(diff, this.deletedByExclusionDiffs)) continue;
                DiffInfo renameItem = this.getRenameItem(diff, renameDiffArray);
                if (renameItem == null) {
                    diff.setTarget(diff.getSource());
                } else {
                    diff.setTarget(this.translateRenamedPath(diff.getSource(), renameItem));
                }
                finalListWithTarget.add(diff);
            }
        }
        return finalListWithTarget;
    }

    public HashSet<String> getTraverseExcludeList(Path newDir, Path prefix) {
        if (this.renameDiffs == null) {
            List<DiffInfo> renameList = this.diffMap.get(SnapshotDiffReport.DiffType.RENAME);
            this.renameDiffs = renameList.toArray(new DiffInfo[renameList.size()]);
            Arrays.sort(this.renameDiffs, DiffInfo.targetComparator);
        }
        if (this.renameDiffs.length <= 0) {
            return null;
        }
        boolean foundChild = false;
        HashSet<String> excludeList = new HashSet<String>();
        for (DiffInfo diff : this.renameDiffs) {
            if (this.isParentOf(newDir, diff.getTarget())) {
                foundChild = true;
                excludeList.add(new Path(prefix, diff.getTarget()).toUri().getPath());
                continue;
            }
            if (foundChild) break;
        }
        return excludeList;
    }
}

