/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.util;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.nio.file.NoSuchFileException;
import java.util.Map;
import org.apache.hadoop.test.GenericTestUtils;
import org.apache.hadoop.util.HostsFileReader;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

public class TestHostsFileReader {
    final String HOSTS_TEST_DIR = GenericTestUtils.getTestDir().getAbsolutePath();
    File EXCLUDES_FILE = new File(this.HOSTS_TEST_DIR, "dfs.exclude");
    File INCLUDES_FILE = new File(this.HOSTS_TEST_DIR, "dfs.include");
    String excludesFile = this.HOSTS_TEST_DIR + "/dfs.exclude";
    String includesFile = this.HOSTS_TEST_DIR + "/dfs.include";
    private String excludesXmlFile = this.HOSTS_TEST_DIR + "/dfs.exclude.xml";

    @BeforeEach
    public void setUp() throws Exception {
    }

    @AfterEach
    public void tearDown() throws Exception {
        this.EXCLUDES_FILE.delete();
        this.INCLUDES_FILE.delete();
    }

    @Test
    public void testHostsFileReader() throws Exception {
        FileWriter efw = new FileWriter(this.excludesFile);
        FileWriter ifw = new FileWriter(this.includesFile);
        efw.write("#DFS-Hosts-excluded\n");
        efw.write("somehost1\n");
        efw.write("#This-is-comment\n");
        efw.write("somehost2\n");
        efw.write("somehost3 # host3\n");
        efw.write("somehost4\n");
        efw.write("somehost4 somehost5\n");
        efw.close();
        ifw.write("#Hosts-in-DFS\n");
        ifw.write("somehost1\n");
        ifw.write("somehost2\n");
        ifw.write("somehost3\n");
        ifw.write("#This-is-comment\n");
        ifw.write("somehost4 # host4\n");
        ifw.write("somehost4 somehost5\n");
        ifw.close();
        HostsFileReader hfp = new HostsFileReader(this.includesFile, this.excludesFile);
        int includesLen = hfp.getHosts().size();
        int excludesLen = hfp.getExcludedHosts().size();
        Assertions.assertEquals((int)5, (int)includesLen);
        Assertions.assertEquals((int)5, (int)excludesLen);
        Assertions.assertTrue((boolean)hfp.getHosts().contains("somehost5"));
        Assertions.assertFalse((boolean)hfp.getHosts().contains("host3"));
        Assertions.assertTrue((boolean)hfp.getExcludedHosts().contains("somehost5"));
        Assertions.assertFalse((boolean)hfp.getExcludedHosts().contains("host4"));
        String newExcludesFile = this.HOSTS_TEST_DIR + "/dfs1.exclude";
        String newIncludesFile = this.HOSTS_TEST_DIR + "/dfs1.include";
        efw = new FileWriter(newExcludesFile);
        ifw = new FileWriter(newIncludesFile);
        efw.write("#DFS-Hosts-excluded\n");
        efw.write("node1\n");
        efw.close();
        ifw.write("#Hosts-in-DFS\n");
        ifw.write("node2\n");
        ifw.close();
        hfp.refresh(newIncludesFile, newExcludesFile);
        Assertions.assertTrue((boolean)hfp.getExcludedHosts().contains("node1"));
        Assertions.assertTrue((boolean)hfp.getHosts().contains("node2"));
        HostsFileReader.HostDetails hostDetails = hfp.getHostDetails();
        Assertions.assertTrue((boolean)hostDetails.getExcludedHosts().contains("node1"));
        Assertions.assertTrue((boolean)hostDetails.getIncludedHosts().contains("node2"));
        Assertions.assertEquals((Object)newIncludesFile, (Object)hostDetails.getIncludesFile());
        Assertions.assertEquals((Object)newExcludesFile, (Object)hostDetails.getExcludesFile());
    }

    @Test
    public void testCreateHostFileReaderWithNonexistentFile() throws Exception {
        try {
            new HostsFileReader(this.HOSTS_TEST_DIR + "/doesnt-exist", this.HOSTS_TEST_DIR + "/doesnt-exist");
            Assertions.fail((String)"Should throw NoSuchFileException");
        }
        catch (NoSuchFileException noSuchFileException) {
            // empty catch block
        }
    }

    @Test
    public void testRefreshHostFileReaderWithNonexistentFile() throws Exception {
        FileWriter efw = new FileWriter(this.excludesFile);
        FileWriter ifw = new FileWriter(this.includesFile);
        efw.close();
        ifw.close();
        HostsFileReader hfp = new HostsFileReader(this.includesFile, this.excludesFile);
        Assertions.assertTrue((boolean)this.INCLUDES_FILE.delete());
        try {
            hfp.refresh();
            Assertions.fail((String)"Should throw NoSuchFileException");
        }
        catch (NoSuchFileException noSuchFileException) {
            // empty catch block
        }
    }

    @Test
    public void testHostFileReaderWithNull() throws Exception {
        FileWriter efw = new FileWriter(this.excludesFile);
        FileWriter ifw = new FileWriter(this.includesFile);
        efw.close();
        ifw.close();
        HostsFileReader hfp = new HostsFileReader(this.includesFile, this.excludesFile);
        int includesLen = hfp.getHosts().size();
        int excludesLen = hfp.getExcludedHosts().size();
        Assertions.assertEquals((int)0, (int)includesLen);
        Assertions.assertEquals((int)0, (int)excludesLen);
        Assertions.assertFalse((boolean)hfp.getHosts().contains("somehost5"));
        Assertions.assertFalse((boolean)hfp.getExcludedHosts().contains("somehost5"));
    }

    @Test
    public void testHostFileReaderWithCommentsOnly() throws Exception {
        FileWriter efw = new FileWriter(this.excludesFile);
        FileWriter ifw = new FileWriter(this.includesFile);
        efw.write("#DFS-Hosts-excluded\n");
        efw.close();
        ifw.write("#Hosts-in-DFS\n");
        ifw.close();
        HostsFileReader hfp = new HostsFileReader(this.includesFile, this.excludesFile);
        int includesLen = hfp.getHosts().size();
        int excludesLen = hfp.getExcludedHosts().size();
        Assertions.assertEquals((int)0, (int)includesLen);
        Assertions.assertEquals((int)0, (int)excludesLen);
        Assertions.assertFalse((boolean)hfp.getHosts().contains("somehost5"));
        Assertions.assertFalse((boolean)hfp.getExcludedHosts().contains("somehost5"));
    }

    @Test
    public void testHostFileReaderWithSpaces() throws Exception {
        FileWriter efw = new FileWriter(this.excludesFile);
        FileWriter ifw = new FileWriter(this.includesFile);
        efw.write("#DFS-Hosts-excluded\n");
        efw.write("   somehost somehost2");
        efw.write("   somehost3 # somehost4");
        efw.close();
        ifw.write("#Hosts-in-DFS\n");
        ifw.write("   somehost somehost2");
        ifw.write("   somehost3 # somehost4");
        ifw.close();
        HostsFileReader hfp = new HostsFileReader(this.includesFile, this.excludesFile);
        int includesLen = hfp.getHosts().size();
        int excludesLen = hfp.getExcludedHosts().size();
        Assertions.assertEquals((int)3, (int)includesLen);
        Assertions.assertEquals((int)3, (int)excludesLen);
        Assertions.assertTrue((boolean)hfp.getHosts().contains("somehost3"));
        Assertions.assertFalse((boolean)hfp.getHosts().contains("somehost5"));
        Assertions.assertFalse((boolean)hfp.getHosts().contains("somehost4"));
        Assertions.assertTrue((boolean)hfp.getExcludedHosts().contains("somehost3"));
        Assertions.assertFalse((boolean)hfp.getExcludedHosts().contains("somehost5"));
        Assertions.assertFalse((boolean)hfp.getExcludedHosts().contains("somehost4"));
    }

    @Test
    public void testHostFileReaderWithTabs() throws Exception {
        FileWriter efw = new FileWriter(this.excludesFile);
        FileWriter ifw = new FileWriter(this.includesFile);
        efw.write("#DFS-Hosts-excluded\n");
        efw.write("     \n");
        efw.write("   somehost \t somehost2 \n somehost4");
        efw.write("   somehost3 \t # somehost5");
        efw.close();
        ifw.write("#Hosts-in-DFS\n");
        ifw.write("     \n");
        ifw.write("   somehost \t  somehost2 \n somehost4");
        ifw.write("   somehost3 \t # somehost5");
        ifw.close();
        HostsFileReader hfp = new HostsFileReader(this.includesFile, this.excludesFile);
        int includesLen = hfp.getHosts().size();
        int excludesLen = hfp.getExcludedHosts().size();
        Assertions.assertEquals((int)4, (int)includesLen);
        Assertions.assertEquals((int)4, (int)excludesLen);
        Assertions.assertTrue((boolean)hfp.getHosts().contains("somehost2"));
        Assertions.assertFalse((boolean)hfp.getHosts().contains("somehost5"));
        Assertions.assertTrue((boolean)hfp.getExcludedHosts().contains("somehost2"));
        Assertions.assertFalse((boolean)hfp.getExcludedHosts().contains("somehost5"));
    }

    @Test
    public void testHostFileReaderWithTimeout() throws Exception {
        FileWriter efw = new FileWriter(this.excludesXmlFile);
        FileWriter ifw = new FileWriter(this.includesFile);
        efw.write("<?xml version=\"1.0\"?>\n");
        efw.write("<!-- yarn.nodes.exclude -->\n");
        efw.write("<hosts>\n");
        efw.write("<host><name>host1</name></host>\n");
        efw.write("<host><name>host2</name><timeout>123</timeout></host>\n");
        efw.write("<host><name>host3</name><timeout>-1</timeout></host>\n");
        efw.write("<host><name>10000</name></host>\n");
        efw.write("<host><name>10001</name><timeout>123</timeout></host>\n");
        efw.write("<host><name>10002</name><timeout>-1</timeout></host>\n");
        efw.write("<host><name>host4,host5, host6</name><timeout>1800</timeout></host>\n");
        efw.write("</hosts>\n");
        efw.close();
        ifw.write("#Hosts-in-DFS\n");
        ifw.write("     \n");
        ifw.write("   somehost \t  somehost2 \n somehost4");
        ifw.write("   somehost3 \t # somehost5");
        ifw.close();
        HostsFileReader hfp = new HostsFileReader(this.includesFile, this.excludesXmlFile);
        int includesLen = hfp.getHosts().size();
        int excludesLen = hfp.getExcludedHosts().size();
        Assertions.assertEquals((int)4, (int)includesLen);
        Assertions.assertEquals((int)9, (int)excludesLen);
        HostsFileReader.HostDetails hostDetails = hfp.getHostDetails();
        Map excludes = hostDetails.getExcludedMap();
        Assertions.assertTrue((boolean)excludes.containsKey("host1"));
        Assertions.assertTrue((boolean)excludes.containsKey("host2"));
        Assertions.assertTrue((boolean)excludes.containsKey("host3"));
        Assertions.assertTrue((boolean)excludes.containsKey("10000"));
        Assertions.assertTrue((boolean)excludes.containsKey("10001"));
        Assertions.assertTrue((boolean)excludes.containsKey("10002"));
        Assertions.assertTrue((boolean)excludes.containsKey("host4"));
        Assertions.assertTrue((boolean)excludes.containsKey("host5"));
        Assertions.assertTrue((boolean)excludes.containsKey("host6"));
        Assertions.assertTrue((excludes.get("host1") == null ? 1 : 0) != 0);
        Assertions.assertTrue(((Integer)excludes.get("host2") == 123 ? 1 : 0) != 0);
        Assertions.assertTrue(((Integer)excludes.get("host3") == -1 ? 1 : 0) != 0);
        Assertions.assertTrue((excludes.get("10000") == null ? 1 : 0) != 0);
        Assertions.assertTrue(((Integer)excludes.get("10001") == 123 ? 1 : 0) != 0);
        Assertions.assertTrue(((Integer)excludes.get("10002") == -1 ? 1 : 0) != 0);
        Assertions.assertTrue(((Integer)excludes.get("host4") == 1800 ? 1 : 0) != 0);
        Assertions.assertTrue(((Integer)excludes.get("host5") == 1800 ? 1 : 0) != 0);
        Assertions.assertTrue(((Integer)excludes.get("host6") == 1800 ? 1 : 0) != 0);
    }

    @Test
    public void testLazyRefresh() throws IOException {
        FileWriter efw = new FileWriter(this.excludesFile);
        FileWriter ifw = new FileWriter(this.includesFile);
        efw.write("host1\n");
        efw.write("host2\n");
        efw.close();
        ifw.write("host3\n");
        ifw.write("host4\n");
        ifw.close();
        HostsFileReader hfp = new HostsFileReader(this.includesFile, this.excludesFile);
        ifw = new FileWriter(this.includesFile);
        ifw.close();
        efw = new FileWriter(this.excludesFile, true);
        efw.write("host3\n");
        efw.write("host4\n");
        efw.close();
        hfp.lazyRefresh(this.includesFile, this.excludesFile);
        HostsFileReader.HostDetails details = hfp.getHostDetails();
        HostsFileReader.HostDetails lazyDetails = hfp.getLazyLoadedHostDetails();
        Assertions.assertEquals((int)2, (int)details.getExcludedHosts().size(), (String)"Details: no. of excluded hosts");
        Assertions.assertEquals((int)2, (int)details.getIncludedHosts().size(), (String)"Details: no. of included hosts");
        Assertions.assertEquals((int)4, (int)lazyDetails.getExcludedHosts().size(), (String)"LazyDetails: no. of excluded hosts");
        Assertions.assertEquals((int)0, (int)lazyDetails.getIncludedHosts().size(), (String)"LayDetails: no. of included hosts");
        hfp.finishRefresh();
        details = hfp.getHostDetails();
        Assertions.assertEquals((int)4, (int)details.getExcludedHosts().size(), (String)"Details: no. of excluded hosts");
        Assertions.assertEquals((int)0, (int)details.getIncludedHosts().size(), (String)"Details: no. of included hosts");
        Assertions.assertNull((Object)hfp.getLazyLoadedHostDetails(), (String)"Lazy host details should be null");
    }

    @Test
    public void testFinishRefreshWithoutLazyRefresh() throws IOException {
        Assertions.assertThrows(IllegalStateException.class, () -> {
            FileWriter efw = new FileWriter(this.excludesFile);
            FileWriter ifw = new FileWriter(this.includesFile);
            efw.close();
            ifw.close();
            HostsFileReader hfp = new HostsFileReader(this.includesFile, this.excludesFile);
            hfp.finishRefresh();
        });
    }
}

