/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.security;

import java.io.File;
import java.io.IOException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.URI;
import java.net.UnknownHostException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.concurrent.TimeUnit;
import javax.security.auth.kerberos.KerberosPrincipal;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.net.NetUtils;
import org.apache.hadoop.security.SecurityUtil;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.security.alias.CredentialProvider;
import org.apache.hadoop.security.alias.CredentialProviderFactory;
import org.apache.hadoop.security.alias.LocalJavaKeyStoreProvider;
import org.apache.hadoop.security.token.Token;
import org.apache.hadoop.thirdparty.com.google.common.io.FileWriteMode;
import org.apache.hadoop.thirdparty.com.google.common.io.Files;
import org.apache.hadoop.util.StringUtils;
import org.apache.hadoop.util.ZKUtil;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Test;
import org.mockito.Mockito;
import org.mockito.verification.VerificationMode;

public class TestSecurityUtil {
    private static final String ZK_AUTH_VALUE = "a_scheme:a_password";

    @BeforeAll
    public static void unsetKerberosRealm() {
        System.setProperty("java.security.krb5.kdc", "");
        System.setProperty("java.security.krb5.realm", "NONE");
    }

    @Test
    public void isOriginalTGTReturnsCorrectValues() {
        Assertions.assertTrue((boolean)SecurityUtil.isTGSPrincipal((KerberosPrincipal)new KerberosPrincipal("krbtgt/foo@foo")));
        Assertions.assertTrue((boolean)SecurityUtil.isTGSPrincipal((KerberosPrincipal)new KerberosPrincipal("krbtgt/foo.bar.bat@foo.bar.bat")));
        Assertions.assertFalse((boolean)SecurityUtil.isTGSPrincipal(null));
        Assertions.assertFalse((boolean)SecurityUtil.isTGSPrincipal((KerberosPrincipal)new KerberosPrincipal("blah")));
        Assertions.assertFalse((boolean)SecurityUtil.isTGSPrincipal((KerberosPrincipal)new KerberosPrincipal("krbtgt/hello")));
        Assertions.assertFalse((boolean)SecurityUtil.isTGSPrincipal((KerberosPrincipal)new KerberosPrincipal("krbtgt/foo@FOO")));
    }

    private void verify(String original, String hostname, String expected) throws IOException {
        Assertions.assertEquals((Object)expected, (Object)SecurityUtil.getServerPrincipal((String)original, (String)hostname));
        InetAddress addr = this.mockAddr(hostname);
        Assertions.assertEquals((Object)expected, (Object)SecurityUtil.getServerPrincipal((String)original, (InetAddress)addr));
    }

    private InetAddress mockAddr(String reverseTo) {
        InetAddress mock = (InetAddress)Mockito.mock(InetAddress.class);
        ((InetAddress)Mockito.doReturn((Object)reverseTo).when((Object)mock)).getCanonicalHostName();
        return mock;
    }

    @Test
    public void testGetServerPrincipal() throws IOException {
        String service = "hdfs/";
        String realm = "@REALM";
        String hostname = "foohost";
        String userPrincipal = "foo@FOOREALM";
        String shouldReplace = service + "_HOST" + realm;
        String replaced = service + hostname + realm;
        this.verify(shouldReplace, hostname, replaced);
        String shouldNotReplace = service + "_HOST" + "NAME" + realm;
        this.verify(shouldNotReplace, hostname, shouldNotReplace);
        this.verify(userPrincipal, hostname, userPrincipal);
        InetAddress notUsed = (InetAddress)Mockito.mock(InetAddress.class);
        Assertions.assertEquals((Object)shouldNotReplace, (Object)SecurityUtil.getServerPrincipal((String)shouldNotReplace, (InetAddress)notUsed));
        ((InetAddress)Mockito.verify((Object)notUsed, (VerificationMode)Mockito.never())).getCanonicalHostName();
    }

    @Test
    public void testPrincipalsWithLowerCaseHosts() throws IOException {
        String service = "xyz/";
        String realm = "@REALM";
        String principalInConf = service + "_HOST" + realm;
        String hostname = "FooHost";
        String principal = service + StringUtils.toLowerCase((String)hostname) + realm;
        this.verify(principalInConf, hostname, principal);
    }

    @Test
    public void testLocalHostNameForNullOrWild() throws Exception {
        String local = StringUtils.toLowerCase((String)SecurityUtil.getLocalHostName(null));
        Assertions.assertEquals((Object)("hdfs/" + local + "@REALM"), (Object)SecurityUtil.getServerPrincipal((String)"hdfs/_HOST@REALM", (String)null));
        Assertions.assertEquals((Object)("hdfs/" + local + "@REALM"), (Object)SecurityUtil.getServerPrincipal((String)"hdfs/_HOST@REALM", (String)"0.0.0.0"));
    }

    @Test
    public void testStartsWithIncorrectSettings() throws IOException {
        Configuration conf = new Configuration();
        SecurityUtil.setAuthenticationMethod((UserGroupInformation.AuthenticationMethod)UserGroupInformation.AuthenticationMethod.KERBEROS, (Configuration)conf);
        String keyTabKey = "key";
        conf.set(keyTabKey, "");
        UserGroupInformation.setConfiguration((Configuration)conf);
        boolean gotException = false;
        try {
            SecurityUtil.login((Configuration)conf, (String)keyTabKey, (String)"", (String)"");
        }
        catch (IOException e) {
            gotException = true;
        }
        Assertions.assertTrue((boolean)gotException, (String)"Exception for empty keytabfile name was expected");
    }

    @Test
    public void testGetHostFromPrincipal() {
        Assertions.assertEquals((Object)"host", (Object)SecurityUtil.getHostFromPrincipal((String)"service/host@realm"));
        Assertions.assertEquals(null, (Object)SecurityUtil.getHostFromPrincipal((String)"service@realm"));
    }

    @Test
    public void testBuildDTServiceName() {
        Configuration conf = new Configuration(false);
        conf.setBoolean("hadoop.security.token.service.use_ip", true);
        SecurityUtil.setConfiguration((Configuration)conf);
        Assertions.assertEquals((Object)"127.0.0.1:123", (Object)SecurityUtil.buildDTServiceName((URI)URI.create("test://LocalHost"), (int)123));
        Assertions.assertEquals((Object)"127.0.0.1:123", (Object)SecurityUtil.buildDTServiceName((URI)URI.create("test://LocalHost:123"), (int)456));
        Assertions.assertEquals((Object)"127.0.0.1:123", (Object)SecurityUtil.buildDTServiceName((URI)URI.create("test://127.0.0.1"), (int)123));
        Assertions.assertEquals((Object)"127.0.0.1:123", (Object)SecurityUtil.buildDTServiceName((URI)URI.create("test://127.0.0.1:123"), (int)456));
    }

    @Test
    public void testBuildTokenServiceSockAddr() {
        Configuration conf = new Configuration(false);
        conf.setBoolean("hadoop.security.token.service.use_ip", true);
        SecurityUtil.setConfiguration((Configuration)conf);
        Assertions.assertEquals((Object)"127.0.0.1:123", (Object)SecurityUtil.buildTokenService((InetSocketAddress)new InetSocketAddress("LocalHost", 123)).toString());
        Assertions.assertEquals((Object)"127.0.0.1:123", (Object)SecurityUtil.buildTokenService((InetSocketAddress)new InetSocketAddress("127.0.0.1", 123)).toString());
        Assertions.assertEquals((Object)"127.0.0.1:123", (Object)SecurityUtil.buildTokenService((InetSocketAddress)NetUtils.createSocketAddr((String)"127.0.0.1", (int)123)).toString());
    }

    @Test
    public void testGoodHostsAndPorts() {
        InetSocketAddress compare = NetUtils.createSocketAddrForHost((String)"localhost", (int)123);
        this.runGoodCases(compare, "localhost", 123);
        this.runGoodCases(compare, "localhost:", 123);
        this.runGoodCases(compare, "localhost:123", 456);
    }

    void runGoodCases(InetSocketAddress addr, String host, int port) {
        Assertions.assertEquals((Object)addr, (Object)NetUtils.createSocketAddr((String)host, (int)port));
        Assertions.assertEquals((Object)addr, (Object)NetUtils.createSocketAddr((String)("hdfs://" + host), (int)port));
        Assertions.assertEquals((Object)addr, (Object)NetUtils.createSocketAddr((String)("hdfs://" + host + "/path"), (int)port));
    }

    @Test
    public void testBadHostsAndPorts() {
        this.runBadCases("", true);
        this.runBadCases(":", false);
        this.runBadCases("hdfs/", false);
        this.runBadCases("hdfs:/", false);
        this.runBadCases("hdfs://", true);
    }

    void runBadCases(String prefix, boolean validIfPosPort) {
        this.runBadPortPermutes(prefix, false);
        this.runBadPortPermutes(prefix + "*", false);
        this.runBadPortPermutes(prefix + "localhost", validIfPosPort);
        this.runBadPortPermutes(prefix + "localhost:-1", false);
        this.runBadPortPermutes(prefix + "localhost:-123", false);
        this.runBadPortPermutes(prefix + "localhost:xyz", false);
        this.runBadPortPermutes(prefix + "localhost/xyz", validIfPosPort);
        this.runBadPortPermutes(prefix + "localhost/:123", validIfPosPort);
        this.runBadPortPermutes(prefix + ":123", false);
        this.runBadPortPermutes(prefix + ":xyz", false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void runBadPortPermutes(String arg, boolean validIfPosPort) {
        int[] ports = new int[]{-123, -1, 123};
        boolean bad = false;
        try {
            NetUtils.createSocketAddr((String)arg);
        }
        catch (IllegalArgumentException e) {
            bad = true;
        }
        finally {
            Assertions.assertTrue((boolean)bad, (String)("should be bad: '" + arg + "'"));
        }
        for (int port : ports) {
            if (validIfPosPort && port > 0) continue;
            bad = false;
            try {
                NetUtils.createSocketAddr((String)arg, (int)port);
            }
            catch (IllegalArgumentException e) {
                bad = true;
            }
            finally {
                Assertions.assertTrue((boolean)bad, (String)("should be bad: '" + arg + "' (default port:" + port + ")"));
            }
        }
    }

    private void verifyValues(InetSocketAddress addr, String host, String ip, int port) {
        Assertions.assertTrue((!addr.isUnresolved() ? 1 : 0) != 0);
        if (!SecurityUtil.useIpForTokenService) {
            Assertions.assertEquals((Object)host, (Object)addr.getHostName());
            Assertions.assertEquals((Object)host, (Object)addr.getAddress().getHostName());
        }
        Assertions.assertEquals((Object)ip, (Object)addr.getAddress().getHostAddress());
        Assertions.assertEquals((int)port, (int)addr.getPort());
    }

    private void verifyTokenService(InetSocketAddress addr, String host, String ip, int port, boolean useIp) {
        Configuration conf = new Configuration(false);
        conf.setBoolean("hadoop.security.token.service.use_ip", useIp);
        SecurityUtil.setConfiguration((Configuration)conf);
        String serviceHost = useIp ? ip : StringUtils.toLowerCase((String)host);
        Token token = new Token();
        Text service = new Text(serviceHost + ":" + port);
        Assertions.assertEquals((Object)service, (Object)SecurityUtil.buildTokenService((InetSocketAddress)addr));
        SecurityUtil.setTokenService((Token)token, (InetSocketAddress)addr);
        Assertions.assertEquals((Object)service, (Object)token.getService());
        InetSocketAddress serviceAddr = SecurityUtil.getTokenServiceAddr((Token)token);
        Assertions.assertNotNull((Object)serviceAddr);
        this.verifyValues(serviceAddr, serviceHost, ip, port);
    }

    private void verifyAddress(InetSocketAddress addr, String host, String ip, int port) {
        this.verifyValues(addr, host, ip, port);
        this.verifyTokenService(addr, host, ip, port, true);
        this.verifyTokenService(addr, host, ip, port, false);
    }

    private void verifyServiceAddr(String host, String ip) {
        int port = 123;
        InetSocketAddress addr = NetUtils.createSocketAddrForHost((String)host, (int)port);
        this.verifyAddress(addr, host, ip, port);
        addr = NetUtils.createSocketAddr((String)(host + ":" + port));
        this.verifyAddress(addr, host, ip, port);
        addr = NetUtils.createSocketAddr((String)(host + ":" + port), (int)(port + 1));
        this.verifyAddress(addr, host, ip, port);
        addr = NetUtils.createSocketAddr((String)host, (int)port);
        this.verifyAddress(addr, host, ip, port);
    }

    @Test
    public void testSocketAddrWithName() {
        String staticHost = "my";
        NetUtils.addStaticResolution((String)staticHost, (String)"localhost");
        this.verifyServiceAddr("LocalHost", "127.0.0.1");
    }

    @Test
    public void testSocketAddrWithIP() {
        String staticHost = "127.0.0.1";
        NetUtils.addStaticResolution((String)staticHost, (String)"localhost");
        this.verifyServiceAddr(staticHost, "127.0.0.1");
    }

    @Test
    public void testSocketAddrWithNameToStaticName() {
        String staticHost = "host1";
        NetUtils.addStaticResolution((String)staticHost, (String)"localhost");
        this.verifyServiceAddr(staticHost, "127.0.0.1");
    }

    @Test
    public void testSocketAddrWithNameToStaticIP() {
        String staticHost = "host3";
        NetUtils.addStaticResolution((String)staticHost, (String)"255.255.255.255");
        this.verifyServiceAddr(staticHost, "255.255.255.255");
    }

    @Test
    public void testSocketAddrWithChangeIP() {
        String staticHost = "host4";
        NetUtils.addStaticResolution((String)staticHost, (String)"255.255.255.255");
        this.verifyServiceAddr(staticHost, "255.255.255.255");
        NetUtils.addStaticResolution((String)staticHost, (String)"255.255.255.254");
        this.verifyServiceAddr(staticHost, "255.255.255.254");
    }

    @Test
    public void testSocketAddrWithIPToStaticIP() {
        String staticHost = "1.2.3.4";
        NetUtils.addStaticResolution((String)staticHost, (String)"255.255.255.255");
        this.verifyServiceAddr(staticHost, "255.255.255.255");
    }

    @Test
    public void testGetAuthenticationMethod() {
        Configuration conf = new Configuration();
        conf.unset("hadoop.security.authentication");
        Assertions.assertEquals((Object)UserGroupInformation.AuthenticationMethod.SIMPLE, (Object)SecurityUtil.getAuthenticationMethod((Configuration)conf));
        conf.set("hadoop.security.authentication", "simple");
        Assertions.assertEquals((Object)UserGroupInformation.AuthenticationMethod.SIMPLE, (Object)SecurityUtil.getAuthenticationMethod((Configuration)conf));
        conf.set("hadoop.security.authentication", "kerberos");
        Assertions.assertEquals((Object)UserGroupInformation.AuthenticationMethod.KERBEROS, (Object)SecurityUtil.getAuthenticationMethod((Configuration)conf));
        conf.set("hadoop.security.authentication", "kaboom");
        String error = null;
        try {
            SecurityUtil.getAuthenticationMethod((Configuration)conf);
        }
        catch (Exception e) {
            error = e.toString();
        }
        Assertions.assertEquals((Object)"java.lang.IllegalArgumentException: Invalid attribute value for hadoop.security.authentication of kaboom", (Object)error);
    }

    @Test
    public void testSetAuthenticationMethod() {
        Configuration conf = new Configuration();
        SecurityUtil.setAuthenticationMethod(null, (Configuration)conf);
        Assertions.assertEquals((Object)"simple", (Object)conf.get("hadoop.security.authentication"));
        SecurityUtil.setAuthenticationMethod((UserGroupInformation.AuthenticationMethod)UserGroupInformation.AuthenticationMethod.SIMPLE, (Configuration)conf);
        Assertions.assertEquals((Object)"simple", (Object)conf.get("hadoop.security.authentication"));
        SecurityUtil.setAuthenticationMethod((UserGroupInformation.AuthenticationMethod)UserGroupInformation.AuthenticationMethod.KERBEROS, (Configuration)conf);
        Assertions.assertEquals((Object)"kerberos", (Object)conf.get("hadoop.security.authentication"));
    }

    @Test
    public void testAuthPlainPasswordProperty() throws Exception {
        Configuration conf = new Configuration();
        conf.set("hadoop.zk.auth", ZK_AUTH_VALUE);
        List zkAuths = SecurityUtil.getZKAuthInfos((Configuration)conf, (String)"hadoop.zk.auth");
        Assertions.assertEquals((int)1, (int)zkAuths.size());
        ZKUtil.ZKAuthInfo zkAuthInfo = (ZKUtil.ZKAuthInfo)zkAuths.get(0);
        Assertions.assertEquals((Object)"a_scheme", (Object)zkAuthInfo.getScheme());
        Assertions.assertArrayEquals((byte[])"a_password".getBytes(), (byte[])zkAuthInfo.getAuth());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testAuthPlainTextFile() throws Exception {
        Configuration conf = new Configuration();
        File passwordTxtFile = File.createTempFile(this.getClass().getSimpleName() + ".testAuthAtPathNotation-", ".txt");
        Files.asCharSink((File)passwordTxtFile, (Charset)StandardCharsets.UTF_8, (FileWriteMode[])new FileWriteMode[0]).write((CharSequence)ZK_AUTH_VALUE);
        try {
            conf.set("hadoop.zk.auth", "@" + passwordTxtFile.getAbsolutePath());
            List zkAuths = SecurityUtil.getZKAuthInfos((Configuration)conf, (String)"hadoop.zk.auth");
            Assertions.assertEquals((int)1, (int)zkAuths.size());
            ZKUtil.ZKAuthInfo zkAuthInfo = (ZKUtil.ZKAuthInfo)zkAuths.get(0);
            Assertions.assertEquals((Object)"a_scheme", (Object)zkAuthInfo.getScheme());
            Assertions.assertArrayEquals((byte[])"a_password".getBytes(), (byte[])zkAuthInfo.getAuth());
        }
        finally {
            boolean deleted = passwordTxtFile.delete();
            Assertions.assertTrue((boolean)deleted);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testAuthLocalJceks() throws Exception {
        File localJceksFile = File.createTempFile(this.getClass().getSimpleName() + ".testAuthLocalJceks-", ".localjceks");
        this.populateLocalJceksTestFile(localJceksFile.getAbsolutePath());
        try {
            String localJceksUri = "localjceks://file/" + localJceksFile.getAbsolutePath();
            Configuration conf = new Configuration();
            conf.set("hadoop.security.credential.provider.path", localJceksUri);
            List zkAuths = SecurityUtil.getZKAuthInfos((Configuration)conf, (String)"hadoop.zk.auth");
            Assertions.assertEquals((int)1, (int)zkAuths.size());
            ZKUtil.ZKAuthInfo zkAuthInfo = (ZKUtil.ZKAuthInfo)zkAuths.get(0);
            Assertions.assertEquals((Object)"a_scheme", (Object)zkAuthInfo.getScheme());
            Assertions.assertArrayEquals((byte[])"a_password".getBytes(), (byte[])zkAuthInfo.getAuth());
        }
        finally {
            boolean deleted = localJceksFile.delete();
            Assertions.assertTrue((boolean)deleted);
        }
    }

    private void populateLocalJceksTestFile(String path) throws IOException {
        Configuration conf = new Configuration();
        conf.set("hadoop.security.credential.provider.path", "localjceks://file/" + path);
        CredentialProvider provider = (CredentialProvider)CredentialProviderFactory.getProviders((Configuration)conf).get(0);
        Assertions.assertEquals((Object)LocalJavaKeyStoreProvider.class.getName(), (Object)provider.getClass().getName());
        provider.createCredentialEntry("hadoop.zk.auth", ZK_AUTH_VALUE.toCharArray());
        provider.flush();
    }

    @Test
    public void testInitiateHostResolver() throws Exception {
        Configuration conf = new Configuration();
        conf.setBoolean("hadoop.security.token.service.use_ip", false);
        conf.setTimeDuration("hadoop.security.hostname.cache.expire-interval.seconds", 0L, TimeUnit.SECONDS);
        SecurityUtil.setConfiguration((Configuration)conf);
        SecurityUtil.HostResolver hostResolver = SecurityUtil.hostResolver;
        Assertions.assertTrue((boolean)(hostResolver instanceof SecurityUtil.QualifiedHostResolver), (String)"Resolver should be a QualifiedHostResolver");
        SecurityUtil.QualifiedHostResolver cacheableHostResolver = (SecurityUtil.QualifiedHostResolver)hostResolver;
        Assertions.assertNull((Object)cacheableHostResolver.getCache(), (String)"Cache should be null when caching interval is less than or equal 0");
        conf.setBoolean("hadoop.security.token.service.use_ip", false);
        conf.setTimeDuration("hadoop.security.hostname.cache.expire-interval.seconds", 10L, TimeUnit.SECONDS);
        SecurityUtil.setConfiguration((Configuration)conf);
        hostResolver = SecurityUtil.hostResolver;
        Assertions.assertTrue((boolean)(hostResolver instanceof SecurityUtil.QualifiedHostResolver), (String)"Resolver should be a QualifiedHostResolver");
        cacheableHostResolver = (SecurityUtil.QualifiedHostResolver)hostResolver;
        Assertions.assertNotNull((Object)cacheableHostResolver.getCache(), (String)"Cache should be set when caching interval is enabled");
        conf.setBoolean("hadoop.security.token.service.use_ip", true);
        conf.setTimeDuration("hadoop.security.hostname.cache.expire-interval.seconds", 0L, TimeUnit.SECONDS);
        SecurityUtil.setConfiguration((Configuration)conf);
        hostResolver = SecurityUtil.hostResolver;
        Assertions.assertTrue((boolean)(hostResolver instanceof SecurityUtil.StandardHostResolver), (String)"Resolver should be a StandardHostResolver");
        cacheableHostResolver = (SecurityUtil.StandardHostResolver)hostResolver;
        Assertions.assertNull((Object)cacheableHostResolver.getCache(), (String)"Cache should be null when caching interval is less than or equal 0");
        conf.setBoolean("hadoop.security.token.service.use_ip", true);
        conf.setTimeDuration("hadoop.security.hostname.cache.expire-interval.seconds", 10L, TimeUnit.SECONDS);
        SecurityUtil.setConfiguration((Configuration)conf);
        hostResolver = SecurityUtil.hostResolver;
        Assertions.assertTrue((boolean)(hostResolver instanceof SecurityUtil.StandardHostResolver), (String)"Resolver should be a StandardHostResolver");
        cacheableHostResolver = (SecurityUtil.StandardHostResolver)hostResolver;
        Assertions.assertNotNull((Object)cacheableHostResolver.getCache(), (String)"Cache should be set when caching interval is enabled");
    }

    @Test
    public void testQualifiedHostResolverCachingEnabled() throws Exception {
        SecurityUtil.QualifiedHostResolver resolver = new SecurityUtil.QualifiedHostResolver(1L);
        this.testCacheableResolve((SecurityUtil.CacheableHostResolver)resolver);
    }

    @Test
    public void testStandardHostResolverCachingEnabled() throws Exception {
        SecurityUtil.StandardHostResolver resolver = new SecurityUtil.StandardHostResolver(1L);
        this.testCacheableResolve((SecurityUtil.CacheableHostResolver)resolver);
    }

    private void testCacheableResolve(SecurityUtil.CacheableHostResolver resolver) throws Exception {
        InetAddress addr1 = resolver.getByName("127.0.0.1");
        InetAddress addr2 = resolver.getByName("127.0.0.1");
        Assertions.assertNotNull((Object)addr1);
        Assertions.assertNotNull((Object)addr2);
        Assertions.assertSame((Object)addr1, (Object)addr2);
        Thread.sleep(1500L);
        InetAddress addr3 = resolver.getByName("127.0.0.1");
        Assertions.assertNotNull((Object)addr3);
        Assertions.assertNotSame((Object)addr1, (Object)addr3);
    }

    @Test
    public void testInvalidHostThrowsException() {
        SecurityUtil.StandardHostResolver standardHostResolver = new SecurityUtil.StandardHostResolver(10L);
        String invalidHost = "invalid_host_name_which_does_not_exist";
        Assertions.assertThrows(UnknownHostException.class, () -> standardHostResolver.getByName(invalidHost), (String)"Resolving an invalid host should throw UnknownHostException");
        SecurityUtil.QualifiedHostResolver qualifiedHostResolver = new SecurityUtil.QualifiedHostResolver(10L);
        Assertions.assertThrows(UnknownHostException.class, () -> qualifiedHostResolver.getByName(invalidHost), (String)"Resolving an invalid host should throw UnknownHostException");
    }
}

