/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.net;

import java.io.Closeable;
import java.io.EOFException;
import java.io.IOException;
import java.net.BindException;
import java.net.ConnectException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.NetworkInterface;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketAddress;
import java.net.SocketException;
import java.net.SocketTimeoutException;
import java.net.URI;
import java.net.UnknownHostException;
import java.nio.charset.CharacterCodingException;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.zip.ZipException;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.net.NetUtils;
import org.apache.hadoop.net.SocketInputWrapper;
import org.apache.hadoop.security.KerberosAuthException;
import org.apache.hadoop.security.NetUtilsTestResolver;
import org.apache.hadoop.test.GenericTestUtils;
import org.apache.hadoop.test.LambdaTestUtils;
import org.apache.hadoop.util.Shell;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Assumptions;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TestNetUtils {
    private static final Logger LOG = LoggerFactory.getLogger(TestNetUtils.class);
    private static final int DEST_PORT = 4040;
    private static final String DEST_PORT_NAME = Integer.toString(4040);
    private static final int LOCAL_PORT = 8080;
    private static final String LOCAL_PORT_NAME = Integer.toString(8080);
    static final long TIME_FUDGE_MILLIS = 200L;
    static NetUtilsTestResolver resolver;
    static Configuration config;

    @Test
    public void testAvoidLoopbackTcpSockets() throws Throwable {
        Configuration conf = new Configuration();
        Socket socket = NetUtils.getDefaultSocketFactory((Configuration)conf).createSocket();
        socket.bind(new InetSocketAddress("127.0.0.1", 0));
        System.err.println("local address: " + socket.getLocalAddress());
        System.err.println("local port: " + socket.getLocalPort());
        try {
            NetUtils.connect((Socket)socket, (SocketAddress)new InetSocketAddress(socket.getLocalAddress(), socket.getLocalPort()), (int)20000);
            socket.close();
            Assertions.fail((String)"Should not have connected");
        }
        catch (ConnectException ce) {
            System.err.println("Got exception: " + ce);
            this.assertInException(ce, "resulted in a loopback");
        }
        catch (SocketException se) {
            this.assertInException(se, "Invalid argument");
        }
    }

    @Test
    public void testInvalidAddress() throws Throwable {
        Configuration conf = new Configuration();
        Socket socket = NetUtils.getDefaultSocketFactory((Configuration)conf).createSocket();
        socket.bind(new InetSocketAddress("127.0.0.1", 0));
        try {
            NetUtils.connect((Socket)socket, (SocketAddress)new InetSocketAddress("invalid-test-host", 0), (int)20000);
            socket.close();
            Assertions.fail((String)"Should not have connected");
        }
        catch (UnknownHostException uhe) {
            LOG.info("Got exception: ", (Throwable)uhe);
            if (Shell.isJavaVersionAtLeast((int)17)) {
                GenericTestUtils.assertExceptionContains("invalid-test-host/<unresolved>:0", uhe);
            }
            GenericTestUtils.assertExceptionContains("invalid-test-host:0", uhe);
        }
    }

    @Test
    public void testSocketReadTimeoutWithChannel() throws Exception {
        this.doSocketReadTimeoutTest(true);
    }

    @Test
    public void testSocketReadTimeoutWithoutChannel() throws Exception {
        this.doSocketReadTimeoutTest(false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void doSocketReadTimeoutTest(boolean withChannel) throws IOException {
        SocketInputWrapper stm;
        Socket s;
        ServerSocket ss;
        block5: {
            ss = new ServerSocket(0);
            if (withChannel) {
                s = NetUtils.getDefaultSocketFactory((Configuration)new Configuration()).createSocket();
                Assumptions.assumeTrue((s.getChannel() != null ? 1 : 0) != 0);
            } else {
                s = new Socket();
                Assertions.assertNull((Object)s.getChannel());
            }
            stm = null;
            try {
                NetUtils.connect((Socket)s, (SocketAddress)ss.getLocalSocketAddress(), (int)1000);
                stm = NetUtils.getInputStream((Socket)s, (long)1000L);
                this.assertReadTimeout(stm, 1000);
                stm.setTimeout(1L);
                this.assertReadTimeout(stm, 1);
                s.setSoTimeout(1000);
                if (withChannel) {
                    this.assertReadTimeout(stm, 1);
                    break block5;
                }
                this.assertReadTimeout(stm, 1000);
            }
            catch (Throwable throwable) {
                IOUtils.closeStream(stm);
                IOUtils.closeSocket((Socket)s);
                ss.close();
                throw throwable;
            }
        }
        IOUtils.closeStream((Closeable)stm);
        IOUtils.closeSocket((Socket)s);
        ss.close();
    }

    private void assertReadTimeout(SocketInputWrapper stm, int timeoutMillis) throws IOException {
        long st = System.nanoTime();
        try {
            stm.read();
            Assertions.fail((String)"Didn't time out");
        }
        catch (SocketTimeoutException ste) {
            this.assertTimeSince(st, timeoutMillis);
        }
    }

    private void assertTimeSince(long startNanos, int expectedMillis) {
        long durationNano = System.nanoTime() - startNanos;
        long millis = TimeUnit.MILLISECONDS.convert(durationNano, TimeUnit.NANOSECONDS);
        Assertions.assertTrue((Math.abs(millis - (long)expectedMillis) < 200L ? 1 : 0) != 0, (String)("Expected " + expectedMillis + "ms, but took " + millis));
    }

    @Test
    public void testGetLocalInetAddress() throws Exception {
        Assertions.assertNotNull((Object)NetUtils.getLocalInetAddress((String)"127.0.0.1"));
        Assertions.assertNull((Object)NetUtils.getLocalInetAddress((String)"invalid-address-for-test"));
        Assertions.assertNull((Object)NetUtils.getLocalInetAddress(null));
    }

    @Test
    public void testVerifyHostnamesException() throws UnknownHostException {
        Assertions.assertThrows(UnknownHostException.class, () -> {
            String[] names = new String[]{"valid.host.com", "1.com", "invalid host here"};
            NetUtils.verifyHostnames((String[])names);
        });
    }

    @Test
    public void testVerifyHostnamesNoException() throws UnknownHostException {
        String[] names = new String[]{"valid.host.com", "1.com"};
        NetUtils.verifyHostnames((String[])names);
    }

    @Test
    public void testIsLocalAddress() throws Exception {
        Assertions.assertTrue((boolean)NetUtils.isLocalAddress((InetAddress)InetAddress.getLocalHost()));
        Enumeration<NetworkInterface> interfaces = NetworkInterface.getNetworkInterfaces();
        if (interfaces != null) {
            while (interfaces.hasMoreElements()) {
                NetworkInterface i = interfaces.nextElement();
                Enumeration<InetAddress> addrs = i.getInetAddresses();
                if (addrs == null) continue;
                while (addrs.hasMoreElements()) {
                    InetAddress addr = addrs.nextElement();
                    Assertions.assertTrue((boolean)NetUtils.isLocalAddress((InetAddress)addr));
                }
            }
        }
        Assertions.assertFalse((boolean)NetUtils.isLocalAddress((InetAddress)InetAddress.getByName("8.8.8.8")));
    }

    @Test
    public void testWrapConnectException() throws Throwable {
        ConnectException e = new ConnectException("failed");
        IOException wrapped = this.verifyExceptionClass(e, ConnectException.class);
        this.assertInException(wrapped, "failed");
        this.assertWikified(wrapped);
        this.assertInException(wrapped, "localhost");
        this.assertRemoteDetailsIncluded(wrapped);
        this.assertInException(wrapped, "/ConnectionRefused");
    }

    @Test
    public void testWrapBindException() throws Throwable {
        BindException e = new BindException("failed");
        IOException wrapped = this.verifyExceptionClass(e, BindException.class);
        this.assertInException(wrapped, "failed");
        this.assertLocalDetailsIncluded(wrapped);
        this.assertNotInException(wrapped, DEST_PORT_NAME);
        this.assertInException(wrapped, "/BindException");
    }

    @Test
    public void testWrapUnknownHostException() throws Throwable {
        UnknownHostException e = new UnknownHostException("failed");
        IOException wrapped = this.verifyExceptionClass(e, UnknownHostException.class);
        this.assertInException(wrapped, "failed");
        this.assertWikified(wrapped);
        this.assertInException(wrapped, "localhost");
        this.assertRemoteDetailsIncluded(wrapped);
        this.assertInException(wrapped, "/UnknownHost");
    }

    @Test
    public void testWrapEOFException() throws Throwable {
        EOFException e = new EOFException("eof");
        IOException wrapped = this.verifyExceptionClass(e, EOFException.class);
        this.assertInException(wrapped, "eof");
        this.assertWikified(wrapped);
        this.assertInException(wrapped, "localhost");
        this.assertRemoteDetailsIncluded(wrapped);
        this.assertInException(wrapped, "/EOFException");
    }

    @Test
    public void testWrapKerbAuthException() throws Throwable {
        KerberosAuthException e = new KerberosAuthException("socket timeout on connection");
        IOException wrapped = this.verifyExceptionClass((IOException)e, KerberosAuthException.class);
        this.assertInException(wrapped, "socket timeout on connection");
        this.assertInException(wrapped, "localhost");
        this.assertInException(wrapped, "DestHost:destPort ");
        this.assertInException(wrapped, "LocalHost:localPort");
        this.assertRemoteDetailsIncluded(wrapped);
        this.assertInException(wrapped, "KerberosAuthException");
    }

    @Test
    public void testWrapIOEWithNoStringConstructor() throws Throwable {
        CharacterCodingException e = new CharacterCodingException();
        IOException wrapped = this.verifyExceptionClass(e, CharacterCodingException.class);
        Assertions.assertEquals(null, (Object)wrapped.getMessage());
    }

    @Test
    public void testWrapIOEWithPrivateStringConstructor() throws Throwable {
        class TestIOException
        extends CharacterCodingException {
            private TestIOException(String cause) {
            }

            TestIOException() {
            }
        }
        TestIOException e = new TestIOException();
        IOException wrapped = this.verifyExceptionClass(e, TestIOException.class);
        Assertions.assertEquals(null, (Object)wrapped.getMessage());
    }

    @Test
    public void testWrapSocketException() throws Throwable {
        IOException wrapped = this.verifyExceptionClass(new SocketException("failed"), SocketException.class);
        this.assertInException(wrapped, "failed");
        this.assertWikified(wrapped);
        this.assertInException(wrapped, "localhost");
        this.assertRemoteDetailsIncluded(wrapped);
        this.assertInException(wrapped, "/SocketException");
    }

    @Test
    public void testGetConnectAddress() throws IOException {
        NetUtils.addStaticResolution((String)"host", (String)"127.0.0.1");
        InetSocketAddress addr = NetUtils.createSocketAddrForHost((String)"host", (int)1);
        InetSocketAddress connectAddr = NetUtils.getConnectAddress((InetSocketAddress)addr);
        Assertions.assertEquals((Object)addr.getHostName(), (Object)connectAddr.getHostName());
        addr = new InetSocketAddress(1);
        connectAddr = NetUtils.getConnectAddress((InetSocketAddress)addr);
        Assertions.assertEquals((Object)InetAddress.getLocalHost().getHostName(), (Object)connectAddr.getHostName());
    }

    @Test
    public void testCreateSocketAddress() throws Throwable {
        InetSocketAddress addr = NetUtils.createSocketAddr((String)"127.0.0.1:12345", (int)1000, (String)"myconfig");
        Assertions.assertEquals((Object)"127.0.0.1", (Object)addr.getAddress().getHostAddress());
        Assertions.assertEquals((int)12345, (int)addr.getPort());
        addr = NetUtils.createSocketAddr((String)"127.0.0.1", (int)1000, (String)"myconfig");
        Assertions.assertEquals((Object)"127.0.0.1", (Object)addr.getAddress().getHostAddress());
        Assertions.assertEquals((int)1000, (int)addr.getPort());
        try {
            NetUtils.createSocketAddr((String)"127.0.0.1:blahblah", (int)1000, (String)"myconfig");
            Assertions.fail((String)"Should have failed to parse bad port");
        }
        catch (IllegalArgumentException iae) {
            this.assertInException(iae, "myconfig");
        }
    }

    @Test
    public void testCreateSocketAddressWithURICache() throws Throwable {
        InetSocketAddress addr = NetUtils.createSocketAddr((String)"127.0.0.1:12345", (int)1000, (String)"myconfig", (boolean)true);
        Assertions.assertEquals((Object)"127.0.0.1", (Object)addr.getAddress().getHostAddress());
        Assertions.assertEquals((int)12345, (int)addr.getPort());
        addr = NetUtils.createSocketAddr((String)"127.0.0.1:12345", (int)1000, (String)"myconfig", (boolean)true);
        Assertions.assertEquals((Object)"127.0.0.1", (Object)addr.getAddress().getHostAddress());
        Assertions.assertEquals((int)12345, (int)addr.getPort());
        addr = NetUtils.createSocketAddr((String)"127.0.0.1", (int)1000, (String)"myconfig", (boolean)true);
        Assertions.assertEquals((Object)"127.0.0.1", (Object)addr.getAddress().getHostAddress());
        Assertions.assertEquals((int)1000, (int)addr.getPort());
        addr = NetUtils.createSocketAddr((String)"127.0.0.1", (int)1000, (String)"myconfig", (boolean)true);
        Assertions.assertEquals((Object)"127.0.0.1", (Object)addr.getAddress().getHostAddress());
        Assertions.assertEquals((int)1000, (int)addr.getPort());
        try {
            NetUtils.createSocketAddr((String)"127.0.0.1:blahblah", (int)1000, (String)"myconfig", (boolean)true);
            Assertions.fail((String)"Should have failed to parse bad port");
        }
        catch (IllegalArgumentException iae) {
            this.assertInException(iae, "myconfig");
        }
        try {
            NetUtils.createSocketAddr((String)"127.0.0.1:blahblah", (int)1000, (String)"myconfig", (boolean)true);
            Assertions.fail((String)"Should have failed to parse bad port");
        }
        catch (IllegalArgumentException iae) {
            this.assertInException(iae, "myconfig");
        }
    }

    private void assertRemoteDetailsIncluded(IOException wrapped) throws Throwable {
        this.assertInException(wrapped, "desthost");
        this.assertInException(wrapped, DEST_PORT_NAME);
    }

    private void assertLocalDetailsIncluded(IOException wrapped) throws Throwable {
        this.assertInException(wrapped, "localhost");
        this.assertInException(wrapped, LOCAL_PORT_NAME);
    }

    private void assertWikified(Exception e) throws Throwable {
        this.assertInException(e, "http://wiki.apache.org/hadoop/");
    }

    private void assertInException(Exception e, String text) throws Throwable {
        String message = this.extractExceptionMessage(e);
        if (!message.contains(text)) {
            throw ((Throwable)((Object)new AssertionError((Object)("Wrong text in message \"" + message + "\" expected \"" + text + "\"")))).initCause(e);
        }
    }

    private String extractExceptionMessage(Exception e) throws Throwable {
        Assertions.assertNotNull((Object)e, (String)"Null Exception");
        String message = e.getMessage();
        if (message == null) {
            throw ((Throwable)((Object)new AssertionError((Object)("Empty text in exception " + e)))).initCause(e);
        }
        return message;
    }

    private void assertNotInException(Exception e, String text) throws Throwable {
        String message = this.extractExceptionMessage(e);
        if (message.contains(text)) {
            throw ((Throwable)((Object)new AssertionError((Object)("Wrong text in message \"" + message + "\" did not expect \"" + text + "\"")))).initCause(e);
        }
    }

    private IOException verifyExceptionClass(IOException e, Class expectedClass) throws Throwable {
        Assertions.assertNotNull((Object)e, (String)"Null Exception");
        IOException wrapped = NetUtils.wrapException((String)"desthost", (int)4040, (String)"localhost", (int)8080, (IOException)e);
        LOG.info(wrapped.toString(), (Throwable)wrapped);
        if (!wrapped.getClass().equals(expectedClass)) {
            throw ((Throwable)((Object)new AssertionError((Object)("Wrong exception class; expected " + expectedClass + " got " + wrapped.getClass() + ": " + wrapped)))).initCause(wrapped);
        }
        return wrapped;
    }

    @BeforeAll
    public static void setupResolver() {
        resolver = NetUtilsTestResolver.install();
    }

    @BeforeEach
    public void resetResolver() {
        resolver.reset();
        config = new Configuration();
    }

    private void verifyGetByExactNameSearch(String host, String ... searches) {
        Assertions.assertNull((Object)resolver.getByExactName(host));
        this.assertBetterArrayEquals(searches, resolver.getHostSearches());
    }

    @Test
    public void testResolverGetByExactNameUnqualified() {
        this.verifyGetByExactNameSearch("unknown", "unknown.");
    }

    @Test
    public void testResolverGetByExactNameUnqualifiedWithDomain() {
        this.verifyGetByExactNameSearch("unknown.domain", "unknown.domain.");
    }

    @Test
    public void testResolverGetByExactNameQualified() {
        this.verifyGetByExactNameSearch("unknown.", "unknown.");
    }

    @Test
    public void testResolverGetByExactNameQualifiedWithDomain() {
        this.verifyGetByExactNameSearch("unknown.domain.", "unknown.domain.");
    }

    private void verifyGetByNameWithSearch(String host, String ... searches) {
        Assertions.assertNull((Object)resolver.getByNameWithSearch(host));
        this.assertBetterArrayEquals(searches, resolver.getHostSearches());
    }

    @Test
    public void testResolverGetByNameWithSearchUnqualified() {
        String host = "unknown";
        this.verifyGetByNameWithSearch(host, host + ".a.b.", host + ".b.", host + ".c.");
    }

    @Test
    public void testResolverGetByNameWithSearchUnqualifiedWithDomain() {
        String host = "unknown.domain";
        this.verifyGetByNameWithSearch(host, host + ".a.b.", host + ".b.", host + ".c.");
    }

    @Test
    public void testResolverGetByNameWithSearchQualified() {
        String host = "unknown.";
        this.verifyGetByNameWithSearch(host, host);
    }

    @Test
    public void testResolverGetByNameWithSearchQualifiedWithDomain() {
        String host = "unknown.domain.";
        this.verifyGetByNameWithSearch(host, host);
    }

    private void verifyGetByName(String host, String ... searches) {
        InetAddress addr = null;
        try {
            addr = resolver.getByName(host);
        }
        catch (UnknownHostException unknownHostException) {
            // empty catch block
        }
        Assertions.assertNull((Object)addr);
        this.assertBetterArrayEquals(searches, resolver.getHostSearches());
    }

    @Test
    public void testResolverGetByNameQualified() {
        String host = "unknown.";
        this.verifyGetByName(host, host);
    }

    @Test
    public void testResolverGetByNameQualifiedWithDomain() {
        this.verifyGetByName("unknown.domain.", "unknown.domain.");
    }

    @Test
    public void testResolverGetByNameUnqualified() {
        String host = "unknown";
        this.verifyGetByName(host, host + ".a.b.", host + ".b.", host + ".c.", host + ".");
    }

    @Test
    public void testResolverGetByNameUnqualifiedWithDomain() {
        String host = "unknown.domain";
        this.verifyGetByName(host, host + ".", host + ".a.b.", host + ".b.", host + ".c.");
    }

    private InetAddress verifyResolve(String host, String ... searches) {
        InetAddress addr = null;
        try {
            addr = resolver.getByName(host);
        }
        catch (UnknownHostException unknownHostException) {
            // empty catch block
        }
        Assertions.assertNotNull((Object)addr);
        this.assertBetterArrayEquals(searches, resolver.getHostSearches());
        return addr;
    }

    private void verifyInetAddress(InetAddress addr, String host, String ip) {
        Assertions.assertNotNull((Object)addr);
        Assertions.assertEquals((Object)host, (Object)addr.getHostName());
        Assertions.assertEquals((Object)ip, (Object)addr.getHostAddress());
    }

    @Test
    public void testResolverUnqualified() {
        String host = "host";
        InetAddress addr = this.verifyResolve(host, host + ".a.b.");
        this.verifyInetAddress(addr, "host.a.b", "1.1.1.1");
    }

    @Test
    public void testResolverUnqualifiedWithDomain() {
        String host = "host.a";
        InetAddress addr = this.verifyResolve(host, host + ".", host + ".a.b.", host + ".b.");
        this.verifyInetAddress(addr, "host.a.b", "1.1.1.1");
    }

    @Test
    public void testResolverUnqualifedFull() {
        String host = "host.a.b";
        InetAddress addr = this.verifyResolve(host, host + ".");
        this.verifyInetAddress(addr, host, "1.1.1.1");
    }

    @Test
    public void testResolverQualifed() {
        String host = "host.a.b.";
        InetAddress addr = this.verifyResolve(host, host);
        this.verifyInetAddress(addr, host, "1.1.1.1");
    }

    @Test
    public void testResolverLoopback() {
        String host = "Localhost";
        InetAddress addr = this.verifyResolve(host, new String[0]);
        this.verifyInetAddress(addr, "Localhost", "127.0.0.1");
    }

    @Test
    public void testResolverIP() {
        String host = "1.1.1.1";
        InetAddress addr = this.verifyResolve(host, new String[0]);
        this.verifyInetAddress(addr, host, host);
    }

    @Test
    public void testCanonicalUriWithPort() {
        URI uri = NetUtils.getCanonicalUri((URI)URI.create("scheme://host:123"), (int)456);
        Assertions.assertEquals((Object)"scheme://host.a.b:123", (Object)uri.toString());
        uri = NetUtils.getCanonicalUri((URI)URI.create("scheme://host:123/"), (int)456);
        Assertions.assertEquals((Object)"scheme://host.a.b:123/", (Object)uri.toString());
        uri = NetUtils.getCanonicalUri((URI)URI.create("scheme://host:123/path"), (int)456);
        Assertions.assertEquals((Object)"scheme://host.a.b:123/path", (Object)uri.toString());
        uri = NetUtils.getCanonicalUri((URI)URI.create("scheme://host:123/path?q#frag"), (int)456);
        Assertions.assertEquals((Object)"scheme://host.a.b:123/path?q#frag", (Object)uri.toString());
    }

    @Test
    public void testCanonicalUriWithDefaultPort() {
        URI uri = NetUtils.getCanonicalUri((URI)URI.create("scheme://host"), (int)123);
        Assertions.assertEquals((Object)"scheme://host.a.b:123", (Object)uri.toString());
        uri = NetUtils.getCanonicalUri((URI)URI.create("scheme://host/"), (int)123);
        Assertions.assertEquals((Object)"scheme://host.a.b:123/", (Object)uri.toString());
        uri = NetUtils.getCanonicalUri((URI)URI.create("scheme://host/path"), (int)123);
        Assertions.assertEquals((Object)"scheme://host.a.b:123/path", (Object)uri.toString());
        uri = NetUtils.getCanonicalUri((URI)URI.create("scheme://host/path?q#frag"), (int)123);
        Assertions.assertEquals((Object)"scheme://host.a.b:123/path?q#frag", (Object)uri.toString());
    }

    @Test
    public void testCanonicalUriWithPath() {
        URI uri = NetUtils.getCanonicalUri((URI)URI.create("path"), (int)2);
        Assertions.assertEquals((Object)"path", (Object)uri.toString());
        uri = NetUtils.getCanonicalUri((URI)URI.create("/path"), (int)2);
        Assertions.assertEquals((Object)"/path", (Object)uri.toString());
    }

    @Test
    public void testCanonicalUriWithNoAuthority() {
        URI uri = NetUtils.getCanonicalUri((URI)URI.create("scheme:/"), (int)2);
        Assertions.assertEquals((Object)"scheme:/", (Object)uri.toString());
        uri = NetUtils.getCanonicalUri((URI)URI.create("scheme:/path"), (int)2);
        Assertions.assertEquals((Object)"scheme:/path", (Object)uri.toString());
        uri = NetUtils.getCanonicalUri((URI)URI.create("scheme:///"), (int)2);
        Assertions.assertEquals((Object)"scheme:///", (Object)uri.toString());
        uri = NetUtils.getCanonicalUri((URI)URI.create("scheme:///path"), (int)2);
        Assertions.assertEquals((Object)"scheme:///path", (Object)uri.toString());
    }

    @Test
    public void testCanonicalUriWithNoHost() {
        URI uri = NetUtils.getCanonicalUri((URI)URI.create("scheme://:123/path"), (int)2);
        Assertions.assertEquals((Object)"scheme://:123/path", (Object)uri.toString());
    }

    @Test
    public void testCanonicalUriWithNoPortNoDefaultPort() {
        URI uri = NetUtils.getCanonicalUri((URI)URI.create("scheme://host/path"), (int)-1);
        Assertions.assertEquals((Object)"scheme://host.a.b/path", (Object)uri.toString());
    }

    @Test
    public void testNormalizeHostName() {
        String oneHost = "1.kanyezone.appspot.com";
        try {
            InetAddress.getByName(oneHost);
        }
        catch (UnknownHostException e) {
            Assumptions.assumeTrue((boolean)false, (String)("Network not resolving " + oneHost));
        }
        List<String> hosts = Arrays.asList("127.0.0.1", "localhost", oneHost, "UnknownHost123");
        List normalizedHosts = NetUtils.normalizeHostNames(hosts);
        String summary = "original [" + StringUtils.join(hosts, (String)", ") + "] normalized [" + StringUtils.join((Iterable)normalizedHosts, (String)", ") + "]";
        Assertions.assertEquals((Object)hosts.get(0), normalizedHosts.get(0), (String)summary);
        Assertions.assertFalse((boolean)((String)normalizedHosts.get(1)).equals(hosts.get(1)), (String)("Element 1 equal " + summary));
        Assertions.assertEquals((Object)hosts.get(0), normalizedHosts.get(1), (String)summary);
        Assertions.assertFalse((boolean)((String)normalizedHosts.get(2)).equals(hosts.get(2)), (String)("Element 2 equal " + summary));
        Assertions.assertEquals((Object)hosts.get(3), normalizedHosts.get(3), (String)summary);
    }

    @Test
    public void testGetHostNameOfIP() {
        Assertions.assertNull((Object)NetUtils.getHostNameOfIP(null));
        Assertions.assertNull((Object)NetUtils.getHostNameOfIP((String)""));
        Assertions.assertNull((Object)NetUtils.getHostNameOfIP((String)"crazytown"));
        Assertions.assertNull((Object)NetUtils.getHostNameOfIP((String)"127.0.0.1:"));
        Assertions.assertNull((Object)NetUtils.getHostNameOfIP((String)"127.0.0.1:-1"));
        Assertions.assertNull((Object)NetUtils.getHostNameOfIP((String)"127.0.0.1:A"));
        Assertions.assertNotNull((Object)NetUtils.getHostNameOfIP((String)"127.0.0.1"));
        Assertions.assertNotNull((Object)NetUtils.getHostNameOfIP((String)"127.0.0.1:1"));
    }

    @Test
    public void testTrimCreateSocketAddress() {
        Configuration conf = new Configuration();
        NetUtils.addStaticResolution((String)"host", (String)"127.0.0.1");
        String defaultAddr = "host:1  ";
        InetSocketAddress addr = NetUtils.createSocketAddr((String)"host:1  ");
        conf.setSocketAddr("myAddress", addr);
        Assertions.assertEquals((Object)"host:1  ".trim(), (Object)NetUtils.getHostPortString((InetSocketAddress)addr));
    }

    @Test
    public void testGetPortFromHostPortString() throws Exception {
        Assertions.assertEquals((int)1002, (int)NetUtils.getPortFromHostPortString((String)"testHost:1002"));
        LambdaTestUtils.intercept(IllegalArgumentException.class, () -> NetUtils.getPortFromHostPortString((String)"testHost"));
        LambdaTestUtils.intercept(IllegalArgumentException.class, () -> NetUtils.getPortFromHostPortString((String)"testHost:randomString"));
    }

    @Test
    public void testBindToLocalAddress() throws Exception {
        Assertions.assertNotNull((Object)NetUtils.bindToLocalAddress((InetAddress)NetUtils.getLocalInetAddress((String)"127.0.0.1"), (boolean)false));
        Assertions.assertNull((Object)NetUtils.bindToLocalAddress((InetAddress)NetUtils.getLocalInetAddress((String)"127.0.0.1"), (boolean)true));
    }

    @Test
    public void testAddNodeNameToIOException() {
        IOException e0 = new IOException("test123");
        this.assertNullCause(e0);
        IOException new0 = NetUtils.addNodeNameToIOException((IOException)e0, (String)"node123");
        this.assertNullCause(new0);
        Assertions.assertEquals((Object)"node123: test123", (Object)new0.getMessage());
        IOException e1 = new IOException("test456", new IllegalStateException("deliberate"));
        IOException new1 = NetUtils.addNodeNameToIOException((IOException)e1, (String)"node456");
        Assertions.assertSame((Object)e1.getCause(), (Object)new1.getCause());
        Assertions.assertEquals((Object)"node456: test456", (Object)new1.getMessage());
        ZipException e2 = new ZipException("test789");
        this.assertNullCause(e2);
        IOException new2 = NetUtils.addNodeNameToIOException((IOException)e2, (String)"node789");
        this.assertNullCause(new2);
        Assertions.assertEquals((Object)"node789: test789", (Object)new2.getMessage());
        WrappedIOException e3 = new WrappedIOException("test987", new IllegalStateException("deliberate"));
        IOException new3 = NetUtils.addNodeNameToIOException((IOException)e3, (String)"node987");
        Assertions.assertSame((Object)e3.getCause(), (Object)new3.getCause());
        Assertions.assertEquals((Object)"node987: test987", (Object)new3.getMessage());
        PrivateIOException e4 = new PrivateIOException("test654", new IllegalStateException("deliberate"));
        IOException new4 = NetUtils.addNodeNameToIOException((IOException)e4, (String)"node654");
        Assertions.assertSame((Object)e4, (Object)new4);
    }

    private <T> void assertBetterArrayEquals(T[] expect, T[] got) {
        String expectStr = StringUtils.join((Object[])expect, (String)", ");
        String gotStr = StringUtils.join((Object[])got, (String)", ");
        Assertions.assertEquals((Object)expectStr, (Object)gotStr);
    }

    private void assertNullCause(Exception e) {
        Assertions.assertNull((Object)e.getCause(), (String)"Expected exception to have null cause");
    }

    private static class PrivateIOException
    extends IOException {
        PrivateIOException(String msg, Throwable cause) {
            super(msg, cause);
        }
    }

    public static class WrappedIOException
    extends IOException {
        public WrappedIOException(String msg, Throwable cause) {
            super(msg, cause);
        }
    }
}

