/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FsShell;
import org.apache.hadoop.fs.LocalFileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.test.GenericTestUtils;
import org.apache.hadoop.test.PlatformAssumptions;
import org.apache.hadoop.util.StringUtils;
import org.assertj.core.api.AbstractIntegerAssert;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TestFsShellCopy {
    static final Logger LOG = LoggerFactory.getLogger(TestFsShellCopy.class);
    static Configuration conf;
    static FsShell shell;
    static LocalFileSystem lfs;
    static Path testRootDir;
    static Path srcPath;
    static Path dstPath;

    @BeforeAll
    public static void setup() throws Exception {
        conf = new Configuration();
        shell = new FsShell(conf);
        lfs = FileSystem.getLocal((Configuration)conf);
        testRootDir = lfs.makeQualified(new Path(GenericTestUtils.getTempPath("testFsShellCopy")));
        lfs.mkdirs(testRootDir);
        lfs.setWorkingDirectory(testRootDir);
        srcPath = new Path(testRootDir, "srcFile");
        dstPath = new Path(testRootDir, "dstFile");
    }

    @BeforeEach
    public void prepFiles() throws Exception {
        lfs.setVerifyChecksum(true);
        lfs.setWriteChecksum(true);
        lfs.delete(srcPath, true);
        lfs.delete(dstPath, true);
        FSDataOutputStream out = lfs.create(srcPath);
        out.writeChars("hi");
        out.close();
        org.junit.jupiter.api.Assertions.assertTrue((boolean)lfs.exists(lfs.getChecksumFile(srcPath)));
    }

    private void shellRun(int n, String ... args) throws Exception {
        org.junit.jupiter.api.Assertions.assertEquals((int)n, (int)shell.run(args));
    }

    private int shellRun(String ... args) throws Exception {
        int exitCode = shell.run(args);
        LOG.info("exit " + exitCode + " - " + StringUtils.join((CharSequence)" ", (String[])args));
        return exitCode;
    }

    @Test
    public void testCopyNoCrc() throws Exception {
        this.shellRun(0, "-get", srcPath.toString(), dstPath.toString());
        this.checkPath(dstPath, false);
    }

    @Test
    public void testCopyCrc() throws Exception {
        this.shellRun(0, "-get", "-crc", srcPath.toString(), dstPath.toString());
        this.checkPath(dstPath, true);
    }

    @Test
    public void testCorruptedCopyCrc() throws Exception {
        FSDataOutputStream out = lfs.getRawFileSystem().create(srcPath);
        out.writeChars("bang");
        out.close();
        this.shellRun(1, "-get", srcPath.toString(), dstPath.toString());
    }

    @Test
    public void testCorruptedCopyIgnoreCrc() throws Exception {
        this.shellRun(0, "-get", "-ignoreCrc", srcPath.toString(), dstPath.toString());
        this.checkPath(dstPath, false);
    }

    private void checkPath(Path p, boolean expectChecksum) throws IOException {
        org.junit.jupiter.api.Assertions.assertTrue((boolean)lfs.exists(p));
        boolean hasChecksum = lfs.exists(lfs.getChecksumFile(p));
        org.junit.jupiter.api.Assertions.assertEquals((Object)expectChecksum, (Object)hasChecksum);
    }

    @Test
    public void testCopyFileFromLocal() throws Exception {
        Path testRoot = new Path(testRootDir, "testPutFile");
        lfs.delete(testRoot, true);
        lfs.mkdirs(testRoot);
        Path targetDir = new Path(testRoot, "target");
        Path filePath = new Path(testRoot, new Path("srcFile"));
        lfs.create(filePath).close();
        this.checkPut(filePath, targetDir, false);
    }

    @Test
    public void testCopyDirFromLocal() throws Exception {
        Path testRoot = new Path(testRootDir, "testPutDir");
        lfs.delete(testRoot, true);
        lfs.mkdirs(testRoot);
        Path targetDir = new Path(testRoot, "target");
        Path dirPath = new Path(testRoot, new Path("srcDir"));
        lfs.mkdirs(dirPath);
        lfs.create(new Path(dirPath, "srcFile")).close();
        this.checkPut(dirPath, targetDir, false);
    }

    @Test
    public void testCopyFileFromWindowsLocalPath() throws Exception {
        PlatformAssumptions.assumeWindows();
        String windowsTestRootPath = new File(testRootDir.toUri().getPath().toString()).getAbsolutePath();
        Path testRoot = new Path(windowsTestRootPath, "testPutFile");
        lfs.delete(testRoot, true);
        lfs.mkdirs(testRoot);
        Path targetDir = new Path(testRoot, "target");
        Path filePath = new Path(testRoot, new Path("srcFile"));
        lfs.create(filePath).close();
        this.checkPut(filePath, targetDir, true);
    }

    @Test
    public void testCopyDirFromWindowsLocalPath() throws Exception {
        PlatformAssumptions.assumeWindows();
        String windowsTestRootPath = new File(testRootDir.toUri().getPath().toString()).getAbsolutePath();
        Path testRoot = new Path(windowsTestRootPath, "testPutDir");
        lfs.delete(testRoot, true);
        lfs.mkdirs(testRoot);
        Path targetDir = new Path(testRoot, "target");
        Path dirPath = new Path(testRoot, new Path("srcDir"));
        lfs.mkdirs(dirPath);
        lfs.create(new Path(dirPath, "srcFile")).close();
        this.checkPut(dirPath, targetDir, true);
    }

    private void checkPut(Path srcPath, Path targetDir, boolean useWindowsPath) throws Exception {
        lfs.delete(targetDir, true);
        lfs.mkdirs(targetDir);
        lfs.setWorkingDirectory(targetDir);
        Path dstPath = new Path("path");
        Path childPath = new Path(dstPath, "childPath");
        lfs.setWorkingDirectory(targetDir);
        this.prepPut(dstPath, false, false);
        this.checkPut(0, srcPath, dstPath, useWindowsPath);
        if (lfs.getFileStatus(srcPath).isFile()) {
            this.checkPut(1, srcPath, dstPath, useWindowsPath);
        } else {
            this.prepPut(dstPath, true, true);
            this.checkPut(0, srcPath, dstPath, useWindowsPath);
        }
        this.prepPut(dstPath, false, false);
        this.checkPut(1, srcPath, childPath, useWindowsPath);
        this.prepPut(dstPath, true, true);
        this.checkPut(0, srcPath, dstPath, useWindowsPath);
        this.prepPut(childPath, true, true);
        this.checkPut(0, srcPath, childPath, useWindowsPath);
        this.prepPut(targetDir, true, true);
        this.checkPut(0, srcPath, null, useWindowsPath);
        this.prepPut(targetDir, true, true);
        this.checkPut(0, srcPath, new Path("."), useWindowsPath);
        this.prepPut(dstPath, false, true);
        lfs.setWorkingDirectory(dstPath);
        this.checkPut(1, srcPath, null, useWindowsPath);
        this.prepPut(dstPath, false, true);
        this.checkPut(1, srcPath, new Path("."), useWindowsPath);
    }

    private void prepPut(Path dst, boolean create, boolean isDir) throws IOException {
        lfs.delete(dst, true);
        org.junit.jupiter.api.Assertions.assertFalse((boolean)lfs.exists(dst));
        if (create) {
            if (isDir) {
                lfs.mkdirs(dst);
                org.junit.jupiter.api.Assertions.assertTrue((boolean)lfs.getFileStatus(dst).isDirectory());
            } else {
                lfs.mkdirs(new Path(dst.getName()));
                lfs.create(dst).close();
                org.junit.jupiter.api.Assertions.assertTrue((boolean)lfs.getFileStatus(dst).isFile());
            }
        }
    }

    private void checkPut(int exitCode, Path src, Path dest, boolean useWindowsPath) throws Exception {
        String[] argv = null;
        String srcPath = src.toString();
        if (useWindowsPath) {
            srcPath = new File(srcPath).getAbsolutePath();
        }
        if (dest != null) {
            argv = new String[]{"-put", srcPath, this.pathAsString(dest)};
        } else {
            argv = new String[]{"-put", srcPath};
            dest = new Path(".");
        }
        Path target = lfs.exists(dest) ? (lfs.getFileStatus(dest).isDirectory() ? new Path(this.pathAsString(dest), src.getName()) : dest) : new Path(lfs.getWorkingDirectory(), dest);
        boolean targetExists = lfs.exists(target);
        Path parent = lfs.makeQualified(target).getParent();
        System.out.println("COPY src[" + src.getName() + "] -> [" + dest + "] as [" + target + "]");
        String[] lsArgv = new String[]{"-ls", "-R", this.pathAsString(parent)};
        shell.run(lsArgv);
        int gotExit = shell.run(argv);
        System.out.println("copy exit:" + gotExit);
        lsArgv = new String[]{"-ls", "-R", this.pathAsString(parent)};
        shell.run(lsArgv);
        if (exitCode == 0) {
            org.junit.jupiter.api.Assertions.assertTrue((boolean)lfs.exists(target));
            org.junit.jupiter.api.Assertions.assertTrue((lfs.getFileStatus(src).isFile() == lfs.getFileStatus(target).isFile() ? 1 : 0) != 0);
            org.junit.jupiter.api.Assertions.assertEquals((int)1, (int)lfs.listStatus(lfs.makeQualified(target).getParent()).length);
        } else {
            org.junit.jupiter.api.Assertions.assertEquals((Object)targetExists, (Object)lfs.exists(target));
        }
        org.junit.jupiter.api.Assertions.assertEquals((int)exitCode, (int)gotExit);
    }

    @Test
    public void testRepresentsDir() throws Exception {
        Path subdirDstPath = new Path(dstPath, srcPath.getName());
        String[] argv = null;
        lfs.delete(dstPath, true);
        org.junit.jupiter.api.Assertions.assertFalse((boolean)lfs.exists(dstPath));
        argv = new String[]{"-put", srcPath.toString(), dstPath.toString()};
        org.junit.jupiter.api.Assertions.assertEquals((int)0, (int)shell.run(argv));
        org.junit.jupiter.api.Assertions.assertTrue((lfs.exists(dstPath) && lfs.getFileStatus(dstPath).isFile() ? 1 : 0) != 0);
        lfs.delete(dstPath, true);
        org.junit.jupiter.api.Assertions.assertFalse((boolean)lfs.exists(dstPath));
        lfs.delete(dstPath, true);
        for (String suffix : new String[]{"/", "/."}) {
            argv = new String[]{"-put", srcPath.toString(), dstPath.toString() + suffix};
            org.junit.jupiter.api.Assertions.assertEquals((int)1, (int)shell.run(argv));
            org.junit.jupiter.api.Assertions.assertFalse((boolean)lfs.exists(dstPath));
            org.junit.jupiter.api.Assertions.assertFalse((boolean)lfs.exists(subdirDstPath));
        }
        for (String suffix : new String[]{"/", "/."}) {
            lfs.delete(dstPath, true);
            lfs.mkdirs(dstPath);
            argv = new String[]{"-put", srcPath.toString(), dstPath.toString() + suffix};
            org.junit.jupiter.api.Assertions.assertEquals((int)0, (int)shell.run(argv));
            org.junit.jupiter.api.Assertions.assertTrue((boolean)lfs.exists(subdirDstPath));
            org.junit.jupiter.api.Assertions.assertTrue((boolean)lfs.getFileStatus(subdirDstPath).isFile());
        }
        String dotdotDst = dstPath + "/foo/..";
        lfs.delete(dstPath, true);
        lfs.mkdirs(new Path(dstPath, "foo"));
        argv = new String[]{"-put", srcPath.toString(), dotdotDst};
        org.junit.jupiter.api.Assertions.assertEquals((int)0, (int)shell.run(argv));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)lfs.exists(subdirDstPath));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)lfs.getFileStatus(subdirDstPath).isFile());
    }

    @Test
    public void testCopyMerge() throws Exception {
        Path root = new Path(testRootDir, "TestMerge");
        Path f1 = new Path(root, "f1");
        Path f2 = new Path(root, "f2");
        Path f3 = new Path(root, "f3");
        Path empty = new Path(root, "empty");
        Path fnf = new Path(root, "fnf");
        Path d = new Path(root, "dir");
        Path df1 = new Path(d, "df1");
        Path df2 = new Path(d, "df2");
        Path df3 = new Path(d, "df3");
        this.createFile(f1, f2, f3, df1, df2, df3);
        this.createEmptyFile(empty);
        int exit = shell.run(new String[]{"-getmerge", f1.toString(), "out"});
        org.junit.jupiter.api.Assertions.assertEquals((int)0, (int)exit);
        org.junit.jupiter.api.Assertions.assertEquals((Object)"f1", (Object)this.readFile("out"));
        exit = shell.run(new String[]{"-getmerge", fnf.toString(), "out"});
        org.junit.jupiter.api.Assertions.assertEquals((int)1, (int)exit);
        org.junit.jupiter.api.Assertions.assertFalse((boolean)lfs.exists(new Path("out")));
        exit = shell.run(new String[]{"-getmerge", f1.toString(), f2.toString(), "out"});
        org.junit.jupiter.api.Assertions.assertEquals((int)0, (int)exit);
        org.junit.jupiter.api.Assertions.assertEquals((Object)"f1f2", (Object)this.readFile("out"));
        exit = shell.run(new String[]{"-getmerge", f2.toString(), f1.toString(), "out"});
        org.junit.jupiter.api.Assertions.assertEquals((int)0, (int)exit);
        org.junit.jupiter.api.Assertions.assertEquals((Object)"f2f1", (Object)this.readFile("out"));
        exit = shell.run(new String[]{"-getmerge", "-nl", f1.toString(), f2.toString(), "out"});
        org.junit.jupiter.api.Assertions.assertEquals((int)0, (int)exit);
        org.junit.jupiter.api.Assertions.assertEquals((Object)"f1\nf2\n", (Object)this.readFile("out"));
        exit = shell.run(new String[]{"-getmerge", "-nl", "-skip-empty-file", f1.toString(), f2.toString(), empty.toString(), "out"});
        org.junit.jupiter.api.Assertions.assertEquals((int)0, (int)exit);
        org.junit.jupiter.api.Assertions.assertEquals((Object)"f1\nf2\n", (Object)this.readFile("out"));
        shell.run(new String[]{"-getmerge", "-nl", new Path(root, "f*").toString(), "out"});
        org.junit.jupiter.api.Assertions.assertEquals((int)0, (int)exit);
        org.junit.jupiter.api.Assertions.assertEquals((Object)"f1\nf2\nf3\n", (Object)this.readFile("out"));
        shell.run(new String[]{"-getmerge", "-nl", root.toString(), "out"});
        org.junit.jupiter.api.Assertions.assertEquals((int)0, (int)exit);
        org.junit.jupiter.api.Assertions.assertEquals((Object)"\nf1\nf2\nf3\n", (Object)this.readFile("out"));
        shell.run(new String[]{"-getmerge", "-nl", d.toString(), "out"});
        org.junit.jupiter.api.Assertions.assertEquals((int)0, (int)exit);
        org.junit.jupiter.api.Assertions.assertEquals((Object)"df1\ndf2\ndf3\n", (Object)this.readFile("out"));
        shell.run(new String[]{"-getmerge", "-nl", f1.toString(), d.toString(), f2.toString(), "out"});
        org.junit.jupiter.api.Assertions.assertEquals((int)0, (int)exit);
        org.junit.jupiter.api.Assertions.assertEquals((Object)"f1\ndf1\ndf2\ndf3\nf2\n", (Object)this.readFile("out"));
    }

    @Test
    public void testMoveFileFromLocal() throws Exception {
        Path testRoot = new Path(testRootDir, "testPutFile");
        lfs.delete(testRoot, true);
        lfs.mkdirs(testRoot);
        Path target = new Path(testRoot, "target");
        Path srcFile = new Path(testRoot, new Path("srcFile"));
        lfs.createNewFile(srcFile);
        int exit = shell.run(new String[]{"-moveFromLocal", srcFile.toString(), target.toString()});
        org.junit.jupiter.api.Assertions.assertEquals((int)0, (int)exit);
        org.junit.jupiter.api.Assertions.assertFalse((boolean)lfs.exists(srcFile));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)lfs.exists(target));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)lfs.getFileStatus(target).isFile());
    }

    @Test
    public void testMoveFileFromLocalDestExists() throws Exception {
        Path testRoot = new Path(testRootDir, "testPutFile");
        lfs.delete(testRoot, true);
        lfs.mkdirs(testRoot);
        Path target = new Path(testRoot, "target");
        Path srcFile = new Path(testRoot, new Path("srcFile"));
        lfs.createNewFile(srcFile);
        int exit = shell.run(new String[]{"-moveFromLocal", srcFile.toString(), target.toString()});
        org.junit.jupiter.api.Assertions.assertEquals((int)0, (int)exit);
        org.junit.jupiter.api.Assertions.assertFalse((boolean)lfs.exists(srcFile));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)lfs.exists(target));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)lfs.getFileStatus(target).isFile());
        lfs.createNewFile(srcFile);
        exit = shell.run(new String[]{"-moveFromLocal", srcFile.toString(), target.toString()});
        org.junit.jupiter.api.Assertions.assertEquals((int)1, (int)exit);
        org.junit.jupiter.api.Assertions.assertTrue((boolean)lfs.exists(srcFile));
    }

    @Test
    public void testMoveDirFromLocal() throws Exception {
        Path testRoot = new Path(testRootDir, "testPutDir");
        lfs.delete(testRoot, true);
        lfs.mkdirs(testRoot);
        Path srcDir = new Path(testRoot, "srcDir");
        lfs.mkdirs(srcDir);
        Path targetDir = new Path(testRoot, "target");
        int exit = shell.run(new String[]{"-moveFromLocal", srcDir.toString(), targetDir.toString()});
        org.junit.jupiter.api.Assertions.assertEquals((int)0, (int)exit);
        org.junit.jupiter.api.Assertions.assertFalse((boolean)lfs.exists(srcDir));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)lfs.exists(targetDir));
    }

    @Test
    public void testMoveDirFromLocalDestExists() throws Exception {
        Path testRoot = new Path(testRootDir, "testPutDir");
        lfs.delete(testRoot, true);
        lfs.mkdirs(testRoot);
        Path srcDir = new Path(testRoot, "srcDir");
        lfs.mkdirs(srcDir);
        Path targetDir = new Path(testRoot, "target");
        lfs.mkdirs(targetDir);
        int exit = shell.run(new String[]{"-moveFromLocal", srcDir.toString(), targetDir.toString()});
        org.junit.jupiter.api.Assertions.assertEquals((int)0, (int)exit);
        org.junit.jupiter.api.Assertions.assertFalse((boolean)lfs.exists(srcDir));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)lfs.exists(new Path(targetDir, srcDir.getName())));
        lfs.mkdirs(srcDir);
        exit = shell.run(new String[]{"-moveFromLocal", srcDir.toString(), targetDir.toString()});
        org.junit.jupiter.api.Assertions.assertEquals((int)1, (int)exit);
        org.junit.jupiter.api.Assertions.assertTrue((boolean)lfs.exists(srcDir));
    }

    @Test
    public void testMoveFromWindowsLocalPath() throws Exception {
        PlatformAssumptions.assumeWindows();
        Path testRoot = new Path(testRootDir, "testPutFile");
        lfs.delete(testRoot, true);
        lfs.mkdirs(testRoot);
        Path target = new Path(testRoot, "target");
        Path srcFile = new Path(testRoot, new Path("srcFile"));
        lfs.createNewFile(srcFile);
        String winSrcFile = new File(srcFile.toUri().getPath().toString()).getAbsolutePath();
        this.shellRun(0, "-moveFromLocal", winSrcFile, target.toString());
        org.junit.jupiter.api.Assertions.assertFalse((boolean)lfs.exists(srcFile));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)lfs.exists(target));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)lfs.getFileStatus(target).isFile());
    }

    @Test
    public void testGetWindowsLocalPath() throws Exception {
        PlatformAssumptions.assumeWindows();
        String winDstFile = new File(dstPath.toUri().getPath().toString()).getAbsolutePath();
        this.shellRun(0, "-get", srcPath.toString(), winDstFile);
        this.checkPath(dstPath, false);
    }

    @Test
    public void testDirectCopy() throws Exception {
        Path testRoot = new Path(testRootDir, "testPutFile");
        lfs.delete(testRoot, true);
        lfs.mkdirs(testRoot);
        Path target_COPYING_File = new Path(testRoot, "target._COPYING_");
        Path target_File = new Path(testRoot, "target");
        Path srcFile = new Path(testRoot, new Path("srcFile"));
        lfs.createNewFile(srcFile);
        this.checkDirectCopy(srcFile, target_File, target_COPYING_File, false);
        shell.run(new String[]{"-rm", target_File.toString()});
        this.checkDirectCopy(srcFile, target_File, target_COPYING_File, true);
    }

    private void checkDirectCopy(Path srcFile, Path target_File, Path target_COPYING_File, boolean direct) throws Exception {
        int directWriteExitCode = direct ? 0 : 1;
        shell.run(new String[]{"-copyFromLocal", srcFile.toString(), target_COPYING_File.toString()});
        int srcFileexist = shell.run(new String[]{"-cat", target_COPYING_File.toString()});
        org.junit.jupiter.api.Assertions.assertEquals((int)0, (int)srcFileexist);
        if (!direct) {
            shell.run(new String[]{"-copyFromLocal", srcFile.toString(), target_File.toString()});
        } else {
            shell.run(new String[]{"-copyFromLocal", "-d", srcFile.toString(), target_File.toString()});
        }
        srcFileexist = shell.run(new String[]{"-cat", target_COPYING_File.toString()});
        org.junit.jupiter.api.Assertions.assertEquals((int)directWriteExitCode, (int)srcFileexist);
    }

    private void createFile(Path ... paths) throws IOException {
        for (Path path : paths) {
            FSDataOutputStream out = lfs.create(path);
            out.write(path.getName().getBytes());
            out.close();
        }
    }

    private void createEmptyFile(Path ... paths) throws IOException {
        for (Path path : paths) {
            FSDataOutputStream out = lfs.create(path);
            out.close();
        }
    }

    private String readFile(String out) throws IOException {
        Path path = new Path(out);
        FileStatus stat = lfs.getFileStatus(path);
        FSDataInputStream in = lfs.open(path);
        byte[] buffer = new byte[(int)stat.getLen()];
        in.readFully(buffer);
        in.close();
        lfs.delete(path, false);
        return new String(buffer);
    }

    private String pathAsString(Path p) {
        String s = p == null ? "." : p.toString();
        return s.isEmpty() ? "." : s;
    }

    @Test
    public void testCopyNoParent() throws Exception {
        String noDirName = "noDir";
        Path noDir = new Path("noDir");
        lfs.delete(noDir, true);
        Assertions.assertThat((boolean)lfs.exists(noDir)).isFalse();
        ((AbstractIntegerAssert)Assertions.assertThat((int)this.shellRun("-put", srcPath.toString(), "noDir/foo")).as("Expected failed put to a path without parent directory", new Object[0])).isNotEqualTo(0);
        ((AbstractIntegerAssert)Assertions.assertThat((int)this.shellRun("-copyFromLocal", srcPath.toString(), "noDir/")).as("Expected failed copyFromLocal to a non-existent directory", new Object[0])).isNotEqualTo(0);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testPutSrcDirNoPerm() throws Exception {
        Path src = new Path(testRootDir, "srcNoPerm");
        Path dst = new Path(testRootDir, "dst");
        lfs.delete(src, true);
        lfs.mkdirs(src, new FsPermission(0));
        lfs.delete(dst, true);
        try {
            ByteArrayOutputStream err = new ByteArrayOutputStream();
            PrintStream oldErr = System.err;
            System.setErr(new PrintStream(err));
            this.shellRun(1, "-put", src.toString(), dst.toString());
            System.setErr(oldErr);
            System.err.print(err.toString());
            org.junit.jupiter.api.Assertions.assertTrue((boolean)err.toString().contains("Permission denied"));
        }
        finally {
            lfs.setPermission(src, new FsPermission(493));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testPutSrcFileNoPerm() throws Exception {
        Path src = new Path(testRootDir, "srcNoPerm");
        Path dst = new Path(testRootDir, "dst");
        lfs.delete(src, true);
        lfs.create(src);
        lfs.setPermission(src, new FsPermission(0));
        lfs.delete(dst, true);
        try {
            ByteArrayOutputStream err = new ByteArrayOutputStream();
            PrintStream oldErr = System.err;
            System.setErr(new PrintStream(err));
            this.shellRun(1, "-put", src.toString(), dst.toString());
            System.setErr(oldErr);
            System.err.print(err.toString());
            org.junit.jupiter.api.Assertions.assertTrue((boolean)err.toString().contains("(Permission denied)"));
        }
        finally {
            lfs.setPermission(src, new FsPermission(493));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testLazyPersistDirectOverwrite() throws Exception {
        Path testRoot = new Path(testRootDir, "testLazyPersistDirectOverwrite");
        try {
            lfs.delete(testRoot, true);
            lfs.mkdirs(testRoot);
            Path filePath = new Path(testRoot, new Path("srcFile"));
            lfs.create(filePath).close();
            String[] argv = new String[]{"-put", "-f", "-l", "-d", filePath.toString(), filePath.toString()};
            org.junit.jupiter.api.Assertions.assertEquals((int)0, (int)shell.run(argv));
            argv = new String[]{"-put", "-l", "-d", filePath.toString(), filePath.toString()};
            org.junit.jupiter.api.Assertions.assertNotEquals((int)0, (int)shell.run(argv));
        }
        finally {
            lfs.delete(testRoot, true);
        }
    }
}

