/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.security;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeoutException;
import java.util.function.Supplier;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.security.Groups;
import org.apache.hadoop.security.ShellBasedUnixGroupsMapping;
import org.apache.hadoop.test.GenericTestUtils;
import org.apache.hadoop.util.FakeTimer;
import org.apache.hadoop.util.Timer;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TestGroupsCaching {
    public static final Logger TESTLOG = LoggerFactory.getLogger(TestGroupsCaching.class);
    private static String[] myGroups = new String[]{"grp1", "grp2"};
    private Configuration conf;

    @BeforeEach
    public void setup() throws IOException {
        FakeGroupMapping.clearAll();
        ExceptionalGroupMapping.resetRequestCount();
        this.conf = new Configuration();
        this.conf.setClass("hadoop.security.group.mapping", FakeGroupMapping.class, ShellBasedUnixGroupsMapping.class);
    }

    @Test
    public void testGroupsCaching() throws Exception {
        Groups groups;
        block2: {
            this.conf.setLong("hadoop.security.groups.negative-cache.secs", 0L);
            groups = new Groups(this.conf);
            groups.cacheGroupsAdd(Arrays.asList(myGroups));
            groups.refresh();
            FakeGroupMapping.clearBlackList();
            FakeGroupMapping.addToBlackList("user1");
            org.junit.jupiter.api.Assertions.assertTrue((groups.getGroups("me").size() == 2 ? 1 : 0) != 0);
            FakeGroupMapping.addToBlackList("me");
            org.junit.jupiter.api.Assertions.assertTrue((groups.getGroups("me").size() == 2 ? 1 : 0) != 0);
            try {
                TESTLOG.error("We are not supposed to get here." + groups.getGroups("user1").toString());
                org.junit.jupiter.api.Assertions.fail();
            }
            catch (IOException ioe) {
                if (ioe.getMessage().startsWith("No groups found")) break block2;
                TESTLOG.error("Got unexpected exception: " + ioe.getMessage());
                org.junit.jupiter.api.Assertions.fail();
            }
        }
        FakeGroupMapping.clearBlackList();
        org.junit.jupiter.api.Assertions.assertTrue((groups.getGroups("user1").size() == 2 ? 1 : 0) != 0);
    }

    @Test
    public void testGroupLookupForStaticUsers() throws Exception {
        this.conf.setClass("hadoop.security.group.mapping", FakeunPrivilegedGroupMapping.class, ShellBasedUnixGroupsMapping.class);
        this.conf.set("hadoop.user.group.static.mapping.overrides", "me=;user1=group1;user2=group1,group2");
        Groups groups = new Groups(this.conf);
        List userGroups = groups.getGroups("me");
        org.junit.jupiter.api.Assertions.assertTrue((boolean)userGroups.isEmpty(), (String)"non-empty groups for static user");
        org.junit.jupiter.api.Assertions.assertFalse((boolean)FakeunPrivilegedGroupMapping.invoked, (String)"group lookup done for static user");
        ArrayList<String> expected = new ArrayList<String>();
        expected.add("group1");
        FakeunPrivilegedGroupMapping.invoked = false;
        userGroups = groups.getGroups("user1");
        org.junit.jupiter.api.Assertions.assertTrue((boolean)expected.equals(userGroups), (String)"groups not correct");
        org.junit.jupiter.api.Assertions.assertFalse((boolean)FakeunPrivilegedGroupMapping.invoked, (String)"group lookup done for unprivileged user");
        expected.add("group2");
        FakeunPrivilegedGroupMapping.invoked = false;
        userGroups = groups.getGroups("user2");
        org.junit.jupiter.api.Assertions.assertTrue((boolean)expected.equals(userGroups), (String)"groups not correct");
        org.junit.jupiter.api.Assertions.assertFalse((boolean)FakeunPrivilegedGroupMapping.invoked, (String)"group lookup done for unprivileged user");
    }

    @Test
    public void testNegativeGroupCaching() throws Exception {
        String user = "negcache";
        String failMessage = "Did not throw IOException: ";
        this.conf.setLong("hadoop.security.groups.negative-cache.secs", 2L);
        FakeTimer timer = new FakeTimer();
        Groups groups = new Groups(this.conf, (Timer)timer);
        groups.cacheGroupsAdd(Arrays.asList(myGroups));
        groups.refresh();
        FakeGroupMapping.addToBlackList("negcache");
        try {
            groups.getGroups("negcache");
            org.junit.jupiter.api.Assertions.fail((String)"Did not throw IOException: Failed to obtain groups from FakeGroupMapping.");
        }
        catch (IOException e) {
            GenericTestUtils.assertExceptionContains("No groups found for user", e);
        }
        try {
            groups.getGroups("negcache");
            org.junit.jupiter.api.Assertions.fail((String)"Did not throw IOException: The user is in the negative cache.");
        }
        catch (IOException e) {
            GenericTestUtils.assertExceptionContains("No groups found for user", e);
        }
        FakeGroupMapping.clearBlackList();
        try {
            groups.getGroups("negcache");
            org.junit.jupiter.api.Assertions.fail((String)"Did not throw IOException: The user is still in the negative cache, even FakeGroupMapping has resumed.");
        }
        catch (IOException e) {
            GenericTestUtils.assertExceptionContains("No groups found for user", e);
        }
        timer.advance(4000L);
        org.junit.jupiter.api.Assertions.assertEquals(Arrays.asList(myGroups), (Object)groups.getGroups("negcache"));
    }

    @Test
    public void testCachePreventsImplRequest() throws Exception {
        this.conf.setLong("hadoop.security.groups.negative-cache.secs", 0L);
        Groups groups = new Groups(this.conf);
        groups.cacheGroupsAdd(Arrays.asList(myGroups));
        groups.refresh();
        FakeGroupMapping.clearBlackList();
        org.junit.jupiter.api.Assertions.assertEquals((int)0, (int)FakeGroupMapping.getRequestCount());
        org.junit.jupiter.api.Assertions.assertTrue((groups.getGroups("me").size() == 2 ? 1 : 0) != 0);
        org.junit.jupiter.api.Assertions.assertEquals((int)1, (int)FakeGroupMapping.getRequestCount());
        org.junit.jupiter.api.Assertions.assertTrue((groups.getGroups("me").size() == 2 ? 1 : 0) != 0);
        org.junit.jupiter.api.Assertions.assertEquals((int)1, (int)FakeGroupMapping.getRequestCount());
    }

    @Test
    public void testExceptionsFromImplNotCachedInNegativeCache() {
        this.conf.setClass("hadoop.security.group.mapping", ExceptionalGroupMapping.class, ShellBasedUnixGroupsMapping.class);
        this.conf.setLong("hadoop.security.groups.negative-cache.secs", 10000L);
        Groups groups = new Groups(this.conf);
        groups.cacheGroupsAdd(Arrays.asList(myGroups));
        groups.refresh();
        org.junit.jupiter.api.Assertions.assertEquals((int)0, (int)ExceptionalGroupMapping.getRequestCount());
        try {
            groups.getGroups("anything");
            org.junit.jupiter.api.Assertions.fail((String)"Should have thrown");
        }
        catch (IOException iOException) {
            // empty catch block
        }
        org.junit.jupiter.api.Assertions.assertEquals((int)1, (int)ExceptionalGroupMapping.getRequestCount());
        try {
            groups.getGroups("anything");
            org.junit.jupiter.api.Assertions.fail((String)"Should have thrown");
        }
        catch (IOException iOException) {
            // empty catch block
        }
        org.junit.jupiter.api.Assertions.assertEquals((int)2, (int)ExceptionalGroupMapping.getRequestCount());
    }

    @Test
    public void testOnlyOneRequestWhenNoEntryIsCached() throws Exception {
        this.conf.setLong("hadoop.security.groups.negative-cache.secs", 0L);
        final Groups groups = new Groups(this.conf);
        groups.cacheGroupsAdd(Arrays.asList(myGroups));
        groups.refresh();
        FakeGroupMapping.clearBlackList();
        FakeGroupMapping.setGetGroupsDelayMs(100L);
        ArrayList<1> threads = new ArrayList<1>();
        for (int i = 0; i < 10; ++i) {
            threads.add(new Thread(){

                @Override
                public void run() {
                    try {
                        org.junit.jupiter.api.Assertions.assertEquals((int)2, (int)groups.getGroups("me").size());
                    }
                    catch (IOException e) {
                        org.junit.jupiter.api.Assertions.fail((String)"Should not happen");
                    }
                }
            });
        }
        for (Thread thread : threads) {
            thread.start();
        }
        for (Thread thread : threads) {
            thread.join();
        }
        org.junit.jupiter.api.Assertions.assertEquals((int)1, (int)FakeGroupMapping.getRequestCount());
    }

    @Test
    public void testOnlyOneRequestWhenExpiredEntryExists() throws Exception {
        this.conf.setLong("hadoop.security.groups.cache.secs", 1L);
        FakeTimer timer = new FakeTimer();
        final Groups groups = new Groups(this.conf, (Timer)timer);
        groups.cacheGroupsAdd(Arrays.asList(myGroups));
        groups.refresh();
        FakeGroupMapping.clearBlackList();
        FakeGroupMapping.setGetGroupsDelayMs(100L);
        groups.getGroups("me");
        int startingRequestCount = FakeGroupMapping.getRequestCount();
        timer.advance(400000L);
        Thread.sleep(100L);
        ArrayList<2> threads = new ArrayList<2>();
        for (int i = 0; i < 10; ++i) {
            threads.add(new Thread(){

                @Override
                public void run() {
                    try {
                        org.junit.jupiter.api.Assertions.assertEquals((int)2, (int)groups.getGroups("me").size());
                    }
                    catch (IOException e) {
                        org.junit.jupiter.api.Assertions.fail((String)"Should not happen");
                    }
                }
            });
        }
        for (Thread thread : threads) {
            thread.start();
        }
        for (Thread thread : threads) {
            thread.join();
        }
        org.junit.jupiter.api.Assertions.assertEquals((int)(startingRequestCount + 1), (int)FakeGroupMapping.getRequestCount());
    }

    @Test
    public void testThreadNotBlockedWhenExpiredEntryExistsWithBackgroundRefresh() throws Exception {
        this.conf.setLong("hadoop.security.groups.cache.secs", 1L);
        this.conf.setBoolean("hadoop.security.groups.cache.background.reload", true);
        FakeTimer timer = new FakeTimer();
        Groups groups = new Groups(this.conf, (Timer)timer);
        groups.cacheGroupsAdd(Arrays.asList(myGroups));
        groups.refresh();
        FakeGroupMapping.clearBlackList();
        groups.getGroups("me");
        FakeGroupMapping.setGetGroupsDelayMs(100L);
        groups.cacheGroupsAdd(Arrays.asList("grp3"));
        int startingRequestCount = FakeGroupMapping.getRequestCount();
        timer.advance(4000L);
        Assertions.assertThat((int)groups.getGroups("me").size()).isEqualTo(2);
        org.junit.jupiter.api.Assertions.assertEquals((int)startingRequestCount, (int)FakeGroupMapping.getRequestCount());
        Thread.sleep(110L);
        org.junit.jupiter.api.Assertions.assertEquals((int)(startingRequestCount + 1), (int)FakeGroupMapping.getRequestCount());
        Assertions.assertThat((int)groups.getGroups("me").size()).isEqualTo(3);
    }

    @Test
    public void testThreadBlockedWhenExpiredEntryExistsWithoutBackgroundRefresh() throws Exception {
        this.conf.setLong("hadoop.security.groups.cache.secs", 1L);
        this.conf.setBoolean("hadoop.security.groups.cache.background.reload", false);
        FakeTimer timer = new FakeTimer();
        Groups groups = new Groups(this.conf, (Timer)timer);
        groups.cacheGroupsAdd(Arrays.asList(myGroups));
        groups.refresh();
        FakeGroupMapping.clearBlackList();
        groups.getGroups("me");
        FakeGroupMapping.setGetGroupsDelayMs(100L);
        groups.cacheGroupsAdd(Arrays.asList("grp3"));
        int startingRequestCount = FakeGroupMapping.getRequestCount();
        timer.advance(4000L);
        Assertions.assertThat((int)groups.getGroups("me").size()).isEqualTo(3);
        org.junit.jupiter.api.Assertions.assertEquals((int)(startingRequestCount + 1), (int)FakeGroupMapping.getRequestCount());
    }

    @Test
    public void testExceptionOnBackgroundRefreshHandled() throws Exception {
        this.conf.setLong("hadoop.security.groups.cache.secs", 1L);
        this.conf.setBoolean("hadoop.security.groups.cache.background.reload", true);
        FakeTimer timer = new FakeTimer();
        Groups groups = new Groups(this.conf, (Timer)timer);
        groups.cacheGroupsAdd(Arrays.asList(myGroups));
        groups.refresh();
        FakeGroupMapping.clearBlackList();
        List g1 = groups.getGroups("me");
        groups.cacheGroupsAdd(Arrays.asList("grp3"));
        int startingRequestCount = FakeGroupMapping.getRequestCount();
        FakeGroupMapping.setThrowException(true);
        timer.advance(4000L);
        FakeGroupMapping.pause();
        Assertions.assertThat((int)groups.getGroups("me").size()).isEqualTo(2);
        org.junit.jupiter.api.Assertions.assertEquals((int)startingRequestCount, (int)FakeGroupMapping.getRequestCount());
        FakeGroupMapping.resume();
        this.waitForGroupCounters(groups, 0L, 0L, 0L, 1L);
        FakeGroupMapping.setThrowException(false);
        org.junit.jupiter.api.Assertions.assertEquals((int)(startingRequestCount + 1), (int)FakeGroupMapping.getRequestCount());
        Assertions.assertThat((int)groups.getGroups("me").size()).isEqualTo(2);
        this.waitForGroupCounters(groups, 0L, 0L, 1L, 1L);
        org.junit.jupiter.api.Assertions.assertEquals((int)(startingRequestCount + 2), (int)FakeGroupMapping.getRequestCount());
        Assertions.assertThat((int)groups.getGroups("me").size()).isEqualTo(3);
    }

    @Test
    public void testEntriesExpireIfBackgroundRefreshFails() throws Exception {
        this.conf.setLong("hadoop.security.groups.cache.secs", 1L);
        this.conf.setBoolean("hadoop.security.groups.cache.background.reload", true);
        FakeTimer timer = new FakeTimer();
        Groups groups = new Groups(this.conf, (Timer)timer);
        groups.cacheGroupsAdd(Arrays.asList(myGroups));
        groups.refresh();
        FakeGroupMapping.clearBlackList();
        groups.getGroups("me");
        FakeGroupMapping.setThrowException(true);
        for (int i = 0; i < 9; ++i) {
            Assertions.assertThat((int)groups.getGroups("me").size()).isEqualTo(2);
            timer.advance(1000L);
        }
        timer.advance(2000L);
        try {
            groups.getGroups("me");
            org.junit.jupiter.api.Assertions.fail((String)"Should have thrown an exception here");
        }
        catch (Exception exception) {
            // empty catch block
        }
        FakeGroupMapping.setThrowException(false);
        Assertions.assertThat((int)groups.getGroups("me").size()).isEqualTo(2);
    }

    @Test
    public void testBackgroundRefreshCounters() throws IOException, InterruptedException {
        String[] grps;
        this.conf.setLong("hadoop.security.groups.cache.secs", 1L);
        this.conf.setBoolean("hadoop.security.groups.cache.background.reload", true);
        this.conf.setInt("hadoop.security.groups.cache.background.reload.threads", 2);
        FakeTimer timer = new FakeTimer();
        Groups groups = new Groups(this.conf, (Timer)timer);
        groups.cacheGroupsAdd(Arrays.asList(myGroups));
        groups.refresh();
        FakeGroupMapping.clearBlackList();
        for (String g : grps = new String[]{"one", "two", "three", "four", "five"}) {
            groups.getGroups(g);
        }
        timer.advance(2000L);
        FakeGroupMapping.pause();
        for (String g : grps) {
            groups.getGroups(g);
        }
        this.waitForGroupCounters(groups, 3L, 2L, 0L, 0L);
        FakeGroupMapping.resume();
        this.waitForGroupCounters(groups, 0L, 0L, 5L, 0L);
        timer.advance(2000L);
        FakeGroupMapping.setGetGroupsDelayMs(0L);
        FakeGroupMapping.setThrowException(true);
        for (String g : grps) {
            groups.getGroups(g);
        }
        this.waitForGroupCounters(groups, 0L, 0L, 5L, 5L);
    }

    private void waitForGroupCounters(final Groups groups, long expectedQueued, long expectedRunning, long expectedSuccess, long expectedExpection) throws InterruptedException {
        final long[] expected = new long[]{expectedQueued, expectedRunning, expectedSuccess, expectedExpection};
        final long[] actual = new long[expected.length];
        try {
            GenericTestUtils.waitFor(new Supplier<Boolean>(){

                @Override
                public Boolean get() {
                    actual[0] = groups.getBackgroundRefreshQueued();
                    actual[1] = groups.getBackgroundRefreshRunning();
                    actual[2] = groups.getBackgroundRefreshSuccess();
                    actual[3] = groups.getBackgroundRefreshException();
                    return Arrays.equals(actual, expected);
                }
            }, 20L, 1000L);
        }
        catch (TimeoutException e) {
            org.junit.jupiter.api.Assertions.fail((String)("Excepted group counter values are not reached in given time, expecting (Queued, Running, Success, Exception) : " + Arrays.toString(expected) + " but actual : " + Arrays.toString(actual)));
        }
    }

    @Test
    public void testExceptionCallingLoadWithoutBackgroundRefreshReturnsOldValue() throws Exception {
        this.conf.setLong("hadoop.security.groups.cache.secs", 1L);
        this.conf.setBoolean("hadoop.security.groups.cache.background.reload", false);
        FakeTimer timer = new FakeTimer();
        Groups groups = new Groups(this.conf, (Timer)timer);
        groups.cacheGroupsAdd(Arrays.asList(myGroups));
        groups.refresh();
        FakeGroupMapping.clearBlackList();
        Assertions.assertThat((int)groups.getGroups("me").size()).isEqualTo(2);
        timer.advance(2000L);
        FakeGroupMapping.setThrowException(true);
        Assertions.assertThat((int)groups.getGroups("me").size()).isEqualTo(2);
    }

    @Test
    public void testCacheEntriesExpire() throws Exception {
        this.conf.setLong("hadoop.security.groups.cache.secs", 1L);
        FakeTimer timer = new FakeTimer();
        Groups groups = new Groups(this.conf, (Timer)timer);
        groups.cacheGroupsAdd(Arrays.asList(myGroups));
        groups.refresh();
        FakeGroupMapping.clearBlackList();
        groups.getGroups("me");
        int startingRequestCount = FakeGroupMapping.getRequestCount();
        timer.advance(20000L);
        groups.getGroups("me");
        org.junit.jupiter.api.Assertions.assertEquals((int)(startingRequestCount + 1), (int)FakeGroupMapping.getRequestCount());
    }

    @Test
    public void testNegativeCacheClearedOnRefresh() throws Exception {
        this.conf.setLong("hadoop.security.groups.negative-cache.secs", 100L);
        Groups groups = new Groups(this.conf);
        groups.cacheGroupsAdd(Arrays.asList(myGroups));
        groups.refresh();
        FakeGroupMapping.clearBlackList();
        FakeGroupMapping.addToBlackList("dne");
        try {
            groups.getGroups("dne");
            org.junit.jupiter.api.Assertions.fail((String)"Should have failed to find this group");
        }
        catch (IOException iOException) {
            // empty catch block
        }
        int startingRequestCount = FakeGroupMapping.getRequestCount();
        groups.refresh();
        FakeGroupMapping.addToBlackList("dne");
        try {
            List g = groups.getGroups("dne");
            org.junit.jupiter.api.Assertions.fail((String)"Should have failed to find this group");
        }
        catch (IOException iOException) {
            // empty catch block
        }
        org.junit.jupiter.api.Assertions.assertEquals((int)(startingRequestCount + 1), (int)FakeGroupMapping.getRequestCount());
    }

    @Test
    public void testNegativeCacheEntriesExpire() throws Exception {
        this.conf.setLong("hadoop.security.groups.negative-cache.secs", 2L);
        FakeTimer timer = new FakeTimer();
        Groups groups = new Groups(this.conf, (Timer)timer);
        groups.cacheGroupsAdd(Arrays.asList(myGroups));
        groups.refresh();
        FakeGroupMapping.addToBlackList("user1");
        FakeGroupMapping.addToBlackList("user2");
        try {
            groups.getGroups("user1");
            org.junit.jupiter.api.Assertions.fail((String)"Did not throw IOException : Failed to obtain groups from FakeGroupMapping.");
        }
        catch (IOException e) {
            GenericTestUtils.assertExceptionContains("No groups found for user", e);
        }
        org.junit.jupiter.api.Assertions.assertTrue((boolean)groups.getNegativeCache().contains("user1"));
        timer.advance(1000L);
        try {
            groups.getGroups("user2");
            org.junit.jupiter.api.Assertions.fail((String)"Did not throw IOException : Failed to obtain groups from FakeGroupMapping.");
        }
        catch (IOException e) {
            GenericTestUtils.assertExceptionContains("No groups found for user", e);
        }
        org.junit.jupiter.api.Assertions.assertTrue((boolean)groups.getNegativeCache().contains("user2"));
        timer.advance(1100L);
        org.junit.jupiter.api.Assertions.assertFalse((boolean)groups.getNegativeCache().contains("user1"));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)groups.getNegativeCache().contains("user2"));
        timer.advance(1000L);
        org.junit.jupiter.api.Assertions.assertFalse((boolean)groups.getNegativeCache().contains("user2"));
    }

    public static class FakeGroupMapping
    extends ShellBasedUnixGroupsMapping {
        private static Set<String> allGroups = new LinkedHashSet<String>();
        private static Set<String> blackList = new HashSet<String>();
        private static int requestCount = 0;
        private static long getGroupsDelayMs = 0L;
        private static boolean throwException;
        private static volatile CountDownLatch latch;

        public Set<String> getGroupsSet(String user) throws IOException {
            TESTLOG.info("Getting groups for " + user);
            this.delayIfNecessary();
            ++requestCount;
            if (throwException) {
                throw new IOException("For test");
            }
            if (blackList.contains(user)) {
                return Collections.emptySet();
            }
            return new LinkedHashSet<String>(allGroups);
        }

        public List<String> getGroups(String user) throws IOException {
            return new ArrayList<String>(this.getGroupsSet(user));
        }

        private void delayIfNecessary() {
            if (latch != null) {
                try {
                    latch.await();
                    return;
                }
                catch (InterruptedException interruptedException) {
                    // empty catch block
                }
            }
            if (getGroupsDelayMs > 0L) {
                try {
                    Thread.sleep(getGroupsDelayMs);
                }
                catch (InterruptedException e) {
                    throw new RuntimeException(e);
                }
            }
        }

        public void cacheGroupsRefresh() throws IOException {
            TESTLOG.info("Cache is being refreshed.");
            FakeGroupMapping.clearBlackList();
        }

        public static void clearBlackList() throws IOException {
            TESTLOG.info("Clearing the blacklist");
            blackList.clear();
        }

        public static void clearAll() throws IOException {
            TESTLOG.info("Resetting FakeGroupMapping");
            blackList.clear();
            allGroups.clear();
            FakeGroupMapping.resetRequestCount();
            getGroupsDelayMs = 0L;
            throwException = false;
            latch = null;
        }

        public void cacheGroupsAdd(List<String> groups) throws IOException {
            TESTLOG.info("Adding " + groups + " to groups.");
            allGroups.addAll(groups);
        }

        public static void addToBlackList(String user) throws IOException {
            TESTLOG.info("Adding " + user + " to the blacklist");
            blackList.add(user);
        }

        public static int getRequestCount() {
            return requestCount;
        }

        public static void resetRequestCount() {
            requestCount = 0;
        }

        public static void setGetGroupsDelayMs(long delayMs) {
            getGroupsDelayMs = delayMs;
        }

        public static void setThrowException(boolean throwIfTrue) {
            throwException = throwIfTrue;
        }

        public static void pause() {
            latch = new CountDownLatch(1);
        }

        public static void resume() {
            if (latch != null) {
                latch.countDown();
            }
        }

        static {
            latch = null;
        }
    }

    public static class ExceptionalGroupMapping
    extends ShellBasedUnixGroupsMapping {
        private static int requestCount = 0;

        public List<String> getGroups(String user) throws IOException {
            ++requestCount;
            throw new IOException("For test");
        }

        public Set<String> getGroupsSet(String user) throws IOException {
            ++requestCount;
            throw new IOException("For test");
        }

        public static int getRequestCount() {
            return requestCount;
        }

        public static void resetRequestCount() {
            requestCount = 0;
        }
    }

    public static class FakeunPrivilegedGroupMapping
    extends FakeGroupMapping {
        private static boolean invoked = false;

        @Override
        public List<String> getGroups(String user) throws IOException {
            invoked = true;
            return super.getGroups(user);
        }
    }
}

