/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.contract;

import java.io.Closeable;
import java.io.EOFException;
import java.io.IOException;
import java.util.Random;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.contract.AbstractFSContractTestBase;
import org.apache.hadoop.fs.contract.ContractTestUtils;
import org.apache.hadoop.io.IOUtils;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractContractSeekTest
extends AbstractFSContractTestBase {
    private static final Logger LOG = LoggerFactory.getLogger(AbstractContractSeekTest.class);
    public static final int DEFAULT_RANDOM_SEEK_COUNT = 100;
    private Path smallSeekFile;
    private Path zeroByteFile;
    private FSDataInputStream instream;

    @Override
    @BeforeEach
    public void setup() throws Exception {
        super.setup();
        this.skipIfUnsupported("supports-seek");
        this.smallSeekFile = this.path("seekfile.txt");
        this.zeroByteFile = this.path("zero.txt");
        byte[] block = ContractTestUtils.dataset(1024, 0, 255);
        FileSystem fs = this.getFileSystem();
        ContractTestUtils.createFile(fs, this.smallSeekFile, true, block);
        ContractTestUtils.touch(fs, this.zeroByteFile);
    }

    @Override
    protected Configuration createConfiguration() {
        Configuration conf = super.createConfiguration();
        conf.setInt("io.file.buffer.size", 4096);
        return conf;
    }

    @Override
    @AfterEach
    public void teardown() throws Exception {
        IOUtils.closeStream((Closeable)this.instream);
        this.instream = null;
        super.teardown();
    }

    protected void assumeSupportsPositionedReadable() throws IOException {
        if (!this.getContract().isSupported("supports-positioned-readable", true)) {
            ContractTestUtils.skip("Skipping as unsupported feature: supports-positioned-readable");
        }
    }

    @Test
    public void testSeekZeroByteFile() throws Throwable {
        this.describe("seek and read a 0 byte file");
        this.instream = this.getFileSystem().open(this.zeroByteFile);
        AbstractContractSeekTest.assertEquals((long)0L, (long)this.instream.getPos());
        int result = this.instream.read();
        this.assertMinusOne("initial byte read", result);
        byte[] buffer = new byte[1];
        this.instream.seek(0L);
        result = this.instream.read();
        this.assertMinusOne("post-seek byte read", result);
        result = this.instream.read(buffer, 0, 1);
        this.assertMinusOne("post-seek buffer read", result);
    }

    @Test
    public void testBlockReadZeroByteFile() throws Throwable {
        this.describe("do a block read on a 0 byte file");
        this.instream = this.getFileSystem().open(this.zeroByteFile);
        AbstractContractSeekTest.assertEquals((long)0L, (long)this.instream.getPos());
        byte[] buffer = new byte[1];
        int result = this.instream.read(buffer, 0, 1);
        this.assertMinusOne("block read zero byte file", result);
    }

    @Test
    public void testSeekReadClosedFile() throws Throwable {
        this.instream = this.getFileSystem().open(this.smallSeekFile);
        AbstractContractSeekTest.getLogger().debug("Stream is of type " + this.instream.getClass().getCanonicalName());
        this.instream.close();
        try {
            this.instream.seek(0L);
            if (!this.isSupported("supports-seek-on-closed-file")) {
                AbstractContractSeekTest.fail((String)"seek succeeded on a closed stream");
            }
        }
        catch (IOException iOException) {
            // empty catch block
        }
        try {
            int data = this.instream.available();
            if (!this.isSupported("supports-available-on-closed-file")) {
                AbstractContractSeekTest.fail((String)("available() succeeded on a closed stream, got " + data));
            }
        }
        catch (IOException data) {
            // empty catch block
        }
        try {
            int data = this.instream.read();
            AbstractContractSeekTest.fail((String)("read() succeeded on a closed stream, got " + data));
        }
        catch (IOException data) {
            // empty catch block
        }
        try {
            byte[] buffer = new byte[1];
            int result = this.instream.read(buffer, 0, 1);
            AbstractContractSeekTest.fail((String)("read(buffer, 0, 1) succeeded on a closed stream, got " + result));
        }
        catch (IOException iOException) {
            // empty catch block
        }
        try {
            long l = this.instream.getPos();
        }
        catch (IOException iOException) {
            // empty catch block
        }
        this.instream.close();
    }

    @Test
    public void testNegativeSeek() throws Throwable {
        this.instream = this.getFileSystem().open(this.smallSeekFile);
        AbstractContractSeekTest.assertEquals((long)0L, (long)this.instream.getPos());
        try {
            this.instream.seek(-1L);
            long p = this.instream.getPos();
            LOG.warn("Seek to -1 returned a position of " + p);
            int result = this.instream.read();
            AbstractContractSeekTest.fail((String)("expected an exception, got data " + result + " at a position of " + p));
        }
        catch (EOFException e) {
            this.handleExpectedException(e);
        }
        catch (IOException e) {
            this.handleRelaxedException("a negative seek", "EOFException", e);
        }
        AbstractContractSeekTest.assertEquals((long)0L, (long)this.instream.getPos());
    }

    @Test
    public void testSeekFile() throws Throwable {
        this.describe("basic seek operations");
        this.instream = this.getFileSystem().open(this.smallSeekFile);
        AbstractContractSeekTest.assertEquals((long)0L, (long)this.instream.getPos());
        this.instream.seek(0L);
        int result = this.instream.read();
        AbstractContractSeekTest.assertEquals((int)0, (int)result);
        AbstractContractSeekTest.assertEquals((int)1, (int)this.instream.read());
        AbstractContractSeekTest.assertEquals((long)2L, (long)this.instream.getPos());
        AbstractContractSeekTest.assertEquals((int)2, (int)this.instream.read());
        AbstractContractSeekTest.assertEquals((long)3L, (long)this.instream.getPos());
        this.instream.seek(128L);
        AbstractContractSeekTest.assertEquals((long)128L, (long)this.instream.getPos());
        AbstractContractSeekTest.assertEquals((int)128, (int)this.instream.read());
        this.instream.seek(63L);
        AbstractContractSeekTest.assertEquals((int)63, (int)this.instream.read());
    }

    @Test
    public void testSeekAndReadPastEndOfFile() throws Throwable {
        this.describe("verify that reading past the last bytes in the file returns -1");
        this.instream = this.getFileSystem().open(this.smallSeekFile);
        AbstractContractSeekTest.assertEquals((long)0L, (long)this.instream.getPos());
        this.instream.seek(1022L);
        AbstractContractSeekTest.assertTrue((this.instream.read() != -1 ? 1 : 0) != 0, (String)"Premature EOF");
        AbstractContractSeekTest.assertTrue((this.instream.read() != -1 ? 1 : 0) != 0, (String)"Premature EOF");
        this.assertMinusOne("read past end of file", this.instream.read());
    }

    @Test
    public void testSeekPastEndOfFileThenReseekAndRead() throws Throwable {
        this.describe("do a seek past the EOF, then verify the stream recovers");
        this.instream = this.getFileSystem().open(this.smallSeekFile);
        boolean canSeekPastEOF = !this.getContract().isSupported("rejects-seek-past-eof", true);
        try {
            this.instream.seek(1025L);
            this.assertMinusOne("read after seeking past EOF", this.instream.read());
        }
        catch (EOFException e) {
            if (canSeekPastEOF) {
                throw e;
            }
            this.handleExpectedException(e);
        }
        catch (IOException e) {
            if (canSeekPastEOF) {
                throw e;
            }
            this.handleRelaxedException("a seek past the end of the file", "EOFException", e);
        }
        this.instream.seek(1L);
        AbstractContractSeekTest.assertTrue((this.instream.read() != -1 ? 1 : 0) != 0, (String)"Premature EOF");
    }

    @Test
    public void testSeekBigFile() throws Throwable {
        this.describe("Seek round a large file and verify the bytes are what is expected");
        Path testSeekFile = this.path("bigseekfile.txt");
        byte[] block = ContractTestUtils.dataset(102400, 0, 255);
        ContractTestUtils.createFile(this.getFileSystem(), testSeekFile, true, block);
        this.instream = this.getFileSystem().open(testSeekFile);
        AbstractContractSeekTest.assertEquals((long)0L, (long)this.instream.getPos());
        this.instream.seek(0L);
        int result = this.instream.read();
        AbstractContractSeekTest.assertEquals((int)0, (int)result);
        AbstractContractSeekTest.assertEquals((int)1, (int)this.instream.read());
        AbstractContractSeekTest.assertEquals((int)2, (int)this.instream.read());
        this.instream.seek(32768L);
        AbstractContractSeekTest.assertEquals((byte)block[32768], (byte)((byte)this.instream.read()), (String)"@32768");
        this.instream.seek(40000L);
        AbstractContractSeekTest.assertEquals((byte)block[40000], (byte)((byte)this.instream.read()), (String)"@40000");
        this.instream.seek(8191L);
        AbstractContractSeekTest.assertEquals((byte)block[8191], (byte)((byte)this.instream.read()), (String)"@8191");
        this.instream.seek(0L);
        AbstractContractSeekTest.assertEquals((int)0, (int)((byte)this.instream.read()), (String)"@0");
        this.instream.seek(0L);
        AbstractContractSeekTest.assertEquals((long)0L, (long)this.instream.getPos());
        this.instream.read();
        AbstractContractSeekTest.assertEquals((long)1L, (long)this.instream.getPos());
        byte[] buf = new byte[81920];
        this.instream.readFully(1L, buf, 0, buf.length);
        AbstractContractSeekTest.assertEquals((long)1L, (long)this.instream.getPos());
    }

    @Test
    public void testPositionedBulkReadDoesntChangePosition() throws Throwable {
        this.describe("verify that a positioned read does not change the getPos() value");
        this.assumeSupportsPositionedReadable();
        Path testSeekFile = this.path("bigseekfile.txt");
        byte[] block = ContractTestUtils.dataset(65536, 0, 255);
        ContractTestUtils.createFile(this.getFileSystem(), testSeekFile, true, block);
        this.instream = this.getFileSystem().open(testSeekFile);
        this.instream.seek(39999L);
        AbstractContractSeekTest.assertTrue((-1 != this.instream.read() ? 1 : 0) != 0);
        AbstractContractSeekTest.assertEquals((long)40000L, (long)this.instream.getPos());
        int v = 256;
        byte[] readBuffer = new byte[v];
        this.instream.readFully(128L, readBuffer, 0, v);
        AbstractContractSeekTest.assertEquals((long)40000L, (long)this.instream.getPos());
        AbstractContractSeekTest.assertEquals((byte)block[40000], (byte)((byte)this.instream.read()), (String)"@40000");
        for (int i = 0; i < 256; ++i) {
            AbstractContractSeekTest.assertEquals((byte)block[i + 128], (byte)readBuffer[i], (String)("@" + i));
        }
    }

    @Test
    public void testRandomSeeks() throws Throwable {
        int limit = this.getContract().getLimit("test.random-seek-count", 100);
        this.describe("Testing " + limit + " random seeks");
        int filesize = 10240;
        byte[] buf = ContractTestUtils.dataset(filesize, 0, 255);
        Path randomSeekFile = this.path("testrandomseeks.bin");
        ContractTestUtils.createFile(this.getFileSystem(), randomSeekFile, true, buf);
        Random r = new Random();
        int[] seeks = new int[10];
        int[] reads = new int[10];
        try (FSDataInputStream stm = this.getFileSystem().open(randomSeekFile);){
            for (int i = 0; i < limit; ++i) {
                int seekOff = r.nextInt(buf.length);
                int toRead = r.nextInt(Math.min(buf.length - seekOff, 32000));
                seeks[i % seeks.length] = seekOff;
                reads[i % reads.length] = toRead;
                ContractTestUtils.verifyRead(stm, buf, seekOff, toRead);
            }
        }
        catch (AssertionError afe) {
            StringBuilder sb = new StringBuilder();
            sb.append("Sequence of actions:\n");
            for (int j = 0; j < seeks.length; ++j) {
                sb.append("seek @ ").append(seeks[j]).append("  ").append("read ").append(reads[j]).append("\n");
            }
            LOG.error(sb.toString());
            throw afe;
        }
    }

    @Test
    public void testReadFullyZeroByteFile() throws Throwable {
        this.describe("readFully against a 0 byte file");
        this.assumeSupportsPositionedReadable();
        this.instream = this.getFileSystem().open(this.zeroByteFile);
        AbstractContractSeekTest.assertEquals((long)0L, (long)this.instream.getPos());
        byte[] buffer = new byte[1];
        this.instream.readFully(0L, buffer, 0, 0);
        AbstractContractSeekTest.assertEquals((long)0L, (long)this.instream.getPos());
        this.instream.seek(0L);
        AbstractContractSeekTest.assertEquals((int)0, (int)this.instream.read(buffer, 0, 0));
    }

    @Test
    public void testReadFullyPastEOFZeroByteFile() throws Throwable {
        this.assumeSupportsPositionedReadable();
        this.describe("readFully past the EOF of a 0 byte file");
        this.instream = this.getFileSystem().open(this.zeroByteFile);
        byte[] buffer = new byte[1];
        try {
            this.instream.readFully(0L, buffer, 0, 16);
            AbstractContractSeekTest.fail((String)"Expected an exception");
        }
        catch (EOFException | IllegalArgumentException | IndexOutOfBoundsException exception) {
            // empty catch block
        }
    }

    @Test
    public void testReadFullySmallFile() throws Throwable {
        this.describe("readFully operations");
        this.assumeSupportsPositionedReadable();
        this.instream = this.getFileSystem().open(this.smallSeekFile);
        byte[] buffer = new byte[256];
        try {
            this.instream.readFully(0L, buffer, 0, -16);
            AbstractContractSeekTest.fail((String)"Expected an exception");
        }
        catch (IllegalArgumentException | IndexOutOfBoundsException runtimeException) {
            // empty catch block
        }
        try {
            this.instream.readFully(0L, buffer, -1, 16);
            AbstractContractSeekTest.fail((String)"Expected an exception");
        }
        catch (IllegalArgumentException | IndexOutOfBoundsException runtimeException) {
            // empty catch block
        }
        try {
            this.instream.readFully(-1L, buffer);
            AbstractContractSeekTest.fail((String)"Expected an exception");
        }
        catch (EOFException e) {
            this.handleExpectedException(e);
        }
        catch (IOException | IllegalArgumentException | IndexOutOfBoundsException e) {
            this.handleRelaxedException("readFully with a negative position ", "EOFException", e);
        }
        try {
            this.instream.readFully(0L, buffer, buffer.length - 8, 16);
            AbstractContractSeekTest.fail((String)"Expected an exception");
        }
        catch (IllegalArgumentException | IndexOutOfBoundsException e) {
            // empty catch block
        }
        AbstractContractSeekTest.assertEquals((long)0L, (long)this.instream.getPos());
        this.instream.readFully(0L, buffer);
        AbstractContractSeekTest.assertEquals((long)0L, (long)this.instream.getPos());
        byte[] fullFile = new byte[1024];
        this.instream.readFully(0L, fullFile);
        AbstractContractSeekTest.assertEquals((long)0L, (long)this.instream.getPos());
        try {
            this.instream.readFully(16L, fullFile);
            AbstractContractSeekTest.fail((String)"Expected an exception");
        }
        catch (EOFException e) {
            this.handleExpectedException(e);
        }
        catch (IOException e) {
            this.handleRelaxedException("readFully which reads past EOF ", "EOFException", e);
        }
    }

    @Test
    public void testReadFullyPastEOF() throws Throwable {
        this.describe("readFully past the EOF of a file");
        this.assumeSupportsPositionedReadable();
        this.instream = this.getFileSystem().open(this.smallSeekFile);
        byte[] buffer = new byte[256];
        try {
            this.instream.readFully(1025L, buffer);
            AbstractContractSeekTest.fail((String)"Expected an exception");
        }
        catch (EOFException e) {
            this.handleExpectedException(e);
        }
        catch (IOException e) {
            this.handleRelaxedException("readFully with an offset past EOF ", "EOFException", e);
        }
        try {
            this.instream.readFully(1025L, buffer, 0, 0);
            LOG.info("Filesystem short-circuits 0-byte reads");
        }
        catch (EOFException e) {
            this.handleExpectedException(e);
        }
        catch (IOException e) {
            this.handleRelaxedException("readFully(0 bytes) with an offset past EOF ", "EOFException", e);
        }
    }

    @Test
    public void testReadFullyZeroBytebufferPastEOF() throws Throwable {
        this.describe("readFully zero bytes from an offset past EOF");
        this.assumeSupportsPositionedReadable();
        this.instream = this.getFileSystem().open(this.smallSeekFile);
        byte[] buffer = new byte[256];
        try {
            this.instream.readFully(1025L, buffer, 0, 0);
            LOG.info("Filesystem short-circuits 0-byte reads");
        }
        catch (EOFException e) {
            this.handleExpectedException(e);
        }
        catch (IOException e) {
            this.handleRelaxedException("readFully(0 bytes) with an offset past EOF ", "EOFException", e);
        }
    }

    @Test
    public void testReadNullBuffer() throws Throwable {
        this.describe("try to read a null buffer ");
        this.assumeSupportsPositionedReadable();
        try (FSDataInputStream in = this.getFileSystem().open(this.smallSeekFile);){
            int r = in.read(0L, null, 0, 16);
            AbstractContractSeekTest.fail((String)("Expected an exception from a read into a null buffer, got " + r));
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
    }

    @Test
    public void testReadSmallFile() throws Throwable {
        int r;
        this.describe("PositionedRead.read operations");
        this.assumeSupportsPositionedReadable();
        this.instream = this.getFileSystem().open(this.smallSeekFile);
        byte[] buffer = new byte[256];
        try {
            r = this.instream.read(0L, buffer, 0, -16);
            AbstractContractSeekTest.fail((String)("Expected an exception, got " + r));
        }
        catch (IllegalArgumentException | IndexOutOfBoundsException runtimeException) {
            // empty catch block
        }
        try {
            r = this.instream.read(0L, buffer, -1, 16);
            AbstractContractSeekTest.fail((String)("Expected an exception, got " + r));
        }
        catch (IllegalArgumentException | IndexOutOfBoundsException runtimeException) {
            // empty catch block
        }
        try {
            r = this.instream.read(-1L, buffer, 0, 16);
            AbstractContractSeekTest.fail((String)("Expected an exception, got " + r));
        }
        catch (EOFException e) {
            this.handleExpectedException(e);
        }
        catch (IOException | IllegalArgumentException | IndexOutOfBoundsException e) {
            this.handleRelaxedException("read() with a negative position ", "EOFException", e);
        }
        try {
            r = this.instream.read(0L, buffer, buffer.length - 8, 16);
            AbstractContractSeekTest.fail((String)("Expected an exception, got " + r));
        }
        catch (IllegalArgumentException | IndexOutOfBoundsException e) {
            // empty catch block
        }
        AbstractContractSeekTest.assertEquals((long)0L, (long)this.instream.getPos());
        this.instream.readFully(0L, buffer);
        AbstractContractSeekTest.assertEquals((long)0L, (long)this.instream.getPos());
        byte[] fullFile = new byte[1024];
        this.instream.readFully(0L, fullFile, 0, fullFile.length);
        AbstractContractSeekTest.assertEquals((long)0L, (long)this.instream.getPos());
        AbstractContractSeekTest.assertEquals((int)-1, (int)this.instream.read(1040L, buffer, 0, 1));
    }

    @Test
    public void testReadAtExactEOF() throws Throwable {
        this.describe("read at the end of the file");
        this.instream = this.getFileSystem().open(this.smallSeekFile);
        this.instream.seek(1023L);
        AbstractContractSeekTest.assertTrue((this.instream.read() > 0 ? 1 : 0) != 0, (String)"read at last byte");
        AbstractContractSeekTest.assertEquals((int)-1, (int)this.instream.read(), (String)"read just past EOF");
    }
}

