/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FilterFileSystem;
import org.apache.hadoop.fs.LocalFileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.UnsupportedFileSystemException;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.security.token.Token;
import org.apache.hadoop.security.token.TokenIdentifier;
import org.apache.hadoop.test.HadoopTestBase;
import org.apache.hadoop.test.LambdaTestUtils;
import org.apache.hadoop.thirdparty.com.google.common.util.concurrent.ListenableFuture;
import org.apache.hadoop.thirdparty.com.google.common.util.concurrent.ListeningExecutorService;
import org.apache.hadoop.thirdparty.com.google.common.util.concurrent.MoreExecutors;
import org.apache.hadoop.util.BlockingThreadPoolExecutorService;
import org.assertj.core.api.AbstractLongAssert;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.Test;
import org.mockito.ArgumentMatchers;
import org.mockito.Mockito;
import org.mockito.verification.VerificationMode;

public class TestFileSystemCaching
extends HadoopTestBase {
    @Test
    public void testCacheEnabled() throws Exception {
        Configuration conf = this.newConf();
        FileSystem fs1 = FileSystem.get((URI)new URI("cachedfile://a"), (Configuration)conf);
        FileSystem fs2 = FileSystem.get((URI)new URI("cachedfile://a"), (Configuration)conf);
        TestFileSystemCaching.assertSame((Object)fs1, (Object)fs2);
    }

    @Test
    public void testDefaultFsUris() throws Exception {
        Configuration conf = new Configuration();
        conf.set("fs.defaultfs.impl", DefaultFs.class.getName());
        URI defaultUri = URI.create("defaultfs://host");
        FileSystem.setDefaultUri((Configuration)conf, (URI)defaultUri);
        FileSystem defaultFs = FileSystem.get((Configuration)conf);
        TestFileSystemCaching.assertEquals((Object)defaultUri, (Object)defaultFs.getUri());
        TestFileSystemCaching.assertSame((Object)defaultFs, (Object)FileSystem.get((URI)URI.create("defaultfs:/"), (Configuration)conf));
        TestFileSystemCaching.assertSame((Object)defaultFs, (Object)FileSystem.get((URI)URI.create("defaultfs:///"), (Configuration)conf));
        TestFileSystemCaching.assertSame((Object)defaultFs, (Object)FileSystem.get((URI)URI.create("defaultfs://host"), (Configuration)conf));
        TestFileSystemCaching.assertNotSame((Object)defaultFs, (Object)FileSystem.get((URI)URI.create("defaultfs://host2"), (Configuration)conf));
        TestFileSystemCaching.assertSame((Object)defaultFs, (Object)FileSystem.get((URI)URI.create("/"), (Configuration)conf));
        LambdaTestUtils.intercept(UnsupportedFileSystemException.class, () -> FileSystem.get((URI)URI.create("//host"), (Configuration)conf));
        LambdaTestUtils.intercept(UnsupportedFileSystemException.class, () -> FileSystem.get((URI)URI.create("//host2"), (Configuration)conf));
    }

    @Test
    public void testCacheEnabledWithInitializeForeverFS() throws Exception {
        final Configuration conf = new Configuration();
        Thread t = new Thread(){

            @Override
            public void run() {
                conf.set("fs.localfs1.impl", "org.apache.hadoop.fs.TestFileSystemCaching$InitializeForeverFileSystem");
                try {
                    FileSystem.get((URI)new URI("localfs1://a"), (Configuration)conf);
                }
                catch (IOException | URISyntaxException e) {
                    e.printStackTrace();
                }
            }
        };
        t.start();
        InitializeForeverFileSystem.sem.acquire();
        conf.set("fs.cachedfile.impl", FileSystem.getFileSystemClass((String)"file", null).getName());
        FileSystem.get((URI)new URI("cachedfile://a"), (Configuration)conf);
        t.interrupt();
        t.join();
    }

    @Test
    public void testCacheDisabled() throws Exception {
        Configuration conf = new Configuration();
        conf.set("fs.uncachedfile.impl", FileSystem.getFileSystemClass((String)"file", null).getName());
        conf.setBoolean("fs.uncachedfile.impl.disable.cache", true);
        FileSystem fs1 = FileSystem.get((URI)new URI("uncachedfile://a"), (Configuration)conf);
        FileSystem fs2 = FileSystem.get((URI)new URI("uncachedfile://a"), (Configuration)conf);
        TestFileSystemCaching.assertNotSame((Object)fs1, (Object)fs2);
    }

    @Test
    public <T extends TokenIdentifier> void testCacheForUgi() throws Exception {
        Configuration conf = this.newConf();
        UserGroupInformation ugiA = UserGroupInformation.createRemoteUser((String)"foo");
        UserGroupInformation ugiB = UserGroupInformation.createRemoteUser((String)"bar");
        FileSystem fsA = this.getCachedFS(ugiA, conf);
        FileSystem fsA1 = this.getCachedFS(ugiA, conf);
        TestFileSystemCaching.assertSame((Object)fsA, (Object)fsA1);
        FileSystem fsB = this.getCachedFS(ugiB, conf);
        TestFileSystemCaching.assertNotSame((Object)fsA, (Object)fsB);
        Token t1 = (Token)Mockito.mock(Token.class);
        UserGroupInformation ugiA2 = UserGroupInformation.createRemoteUser((String)"foo");
        fsA = this.getCachedFS(ugiA2, conf);
        TestFileSystemCaching.assertNotSame((Object)fsA, (Object)fsA1);
        ugiA.addToken(t1);
        fsA = this.getCachedFS(ugiA, conf);
        TestFileSystemCaching.assertSame((Object)fsA, (Object)fsA1);
    }

    private FileSystem getCachedFS(UserGroupInformation ugi, Configuration conf) throws IOException, InterruptedException {
        return (FileSystem)ugi.doAs(() -> FileSystem.get((URI)new URI("cachedfile://a"), (Configuration)conf));
    }

    @Test
    public void testUserFS() throws Exception {
        Configuration conf = this.newConf();
        FileSystem fsU1 = FileSystem.get((URI)new URI("cachedfile://a"), (Configuration)conf, (String)"bar");
        FileSystem fsU2 = FileSystem.get((URI)new URI("cachedfile://a"), (Configuration)conf, (String)"foo");
        TestFileSystemCaching.assertNotSame((Object)fsU1, (Object)fsU2);
    }

    private Configuration newConf() throws IOException {
        Configuration conf = new Configuration();
        conf.set("fs.cachedfile.impl", FileSystem.getFileSystemClass((String)"file", null).getName());
        return conf;
    }

    @Test
    public void testFsUniqueness() throws Exception {
        Configuration conf = this.newConf();
        FileSystem fs1 = FileSystem.get((Configuration)conf);
        FileSystem fs2 = FileSystem.get((Configuration)conf);
        TestFileSystemCaching.assertSame((Object)fs1, (Object)fs2);
        fs1 = FileSystem.newInstance((URI)new URI("cachedfile://a"), (Configuration)conf, (String)"bar");
        fs2 = FileSystem.newInstance((URI)new URI("cachedfile://a"), (Configuration)conf, (String)"bar");
        TestFileSystemCaching.assertTrue((fs1 != fs2 && !fs1.equals(fs2) ? 1 : 0) != 0);
        fs1.close();
        fs2.close();
    }

    @Test
    public void testCloseAllForUGI() throws Exception {
        Configuration conf = this.newConf();
        UserGroupInformation ugiA = UserGroupInformation.createRemoteUser((String)"foo");
        FileSystem fsA = this.getCachedFS(ugiA, conf);
        FileSystem fsA1 = this.getCachedFS(ugiA, conf);
        TestFileSystemCaching.assertSame((Object)fsA, (Object)fsA1);
        FileSystem.closeAllForUGI((UserGroupInformation)ugiA);
        fsA1 = this.getCachedFS(ugiA, conf);
        TestFileSystemCaching.assertNotSame((Object)fsA, (Object)fsA1);
    }

    @Test
    public void testDelete() throws IOException {
        FileSystem mockFs = (FileSystem)Mockito.mock(FileSystem.class);
        FilterFileSystem fs = new FilterFileSystem(mockFs);
        Path path = new Path("/a");
        fs.delete(path, false);
        ((FileSystem)Mockito.verify((Object)mockFs)).delete((Path)Mockito.eq((Object)path), Mockito.eq((boolean)false));
        Mockito.reset((Object[])new FileSystem[]{mockFs});
        fs.delete(path, true);
        ((FileSystem)Mockito.verify((Object)mockFs)).delete((Path)Mockito.eq((Object)path), Mockito.eq((boolean)true));
    }

    @Test
    public void testDeleteOnExit() throws IOException {
        FileSystem mockFs = (FileSystem)Mockito.mock(FileSystem.class);
        Path path = new Path("/a");
        try (FilterFileSystem fs = new FilterFileSystem(mockFs);){
            Mockito.when((Object)mockFs.getFileStatus((Path)Mockito.eq((Object)path))).thenReturn((Object)new FileStatus());
            TestFileSystemCaching.assertTrue((boolean)fs.deleteOnExit(path));
            ((FileSystem)Mockito.verify((Object)mockFs)).getFileStatus((Path)Mockito.eq((Object)path));
            Mockito.reset((Object[])new FileSystem[]{mockFs});
            Mockito.when((Object)mockFs.getFileStatus((Path)Mockito.eq((Object)path))).thenReturn((Object)new FileStatus());
            fs.close();
        }
        ((FileSystem)Mockito.verify((Object)mockFs)).getFileStatus((Path)Mockito.eq((Object)path));
        ((FileSystem)Mockito.verify((Object)mockFs)).delete((Path)Mockito.eq((Object)path), Mockito.eq((boolean)true));
    }

    @Test
    public void testDeleteOnExitFNF() throws IOException {
        Path path;
        FileSystem mockFs = (FileSystem)Mockito.mock(FileSystem.class);
        try (FilterFileSystem fs = new FilterFileSystem(mockFs);){
            path = new Path("/a");
            TestFileSystemCaching.assertFalse((boolean)fs.deleteOnExit(path));
            ((FileSystem)Mockito.verify((Object)mockFs)).getFileStatus((Path)Mockito.eq((Object)path));
            Mockito.reset((Object[])new FileSystem[]{mockFs});
            fs.close();
        }
        ((FileSystem)Mockito.verify((Object)mockFs, (VerificationMode)Mockito.never())).getFileStatus((Path)Mockito.eq((Object)path));
        ((FileSystem)Mockito.verify((Object)mockFs, (VerificationMode)Mockito.never())).delete((Path)ArgumentMatchers.any(Path.class), Mockito.anyBoolean());
    }

    @Test
    public void testDeleteOnExitRemoved() throws IOException {
        Path path;
        FileSystem mockFs = (FileSystem)Mockito.mock(FileSystem.class);
        try (FilterFileSystem fs = new FilterFileSystem(mockFs);){
            path = new Path("/a");
            Mockito.when((Object)mockFs.getFileStatus((Path)Mockito.eq((Object)path))).thenReturn((Object)new FileStatus());
            TestFileSystemCaching.assertTrue((boolean)fs.deleteOnExit(path));
            ((FileSystem)Mockito.verify((Object)mockFs)).getFileStatus((Path)Mockito.eq((Object)path));
            Mockito.reset((Object[])new FileSystem[]{mockFs});
            fs.close();
        }
        ((FileSystem)Mockito.verify((Object)mockFs)).getFileStatus((Path)Mockito.eq((Object)path));
        ((FileSystem)Mockito.verify((Object)mockFs, (VerificationMode)Mockito.never())).delete((Path)ArgumentMatchers.any(Path.class), Mockito.anyBoolean());
    }

    @Test
    public void testCancelDeleteOnExit() throws IOException {
        FileSystem mockFs = (FileSystem)Mockito.mock(FileSystem.class);
        try (FilterFileSystem fs = new FilterFileSystem(mockFs);){
            Path path = new Path("/a");
            Mockito.when((Object)mockFs.getFileStatus((Path)Mockito.eq((Object)path))).thenReturn((Object)new FileStatus());
            TestFileSystemCaching.assertTrue((boolean)fs.deleteOnExit(path));
            ((FileSystem)Mockito.verify((Object)mockFs)).getFileStatus((Path)Mockito.eq((Object)path));
            TestFileSystemCaching.assertTrue((boolean)fs.cancelDeleteOnExit(path));
            TestFileSystemCaching.assertFalse((boolean)fs.cancelDeleteOnExit(path));
            Mockito.reset((Object[])new FileSystem[]{mockFs});
            fs.close();
        }
        ((FileSystem)Mockito.verify((Object)mockFs, (VerificationMode)Mockito.never())).getFileStatus((Path)ArgumentMatchers.any(Path.class));
        ((FileSystem)Mockito.verify((Object)mockFs, (VerificationMode)Mockito.never())).delete((Path)ArgumentMatchers.any(Path.class), Mockito.anyBoolean());
    }

    @Test
    public void testCacheIncludesURIUserInfo() throws Throwable {
        URI containerA = new URI("wasb://a@account.blob.core.windows.net");
        URI containerB = new URI("wasb://b@account.blob.core.windows.net");
        Configuration conf = new Configuration(false);
        FileSystem.Cache.Key keyA = new FileSystem.Cache.Key(containerA, conf);
        FileSystem.Cache.Key keyB = new FileSystem.Cache.Key(containerB, conf);
        TestFileSystemCaching.assertNotEquals((Object)keyA, (Object)keyB);
        TestFileSystemCaching.assertNotEquals((Object)keyA, (Object)new FileSystem.Cache.Key(new URI("wasb://account.blob.core.windows.net"), conf));
        TestFileSystemCaching.assertEquals((Object)keyA, (Object)new FileSystem.Cache.Key(new URI("wasb://A@account.blob.core.windows.net"), conf));
        TestFileSystemCaching.assertNotEquals((Object)keyA, (Object)new FileSystem.Cache.Key(new URI("wasb://a:password@account.blob.core.windows.net"), conf));
    }

    @Test
    public void testCacheSingleSemaphoredConstruction() throws Exception {
        FileSystem.Cache cache = this.semaphoredCache(1);
        this.createFileSystems(cache, 10);
        ((AbstractLongAssert)Assertions.assertThat((long)cache.getDiscardedInstances()).describedAs("Discarded FS instances", new Object[0])).isEqualTo(0L);
    }

    @Test
    public void testCacheDualSemaphoreConstruction() throws Exception {
        FileSystem.Cache cache = this.semaphoredCache(2);
        this.createFileSystems(cache, 10);
        ((AbstractLongAssert)Assertions.assertThat((long)cache.getDiscardedInstances()).describedAs("Discarded FS instances", new Object[0])).isEqualTo(1L);
    }

    @Test
    public void testCacheLargeSemaphoreConstruction() throws Exception {
        FileSystem.Cache cache = this.semaphoredCache(999);
        int count = 10;
        this.createFileSystems(cache, count);
        ((AbstractLongAssert)Assertions.assertThat((long)cache.getDiscardedInstances()).describedAs("Discarded FS instances", new Object[0])).isEqualTo((long)(count - 1));
    }

    private FileSystem.Cache semaphoredCache(int semaphores) {
        Configuration conf1 = new Configuration();
        conf1.setInt("fs.creation.parallel.count", semaphores);
        FileSystem.Cache cache = new FileSystem.Cache(conf1);
        return cache;
    }

    private void createFileSystems(FileSystem.Cache cache, int count) throws URISyntaxException, InterruptedException, ExecutionException {
        Configuration conf = new Configuration();
        conf.set("fs.blocking.impl", BlockingInitializer.NAME);
        URI uri = new URI("blocking://a");
        ListeningExecutorService pool = MoreExecutors.listeningDecorator((ExecutorService)BlockingThreadPoolExecutorService.newInstance((int)(count * 2), (int)0, (long)10L, (TimeUnit)TimeUnit.SECONDS, (String)"creation-threads"));
        ArrayList<ListenableFuture> futures = new ArrayList<ListenableFuture>(count);
        Semaphore semaphore = BlockingInitializer.SEM;
        semaphore.acquire();
        for (int i = 0; i < count; ++i) {
            futures.add(pool.submit(() -> cache.get(uri, conf)));
        }
        semaphore.release();
        FileSystem createdFS = (FileSystem)((ListenableFuture)futures.get(0)).get();
        for (int i = 1; i < count; ++i) {
            FileSystem fs = (FileSystem)((ListenableFuture)futures.get(i)).get();
            Assertions.assertThat((Object)fs).isSameAs((Object)createdFS);
        }
    }

    private static class DefaultFs
    extends LocalFileSystem {
        URI uri;

        private DefaultFs() {
        }

        public void initialize(URI uri, Configuration conf) {
            this.uri = uri;
        }

        public URI getUri() {
            return this.uri;
        }
    }

    public static class InitializeForeverFileSystem
    extends LocalFileSystem {
        static final Semaphore sem = new Semaphore(0);

        public void initialize(URI uri, Configuration conf) throws IOException {
            sem.release();
            try {
                while (true) {
                    Thread.sleep(1000L);
                }
            }
            catch (InterruptedException e) {
                return;
            }
        }
    }

    private static final class BlockingInitializer
    extends LocalFileSystem {
        private static final String NAME = BlockingInitializer.class.getName();
        private static final Semaphore SEM = new Semaphore(1);

        private BlockingInitializer() {
        }

        public void initialize(URI uri, Configuration conf) throws IOException {
            try {
                SEM.acquire();
                SEM.release();
            }
            catch (InterruptedException e) {
                throw new IOException(e.toString(), e);
            }
        }
    }
}

