/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.impl.prefetch;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.DoubleSummaryStatistics;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.hadoop.fs.impl.prefetch.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class BlockOperations {
    private static final Logger LOG = LoggerFactory.getLogger(BlockOperations.class);
    private ArrayList<Operation> ops = new ArrayList();
    private boolean debugMode;

    public synchronized void setDebug(boolean state) {
        this.debugMode = state;
    }

    private synchronized Operation add(Operation op) {
        if (this.debugMode) {
            LOG.info(op.getDebugInfo());
        }
        this.ops.add(op);
        return op;
    }

    public Operation getPrefetched(int blockNumber) {
        Validate.checkNotNegative(blockNumber, "blockNumber");
        return this.add(new Operation(Kind.GET_PREFETCHED, blockNumber));
    }

    public Operation getCached(int blockNumber) {
        Validate.checkNotNegative(blockNumber, "blockNumber");
        return this.add(new Operation(Kind.GET_CACHED, blockNumber));
    }

    public Operation getRead(int blockNumber) {
        Validate.checkNotNegative(blockNumber, "blockNumber");
        return this.add(new Operation(Kind.GET_READ, blockNumber));
    }

    public Operation release(int blockNumber) {
        Validate.checkNotNegative(blockNumber, "blockNumber");
        return this.add(new Operation(Kind.RELEASE, blockNumber));
    }

    public Operation requestPrefetch(int blockNumber) {
        Validate.checkNotNegative(blockNumber, "blockNumber");
        return this.add(new Operation(Kind.REQUEST_PREFETCH, blockNumber));
    }

    public Operation prefetch(int blockNumber) {
        Validate.checkNotNegative(blockNumber, "blockNumber");
        return this.add(new Operation(Kind.PREFETCH, blockNumber));
    }

    public Operation cancelPrefetches() {
        return this.add(new Operation(Kind.CANCEL_PREFETCHES, -1));
    }

    public Operation close() {
        return this.add(new Operation(Kind.CLOSE, -1));
    }

    public Operation requestCaching(int blockNumber) {
        Validate.checkNotNegative(blockNumber, "blockNumber");
        return this.add(new Operation(Kind.REQUEST_CACHING, blockNumber));
    }

    public Operation addToCache(int blockNumber) {
        Validate.checkNotNegative(blockNumber, "blockNumber");
        return this.add(new Operation(Kind.CACHE_PUT, blockNumber));
    }

    public Operation end(Operation op) {
        return this.add(new End(op));
    }

    private static void append(StringBuilder sb, String format, Object ... args) {
        sb.append(String.format(format, args));
    }

    public synchronized String getSummary(boolean showDebugInfo) {
        StringBuilder sb = new StringBuilder();
        for (Operation op : this.ops) {
            if (op == null) continue;
            if (showDebugInfo) {
                sb.append(op.getDebugInfo());
                sb.append("\n");
                continue;
            }
            op.getSummary(sb);
            sb.append(";");
        }
        sb.append("\n");
        this.getDurationInfo(sb);
        return sb.toString();
    }

    public synchronized void getDurationInfo(StringBuilder sb) {
        DoubleSummaryStatistics stats;
        HashMap<Kind, DoubleSummaryStatistics> durations = new HashMap<Kind, DoubleSummaryStatistics>();
        for (Operation op : this.ops) {
            if (!(op instanceof End)) continue;
            End endOp = (End)op;
            stats = (DoubleSummaryStatistics)durations.get((Object)endOp.getKind());
            if (stats == null) {
                stats = new DoubleSummaryStatistics();
                durations.put(endOp.getKind(), stats);
            }
            stats.accept(endOp.duration());
        }
        List<Kind> kinds = Arrays.asList(Kind.GET_CACHED, Kind.GET_PREFETCHED, Kind.GET_READ, Kind.CACHE_PUT, Kind.PREFETCH, Kind.REQUEST_CACHING, Kind.REQUEST_PREFETCH, Kind.CANCEL_PREFETCHES, Kind.RELEASE, Kind.CLOSE);
        for (Kind kind : kinds) {
            BlockOperations.append(sb, "%-18s : ", new Object[]{kind});
            stats = (DoubleSummaryStatistics)durations.get((Object)kind);
            if (stats == null) {
                BlockOperations.append(sb, "--\n", new Object[0]);
                continue;
            }
            BlockOperations.append(sb, "#ops = %3d, total = %5.1f, min: %3.1f, avg: %3.1f, max: %3.1f\n", stats.getCount(), stats.getSum(), stats.getMin(), stats.getAverage(), stats.getMax());
        }
    }

    public synchronized void analyze(StringBuilder sb) {
        HashMap blockOps = new HashMap();
        for (Operation op : this.ops) {
            List<Operation> perBlockOps;
            if (op.blockNumber < 0) continue;
            if (!blockOps.containsKey(op.blockNumber)) {
                perBlockOps = new ArrayList();
                blockOps.put(op.blockNumber, perBlockOps);
            }
            perBlockOps = (List)blockOps.get(op.blockNumber);
            perBlockOps.add(op);
        }
        ArrayList<Integer> prefetchedNotUsed = new ArrayList<Integer>();
        ArrayList<Integer> cachedNotUsed = new ArrayList<Integer>();
        for (Map.Entry entry : blockOps.entrySet()) {
            int count;
            Integer blockNumber = (Integer)entry.getKey();
            List perBlockOps = (List)entry.getValue();
            HashMap<Kind, Integer> kindCounts = new HashMap<Kind, Integer>();
            HashMap<Kind, Integer> endKindCounts = new HashMap<Kind, Integer>();
            for (Operation op : perBlockOps) {
                if (op instanceof End) {
                    int endCount = endKindCounts.getOrDefault((Object)op.kind, 0) + 1;
                    endKindCounts.put(op.kind, endCount);
                    continue;
                }
                count = kindCounts.getOrDefault((Object)op.kind, 0) + 1;
                kindCounts.put(op.kind, count);
            }
            for (Kind kind : kindCounts.keySet()) {
                int endCount;
                count = kindCounts.getOrDefault((Object)kind, 0);
                if (count != (endCount = endKindCounts.getOrDefault((Object)kind, 0).intValue())) {
                    BlockOperations.append(sb, "[%d] %s : #ops(%d) != #end-ops(%d)\n", new Object[]{blockNumber, kind, count, endCount});
                }
                if (count <= 1) continue;
                BlockOperations.append(sb, "[%d] %s = %d\n", new Object[]{blockNumber, kind, count});
            }
            int prefetchCount = kindCounts.getOrDefault((Object)Kind.PREFETCH, 0);
            int getPrefetchedCount = kindCounts.getOrDefault((Object)Kind.GET_PREFETCHED, 0);
            if (prefetchCount > 0 && getPrefetchedCount < prefetchCount) {
                prefetchedNotUsed.add(blockNumber);
            }
            int cacheCount = kindCounts.getOrDefault((Object)Kind.CACHE_PUT, 0);
            int getCachedCount = kindCounts.getOrDefault((Object)Kind.GET_CACHED, 0);
            if (cacheCount <= 0 || getCachedCount >= cacheCount) continue;
            cachedNotUsed.add(blockNumber);
        }
        if (!prefetchedNotUsed.isEmpty()) {
            BlockOperations.append(sb, "Prefetched but not used: %s\n", BlockOperations.getIntList(prefetchedNotUsed));
        }
        if (!cachedNotUsed.isEmpty()) {
            BlockOperations.append(sb, "Cached but not used: %s\n", BlockOperations.getIntList(cachedNotUsed));
        }
    }

    private static String getIntList(Iterable<Integer> nums) {
        ArrayList<String> numList = new ArrayList<String>();
        for (Integer n : nums) {
            numList.add(n.toString());
        }
        return String.join((CharSequence)", ", numList);
    }

    public static BlockOperations fromSummary(String summary) {
        String[] tokens;
        BlockOperations ops = new BlockOperations();
        ops.setDebug(true);
        Pattern blockOpPattern = Pattern.compile("([A-Z+]+)(\\(([0-9]+)?\\))?");
        for (String token : tokens = summary.split(";")) {
            Matcher matcher = blockOpPattern.matcher(token);
            if (!matcher.matches()) {
                String message = String.format("Unknown summary format: %s", token);
                throw new IllegalArgumentException(message);
            }
            String shortName = matcher.group(1);
            String blockNumberStr = matcher.group(3);
            int blockNumber = blockNumberStr == null ? -1 : Integer.parseInt(blockNumberStr);
            Kind kind = Kind.fromShortName(shortName);
            Kind endKind = null;
            if (kind == null && shortName.charAt(0) == 'E') {
                endKind = Kind.fromShortName(shortName.substring(1));
            }
            if (kind == null && endKind == null) {
                String message = String.format("Unknown short name: %s (token = %s)", shortName, token);
                throw new IllegalArgumentException(message);
            }
            if (kind != null) {
                ops.add(new Operation(kind, blockNumber));
                continue;
            }
            Operation op = null;
            for (int i = ops.ops.size() - 1; i >= 0; --i) {
                op = ops.ops.get(i);
                if (op.blockNumber != blockNumber || op.kind != endKind || op instanceof End) continue;
                ops.add(new End(op));
                break;
            }
            if (op != null) continue;
            LOG.warn("Start op not found: {}({})", (Object)endKind, (Object)blockNumber);
        }
        return ops;
    }

    public static class Operation {
        private final Kind kind;
        private final int blockNumber;
        private final long timestamp;

        public Operation(Kind kind, int blockNumber) {
            this.kind = kind;
            this.blockNumber = blockNumber;
            this.timestamp = System.nanoTime();
        }

        public Kind getKind() {
            return this.kind;
        }

        public int getBlockNumber() {
            return this.blockNumber;
        }

        public long getTimestamp() {
            return this.timestamp;
        }

        public void getSummary(StringBuilder sb) {
            if (this.kind.hasBlock) {
                sb.append(String.format("%s(%d)", this.kind.shortName, this.blockNumber));
            } else {
                sb.append(String.format("%s", this.kind.shortName));
            }
        }

        public String getDebugInfo() {
            if (this.kind.hasBlock) {
                return String.format("--- %s(%d)", this.kind.name, this.blockNumber);
            }
            return String.format("... %s()", this.kind.name);
        }
    }

    public static enum Kind {
        UNKNOWN("??", "unknown", false),
        CANCEL_PREFETCHES("CP", "cancelPrefetches", false),
        CLOSE("CX", "close", false),
        CACHE_PUT("C+", "putC", true),
        GET_CACHED("GC", "getCached", true),
        GET_PREFETCHED("GP", "getPrefetched", true),
        GET_READ("GR", "getRead", true),
        PREFETCH("PF", "prefetch", true),
        RELEASE("RL", "release", true),
        REQUEST_CACHING("RC", "requestCaching", true),
        REQUEST_PREFETCH("RP", "requestPrefetch", true);

        private String shortName;
        private String name;
        private boolean hasBlock;
        private static Map<String, Kind> shortNameToKind;

        private Kind(String shortName, String name, boolean hasBlock) {
            this.shortName = shortName;
            this.name = name;
            this.hasBlock = hasBlock;
        }

        public static Kind fromShortName(String shortName) {
            if (shortNameToKind.isEmpty()) {
                for (Kind kind : Kind.values()) {
                    shortNameToKind.put(kind.shortName, kind);
                }
            }
            return shortNameToKind.get(shortName);
        }

        static {
            shortNameToKind = new HashMap<String, Kind>();
        }
    }

    public static class End
    extends Operation {
        private Operation op;

        public End(Operation op) {
            super(op.kind, op.blockNumber);
            this.op = op;
        }

        @Override
        public void getSummary(StringBuilder sb) {
            sb.append("E");
            super.getSummary(sb);
        }

        @Override
        public String getDebugInfo() {
            return "***" + super.getDebugInfo().substring(3);
        }

        public double duration() {
            return (double)(this.getTimestamp() - this.op.getTimestamp()) / 1.0E9;
        }
    }
}

