/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.store;

import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.LocalDirAllocator;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.store.BlockUploadStatistics;
import org.apache.hadoop.fs.store.ByteBufferInputStream;
import org.apache.hadoop.util.DirectBufferPool;
import org.apache.hadoop.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class DataBlocks {
    private static final Logger LOG = LoggerFactory.getLogger(DataBlocks.class);
    public static final String DATA_BLOCKS_BUFFER_DISK = "disk";
    public static final String DATA_BLOCKS_BYTEBUFFER = "bytebuffer";
    public static final String DATA_BLOCKS_BUFFER_ARRAY = "array";

    private DataBlocks() {
    }

    public static void validateWriteArgs(byte[] b, int off, int len) throws IOException {
        Preconditions.checkNotNull(b);
        if (off < 0 || off > b.length || len < 0 || off + len > b.length || off + len < 0) {
            throw new IndexOutOfBoundsException("write (b[" + b.length + "], " + off + ", " + len + ')');
        }
    }

    public static BlockFactory createFactory(String keyToBufferDir, Configuration configuration, String name) {
        LOG.debug("Creating DataFactory of type : {}", (Object)name);
        switch (name) {
            case "array": {
                return new ArrayBlockFactory(keyToBufferDir, configuration);
            }
            case "disk": {
                return new DiskBlockFactory(keyToBufferDir, configuration);
            }
            case "bytebuffer": {
                return new ByteBufferBlockFactory(keyToBufferDir, configuration);
            }
        }
        throw new IllegalArgumentException("Unsupported block buffer \"" + name + '\"');
    }

    static class ArrayBlockFactory
    extends BlockFactory {
        ArrayBlockFactory(String keyToBufferDir, Configuration conf) {
            super(keyToBufferDir, conf);
        }

        @Override
        public DataBlock create(long index, int limit, BlockUploadStatistics statistics) throws IOException {
            return new ByteArrayBlock(0L, limit, statistics);
        }
    }

    static class DiskBlockFactory
    extends BlockFactory {
        private LocalDirAllocator directoryAllocator;

        DiskBlockFactory(String keyToBufferDir, Configuration conf) {
            super(keyToBufferDir, conf);
            String bufferDir = conf.get(keyToBufferDir) != null ? keyToBufferDir : "hadoop.tmp.dir";
            this.directoryAllocator = new LocalDirAllocator(bufferDir);
        }

        @Override
        public DataBlock create(long index, int limit, BlockUploadStatistics statistics) throws IOException {
            File destFile = this.createTmpFileForWrite(String.format("datablock-%04d-", index), limit, this.getConf());
            return new DiskBlock(destFile, limit, index, statistics);
        }

        File createTmpFileForWrite(String pathStr, long size, Configuration conf) throws IOException {
            Path path = this.directoryAllocator.getLocalPathForWrite(pathStr, size, conf);
            File dir = new File(path.getParent().toUri().getPath());
            String prefix = path.getName();
            return File.createTempFile(prefix, null, dir);
        }
    }

    static class ByteBufferBlockFactory
    extends BlockFactory {
        private final DirectBufferPool bufferPool = new DirectBufferPool();
        private final AtomicInteger buffersOutstanding = new AtomicInteger(0);

        ByteBufferBlockFactory(String keyToBufferDir, Configuration conf) {
            super(keyToBufferDir, conf);
        }

        @Override
        public ByteBufferBlock create(long index, int limit, BlockUploadStatistics statistics) throws IOException {
            return new ByteBufferBlock(index, limit, statistics);
        }

        private ByteBuffer requestBuffer(int limit) {
            LOG.debug("Requesting buffer of size {}", (Object)limit);
            this.buffersOutstanding.incrementAndGet();
            return this.bufferPool.getBuffer(limit);
        }

        private void releaseBuffer(ByteBuffer buffer) {
            LOG.debug("Releasing buffer");
            this.bufferPool.returnBuffer(buffer);
            this.buffersOutstanding.decrementAndGet();
        }

        public int getOutstandingBufferCount() {
            return this.buffersOutstanding.get();
        }

        public String toString() {
            return "ByteBufferBlockFactory{buffersOutstanding=" + this.buffersOutstanding + '}';
        }

        class ByteBufferBlock
        extends DataBlock {
            private ByteBuffer blockBuffer;
            private final int bufferSize;
            private Integer dataSize;

            ByteBufferBlock(long index, int bufferSize, BlockUploadStatistics statistics) {
                super(index, statistics);
                this.bufferSize = bufferSize;
                this.blockBuffer = ByteBufferBlockFactory.this.requestBuffer(bufferSize);
                this.blockAllocated();
            }

            @Override
            public int dataSize() {
                return this.dataSize != null ? this.dataSize.intValue() : this.bufferCapacityUsed();
            }

            @Override
            public BlockUploadData startUpload() throws IOException {
                super.startUpload();
                this.dataSize = this.bufferCapacityUsed();
                this.blockBuffer.limit(this.blockBuffer.position());
                this.blockBuffer.position(0);
                return new BlockUploadData(new ByteBufferInputStream(this.dataSize, this.blockBuffer));
            }

            @Override
            public boolean hasCapacity(long bytes) {
                return bytes <= (long)this.remainingCapacity();
            }

            @Override
            public int remainingCapacity() {
                return this.blockBuffer != null ? this.blockBuffer.remaining() : 0;
            }

            private int bufferCapacityUsed() {
                return this.blockBuffer.capacity() - this.blockBuffer.remaining();
            }

            @Override
            public int write(byte[] b, int offset, int len) throws IOException {
                super.write(b, offset, len);
                int written = Math.min(this.remainingCapacity(), len);
                this.blockBuffer.put(b, offset, written);
                return written;
            }

            @Override
            protected void innerClose() {
                if (this.blockBuffer != null) {
                    this.blockReleased();
                    ByteBufferBlockFactory.this.releaseBuffer(this.blockBuffer);
                    this.blockBuffer = null;
                }
            }

            public String toString() {
                return "ByteBufferBlock{index=" + this.getIndex() + ", state=" + (Object)((Object)this.getState()) + ", dataSize=" + this.dataSize() + ", limit=" + this.bufferSize + ", remainingCapacity=" + this.remainingCapacity() + '}';
            }
        }
    }

    static class DiskBlock
    extends DataBlock {
        private int bytesWritten;
        private final File bufferFile;
        private final int limit;
        private BufferedOutputStream out;
        private final AtomicBoolean closed = new AtomicBoolean(false);

        DiskBlock(File bufferFile, int limit, long index, BlockUploadStatistics statistics) throws FileNotFoundException {
            super(index, statistics);
            this.limit = limit;
            this.bufferFile = bufferFile;
            this.blockAllocated();
            this.out = new BufferedOutputStream(new FileOutputStream(bufferFile));
        }

        @Override
        public int dataSize() {
            return this.bytesWritten;
        }

        @Override
        boolean hasCapacity(long bytes) {
            return (long)this.dataSize() + bytes <= (long)this.limit;
        }

        @Override
        public int remainingCapacity() {
            return this.limit - this.bytesWritten;
        }

        @Override
        public int write(byte[] b, int offset, int len) throws IOException {
            super.write(b, offset, len);
            int written = Math.min(this.remainingCapacity(), len);
            this.out.write(b, offset, written);
            this.bytesWritten += written;
            return written;
        }

        @Override
        public BlockUploadData startUpload() throws IOException {
            super.startUpload();
            try {
                this.out.flush();
            }
            finally {
                this.out.close();
                this.out = null;
            }
            return new BlockUploadData(this.bufferFile);
        }

        @Override
        protected void innerClose() throws IOException {
            DataBlock.DestState state = this.getState();
            LOG.debug("Closing {}", (Object)this);
            switch (state) {
                case Writing: {
                    if (!this.bufferFile.exists()) break;
                    LOG.debug("Block[{}]: Deleting buffer file as upload did not start", (Object)this.getIndex());
                    this.closeBlock();
                    break;
                }
                case Upload: {
                    LOG.debug("Block[{}]: Buffer file {} exists \u2014close upload stream", (Object)this.getIndex(), (Object)this.bufferFile);
                    break;
                }
                case Closed: {
                    this.closeBlock();
                    break;
                }
            }
        }

        @Override
        public void flush() throws IOException {
            super.flush();
            this.out.flush();
        }

        public String toString() {
            String sb = "FileBlock{index=" + this.getIndex() + ", destFile=" + this.bufferFile + ", state=" + (Object)((Object)this.getState()) + ", dataSize=" + this.dataSize() + ", limit=" + this.limit + '}';
            return sb;
        }

        void closeBlock() {
            LOG.debug("block[{}]: closeBlock()", (Object)this.getIndex());
            if (!this.closed.getAndSet(true)) {
                this.blockReleased();
                if (!this.bufferFile.delete() && this.bufferFile.exists()) {
                    LOG.warn("delete({}) returned false", (Object)this.bufferFile.getAbsoluteFile());
                }
            } else {
                LOG.debug("block[{}]: skipping re-entrant closeBlock()", (Object)this.getIndex());
            }
        }
    }

    static class ByteArrayBlock
    extends DataBlock {
        private DataBlockByteArrayOutputStream buffer;
        private final int limit;
        private Integer dataSize;

        ByteArrayBlock(long index, int limit, BlockUploadStatistics statistics) {
            super(index, statistics);
            this.limit = limit;
            this.buffer = new DataBlockByteArrayOutputStream(limit);
            this.blockAllocated();
        }

        @Override
        public int dataSize() {
            return this.dataSize != null ? this.dataSize.intValue() : this.buffer.size();
        }

        @Override
        public BlockUploadData startUpload() throws IOException {
            super.startUpload();
            this.dataSize = this.buffer.size();
            ByteArrayInputStream bufferData = this.buffer.getInputStream();
            this.buffer = null;
            return new BlockUploadData(bufferData);
        }

        @Override
        boolean hasCapacity(long bytes) {
            return (long)this.dataSize() + bytes <= (long)this.limit;
        }

        @Override
        public int remainingCapacity() {
            return this.limit - this.dataSize();
        }

        @Override
        public int write(byte[] b, int offset, int len) throws IOException {
            super.write(b, offset, len);
            int written = Math.min(this.remainingCapacity(), len);
            this.buffer.write(b, offset, written);
            return written;
        }

        @Override
        protected void innerClose() {
            this.buffer = null;
            this.blockReleased();
        }

        public String toString() {
            return "ByteArrayBlock{index=" + this.getIndex() + ", state=" + (Object)((Object)this.getState()) + ", limit=" + this.limit + ", dataSize=" + this.dataSize + '}';
        }
    }

    static class DataBlockByteArrayOutputStream
    extends ByteArrayOutputStream {
        DataBlockByteArrayOutputStream(int size) {
            super(size);
        }

        ByteArrayInputStream getInputStream() {
            ByteArrayInputStream bin = new ByteArrayInputStream(this.buf, 0, this.count);
            this.reset();
            this.buf = null;
            return bin;
        }
    }

    public static abstract class DataBlock
    implements Closeable {
        private volatile DestState state = DestState.Writing;
        private final long index;
        private final BlockUploadStatistics statistics;

        protected DataBlock(long index, BlockUploadStatistics statistics) {
            this.index = index;
            this.statistics = statistics;
        }

        protected final synchronized void enterState(DestState current, DestState next) throws IllegalStateException {
            this.verifyState(current);
            LOG.debug("{}: entering state {}", (Object)this, (Object)next);
            this.state = next;
        }

        protected final void verifyState(DestState expected) throws IllegalStateException {
            if (expected != null && this.state != expected) {
                throw new IllegalStateException("Expected stream state " + (Object)((Object)expected) + " -but actual state is " + (Object)((Object)this.state) + " in " + this);
            }
        }

        public final DestState getState() {
            return this.state;
        }

        public abstract int dataSize();

        abstract boolean hasCapacity(long var1);

        public boolean hasData() {
            return this.dataSize() > 0;
        }

        public abstract int remainingCapacity();

        public int write(byte[] buffer, int offset, int length) throws IOException {
            this.verifyState(DestState.Writing);
            Preconditions.checkArgument(buffer != null, "Null buffer");
            Preconditions.checkArgument(length >= 0, "length is negative");
            Preconditions.checkArgument(offset >= 0, "offset is negative");
            Preconditions.checkArgument(buffer.length - offset >= length, "buffer shorter than amount of data to write");
            return 0;
        }

        public void flush() throws IOException {
            this.verifyState(DestState.Writing);
        }

        public BlockUploadData startUpload() throws IOException {
            LOG.debug("Start datablock[{}] upload", (Object)this.index);
            this.enterState(DestState.Writing, DestState.Upload);
            return null;
        }

        protected synchronized boolean enterClosedState() {
            if (!this.state.equals((Object)DestState.Closed)) {
                this.enterState(null, DestState.Closed);
                return true;
            }
            return false;
        }

        @Override
        public void close() throws IOException {
            if (this.enterClosedState()) {
                LOG.debug("Closed {}", (Object)this);
                this.innerClose();
            }
        }

        protected void innerClose() throws IOException {
        }

        protected void blockAllocated() {
            if (this.statistics != null) {
                this.statistics.blockAllocated();
            }
        }

        protected void blockReleased() {
            if (this.statistics != null) {
                this.statistics.blockReleased();
            }
        }

        protected BlockUploadStatistics getStatistics() {
            return this.statistics;
        }

        public long getIndex() {
            return this.index;
        }

        public static enum DestState {
            Writing,
            Upload,
            Closed;

        }
    }

    public static abstract class BlockFactory
    implements Closeable {
        private final String keyToBufferDir;
        private final Configuration conf;

        protected BlockFactory(String keyToBufferDir, Configuration conf) {
            this.keyToBufferDir = keyToBufferDir;
            this.conf = conf;
        }

        public abstract DataBlock create(long var1, int var3, BlockUploadStatistics var4) throws IOException;

        @Override
        public void close() throws IOException {
        }

        protected Configuration getConf() {
            return this.conf;
        }

        public String getKeyToBufferDir() {
            return this.keyToBufferDir;
        }
    }

    public static final class BlockUploadData
    implements Closeable {
        private final File file;
        private InputStream uploadStream;
        private byte[] byteArray;
        private boolean isClosed;

        public BlockUploadData(byte[] byteArray) {
            this.file = null;
            this.uploadStream = null;
            this.byteArray = Objects.requireNonNull(byteArray);
        }

        BlockUploadData(File file) {
            Preconditions.checkArgument(file.exists(), "No file: %s", file);
            this.file = file;
            this.uploadStream = null;
            this.byteArray = null;
        }

        BlockUploadData(InputStream uploadStream) {
            Objects.requireNonNull(uploadStream, "rawUploadStream");
            this.uploadStream = uploadStream;
            this.file = null;
            this.byteArray = null;
        }

        boolean hasFile() {
            return this.file != null;
        }

        File getFile() {
            return this.file;
        }

        InputStream getUploadStream() {
            return this.uploadStream;
        }

        public byte[] toByteArray() throws IOException {
            Preconditions.checkState(!this.isClosed, "Block is closed");
            if (this.byteArray != null) {
                return this.byteArray;
            }
            if (this.file != null) {
                this.byteArray = FileUtils.readFileToByteArray((File)this.file);
                return this.byteArray;
            }
            this.byteArray = IOUtils.toByteArray((InputStream)this.uploadStream);
            IOUtils.close((Closeable)this.uploadStream);
            this.uploadStream = null;
            return this.byteArray;
        }

        @Override
        public void close() throws IOException {
            this.isClosed = true;
            org.apache.hadoop.io.IOUtils.cleanupWithLogger(LOG, this.uploadStream);
            this.byteArray = null;
            if (this.file != null) {
                LOG.debug("File deleted in BlockUploadData close: {}", (Object)this.file.delete());
            }
        }
    }
}

