/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.atomic.LongAdder;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.function.Supplier;
import org.apache.hadoop.classification.VisibleForTesting;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.ipc.ProcessingDetails;
import org.apache.hadoop.ipc.Server;
import org.apache.hadoop.log.LogThrottlingHelper;
import org.apache.hadoop.metrics2.lib.MutableRatesWithAggregation;
import org.apache.hadoop.util.StringUtils;
import org.apache.hadoop.util.Time;
import org.apache.hadoop.util.Timer;

public class FSNamesystemLock {
    @VisibleForTesting
    protected ReentrantReadWriteLock coarseLock;
    private final String lockName;
    private volatile boolean metricsEnabled;
    private final MutableRatesWithAggregation detailedHoldTimeMetrics;
    private final Timer timer;
    private final long lockSuppressWarningIntervalMs;
    private volatile long writeLockReportingThresholdMs;
    private long writeLockHeldTimeStampNanos;
    private final LogThrottlingHelper writeLockReportLogger;
    private volatile long readLockReportingThresholdMs;
    private final ThreadLocal<Long> readLockHeldTimeStampNanos = new ThreadLocal<Long>(){

        @Override
        public Long initialValue() {
            return Long.MAX_VALUE;
        }
    };
    private final AtomicInteger numReadLockWarningsSuppressed = new AtomicInteger(0);
    private final AtomicLong timeStampOfLastReadLockReportMs = new AtomicLong(0L);
    private final AtomicReference<LockHeldInfo> longestReadLockHeldInfo = new AtomicReference<LockHeldInfo>(new LockHeldInfo());
    private LockHeldInfo longestWriteLockHeldInfo = new LockHeldInfo();
    private final LongAdder numReadLockLongHold = new LongAdder();
    private final LongAdder numWriteLockLongHold = new LongAdder();
    @VisibleForTesting
    static final String OP_NAME_OTHER = "OTHER";
    private final String readLockMetricPrefix;
    private final String writeLockMetricPrefix;
    private static final String LOCK_METRIC_SUFFIX = "Nanos";
    private static final String OVERALL_METRIC_NAME = "Overall";

    public FSNamesystemLock(Configuration conf, String lockName, MutableRatesWithAggregation detailedHoldTimeMetrics) {
        this(conf, lockName, detailedHoldTimeMetrics, new Timer());
    }

    @VisibleForTesting
    FSNamesystemLock(Configuration conf, String lockName, MutableRatesWithAggregation detailedHoldTimeMetrics, Timer timer) {
        this.lockName = lockName;
        this.readLockMetricPrefix = this.lockName + "ReadLock";
        this.writeLockMetricPrefix = this.lockName + "WriteLock";
        boolean fair = conf.getBoolean("dfs.namenode.fslock.fair", true);
        FSNamesystem.LOG.info("{}Lock is fair: {}.", (Object)this.lockName, (Object)fair);
        this.coarseLock = new ReentrantReadWriteLock(fair);
        this.timer = timer;
        this.writeLockReportingThresholdMs = conf.getLong("dfs.namenode.write-lock-reporting-threshold-ms", 5000L);
        this.readLockReportingThresholdMs = conf.getLong("dfs.namenode.read-lock-reporting-threshold-ms", 5000L);
        this.lockSuppressWarningIntervalMs = conf.getTimeDuration("dfs.lock.suppress.warning.interval", 10000L, TimeUnit.MILLISECONDS);
        this.writeLockReportLogger = new LogThrottlingHelper(this.lockSuppressWarningIntervalMs);
        this.metricsEnabled = conf.getBoolean("dfs.namenode.lock.detailed-metrics.enabled", false);
        FSNamesystem.LOG.info("Detailed lock hold time metrics of {}Lock is {}.", (Object)this.lockName, (Object)(this.metricsEnabled ? "enabled" : "disabled"));
        this.detailedHoldTimeMetrics = detailedHoldTimeMetrics;
    }

    public void readLock() {
        this.doLock(false);
    }

    public void readLockInterruptibly() throws InterruptedException {
        this.doLockInterruptibly(false);
    }

    public void readUnlock() {
        this.readUnlock(OP_NAME_OTHER, null);
    }

    public void readUnlock(String opName) {
        this.readUnlock(opName, null);
    }

    public void readUnlock(String opName, Supplier<String> lockReportInfoSupplier) {
        boolean needReport = this.coarseLock.getReadHoldCount() == 1;
        long readLockIntervalNanos = this.timer.monotonicNowNanos() - this.readLockHeldTimeStampNanos.get();
        long currentTimeMs = this.timer.now();
        this.coarseLock.readLock().unlock();
        if (needReport) {
            this.addMetric(opName, readLockIntervalNanos, false);
            this.readLockHeldTimeStampNanos.remove();
        }
        long readLockIntervalMs = TimeUnit.NANOSECONDS.toMillis(readLockIntervalNanos);
        if (needReport && readLockIntervalMs >= this.readLockReportingThresholdMs) {
            long nowMs;
            long localTimeStampOfLastReadLockReport;
            this.numReadLockLongHold.increment();
            String lockReportInfo = null;
            boolean done = false;
            while (!done) {
                LockHeldInfo localLockHeldInfo = this.longestReadLockHeldInfo.get();
                if (localLockHeldInfo.getIntervalMs() <= readLockIntervalMs) {
                    if (lockReportInfo == null) {
                        String string = lockReportInfo = lockReportInfoSupplier != null ? " (" + lockReportInfoSupplier.get() + ")" : "";
                    }
                    if (!this.longestReadLockHeldInfo.compareAndSet(localLockHeldInfo, new LockHeldInfo(currentTimeMs, readLockIntervalMs, StringUtils.getStackTrace((Thread)Thread.currentThread()), opName, lockReportInfo))) continue;
                    done = true;
                    continue;
                }
                done = true;
            }
            do {
                if ((nowMs = this.timer.monotonicNow()) - (localTimeStampOfLastReadLockReport = this.timeStampOfLastReadLockReportMs.get()) >= this.lockSuppressWarningIntervalMs) continue;
                this.numReadLockWarningsSuppressed.incrementAndGet();
                return;
            } while (!this.timeStampOfLastReadLockReportMs.compareAndSet(localTimeStampOfLastReadLockReport, nowMs));
            int numSuppressedWarnings = this.numReadLockWarningsSuppressed.getAndSet(0);
            LockHeldInfo lockHeldInfo = this.longestReadLockHeldInfo.getAndSet(new LockHeldInfo());
            FSNamesystem.LOG.info("\tNumber of suppressed read-lock reports of {}Lock is {}\n\tLongest read-lock held at {} for {}ms by {}{} via {}", new Object[]{this.lockName, numSuppressedWarnings, Time.formatTime((long)lockHeldInfo.getStartTimeMs()), lockHeldInfo.getIntervalMs(), lockHeldInfo.getOpName(), lockHeldInfo.getLockReportInfo(), lockHeldInfo.getStackTrace()});
        }
    }

    public void writeLock() {
        this.doLock(true);
    }

    public void writeLockInterruptibly() throws InterruptedException {
        this.doLockInterruptibly(true);
    }

    public void writeUnlock() {
        this.writeUnlock(OP_NAME_OTHER, false, null);
    }

    public void writeUnlock(String opName) {
        this.writeUnlock(opName, false, null);
    }

    public void writeUnlock(String opName, Supplier<String> lockReportInfoSupplier) {
        this.writeUnlock(opName, false, lockReportInfoSupplier);
    }

    public void writeUnlock(String opName, boolean suppressWriteLockReport) {
        this.writeUnlock(opName, suppressWriteLockReport, null);
    }

    private void writeUnlock(String opName, boolean suppressWriteLockReport, Supplier<String> lockReportInfoSupplier) {
        boolean needReport = !suppressWriteLockReport && this.coarseLock.getWriteHoldCount() == 1 && this.coarseLock.isWriteLockedByCurrentThread();
        long writeLockIntervalNanos = this.timer.monotonicNowNanos() - this.writeLockHeldTimeStampNanos;
        long currentTimeMs = this.timer.now();
        long writeLockIntervalMs = TimeUnit.NANOSECONDS.toMillis(writeLockIntervalNanos);
        LogThrottlingHelper.LogAction logAction = LogThrottlingHelper.DO_NOT_LOG;
        if (needReport && writeLockIntervalMs >= this.writeLockReportingThresholdMs) {
            this.numWriteLockLongHold.increment();
            if (this.longestWriteLockHeldInfo.getIntervalMs() <= writeLockIntervalMs) {
                String lockReportInfo = lockReportInfoSupplier != null ? " (" + lockReportInfoSupplier.get() + ")" : "";
                this.longestWriteLockHeldInfo = new LockHeldInfo(currentTimeMs, writeLockIntervalMs, StringUtils.getStackTrace((Thread)Thread.currentThread()), opName, lockReportInfo);
            }
            logAction = this.writeLockReportLogger.record("write", currentTimeMs, new double[]{writeLockIntervalMs});
        }
        LockHeldInfo lockHeldInfo = this.longestWriteLockHeldInfo;
        if (logAction.shouldLog()) {
            this.longestWriteLockHeldInfo = new LockHeldInfo();
        }
        this.coarseLock.writeLock().unlock();
        if (needReport) {
            this.addMetric(opName, writeLockIntervalNanos, true);
        }
        if (logAction.shouldLog()) {
            FSNamesystem.LOG.info("\tNumber of suppressed write-lock reports of {}Lock is {}\n\tLongest write-lock held at {} for {}ms by {}{} via {}\n\tTotal suppressed write-lock held time: {}", new Object[]{this.lockName, logAction.getCount() - 1, Time.formatTime((long)lockHeldInfo.getStartTimeMs()), lockHeldInfo.getIntervalMs(), lockHeldInfo.getOpName(), lockHeldInfo.getLockReportInfo(), lockHeldInfo.getStackTrace(), logAction.getStats(0).getSum() - (double)lockHeldInfo.getIntervalMs().longValue()});
        }
    }

    public int getReadHoldCount() {
        return this.coarseLock.getReadHoldCount();
    }

    public int getWriteHoldCount() {
        return this.coarseLock.getWriteHoldCount();
    }

    public boolean isWriteLockedByCurrentThread() {
        return this.coarseLock.isWriteLockedByCurrentThread();
    }

    public Condition newWriteLockCondition() {
        return this.coarseLock.writeLock().newCondition();
    }

    public int getQueueLength() {
        return this.coarseLock.getQueueLength();
    }

    public long getNumOfReadLockLongHold() {
        return this.numReadLockLongHold.longValue();
    }

    public long getNumOfWriteLockLongHold() {
        return this.numWriteLockLongHold.longValue();
    }

    private void addMetric(String operationName, long value, boolean isWrite) {
        if (this.metricsEnabled) {
            String opMetric = this.getMetricName(operationName, isWrite);
            this.detailedHoldTimeMetrics.add(opMetric, value);
            String overallMetric = this.getMetricName(OVERALL_METRIC_NAME, isWrite);
            this.detailedHoldTimeMetrics.add(overallMetric, value);
        }
        FSNamesystemLock.updateProcessingDetails(isWrite ? ProcessingDetails.Timing.LOCKEXCLUSIVE : ProcessingDetails.Timing.LOCKSHARED, value);
    }

    private void doLock(boolean isWrite) {
        long startNanos = this.timer.monotonicNowNanos();
        if (isWrite) {
            this.coarseLock.writeLock().lock();
        } else {
            this.coarseLock.readLock().lock();
        }
        this.updateLockWait(startNanos, isWrite);
    }

    private void doLockInterruptibly(boolean isWrite) throws InterruptedException {
        long startNanos = this.timer.monotonicNowNanos();
        if (isWrite) {
            this.coarseLock.writeLock().lockInterruptibly();
        } else {
            this.coarseLock.readLock().lockInterruptibly();
        }
        this.updateLockWait(startNanos, isWrite);
    }

    private void updateLockWait(long startNanos, boolean isWrite) {
        long now = this.timer.monotonicNowNanos();
        FSNamesystemLock.updateProcessingDetails(ProcessingDetails.Timing.LOCKWAIT, now - startNanos);
        if (isWrite) {
            if (this.coarseLock.getWriteHoldCount() == 1) {
                this.writeLockHeldTimeStampNanos = now;
            }
        } else if (this.coarseLock.getReadHoldCount() == 1) {
            this.readLockHeldTimeStampNanos.set(now);
        }
    }

    private static void updateProcessingDetails(ProcessingDetails.Timing type, long deltaNanos) {
        Server.Call call = (Server.Call)Server.getCurCall().get();
        if (call != null) {
            call.getProcessingDetails().add(type, deltaNanos, TimeUnit.NANOSECONDS);
        }
    }

    private String getMetricName(String operationName, boolean isWrite) {
        return (isWrite ? this.writeLockMetricPrefix : this.readLockMetricPrefix) + org.apache.hadoop.shaded.org.apache.commons.lang3.StringUtils.capitalize((String)operationName) + LOCK_METRIC_SUFFIX;
    }

    @VisibleForTesting
    public void setMetricsEnabled(boolean metricsEnabled) {
        this.metricsEnabled = metricsEnabled;
    }

    public boolean isMetricsEnabled() {
        return this.metricsEnabled;
    }

    public void setReadLockReportingThresholdMs(long readLockReportingThresholdMs) {
        this.readLockReportingThresholdMs = readLockReportingThresholdMs;
    }

    @VisibleForTesting
    public long getReadLockReportingThresholdMs() {
        return this.readLockReportingThresholdMs;
    }

    public void setWriteLockReportingThresholdMs(long writeLockReportingThresholdMs) {
        this.writeLockReportingThresholdMs = writeLockReportingThresholdMs;
    }

    @VisibleForTesting
    public long getWriteLockReportingThresholdMs() {
        return this.writeLockReportingThresholdMs;
    }

    public void setLockForTests(ReentrantReadWriteLock lock) {
        this.coarseLock = lock;
    }

    public ReentrantReadWriteLock getLockForTests() {
        return this.coarseLock;
    }

    private static class LockHeldInfo {
        private final Long startTimeMs;
        private final Long intervalMs;
        private final String stackTrace;
        private final String opName;
        private final String lockReportInfo;

        LockHeldInfo() {
            this.startTimeMs = 0L;
            this.intervalMs = 0L;
            this.stackTrace = null;
            this.opName = null;
            this.lockReportInfo = null;
        }

        LockHeldInfo(long startTimeMs, long intervalMs, String stackTrace, String opName, String lockReportInfo) {
            this.startTimeMs = startTimeMs;
            this.intervalMs = intervalMs;
            this.stackTrace = stackTrace;
            this.opName = opName;
            this.lockReportInfo = lockReportInfo;
        }

        public Long getStartTimeMs() {
            return this.startTimeMs;
        }

        public Long getIntervalMs() {
            return this.intervalMs;
        }

        public String getStackTrace() {
            return this.stackTrace;
        }

        public String getOpName() {
            return this.opName;
        }

        public String getLockReportInfo() {
            return this.lockReportInfo;
        }
    }
}

