/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.protocol.datatransfer.sasl;

import java.io.ByteArrayInputStream;
import java.io.DataInput;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.NameCallback;
import javax.security.auth.callback.PasswordCallback;
import javax.security.auth.callback.UnsupportedCallbackException;
import javax.security.sasl.AuthorizeCallback;
import javax.security.sasl.RealmCallback;
import javax.security.sasl.SaslException;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.VisibleForTesting;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.crypto.CipherOption;
import org.apache.hadoop.hdfs.net.Peer;
import org.apache.hadoop.hdfs.protocol.DatanodeID;
import org.apache.hadoop.hdfs.protocol.datatransfer.IOStreamPair;
import org.apache.hadoop.hdfs.protocol.datatransfer.InvalidEncryptionKeyException;
import org.apache.hadoop.hdfs.protocol.datatransfer.sasl.DataTransferSaslUtil;
import org.apache.hadoop.hdfs.protocol.datatransfer.sasl.InvalidMagicNumberException;
import org.apache.hadoop.hdfs.protocol.datatransfer.sasl.SaslParticipant;
import org.apache.hadoop.hdfs.protocol.proto.DataTransferProtos;
import org.apache.hadoop.hdfs.security.token.block.BlockPoolTokenSecretManager;
import org.apache.hadoop.hdfs.security.token.block.BlockTokenIdentifier;
import org.apache.hadoop.hdfs.security.token.block.InvalidBlockTokenException;
import org.apache.hadoop.hdfs.server.datanode.DNConf;
import org.apache.hadoop.security.CustomizedCallbackHandler;
import org.apache.hadoop.security.SaslPropertiesResolver;
import org.apache.hadoop.security.SecurityUtil;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.security.token.SecretManager;
import org.apache.hadoop.shaded.org.apache.commons.codec.binary.Base64;
import org.apache.hadoop.util.Lists;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class SaslDataTransferServer {
    private static final Logger LOG = LoggerFactory.getLogger(SaslDataTransferServer.class);
    private final BlockPoolTokenSecretManager blockPoolTokenSecretManager;
    private final DNConf dnConf;
    private String negotiatedQOP;

    public SaslDataTransferServer(DNConf dnConf, BlockPoolTokenSecretManager blockPoolTokenSecretManager) {
        this.blockPoolTokenSecretManager = blockPoolTokenSecretManager;
        this.dnConf = dnConf;
    }

    public IOStreamPair receive(Peer peer, OutputStream underlyingOut, InputStream underlyingIn, int xferPort, DatanodeID datanodeId) throws IOException {
        if (this.dnConf.getEncryptDataTransfer()) {
            LOG.debug("SASL server doing encrypted handshake for peer = {}, datanodeId = {}", (Object)peer, (Object)datanodeId);
            return this.getEncryptedStreams(peer, underlyingOut, underlyingIn);
        }
        if (!UserGroupInformation.isSecurityEnabled()) {
            LOG.debug("SASL server skipping handshake in unsecured configuration for peer = {}, datanodeId = {}", (Object)peer, (Object)datanodeId);
            return new IOStreamPair(underlyingIn, underlyingOut);
        }
        if (SecurityUtil.isPrivilegedPort((int)xferPort)) {
            LOG.debug("SASL server skipping handshake in secured configuration for peer = {}, datanodeId = {}", (Object)peer, (Object)datanodeId);
            return new IOStreamPair(underlyingIn, underlyingOut);
        }
        if (this.dnConf.getSaslPropsResolver() != null) {
            LOG.debug("SASL server doing general handshake for peer = {}, datanodeId = {}", (Object)peer, (Object)datanodeId);
            return this.getSaslStreams(peer, underlyingOut, underlyingIn);
        }
        if (this.dnConf.getIgnoreSecurePortsForTesting()) {
            LOG.debug("SASL server skipping handshake in secured configuration with no SASL protection configured for peer = {}, datanodeId = {}", (Object)peer, (Object)datanodeId);
            return new IOStreamPair(underlyingIn, underlyingOut);
        }
        throw new IOException(String.format("Cannot create a secured connection if DataNode listens on unprivileged port (%d) and no protection is defined in configuration property %s.", datanodeId.getXferPort(), "dfs.data.transfer.protection"));
    }

    private IOStreamPair getEncryptedStreams(Peer peer, OutputStream underlyingOut, InputStream underlyingIn) throws IOException {
        if (peer.hasSecureChannel() || this.dnConf.getTrustedChannelResolver().isTrusted(DataTransferSaslUtil.getPeerAddress((Peer)peer))) {
            return new IOStreamPair(underlyingIn, underlyingOut);
        }
        Map saslProps = DataTransferSaslUtil.createSaslPropertiesForEncryption((String)this.dnConf.getEncryptionAlgorithm());
        if (LOG.isDebugEnabled()) {
            LOG.debug("Server using encryption algorithm " + this.dnConf.getEncryptionAlgorithm());
        }
        SaslServerCallbackHandler callbackHandler = new SaslServerCallbackHandler(this.dnConf.getConf(), new PasswordFunction(){

            @Override
            public char[] apply(String userName) throws IOException {
                return DataTransferSaslUtil.encryptionKeyToPassword((byte[])SaslDataTransferServer.this.getEncryptionKeyFromUserName(userName));
            }
        });
        return this.doSaslHandshake(peer, underlyingOut, underlyingIn, saslProps, callbackHandler);
    }

    private byte[] getEncryptionKeyFromUserName(String userName) throws IOException {
        String[] nameComponents = userName.split(" ");
        if (nameComponents.length != 3) {
            throw new IOException("Provided name '" + userName + "' has " + nameComponents.length + " components instead of the expected 3.");
        }
        int keyId = Integer.parseInt(nameComponents[0]);
        String blockPoolId = nameComponents[1];
        byte[] nonce = Base64.decodeBase64((String)nameComponents[2]);
        return this.blockPoolTokenSecretManager.retrieveDataEncryptionKey(keyId, blockPoolId, nonce);
    }

    private IOStreamPair getSaslStreams(Peer peer, OutputStream underlyingOut, InputStream underlyingIn) throws IOException {
        if (peer.hasSecureChannel() || this.dnConf.getTrustedChannelResolver().isTrusted(DataTransferSaslUtil.getPeerAddress((Peer)peer))) {
            return new IOStreamPair(underlyingIn, underlyingOut);
        }
        SaslPropertiesResolver saslPropsResolver = this.dnConf.getSaslPropsResolver();
        Map saslProps = saslPropsResolver.getServerProperties(DataTransferSaslUtil.getPeerAddress((Peer)peer));
        SaslServerCallbackHandler callbackHandler = new SaslServerCallbackHandler(this.dnConf.getConf(), new PasswordFunction(){

            @Override
            public char[] apply(String userName) throws IOException {
                return SaslDataTransferServer.this.buildServerPassword(userName);
            }
        });
        return this.doSaslHandshake(peer, underlyingOut, underlyingIn, saslProps, callbackHandler);
    }

    private char[] buildServerPassword(String userName) throws IOException {
        BlockTokenIdentifier identifier = this.deserializeIdentifier(userName);
        byte[] tokenPassword = this.blockPoolTokenSecretManager.retrievePassword(identifier);
        return new String(Base64.encodeBase64((byte[])tokenPassword, (boolean)false), StandardCharsets.UTF_8).toCharArray();
    }

    private BlockTokenIdentifier deserializeIdentifier(String str) throws IOException {
        BlockTokenIdentifier identifier = new BlockTokenIdentifier();
        identifier.readFields((DataInput)new DataInputStream(new ByteArrayInputStream(Base64.decodeBase64((String)str))));
        return identifier;
    }

    @VisibleForTesting
    public String getNegotiatedQOP() {
        return this.negotiatedQOP;
    }

    private IOStreamPair doSaslHandshake(Peer peer, OutputStream underlyingOut, InputStream underlyingIn, Map<String, String> saslProps, CallbackHandler callbackHandler) throws IOException {
        DataInputStream in = new DataInputStream(underlyingIn);
        DataOutputStream out = new DataOutputStream(underlyingOut);
        int magicNumber = in.readInt();
        if (magicNumber != -559038737) {
            throw new InvalidMagicNumberException(magicNumber, this.dnConf.getEncryptDataTransfer());
        }
        try {
            DataTransferSaslUtil.SaslMessageWithHandshake message = DataTransferSaslUtil.readSaslMessageWithHandshakeSecret((InputStream)in);
            byte[] secret = message.getSecret();
            String bpid = message.getBpid();
            TreeMap<String, String> dynamicSaslProps = new TreeMap<String, String>(saslProps);
            if (secret != null || bpid != null) {
                assert (secret != null && bpid != null);
                String qop = new String(secret, StandardCharsets.UTF_8);
                saslProps.put("javax.security.sasl.qop", qop);
                dynamicSaslProps.put("javax.security.sasl.qop", qop);
            }
            SaslParticipant sasl = SaslParticipant.createServerSaslParticipant(dynamicSaslProps, (CallbackHandler)callbackHandler);
            byte[] remoteResponse = message.getPayload();
            byte[] localResponse = sasl.evaluateChallengeOrResponse(remoteResponse);
            DataTransferSaslUtil.sendSaslMessage((OutputStream)out, (byte[])localResponse);
            ArrayList cipherOptions = Lists.newArrayList();
            remoteResponse = DataTransferSaslUtil.readSaslMessageAndNegotiationCipherOptions((InputStream)in, (List)cipherOptions);
            localResponse = sasl.evaluateChallengeOrResponse(remoteResponse);
            DataTransferSaslUtil.checkSaslComplete((SaslParticipant)sasl, dynamicSaslProps);
            CipherOption cipherOption = null;
            this.negotiatedQOP = sasl.getNegotiatedQop();
            if (sasl.isNegotiatedQopPrivacy()) {
                Configuration conf = this.dnConf.getConf();
                cipherOption = DataTransferSaslUtil.negotiateCipherOption((Configuration)conf, (List)cipherOptions);
                if (LOG.isDebugEnabled()) {
                    if (cipherOption == null) {
                        String cipherSuites = conf.get("dfs.encrypt.data.transfer.cipher.suites");
                        if (cipherSuites != null && !cipherSuites.isEmpty()) {
                            LOG.debug("Server accepts cipher suites {}, but client {} does not accept any of them", (Object)cipherSuites, (Object)peer.getRemoteAddressString());
                        }
                    } else {
                        LOG.debug("Server using cipher suite {} with client {}", (Object)cipherOption.getCipherSuite().getName(), (Object)peer.getRemoteAddressString());
                    }
                }
            }
            DataTransferSaslUtil.sendSaslMessageAndNegotiatedCipherOption((OutputStream)out, (byte[])localResponse, (CipherOption)DataTransferSaslUtil.wrap(cipherOption, (SaslParticipant)sasl));
            return cipherOption != null ? DataTransferSaslUtil.createStreamPair((Configuration)this.dnConf.getConf(), (CipherOption)cipherOption, (OutputStream)underlyingOut, (InputStream)underlyingIn, (boolean)true) : sasl.createStreamPair(out, in);
        }
        catch (IOException ioe) {
            if (ioe instanceof SaslException && ioe.getCause() != null && ioe.getCause() instanceof InvalidEncryptionKeyException) {
                SaslDataTransferServer.sendInvalidKeySaslErrorMessage(out, ioe.getCause().getMessage());
            } else if (ioe instanceof SaslException && ioe.getCause() != null && (ioe.getCause() instanceof InvalidBlockTokenException || ioe.getCause() instanceof SecretManager.InvalidToken)) {
                SaslDataTransferServer.sendInvalidTokenSaslErrorMessage(out, ioe.getCause().getMessage());
            } else {
                DataTransferSaslUtil.sendGenericSaslErrorMessage((OutputStream)out, (String)ioe.getMessage());
            }
            throw ioe;
        }
    }

    private static void sendInvalidKeySaslErrorMessage(DataOutputStream out, String message) throws IOException {
        DataTransferSaslUtil.sendSaslMessage((OutputStream)out, (DataTransferProtos.DataTransferEncryptorMessageProto.DataTransferEncryptorStatus)DataTransferProtos.DataTransferEncryptorMessageProto.DataTransferEncryptorStatus.ERROR_UNKNOWN_KEY, null, (String)message);
    }

    private static void sendInvalidTokenSaslErrorMessage(DataOutputStream out, String message) throws IOException {
        DataTransferSaslUtil.sendSaslMessage((OutputStream)out, (DataTransferProtos.DataTransferEncryptorMessageProto.DataTransferEncryptorStatus)DataTransferProtos.DataTransferEncryptorMessageProto.DataTransferEncryptorStatus.ERROR, null, (String)message, null, (boolean)true);
    }

    static final class SaslServerCallbackHandler
    implements CallbackHandler {
        private final PasswordFunction passwordFunction;
        private final CustomizedCallbackHandler customizedCallbackHandler;

        SaslServerCallbackHandler(Configuration conf, PasswordFunction passwordFunction) {
            this.passwordFunction = passwordFunction;
            this.customizedCallbackHandler = CustomizedCallbackHandler.get((String)"hadoop.security.sasl.CustomizedCallbackHandler.class", (Configuration)conf);
        }

        @Override
        public void handle(Callback[] callbacks) throws IOException, UnsupportedCallbackException {
            NameCallback nc = null;
            PasswordCallback pc = null;
            AuthorizeCallback ac = null;
            ArrayList<Callback> unknownCallbacks = null;
            for (Callback callback : callbacks) {
                if (callback instanceof AuthorizeCallback) {
                    ac = (AuthorizeCallback)callback;
                    continue;
                }
                if (callback instanceof PasswordCallback) {
                    pc = (PasswordCallback)callback;
                    continue;
                }
                if (callback instanceof NameCallback) {
                    nc = (NameCallback)callback;
                    continue;
                }
                if (callback instanceof RealmCallback) continue;
                if (unknownCallbacks == null) {
                    unknownCallbacks = new ArrayList<Callback>();
                }
                unknownCallbacks.add(callback);
            }
            if (pc != null) {
                pc.setPassword(this.passwordFunction.apply(nc.getDefaultName()));
            }
            if (ac != null) {
                ac.setAuthorized(true);
                ac.setAuthorizedID(ac.getAuthorizationID());
            }
            if (unknownCallbacks != null) {
                String name = nc != null ? nc.getDefaultName() : null;
                char[] password = name != null ? this.passwordFunction.apply(name) : null;
                this.customizedCallbackHandler.handleCallbacks(unknownCallbacks, name, password);
            }
        }
    }

    static interface PasswordFunction {
        public char[] apply(String var1) throws IOException;
    }
}

