/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.regex.Pattern;
import org.apache.hadoop.fs.FileContext;
import org.apache.hadoop.fs.FileContextTestHelper;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FsStatus;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.test.GenericTestUtils;
import org.apache.hadoop.test.LambdaTestUtils;
import org.apache.hadoop.util.Shell;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

public abstract class FileContextURIBase {
    private static final String basePath = GenericTestUtils.getTempPath("testContextURI");
    private static final Path BASE = new Path(basePath);
    private static final Pattern WIN_INVALID_FILE_NAME_PATTERN = Pattern.compile("^(.*?[<>\\:\"\\|\\?\\*].*?)|(.*?[ \\.])$");
    protected FileContext fc1;
    protected FileContext fc2;

    protected Path qualifiedPath(String path, FileContext fc) {
        return fc.makeQualified(new Path(BASE, path));
    }

    @BeforeEach
    public void setUp() throws Exception {
    }

    @AfterEach
    public void tearDown() throws Exception {
        if (this.fc2 != null) {
            this.fc2.delete(BASE, true);
        }
    }

    @Test
    public void testCreateFile() throws IOException {
        String[] fileNames;
        for (String f : fileNames = new String[]{"testFile", "test File", "test*File", "test#File", "test1234", "1234Test", "test)File", "test_File", "()&^%$#@!~_+}{><?", "  ", "^ "}) {
            if (!FileContextURIBase.isTestableFileNameOnPlatform(f)) continue;
            Path testPath = this.qualifiedPath(f, this.fc2);
            Assertions.assertFalse((boolean)FileContextTestHelper.exists(this.fc2, testPath));
            FileContextTestHelper.createFile(this.fc1, testPath);
            Assertions.assertTrue((boolean)FileContextTestHelper.exists(this.fc2, testPath));
        }
    }

    @Test
    public void testCreateFileWithNullName() throws IOException {
        String fileName = null;
        try {
            Path testPath = this.qualifiedPath(fileName, this.fc2);
            Assertions.assertFalse((boolean)FileContextTestHelper.exists(this.fc2, testPath));
            FileContextTestHelper.createFile(this.fc1, testPath);
            Assertions.fail((String)"Create file with null name should throw IllegalArgumentException.");
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
    }

    @Test
    public void testCreateExistingFile() throws Exception {
        String fileName = "testCreateExistingFile";
        Path testPath = this.qualifiedPath(fileName, this.fc2);
        Assertions.assertFalse((boolean)FileContextTestHelper.exists(this.fc2, testPath));
        FileContextTestHelper.createFile(this.fc1, testPath);
        LambdaTestUtils.intercept(IOException.class, () -> FileContextTestHelper.createFile(this.fc2, testPath));
        this.fc2.getFileStatus(testPath);
    }

    @Test
    public void testCreateFileInNonExistingDirectory() throws IOException {
        String fileName = "testCreateFileInNonExistingDirectory/testFile";
        Path testPath = this.qualifiedPath(fileName, this.fc2);
        Assertions.assertFalse((boolean)FileContextTestHelper.exists(this.fc2, testPath));
        FileContextTestHelper.createFile(this.fc1, testPath);
        Assertions.assertTrue((boolean)FileContextTestHelper.isDir(this.fc2, testPath.getParent()));
        Assertions.assertEquals((Object)"testCreateFileInNonExistingDirectory", (Object)testPath.getParent().getName());
        this.fc2.getFileStatus(testPath);
    }

    @Test
    public void testCreateDirectory() throws IOException {
        String[] dirNames;
        Path path = this.qualifiedPath("test/hadoop", this.fc2);
        Path falsePath = this.qualifiedPath("path/doesnot.exist", this.fc2);
        Path subDirPath = this.qualifiedPath("dir0", this.fc2);
        Assertions.assertFalse((boolean)FileContextTestHelper.exists(this.fc1, path));
        Assertions.assertFalse((boolean)FileContextTestHelper.isFile(this.fc1, path));
        Assertions.assertFalse((boolean)FileContextTestHelper.isDir(this.fc1, path));
        this.fc1.mkdir(path, FsPermission.getDefault(), true);
        Assertions.assertTrue((boolean)FileContextTestHelper.isDir(this.fc2, path));
        Assertions.assertTrue((boolean)FileContextTestHelper.exists(this.fc2, path));
        Assertions.assertFalse((boolean)FileContextTestHelper.isFile(this.fc2, path));
        this.fc1.mkdir(subDirPath, FsPermission.getDefault(), true);
        this.fc1.mkdir(subDirPath, FsPermission.getDefault(), true);
        this.fc1.mkdir(subDirPath, FsPermission.getDefault(), true);
        Path parentDir = path.getParent();
        Assertions.assertTrue((boolean)FileContextTestHelper.exists(this.fc2, parentDir));
        Assertions.assertFalse((boolean)FileContextTestHelper.isFile(this.fc2, parentDir));
        Path grandparentDir = parentDir.getParent();
        Assertions.assertTrue((boolean)FileContextTestHelper.exists(this.fc2, grandparentDir));
        Assertions.assertFalse((boolean)FileContextTestHelper.isFile(this.fc2, grandparentDir));
        Assertions.assertFalse((boolean)FileContextTestHelper.exists(this.fc2, falsePath));
        Assertions.assertFalse((boolean)FileContextTestHelper.isDir(this.fc2, falsePath));
        for (String f : dirNames = new String[]{"createTest/testDir", "createTest/test Dir", "createTest/test*Dir", "createTest/test#Dir", "createTest/test1234", "createTest/test_DIr", "createTest/1234Test", "createTest/test)Dir", "createTest/()&^%$#@!~_+}{><?", "createTest/  ", "createTest/^ "}) {
            if (!FileContextURIBase.isTestableFileNameOnPlatform(f)) continue;
            Path testPath = this.qualifiedPath(f, this.fc2);
            Assertions.assertFalse((boolean)FileContextTestHelper.exists(this.fc2, testPath));
            this.fc1.mkdir(testPath, FsPermission.getDefault(), true);
            Assertions.assertTrue((boolean)FileContextTestHelper.exists(this.fc2, testPath));
            Assertions.assertTrue((boolean)FileContextTestHelper.isDir(this.fc2, testPath));
        }
        Path parent = this.qualifiedPath("createTest", this.fc2);
        this.fc2.delete(parent, true);
        Assertions.assertFalse((boolean)FileContextTestHelper.exists(this.fc2, parent));
    }

    @Test
    public void testMkdirsFailsForSubdirectoryOfExistingFile() throws Exception {
        Path testDir = this.qualifiedPath("test/hadoop", this.fc2);
        Assertions.assertFalse((boolean)FileContextTestHelper.exists(this.fc2, testDir));
        this.fc2.mkdir(testDir, FsPermission.getDefault(), true);
        Assertions.assertTrue((boolean)FileContextTestHelper.exists(this.fc2, testDir));
        FileContextTestHelper.createFile(this.fc1, this.qualifiedPath("test/hadoop/file", this.fc2));
        Path testSubDir = this.qualifiedPath("test/hadoop/file/subdir", this.fc2);
        try {
            this.fc1.mkdir(testSubDir, FsPermission.getDefault(), true);
            Assertions.fail((String)"Should throw IOException.");
        }
        catch (IOException iOException) {
            // empty catch block
        }
        Assertions.assertFalse((boolean)FileContextTestHelper.exists(this.fc1, testSubDir));
        Path testDeepSubDir = this.qualifiedPath("test/hadoop/file/deep/sub/dir", this.fc1);
        try {
            this.fc2.mkdir(testDeepSubDir, FsPermission.getDefault(), true);
            Assertions.fail((String)"Should throw IOException.");
        }
        catch (IOException iOException) {
            // empty catch block
        }
        Assertions.assertFalse((boolean)FileContextTestHelper.exists(this.fc1, testDeepSubDir));
    }

    @Test
    public void testIsDirectory() throws IOException {
        String dirName = "dirTest";
        String invalidDir = "nonExistantDir";
        String rootDir = "/";
        Path existingPath = this.qualifiedPath(dirName, this.fc2);
        Path nonExistingPath = this.qualifiedPath(invalidDir, this.fc2);
        Path pathToRootDir = this.qualifiedPath(rootDir, this.fc2);
        this.fc1.mkdir(existingPath, FsPermission.getDefault(), true);
        Assertions.assertTrue((boolean)FileContextTestHelper.isDir(this.fc2, existingPath));
        Assertions.assertTrue((boolean)FileContextTestHelper.isDir(this.fc2, pathToRootDir));
        Assertions.assertFalse((boolean)FileContextTestHelper.isDir(this.fc2, nonExistingPath));
    }

    @Test
    public void testDeleteFile() throws IOException {
        Path testPath = this.qualifiedPath("testDeleteFile", this.fc2);
        Assertions.assertFalse((boolean)FileContextTestHelper.exists(this.fc2, testPath));
        FileContextTestHelper.createFile(this.fc1, testPath);
        Assertions.assertTrue((boolean)FileContextTestHelper.exists(this.fc2, testPath));
        this.fc2.delete(testPath, false);
        Assertions.assertFalse((boolean)FileContextTestHelper.exists(this.fc2, testPath));
    }

    @Test
    public void testDeleteNonExistingFile() throws IOException {
        String testFileName = "testDeleteNonExistingFile";
        Path testPath = this.qualifiedPath(testFileName, this.fc2);
        Assertions.assertFalse((boolean)FileContextTestHelper.exists(this.fc2, testPath));
        Assertions.assertFalse((boolean)this.fc2.delete(testPath, false));
        FileContextTestHelper.createFile(this.fc1, testPath);
        Assertions.assertTrue((boolean)FileContextTestHelper.exists(this.fc2, testPath));
        Assertions.assertTrue((boolean)this.fc2.delete(testPath, false));
        Assertions.assertFalse((boolean)FileContextTestHelper.exists(this.fc2, testPath));
        Assertions.assertFalse((boolean)this.fc2.delete(testPath, false));
    }

    @Test
    public void testDeleteNonExistingFileInDir() throws IOException {
        String testFileInDir = "testDeleteNonExistingFileInDir/testDir/TestFile";
        Path testPath = this.qualifiedPath(testFileInDir, this.fc2);
        Assertions.assertFalse((boolean)FileContextTestHelper.exists(this.fc2, testPath));
        Assertions.assertFalse((boolean)this.fc2.delete(testPath, false));
        FileContextTestHelper.createFile(this.fc1, testPath);
        Assertions.assertTrue((boolean)FileContextTestHelper.exists(this.fc2, testPath));
        Assertions.assertTrue((boolean)this.fc2.delete(testPath, false));
        Assertions.assertFalse((boolean)FileContextTestHelper.exists(this.fc2, testPath));
        Assertions.assertFalse((boolean)this.fc2.delete(testPath, false));
    }

    @Test
    public void testDeleteDirectory() throws IOException {
        String[] dirNames;
        String dirName = "dirTest";
        Path testDirPath = this.qualifiedPath(dirName, this.fc2);
        Assertions.assertFalse((boolean)FileContextTestHelper.exists(this.fc2, testDirPath));
        this.fc1.mkdir(testDirPath, FsPermission.getDefault(), true);
        Assertions.assertTrue((boolean)FileContextTestHelper.exists(this.fc2, testDirPath));
        Assertions.assertTrue((boolean)FileContextTestHelper.isDir(this.fc2, testDirPath));
        this.fc2.delete(testDirPath, true);
        Assertions.assertFalse((boolean)FileContextTestHelper.isDir(this.fc2, testDirPath));
        for (String f : dirNames = new String[]{"deleteTest/testDir", "deleteTest/test Dir", "deleteTest/test*Dir", "deleteTest/test#Dir", "deleteTest/test1234", "deleteTest/1234Test", "deleteTest/test)Dir", "deleteTest/test_DIr", "deleteTest/()&^%$#@!~_+}{><?", "deleteTest/  ", "deleteTest/^ "}) {
            if (!FileContextURIBase.isTestableFileNameOnPlatform(f)) continue;
            Path testPath = this.qualifiedPath(f, this.fc2);
            Assertions.assertFalse((boolean)FileContextTestHelper.exists(this.fc2, testPath));
            this.fc1.mkdir(testPath, FsPermission.getDefault(), true);
            Assertions.assertTrue((boolean)FileContextTestHelper.exists(this.fc2, testPath));
            Assertions.assertTrue((boolean)FileContextTestHelper.isDir(this.fc2, testPath));
            Assertions.assertTrue((boolean)this.fc2.delete(testPath, true));
            Assertions.assertFalse((boolean)FileContextTestHelper.exists(this.fc2, testPath));
            Assertions.assertFalse((boolean)FileContextTestHelper.isDir(this.fc2, testPath));
        }
    }

    @Test
    public void testDeleteNonExistingDirectory() throws IOException {
        String testDirName = "testDeleteNonExistingDirectory";
        Path testPath = this.qualifiedPath(testDirName, this.fc2);
        Assertions.assertFalse((boolean)FileContextTestHelper.exists(this.fc2, testPath));
        Assertions.assertFalse((boolean)this.fc2.delete(testPath, false));
        this.fc1.mkdir(testPath, FsPermission.getDefault(), true);
        Assertions.assertTrue((boolean)FileContextTestHelper.exists(this.fc2, testPath));
        Assertions.assertTrue((boolean)this.fc2.delete(testPath, false));
        Assertions.assertFalse((boolean)FileContextTestHelper.exists(this.fc2, testPath));
        Assertions.assertFalse((boolean)this.fc2.delete(testPath, false));
    }

    @Test
    public void testModificationTime() throws IOException {
        String testFile = "testModificationTime";
        Path testPath = this.qualifiedPath(testFile, this.fc2);
        FileContextTestHelper.createFile(this.fc1, testPath);
        long fc1ModificationTime = this.fc1.getFileStatus(testPath).getModificationTime();
        long fc2ModificationTime = this.fc2.getFileStatus(testPath).getModificationTime();
        Assertions.assertEquals((long)fc1ModificationTime, (long)fc2ModificationTime);
    }

    @Test
    public void testFileStatus() throws IOException {
        String fileName = "testModificationTime";
        Path path2 = this.fc2.makeQualified(new Path(BASE, fileName));
        FileContextTestHelper.createFile(this.fc1, path2);
        FsStatus fc2Status = this.fc2.getFsStatus(path2);
        Assertions.assertNotNull((Object)fc2Status);
        Assertions.assertTrue((fc2Status.getCapacity() > 0L ? 1 : 0) != 0);
        Assertions.assertTrue((fc2Status.getRemaining() > 0L ? 1 : 0) != 0);
        Assertions.assertTrue((fc2Status.getUsed() > 0L ? 1 : 0) != 0);
    }

    @Test
    public void testGetFileStatusThrowsExceptionForNonExistentFile() throws Exception {
        String testFile = "test/hadoop/fileDoesNotExist";
        Path testPath = this.qualifiedPath(testFile, this.fc2);
        try {
            this.fc1.getFileStatus(testPath);
            Assertions.fail((String)"Should throw FileNotFoundException");
        }
        catch (FileNotFoundException fileNotFoundException) {
            // empty catch block
        }
    }

    @Test
    public void testListStatusThrowsExceptionForNonExistentFile() throws Exception {
        String testFile = "test/hadoop/file";
        Path testPath = this.qualifiedPath(testFile, this.fc2);
        try {
            this.fc1.listStatus(testPath);
            Assertions.fail((String)"Should throw FileNotFoundException");
        }
        catch (FileNotFoundException fileNotFoundException) {
            // empty catch block
        }
    }

    @Test
    public void testListStatus() throws Exception {
        String hPrefix = "test/hadoop";
        String[] dirs = new String[]{"test/hadoop/a", "test/hadoop/b", "test/hadoop/c", "test/hadoop/1", "test/hadoop/#@#@", "test/hadoop/&*#$#$@234"};
        ArrayList<Path> testDirs = new ArrayList<Path>();
        for (String d : dirs) {
            if (!FileContextURIBase.isTestableFileNameOnPlatform(d)) continue;
            testDirs.add(this.qualifiedPath(d, this.fc2));
        }
        Assertions.assertFalse((boolean)FileContextTestHelper.exists(this.fc1, (Path)testDirs.get(0)));
        for (Path path : testDirs) {
            this.fc1.mkdir(path, FsPermission.getDefault(), true);
        }
        FileStatus[] paths = this.fc1.util().listStatus(this.qualifiedPath("test", this.fc1));
        Assertions.assertEquals((int)1, (int)paths.length);
        Assertions.assertEquals((Object)this.qualifiedPath("test/hadoop", this.fc1), (Object)paths[0].getPath());
        paths = this.fc1.util().listStatus(this.qualifiedPath("test/hadoop", this.fc1));
        Assertions.assertEquals((int)testDirs.size(), (int)paths.length);
        for (int i = 0; i < testDirs.size(); ++i) {
            boolean found = false;
            for (int j = 0; j < paths.length; ++j) {
                if (!this.qualifiedPath(((Path)testDirs.get(i)).toString(), this.fc1).equals((Object)paths[j].getPath())) continue;
                found = true;
            }
            Assertions.assertTrue((boolean)found, (String)(testDirs.get(i) + " not found"));
        }
        paths = this.fc1.util().listStatus(this.qualifiedPath(dirs[0], this.fc1));
        Assertions.assertEquals((int)0, (int)paths.length);
        RemoteIterator pathsItor = this.fc1.listStatus(this.qualifiedPath("test", this.fc1));
        Assertions.assertEquals((Object)this.qualifiedPath("test/hadoop", this.fc1), (Object)((FileStatus)pathsItor.next()).getPath());
        Assertions.assertFalse((boolean)pathsItor.hasNext());
        pathsItor = this.fc1.listStatus(this.qualifiedPath("test/hadoop", this.fc1));
        int dirLen = 0;
        while (pathsItor.hasNext()) {
            boolean found = false;
            FileStatus stat = (FileStatus)pathsItor.next();
            for (int j = 0; j < dirs.length; ++j) {
                if (!this.qualifiedPath(dirs[j], this.fc1).equals((Object)stat.getPath())) continue;
                found = true;
                break;
            }
            Assertions.assertTrue((boolean)found, (String)(stat.getPath() + " not found"));
            ++dirLen;
        }
        Assertions.assertEquals((int)testDirs.size(), (int)dirLen);
        pathsItor = this.fc1.listStatus(this.qualifiedPath(dirs[0], this.fc1));
        Assertions.assertFalse((boolean)pathsItor.hasNext());
    }

    private static boolean isTestableFileNameOnPlatform(String fileName) {
        boolean valid = true;
        if (Shell.WINDOWS) {
            valid = !WIN_INVALID_FILE_NAME_PATTERN.matcher(fileName).matches();
        }
        return valid;
    }
}

