/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.azurebfs.services;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URL;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.nio.charset.CharacterCodingException;
import java.nio.charset.Charset;
import java.nio.charset.CharsetEncoder;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.classification.VisibleForTesting;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.azurebfs.AbfsConfiguration;
import org.apache.hadoop.fs.azurebfs.AbfsStatistic;
import org.apache.hadoop.fs.azurebfs.AzureBlobFileSystemStore;
import org.apache.hadoop.fs.azurebfs.constants.AbfsHttpConstants;
import org.apache.hadoop.fs.azurebfs.constants.AbfsServiceType;
import org.apache.hadoop.fs.azurebfs.constants.FSOperationType;
import org.apache.hadoop.fs.azurebfs.contracts.exceptions.AbfsDriverException;
import org.apache.hadoop.fs.azurebfs.contracts.exceptions.AbfsInvalidChecksumException;
import org.apache.hadoop.fs.azurebfs.contracts.exceptions.AbfsRestOperationException;
import org.apache.hadoop.fs.azurebfs.contracts.exceptions.AzureBlobFileSystemException;
import org.apache.hadoop.fs.azurebfs.contracts.exceptions.ConcurrentWriteOperationDetectedException;
import org.apache.hadoop.fs.azurebfs.contracts.exceptions.InvalidAbfsRestOperationException;
import org.apache.hadoop.fs.azurebfs.contracts.services.AppendRequestParameters;
import org.apache.hadoop.fs.azurebfs.contracts.services.AzureServiceErrorCode;
import org.apache.hadoop.fs.azurebfs.contracts.services.BlobListResultEntrySchema;
import org.apache.hadoop.fs.azurebfs.contracts.services.BlobListResultSchema;
import org.apache.hadoop.fs.azurebfs.contracts.services.BlobListXmlParser;
import org.apache.hadoop.fs.azurebfs.contracts.services.StorageErrorResponseSchema;
import org.apache.hadoop.fs.azurebfs.extensions.EncryptionContextProvider;
import org.apache.hadoop.fs.azurebfs.extensions.SASTokenProvider;
import org.apache.hadoop.fs.azurebfs.oauth2.AccessTokenProvider;
import org.apache.hadoop.fs.azurebfs.security.ContextEncryptionAdapter;
import org.apache.hadoop.fs.azurebfs.services.AbfsClient;
import org.apache.hadoop.fs.azurebfs.services.AbfsClientContext;
import org.apache.hadoop.fs.azurebfs.services.AbfsClientRenameResult;
import org.apache.hadoop.fs.azurebfs.services.AbfsHttpHeader;
import org.apache.hadoop.fs.azurebfs.services.AbfsHttpOperation;
import org.apache.hadoop.fs.azurebfs.services.AbfsRestOperation;
import org.apache.hadoop.fs.azurebfs.services.AbfsRestOperationType;
import org.apache.hadoop.fs.azurebfs.services.AbfsUriQueryBuilder;
import org.apache.hadoop.fs.azurebfs.services.BlobDeleteHandler;
import org.apache.hadoop.fs.azurebfs.services.BlobRenameHandler;
import org.apache.hadoop.fs.azurebfs.services.ListResponseData;
import org.apache.hadoop.fs.azurebfs.services.RenameAtomicity;
import org.apache.hadoop.fs.azurebfs.services.SharedKeyCredentials;
import org.apache.hadoop.fs.azurebfs.services.VersionedFileStatus;
import org.apache.hadoop.fs.azurebfs.utils.ListUtils;
import org.apache.hadoop.fs.azurebfs.utils.TracingContext;
import org.apache.hadoop.fs.azurebfs.utils.UriUtils;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

public class AbfsBlobClient
extends AbfsClient {
    private final HashSet<String> azureAtomicRenameDirSet;
    private final ThreadLocal<SAXParser> saxParserThreadLocal = ThreadLocal.withInitial(() -> {
        SAXParserFactory factory = SAXParserFactory.newInstance();
        factory.setNamespaceAware(true);
        try {
            return factory.newSAXParser();
        }
        catch (SAXException e) {
            throw new RuntimeException("Unable to create SAXParser", e);
        }
        catch (ParserConfigurationException e) {
            throw new RuntimeException("Check parser configuration", e);
        }
    });

    public AbfsBlobClient(URL baseUrl, SharedKeyCredentials sharedKeyCredentials, AbfsConfiguration abfsConfiguration, AccessTokenProvider tokenProvider, EncryptionContextProvider encryptionContextProvider, AbfsClientContext abfsClientContext) throws IOException {
        super(baseUrl, sharedKeyCredentials, abfsConfiguration, tokenProvider, encryptionContextProvider, abfsClientContext, AbfsServiceType.BLOB);
        this.azureAtomicRenameDirSet = new HashSet<String>(Arrays.asList(abfsConfiguration.getAzureAtomicRenameDirs().split(",")));
    }

    public AbfsBlobClient(URL baseUrl, SharedKeyCredentials sharedKeyCredentials, AbfsConfiguration abfsConfiguration, SASTokenProvider sasTokenProvider, EncryptionContextProvider encryptionContextProvider, AbfsClientContext abfsClientContext) throws IOException {
        super(baseUrl, sharedKeyCredentials, abfsConfiguration, sasTokenProvider, encryptionContextProvider, abfsClientContext, AbfsServiceType.BLOB);
        this.azureAtomicRenameDirSet = new HashSet<String>(Arrays.asList(abfsConfiguration.getAzureAtomicRenameDirs().split(",")));
    }

    @Override
    public List<AbfsHttpHeader> createDefaultHeaders() {
        return this.createDefaultHeaders(this.getxMsVersion());
    }

    @Override
    public List<AbfsHttpHeader> createDefaultHeaders(AbfsHttpConstants.ApiVersion xMsVersion) {
        List<AbfsHttpHeader> requestHeaders = super.createCommonHeaders(xMsVersion);
        requestHeaders.add(new AbfsHttpHeader("Accept", "application/json, application/octet-stream, application/xml"));
        return requestHeaders;
    }

    @Override
    public AbfsRestOperation createFilesystem(TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        AbfsUriQueryBuilder abfsUriQueryBuilder = new AbfsUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("restype", "container");
        URL url = this.createRequestUrl(abfsUriQueryBuilder.toString());
        AbfsRestOperation op = this.getAbfsRestOperation(AbfsRestOperationType.CreateContainer, "PUT", url, requestHeaders);
        op.execute(tracingContext);
        return op;
    }

    @Override
    public AbfsRestOperation setFilesystemProperties(Hashtable<String, String> properties, TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        try {
            List<AbfsHttpHeader> metadataRequestHeaders = this.getMetadataHeadersList(properties);
            requestHeaders.addAll(metadataRequestHeaders);
        }
        catch (CharacterCodingException ex) {
            throw new InvalidAbfsRestOperationException(ex);
        }
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("restype", "container");
        abfsUriQueryBuilder.addQuery("comp", "metadata");
        URL url = this.createRequestUrl(abfsUriQueryBuilder.toString());
        AbfsRestOperation op = this.getAbfsRestOperation(AbfsRestOperationType.SetContainerMetadata, "PUT", url, requestHeaders);
        op.execute(tracingContext);
        return op;
    }

    @Override
    public AbfsRestOperation getFilesystemProperties(TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("restype", "container");
        URL url = this.createRequestUrl(abfsUriQueryBuilder.toString());
        AbfsRestOperation op = this.getAbfsRestOperation(AbfsRestOperationType.GetContainerProperties, "HEAD", url, requestHeaders);
        op.execute(tracingContext);
        return op;
    }

    @Override
    public AbfsRestOperation deleteFilesystem(TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("restype", "container");
        URL url = this.createRequestUrl(abfsUriQueryBuilder.toString());
        AbfsRestOperation op = this.getAbfsRestOperation(AbfsRestOperationType.DeleteContainer, "DELETE", url, requestHeaders);
        op.execute(tracingContext);
        return op;
    }

    @Override
    public ListResponseData listPath(String relativePath, boolean recursive, int listMaxResults, String continuation, TracingContext tracingContext, URI uri) throws AzureBlobFileSystemException {
        boolean isRenameRecovered;
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("restype", "container");
        abfsUriQueryBuilder.addQuery("comp", "list");
        abfsUriQueryBuilder.addQuery("include", "metadata");
        abfsUriQueryBuilder.addQuery("prefix", AbfsBlobClient.getDirectoryQueryParameter(relativePath));
        abfsUriQueryBuilder.addQuery("marker", continuation);
        if (!recursive) {
            abfsUriQueryBuilder.addQuery("delimiter", "/");
        }
        abfsUriQueryBuilder.addQuery("maxresults", String.valueOf(listMaxResults));
        this.appendSASTokenToQuery(relativePath, "list-blob", abfsUriQueryBuilder);
        URL url = this.createRequestUrl(abfsUriQueryBuilder.toString());
        AbfsRestOperation op = this.getAbfsRestOperation(AbfsRestOperationType.ListBlobs, "GET", url, requestHeaders);
        op.execute(tracingContext);
        ListResponseData listResponseData = this.parseListPathResults(op.getResult(), uri);
        listResponseData.setOp(op);
        if (tracingContext.getOpType() == FSOperationType.LISTSTATUS && op.getResult() != null && op.getResult().getStatusCode() == 200 && (isRenameRecovered = this.retryRenameOnAtomicEntriesInListResults(tracingContext, listResponseData.getRenamePendingJsonPaths()))) {
            LOG.debug("Retrying list operation after rename recovery.");
            AbfsRestOperation retryListOp = this.getAbfsRestOperation(AbfsRestOperationType.ListBlobs, "GET", url, requestHeaders);
            retryListOp.execute(tracingContext);
            listResponseData = this.parseListPathResults(retryListOp.getResult(), uri);
            listResponseData.setOp(retryListOp);
        }
        return listResponseData;
    }

    @Override
    public List<FileStatus> postListProcessing(String relativePath, List<FileStatus> fileStatuses, TracingContext tracingContext, URI uri) throws AzureBlobFileSystemException {
        ArrayList<FileStatus> rectifiedFileStatuses = new ArrayList<FileStatus>();
        if (fileStatuses.isEmpty() && !"/".equals(relativePath)) {
            AbfsRestOperation pathStatus = this.getPathStatus(relativePath, tracingContext, null, false);
            BlobListResultSchema listResultSchema = this.getListResultSchemaFromPathStatus(relativePath, pathStatus);
            LOG.debug("ListStatus attempted on a file path {}. Returning file status.", (Object)relativePath);
            for (BlobListResultEntrySchema entry : listResultSchema.paths()) {
                rectifiedFileStatuses.add(this.getVersionedFileStatusFromEntry(entry, uri));
            }
        } else {
            rectifiedFileStatuses.addAll(ListUtils.getUniqueListResult(fileStatuses));
            LOG.debug("ListBlob API returned a total of {} elements including duplicates.Number of unique Elements are {}", (Object)fileStatuses.size(), (Object)rectifiedFileStatuses.size());
        }
        return rectifiedFileStatuses;
    }

    private boolean retryRenameOnAtomicEntriesInListResults(TracingContext tracingContext, Map<Path, Integer> renamePendingJsonPaths) throws AzureBlobFileSystemException {
        if (renamePendingJsonPaths == null || renamePendingJsonPaths.isEmpty()) {
            return false;
        }
        for (Map.Entry<Path, Integer> entry : renamePendingJsonPaths.entrySet()) {
            this.retryRenameOnAtomicKeyPath(entry.getKey(), entry.getValue(), tracingContext);
        }
        return true;
    }

    @Override
    public void createNonRecursivePreCheck(Path parentPath, TracingContext tracingContext) throws IOException {
        try {
            if (this.isAtomicRenameKey(parentPath.toUri().getPath())) {
                this.takeGetPathStatusAtomicRenameKeyAction(parentPath, tracingContext);
            }
            try {
                this.getPathStatus(parentPath.toUri().getPath(), false, tracingContext, null);
            }
            finally {
                this.getAbfsCounters().incrementCounter(AbfsStatistic.CALL_GET_FILE_STATUS, 1L);
            }
        }
        catch (AbfsRestOperationException ex) {
            if (ex.getStatusCode() == 404) {
                throw new FileNotFoundException("Cannot create file " + parentPath.toUri().getPath() + " because parent folder does not exist.");
            }
            throw ex;
        }
    }

    @Override
    public AbfsRestOperation createPath(String path, boolean isFileCreation, boolean overwrite, AzureBlobFileSystemStore.Permissions permissions, boolean isAppendBlob, String eTag, ContextEncryptionAdapter contextEncryptionAdapter, TracingContext tracingContext) throws AzureBlobFileSystemException {
        AbfsRestOperation op;
        if (isFileCreation) {
            if (this.getAbfsConfiguration().getIsCreateIdempotencyEnabled()) {
                AbfsRestOperation statusOp;
                block7: {
                    statusOp = null;
                    try {
                        statusOp = this.getPathStatus(path, tracingContext, null, false);
                    }
                    catch (AbfsRestOperationException ex) {
                        if (ex.getStatusCode() == 404) break block7;
                        throw ex;
                    }
                }
                if (statusOp != null && statusOp.hasResult() && !overwrite) {
                    throw new AbfsRestOperationException(409, AzureServiceErrorCode.PATH_CONFLICT.getErrorCode(), "The specified path, or an element of the path, exists and its resource type is invalid for this operation.", null);
                }
                op = this.createFile(path, true, permissions, isAppendBlob, eTag, contextEncryptionAdapter, tracingContext);
            } else {
                op = this.createFile(path, overwrite, permissions, isAppendBlob, eTag, contextEncryptionAdapter, tracingContext);
            }
        } else {
            op = this.createDirectory(path, permissions, isAppendBlob, eTag, contextEncryptionAdapter, tracingContext);
        }
        return op;
    }

    public AbfsRestOperation createMarkerAtPath(String path, String eTag, ContextEncryptionAdapter contextEncryptionAdapter, TracingContext tracingContext) throws AzureBlobFileSystemException {
        return this.createPathRestOp(path, false, false, false, eTag, contextEncryptionAdapter, tracingContext);
    }

    public AbfsRestOperation createPathRestOp(String path, boolean isFile, boolean overwrite, boolean isAppendBlob, String eTag, ContextEncryptionAdapter contextEncryptionAdapter, TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        if (isFile) {
            this.addEncryptionKeyRequestHeaders(path, requestHeaders, true, contextEncryptionAdapter, tracingContext);
            this.appendSASTokenToQuery(path, "create-file", abfsUriQueryBuilder);
        } else {
            requestHeaders.add(new AbfsHttpHeader("x-ms-meta-hdi_isfolder", "true"));
            this.appendSASTokenToQuery(path, "create-directory", abfsUriQueryBuilder);
        }
        requestHeaders.add(new AbfsHttpHeader("Content-Length", "0"));
        if (isAppendBlob) {
            requestHeaders.add(new AbfsHttpHeader("x-ms-blob-type", "appendblob"));
        } else {
            requestHeaders.add(new AbfsHttpHeader("x-ms-blob-type", "BlockBlob"));
        }
        if (!overwrite) {
            requestHeaders.add(new AbfsHttpHeader("If-None-Match", "*"));
        }
        if (eTag != null && !eTag.isEmpty()) {
            requestHeaders.add(new AbfsHttpHeader("If-Match", eTag));
        }
        URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = this.getAbfsRestOperation(AbfsRestOperationType.PutBlob, "PUT", url, requestHeaders);
        op.execute(tracingContext);
        return op;
    }

    @Override
    public AbfsRestOperation conditionalCreateOverwriteFile(String relativePath, FileSystem.Statistics statistics, AzureBlobFileSystemStore.Permissions permissions, boolean isAppendBlob, ContextEncryptionAdapter contextEncryptionAdapter, TracingContext tracingContext) throws IOException {
        AbfsRestOperation op;
        if (!this.getIsNamespaceEnabled()) {
            if (this.isNonEmptyDirectory(relativePath, tracingContext)) {
                throw new AbfsRestOperationException(409, AzureServiceErrorCode.PATH_CONFLICT.getErrorCode(), "The specified path, or an element of the path, exists and its resource type is invalid for this operation.", null);
            }
            this.tryMarkerCreation(relativePath, isAppendBlob, null, contextEncryptionAdapter, tracingContext);
        }
        try {
            op = this.createPathRestOp(relativePath, true, false, isAppendBlob, null, contextEncryptionAdapter, tracingContext);
        }
        catch (AbfsRestOperationException e) {
            if (e.getStatusCode() == 409) {
                try {
                    op = this.getPathStatus(relativePath, tracingContext, null, false);
                }
                catch (AbfsRestOperationException ex) {
                    if (ex.getStatusCode() == 404) {
                        throw new ConcurrentWriteOperationDetectedException("Parallel access to the create path detected. Failing request as the path which existed before gives not found error");
                    }
                    throw ex;
                }
                boolean isExplicitDir = this.checkIsDir(op.getResult());
                if (isExplicitDir) {
                    throw new AbfsRestOperationException(409, AzureServiceErrorCode.PATH_CONFLICT.getErrorCode(), "The specified path, or an element of the path, exists and its resource type is invalid for this operation.", null);
                }
                String eTag = AzureBlobFileSystemStore.extractEtagHeader(op.getResult());
                try {
                    op = this.createPathRestOp(relativePath, true, true, isAppendBlob, eTag, contextEncryptionAdapter, tracingContext);
                }
                catch (AbfsRestOperationException ex) {
                    if (ex.getStatusCode() == 412) {
                        throw new ConcurrentWriteOperationDetectedException("Parallel access to the create path detected. Failing request due to precondition failure");
                    }
                    throw ex;
                }
            }
            throw e;
        }
        return op;
    }

    @Override
    public AbfsRestOperation acquireLease(String path, int duration, String eTag, TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        requestHeaders.add(new AbfsHttpHeader("x-ms-lease-action", "acquire"));
        requestHeaders.add(new AbfsHttpHeader("x-ms-lease-duration", Integer.toString(duration)));
        requestHeaders.add(new AbfsHttpHeader("x-ms-proposed-lease-id", UUID.randomUUID().toString()));
        if (StringUtils.isNotEmpty((CharSequence)eTag)) {
            requestHeaders.add(new AbfsHttpHeader("If-Match", eTag));
        }
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("comp", "lease");
        this.appendSASTokenToQuery(path, "lease-blob", abfsUriQueryBuilder);
        URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = this.getAbfsRestOperation(AbfsRestOperationType.LeaseBlob, "PUT", url, requestHeaders);
        op.execute(tracingContext);
        return op;
    }

    @Override
    public AbfsRestOperation renewLease(String path, String leaseId, TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        requestHeaders.add(new AbfsHttpHeader("x-ms-lease-action", "renew"));
        requestHeaders.add(new AbfsHttpHeader("x-ms-lease-id", leaseId));
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("comp", "lease");
        this.appendSASTokenToQuery(path, "lease-blob", abfsUriQueryBuilder);
        URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = this.getAbfsRestOperation(AbfsRestOperationType.LeaseBlob, "PUT", url, requestHeaders);
        op.execute(tracingContext);
        return op;
    }

    @Override
    public AbfsRestOperation releaseLease(String path, String leaseId, TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        requestHeaders.add(new AbfsHttpHeader("x-ms-lease-action", "release"));
        requestHeaders.add(new AbfsHttpHeader("x-ms-lease-id", leaseId));
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("comp", "lease");
        this.appendSASTokenToQuery(path, "lease-blob", abfsUriQueryBuilder);
        URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = this.getAbfsRestOperation(AbfsRestOperationType.LeaseBlob, "PUT", url, requestHeaders);
        op.execute(tracingContext);
        return op;
    }

    @Override
    public AbfsRestOperation breakLease(String path, TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        requestHeaders.add(new AbfsHttpHeader("x-ms-lease-action", "break"));
        requestHeaders.add(new AbfsHttpHeader("x-ms-lease-break-period", "0"));
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("comp", "lease");
        this.appendSASTokenToQuery(path, "lease-blob", abfsUriQueryBuilder);
        URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = this.getAbfsRestOperation(AbfsRestOperationType.LeaseBlob, "PUT", url, requestHeaders);
        op.execute(tracingContext);
        return op;
    }

    @Override
    public AbfsClientRenameResult renamePath(String source, String destination, String continuation, TracingContext tracingContext, String sourceEtag, boolean isMetadataIncompleteState) throws IOException {
        BlobRenameHandler blobRenameHandler = this.getBlobRenameHandler(source, destination, sourceEtag, this.isAtomicRenameKey(source), tracingContext);
        try {
            if (blobRenameHandler.execute(false)) {
                AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
                this.appendSASTokenToQuery(source, "rename-source", abfsUriQueryBuilder);
                URL url = this.createRequestUrl(destination, abfsUriQueryBuilder.toString());
                List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
                AbfsRestOperation successOp = this.getSuccessOp(AbfsRestOperationType.RenamePath, "PUT", url, requestHeaders);
                AbfsClientRenameResult abfsClientRenameResult = new AbfsClientRenameResult(successOp, true, false);
                return abfsClientRenameResult;
            }
            throw new AbfsRestOperationException(500, AzureServiceErrorCode.UNKNOWN.getErrorCode(), "FNS-Blob rename was not successful for source and destination path: " + source + " & " + destination, null);
        }
        finally {
            this.incrementAbfsRenamePath();
        }
    }

    @VisibleForTesting
    BlobRenameHandler getBlobRenameHandler(String source, String destination, String sourceEtag, boolean isAtomicRename, TracingContext tracingContext) {
        return new BlobRenameHandler(source, destination, this, sourceEtag, isAtomicRename, false, tracingContext);
    }

    @Override
    public AbfsRestOperation append(String path, byte[] buffer, AppendRequestParameters reqParams, String cachedSasToken, ContextEncryptionAdapter contextEncryptionAdapter, TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        this.addEncryptionKeyRequestHeaders(path, requestHeaders, false, contextEncryptionAdapter, tracingContext);
        requestHeaders.add(new AbfsHttpHeader("Content-Length", String.valueOf(buffer.length)));
        if (reqParams.getLeaseId() != null) {
            requestHeaders.add(new AbfsHttpHeader("x-ms-lease-id", reqParams.getLeaseId()));
        }
        if (reqParams.isExpectHeaderEnabled()) {
            requestHeaders.add(new AbfsHttpHeader("Expect", "100-continue"));
        }
        if (this.isChecksumValidationEnabled()) {
            this.addCheckSumHeaderForWrite(requestHeaders, reqParams);
        }
        if (reqParams.isRetryDueToExpect()) {
            String userAgentRetry = this.getUserAgent();
            userAgentRetry = userAgentRetry.replace(" 100-continue;", "");
            requestHeaders.removeIf(header -> header.getName().equalsIgnoreCase("User-Agent"));
            requestHeaders.add(new AbfsHttpHeader("User-Agent", userAgentRetry));
        }
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("comp", "block");
        abfsUriQueryBuilder.addQuery("blockid", reqParams.getBlockId());
        String sasTokenForReuse = this.appendSASTokenToQuery(path, "write", abfsUriQueryBuilder, cachedSasToken);
        URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = this.getAbfsRestOperation(AbfsRestOperationType.PutBlock, "PUT", url, requestHeaders, buffer, reqParams.getoffset(), reqParams.getLength(), sasTokenForReuse);
        try {
            op.execute(tracingContext);
        }
        catch (AbfsRestOperationException e) {
            int responseStatusCode = e.getStatusCode();
            if (this.checkUserError(responseStatusCode) && reqParams.isExpectHeaderEnabled()) {
                LOG.debug("User error, retrying without 100 continue enabled for the given path {}", (Object)path);
                reqParams.setExpectHeaderEnabled(false);
                reqParams.setRetryDueToExpect(true);
                return this.append(path, buffer, reqParams, cachedSasToken, contextEncryptionAdapter, tracingContext);
            }
            if (!op.hasResult()) {
                throw e;
            }
            if (this.isMd5ChecksumError(e)) {
                throw new AbfsInvalidChecksumException(e);
            }
            throw e;
        }
        catch (AzureBlobFileSystemException e) {
            LOG.debug("Append request failed with non server issues for path: {}, offset: {}, position: {}", new Object[]{path, reqParams.getoffset(), reqParams.getPosition()});
            throw e;
        }
        return op;
    }

    public AbfsRestOperation appendBlock(String path, AppendRequestParameters requestParameters, byte[] data, TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        requestHeaders.add(new AbfsHttpHeader("Content-Length", String.valueOf(data.length)));
        requestHeaders.add(new AbfsHttpHeader("x-ms-blob-type", "appendblob"));
        if (requestParameters.getLeaseId() != null) {
            requestHeaders.add(new AbfsHttpHeader("x-ms-lease-id", requestParameters.getLeaseId()));
        }
        if (this.isChecksumValidationEnabled()) {
            this.addCheckSumHeaderForWrite(requestHeaders, requestParameters);
        }
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("comp", "appendblock");
        String sasTokenForReuse = this.appendSASTokenToQuery(path, "write", abfsUriQueryBuilder);
        URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = this.getAbfsRestOperation(AbfsRestOperationType.AppendBlock, "PUT", url, requestHeaders, data, requestParameters.getoffset(), requestParameters.getLength(), sasTokenForReuse);
        try {
            op.execute(tracingContext);
        }
        catch (AzureBlobFileSystemException ex) {
            LOG.debug("Exception occurred during append block operation for path: {}", (Object)path, (Object)ex);
            if (!op.hasResult()) {
                throw ex;
            }
            throw ex;
        }
        return op;
    }

    @Override
    public AbfsRestOperation flush(String path, long position, boolean retainUncommittedData, boolean isClose, String cachedSasToken, String leaseId, ContextEncryptionAdapter contextEncryptionAdapter, TracingContext tracingContext, String blobMd5) throws AzureBlobFileSystemException {
        throw new UnsupportedOperationException("Flush without blockIds not supported on Blob Endpoint");
    }

    @Override
    public AbfsRestOperation flush(byte[] buffer, String path, boolean isClose, String cachedSasToken, String leaseId, String eTag, ContextEncryptionAdapter contextEncryptionAdapter, TracingContext tracingContext, String blobMd5) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        this.addEncryptionKeyRequestHeaders(path, requestHeaders, false, contextEncryptionAdapter, tracingContext);
        requestHeaders.add(new AbfsHttpHeader("Content-Length", String.valueOf(buffer.length)));
        requestHeaders.add(new AbfsHttpHeader("Content-Type", "application/xml"));
        requestHeaders.add(new AbfsHttpHeader("If-Match", eTag));
        if (leaseId != null) {
            requestHeaders.add(new AbfsHttpHeader("x-ms-lease-id", leaseId));
        }
        String md5Value = this.isFullBlobChecksumValidationEnabled() && blobMd5 != null ? blobMd5 : this.computeMD5Hash(buffer, 0, buffer.length);
        requestHeaders.add(new AbfsHttpHeader("x-ms-blob-content-md5", md5Value));
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("comp", "blocklist");
        abfsUriQueryBuilder.addQuery("close", String.valueOf(isClose));
        String sasTokenForReuse = this.appendSASTokenToQuery(path, "write", abfsUriQueryBuilder, cachedSasToken);
        URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = this.getAbfsRestOperation(AbfsRestOperationType.PutBlockList, "PUT", url, requestHeaders, buffer, 0, buffer.length, sasTokenForReuse);
        try {
            op.execute(tracingContext);
        }
        catch (AbfsRestOperationException ex) {
            if (op.getRetryCount() >= 1 && ex.getStatusCode() == 412) {
                AbfsRestOperation op1 = this.getPathStatus(path, true, tracingContext, contextEncryptionAdapter);
                String metadataMd5 = op1.getResult().getResponseHeader("Content-MD5");
                if (md5Value != null && !md5Value.equals(metadataMd5)) {
                    throw ex;
                }
                return op;
            }
            throw ex;
        }
        return op;
    }

    @Override
    public AbfsRestOperation setPathProperties(String path, Hashtable<String, String> properties, TracingContext tracingContext, ContextEncryptionAdapter contextEncryptionAdapter) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        try {
            List<AbfsHttpHeader> metadataRequestHeaders = this.getMetadataHeadersList(properties);
            requestHeaders.addAll(metadataRequestHeaders);
        }
        catch (CharacterCodingException ex) {
            throw new InvalidAbfsRestOperationException(ex);
        }
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("comp", "metadata");
        this.appendSASTokenToQuery(path, "set-properties", abfsUriQueryBuilder);
        URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = this.getAbfsRestOperation(AbfsRestOperationType.SetPathProperties, "PUT", url, requestHeaders);
        try {
            op.execute(tracingContext);
        }
        catch (AbfsRestOperationException ex) {
            if (!op.hasResult()) {
                throw ex;
            }
            if (op.getResult().getStatusCode() == 404 && this.isNonEmptyDirectory(path, tracingContext)) {
                this.createPathRestOp(path, false, false, false, null, contextEncryptionAdapter, tracingContext);
                boolean hdiIsFolderExists = properties.keySet().stream().anyMatch("hdi_isfolder"::equalsIgnoreCase);
                if (!hdiIsFolderExists) {
                    properties.put("hdi_isfolder", "true");
                }
                return this.setPathProperties(path, properties, tracingContext, contextEncryptionAdapter);
            }
            throw ex;
        }
        return op;
    }

    @Override
    public AbfsRestOperation getPathStatus(String path, boolean includeProperties, TracingContext tracingContext, ContextEncryptionAdapter contextEncryptionAdapter) throws AzureBlobFileSystemException {
        AbfsRestOperation op = this.getPathStatus(path, tracingContext, contextEncryptionAdapter, true);
        if (tracingContext.getOpType() == FSOperationType.GET_FILESTATUS && op.getResult() != null && this.checkIsDir(op.getResult())) {
            this.takeGetPathStatusAtomicRenameKeyAction(new Path(path), tracingContext);
        }
        return op;
    }

    public AbfsRestOperation getPathStatus(String path, TracingContext tracingContext, ContextEncryptionAdapter contextEncryptionAdapter, boolean isImplicitCheckRequired) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        abfsUriQueryBuilder.addQuery("upn", String.valueOf(this.getAbfsConfiguration().isUpnUsed()));
        this.appendSASTokenToQuery(path, "get-properties", abfsUriQueryBuilder);
        URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = this.getAbfsRestOperation(AbfsRestOperationType.GetBlobProperties, "HEAD", url, requestHeaders);
        try {
            op.execute(tracingContext);
        }
        catch (AzureBlobFileSystemException ex) {
            if (!op.hasResult()) {
                throw ex;
            }
            if (op.getResult().getStatusCode() == 404 && isImplicitCheckRequired && this.isNonEmptyDirectory(path, tracingContext)) {
                try {
                    this.createMarkerAtPath(path, null, contextEncryptionAdapter, tracingContext);
                }
                catch (AbfsRestOperationException exception) {
                    LOG.debug("Marker creation failed for path {} during getPathStatus. StatusCode: {}, ErrorCode: {}", new Object[]{path, exception.getStatusCode(), exception.getErrorCode()});
                }
                AbfsRestOperation successOp = this.getSuccessOp(AbfsRestOperationType.GetPathStatus, "HEAD", url, requestHeaders);
                successOp.hardSetGetFileStatusResult(200);
                return successOp;
            }
            if (op.getResult().getStatusCode() == 404) {
                throw new AbfsRestOperationException(404, AzureServiceErrorCode.BLOB_PATH_NOT_FOUND.getErrorCode(), ex.getMessage(), ex);
            }
            throw ex;
        }
        return op;
    }

    @Override
    public AbfsRestOperation read(String path, long position, byte[] buffer, int bufferOffset, int bufferLength, String eTag, String cachedSasToken, ContextEncryptionAdapter contextEncryptionAdapter, TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        AbfsHttpHeader rangeHeader = new AbfsHttpHeader("Range", String.format("bytes=%d-%d", position, position + (long)bufferLength - 1L));
        requestHeaders.add(rangeHeader);
        requestHeaders.add(new AbfsHttpHeader("If-Match", eTag));
        if (this.isChecksumValidationEnabled(requestHeaders, rangeHeader, bufferLength)) {
            requestHeaders.add(new AbfsHttpHeader("x-ms-range-get-content-md5", "true"));
        }
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        String sasTokenForReuse = this.appendSASTokenToQuery(path, "read", abfsUriQueryBuilder, cachedSasToken);
        URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = this.getAbfsRestOperation(AbfsRestOperationType.GetBlob, "GET", url, requestHeaders, buffer, bufferOffset, bufferLength, sasTokenForReuse);
        op.execute(tracingContext);
        if (this.isChecksumValidationEnabled(requestHeaders, rangeHeader, bufferLength)) {
            this.verifyCheckSumForRead(buffer, op.getResult(), bufferOffset);
        }
        return op;
    }

    @Override
    public AbfsRestOperation deletePath(String path, boolean recursive, String continuation, TracingContext tracingContext) throws AzureBlobFileSystemException {
        BlobDeleteHandler blobDeleteHandler = this.getBlobDeleteHandler(path, recursive, tracingContext);
        if (blobDeleteHandler.execute()) {
            AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
            URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
            List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
            return this.getSuccessOp(AbfsRestOperationType.DeletePath, "DELETE", url, requestHeaders);
        }
        throw new AbfsRestOperationException(500, AzureServiceErrorCode.UNKNOWN.getErrorCode(), "FNS-Blob delete was not successful for path: " + path, null);
    }

    @VisibleForTesting
    public BlobDeleteHandler getBlobDeleteHandler(String path, boolean recursive, TracingContext tracingContext) {
        return new BlobDeleteHandler(new Path(path), recursive, this, tracingContext);
    }

    @Override
    public AbfsRestOperation setOwner(String path, String owner, String group, TracingContext tracingContext) throws AzureBlobFileSystemException {
        throw new UnsupportedOperationException("SetOwner operation is only supported on HNS enabled Accounts.");
    }

    @Override
    public AbfsRestOperation setPermission(String path, String permission, TracingContext tracingContext) throws AzureBlobFileSystemException {
        throw new UnsupportedOperationException("SetPermission operation is only supported on HNS enabled Accounts.");
    }

    @Override
    public AbfsRestOperation setAcl(String path, String aclSpecString, String eTag, TracingContext tracingContext) throws AzureBlobFileSystemException {
        throw new UnsupportedOperationException("SetAcl operation is only supported on HNS enabled Accounts.");
    }

    @Override
    public AbfsRestOperation getAclStatus(String path, boolean useUPN, TracingContext tracingContext) throws AzureBlobFileSystemException {
        throw new UnsupportedOperationException("GetAclStatus operation is only supported on HNS enabled Accounts.");
    }

    @Override
    public AbfsRestOperation checkAccess(String path, String rwx, TracingContext tracingContext) throws AzureBlobFileSystemException {
        throw new UnsupportedOperationException("CheckAccess operation is only supported on HNS enabled Accounts.");
    }

    public AbfsRestOperation getBlockList(String path, TracingContext tracingContext) throws AzureBlobFileSystemException {
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        String operation = "read";
        this.appendSASTokenToQuery(path, operation, abfsUriQueryBuilder);
        abfsUriQueryBuilder.addQuery("comp", "blocklist");
        abfsUriQueryBuilder.addQuery("blocklisttype", "committed");
        URL url = this.createRequestUrl(path, abfsUriQueryBuilder.toString());
        AbfsRestOperation op = this.getAbfsRestOperation(AbfsRestOperationType.GetBlockList, "GET", url, requestHeaders);
        op.execute(tracingContext);
        return op;
    }

    public AbfsRestOperation copyBlob(Path sourceBlobPath, Path destinationBlobPath, String srcLeaseId, TracingContext tracingContext) throws AzureBlobFileSystemException {
        AbfsUriQueryBuilder abfsUriQueryBuilderDst = this.createDefaultUriQueryBuilder();
        AbfsUriQueryBuilder abfsUriQueryBuilderSrc = new AbfsUriQueryBuilder();
        String dstBlobRelativePath = destinationBlobPath.toUri().getPath();
        String srcBlobRelativePath = sourceBlobPath.toUri().getPath();
        this.appendSASTokenToQuery(dstBlobRelativePath, "copy-blob-dst", abfsUriQueryBuilderDst);
        this.appendSASTokenToQuery(srcBlobRelativePath, "copy-blob-src", abfsUriQueryBuilderSrc);
        URL url = this.createRequestUrl(dstBlobRelativePath, abfsUriQueryBuilderDst.toString());
        String sourcePathUrl = this.createRequestUrl(srcBlobRelativePath, abfsUriQueryBuilderSrc.toString()).toString();
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        if (srcLeaseId != null) {
            requestHeaders.add(new AbfsHttpHeader("x-ms-source-lease-id", srcLeaseId));
        }
        requestHeaders.add(new AbfsHttpHeader("x-ms-copy-source", sourcePathUrl));
        requestHeaders.add(new AbfsHttpHeader("If-None-Match", "*"));
        AbfsRestOperation op = this.getAbfsRestOperation(AbfsRestOperationType.CopyBlob, "PUT", url, requestHeaders);
        op.execute(tracingContext);
        return op;
    }

    public AbfsRestOperation deleteBlobPath(Path blobPath, String leaseId, TracingContext tracingContext) throws AzureBlobFileSystemException {
        AbfsUriQueryBuilder abfsUriQueryBuilder = this.createDefaultUriQueryBuilder();
        String blobRelativePath = blobPath.toUri().getPath();
        this.appendSASTokenToQuery(blobRelativePath, "delete", abfsUriQueryBuilder);
        URL url = this.createRequestUrl(blobRelativePath, abfsUriQueryBuilder.toString());
        List<AbfsHttpHeader> requestHeaders = this.createDefaultHeaders();
        if (leaseId != null) {
            requestHeaders.add(new AbfsHttpHeader("x-ms-lease-id", leaseId));
        }
        AbfsRestOperation op = this.getAbfsRestOperation(AbfsRestOperationType.DeleteBlob, "DELETE", url, requestHeaders);
        try {
            op.execute(tracingContext);
            return op;
        }
        catch (AzureBlobFileSystemException e) {
            if (!op.hasResult()) {
                throw e;
            }
            AbfsRestOperation idempotencyOp = this.deleteIdempotencyCheckOp(op);
            if (idempotencyOp.getResult().getStatusCode() == op.getResult().getStatusCode()) {
                throw e;
            }
            return idempotencyOp;
        }
    }

    @Override
    public boolean checkIsDir(AbfsHttpOperation result) {
        String resourceType = result.getResponseHeaderIgnoreCase("x-ms-meta-hdi_isfolder");
        return resourceType != null && resourceType.equals("true");
    }

    @Override
    public boolean checkUserError(int responseStatusCode) {
        return responseStatusCode >= 400 && responseStatusCode < 500 && responseStatusCode != 409;
    }

    @Override
    public Hashtable<String, String> getXMSProperties(AbfsHttpOperation result) throws InvalidAbfsRestOperationException {
        Hashtable<String, String> properties = new Hashtable<String, String>();
        Map<String, List<String>> responseHeaders = result.getResponseHeaders();
        for (Map.Entry<String, List<String>> entry : responseHeaders.entrySet()) {
            String value;
            String name = entry.getKey();
            if (name == null || !name.startsWith("x-ms-meta-")) continue;
            try {
                value = AbfsBlobClient.decodeMetadataAttribute(entry.getValue().get(0));
            }
            catch (UnsupportedEncodingException e) {
                throw new InvalidAbfsRestOperationException(e);
            }
            properties.put(name.substring("x-ms-meta-".length()), value);
        }
        return properties;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public ListResponseData parseListPathResults(AbfsHttpOperation result, URI uri) throws AzureBlobFileSystemException {
        try {
            InputStream stream = result.getListResultStream();
            try {
                SAXParser saxParser = this.saxParserThreadLocal.get();
                saxParser.reset();
                BlobListResultSchema listResultSchema = new BlobListResultSchema();
                saxParser.parse(stream, (DefaultHandler)new BlobListXmlParser(listResultSchema, this.getBaseUrl().toString()));
                result.setListResultSchema(listResultSchema);
                LOG.debug("ListBlobs listed {} blobs with {} as continuation token", (Object)listResultSchema.paths().size(), (Object)listResultSchema.getNextMarker());
                ListResponseData listResponseData = this.filterRenamePendingFiles(listResultSchema, uri);
                return listResponseData;
            }
            catch (IOException | SAXException ex) {
                throw new AbfsDriverException("Parsing of XML List Response Failed in BlobClient.", ex);
            }
            finally {
                if (stream != null) {
                    try {
                        stream.close();
                    }
                    catch (Throwable throwable) {
                        Throwable throwable2;
                        throwable2.addSuppressed(throwable);
                    }
                }
            }
        }
        catch (AbfsDriverException ex) {
            LOG.error("Unable to deserialize list results for Uri {}", (Object)(uri != null ? uri.toString() : "NULL"), (Object)ex);
            throw ex;
        }
        catch (Exception ex) {
            LOG.error("Unable to get stream for list results for uri {}", (Object)(uri != null ? uri.toString() : "NULL"), (Object)ex);
            throw new AbfsDriverException("Parsing of XML List Response Failed in BlobClient.", ex);
        }
    }

    @Override
    public List<String> parseBlockListResponse(InputStream stream) throws IOException {
        ArrayList<String> blockIdList = new ArrayList<String>();
        try {
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            Document doc = factory.newDocumentBuilder().parse(stream);
            NodeList committedBlocksList = doc.getElementsByTagName("CommittedBlocks");
            if (committedBlocksList.getLength() > 0) {
                Node committedBlocks = committedBlocksList.item(0);
                NodeList blockList = committedBlocks.getChildNodes();
                for (int i = 0; i < blockList.getLength(); ++i) {
                    Node block = blockList.item(i);
                    if (!block.getNodeName().equals("Block")) continue;
                    NodeList nameList = block.getChildNodes();
                    for (int j = 0; j < nameList.getLength(); ++j) {
                        Node name = nameList.item(j);
                        if (!name.getNodeName().equals("Name")) continue;
                        String blockId = name.getTextContent();
                        blockIdList.add(blockId);
                    }
                }
            }
        }
        catch (ParserConfigurationException | SAXException e) {
            throw new IOException(e);
        }
        return blockIdList;
    }

    @Override
    public StorageErrorResponseSchema processStorageErrorResponse(InputStream stream) throws IOException {
        String data = IOUtils.toString((InputStream)stream, (Charset)StandardCharsets.UTF_8);
        String storageErrorCode = "";
        String storageErrorMessage = "";
        String expectedAppendPos = "";
        int codeStartFirstInstance = data.indexOf("<Code>");
        int codeEndFirstInstance = data.indexOf("</Code>");
        if (codeEndFirstInstance != -1 && codeStartFirstInstance != -1) {
            storageErrorCode = data.substring(codeStartFirstInstance, codeEndFirstInstance).replace("<Code>", "");
        }
        int msgStartFirstInstance = data.indexOf("<Message>");
        int msgEndFirstInstance = data.indexOf("</Message>");
        if (msgEndFirstInstance != -1 && msgStartFirstInstance != -1) {
            storageErrorMessage = data.substring(msgStartFirstInstance, msgEndFirstInstance).replace("<Message>", "");
        }
        return new StorageErrorResponseSchema(storageErrorCode, storageErrorMessage, expectedAppendPos);
    }

    @Override
    public byte[] encodeAttribute(String value) throws UnsupportedEncodingException {
        return value.getBytes("UTF-8");
    }

    @Override
    public String decodeAttribute(byte[] value) throws UnsupportedEncodingException {
        return new String(value, "UTF-8");
    }

    public static String getDirectoryQueryParameter(String path) {
        Object directory = AbfsClient.getDirectoryQueryParameter(path);
        if (((String)directory).isEmpty()) {
            return directory;
        }
        if (!((String)directory).endsWith("/")) {
            directory = (String)directory + "/";
        }
        return directory;
    }

    public boolean isAtomicRenameKey(String key) {
        return UriUtils.isKeyForDirectorySet(key, this.azureAtomicRenameDirSet);
    }

    public void takeGetPathStatusAtomicRenameKeyAction(Path path, TracingContext tracingContext) throws AzureBlobFileSystemException {
        boolean renameSrcHasChanged;
        if (path == null || path.isRoot() || !this.isAtomicRenameKey(path.toUri().getPath())) {
            return;
        }
        Path pendingJsonPath = new Path(path.getParent(), path.toUri().getPath() + "-RenamePending.json");
        int pendingJsonFileContentLength = 0;
        try {
            AbfsRestOperation pendingJsonFileStatus = this.getPathStatus(pendingJsonPath.toUri().getPath(), tracingContext, null, false);
            if (this.checkIsDir(pendingJsonFileStatus.getResult())) {
                return;
            }
            pendingJsonFileContentLength = Integer.parseInt(pendingJsonFileStatus.getResult().getResponseHeader("Content-Length"));
        }
        catch (AbfsRestOperationException ex) {
            if (ex.getStatusCode() == 404) {
                return;
            }
            throw ex;
        }
        try {
            RenameAtomicity renameAtomicity = this.getRedoRenameAtomicity(pendingJsonPath, pendingJsonFileContentLength, tracingContext);
            renameAtomicity.redo();
            renameSrcHasChanged = false;
        }
        catch (AbfsRestOperationException ex) {
            if (ex.getStatusCode() == 404 || ex.getStatusCode() == 409) {
                renameSrcHasChanged = true;
            }
            throw ex;
        }
        if (!renameSrcHasChanged) {
            throw new AbfsRestOperationException(AzureServiceErrorCode.PATH_NOT_FOUND.getStatusCode(), AzureServiceErrorCode.PATH_NOT_FOUND.getErrorCode(), "Path had to be recovered from atomic rename operation.", null);
        }
    }

    private void retryRenameOnAtomicKeyPath(Path path, int renamePendingJsonLen, TracingContext tracingContext) throws AzureBlobFileSystemException {
        block2: {
            try {
                RenameAtomicity renameAtomicity = this.getRedoRenameAtomicity(path, renamePendingJsonLen, tracingContext);
                renameAtomicity.redo();
            }
            catch (AbfsRestOperationException ex) {
                if (ex.getStatusCode() == 404 || ex.getStatusCode() == 409) break block2;
                throw ex;
            }
        }
    }

    @VisibleForTesting
    public RenameAtomicity getRedoRenameAtomicity(Path renamePendingJsonPath, int fileLen, TracingContext tracingContext) {
        return new RenameAtomicity(renamePendingJsonPath, fileLen, tracingContext, null, this);
    }

    private boolean isPureASCII(String value) throws CharacterCodingException {
        CharsetEncoder encoder = Charset.forName("ISO-8859-1").newEncoder();
        boolean canEncodeValue = encoder.canEncode(value);
        if (!canEncodeValue) {
            LOG.debug("Value {} for ne of the metadata is not pure ASCII.", (Object)value);
            throw new CharacterCodingException();
        }
        return true;
    }

    private List<AbfsHttpHeader> getMetadataHeadersList(Hashtable<String, String> properties) throws AbfsRestOperationException, CharacterCodingException {
        ArrayList<AbfsHttpHeader> metadataRequestHeaders = new ArrayList<AbfsHttpHeader>();
        for (Map.Entry<String, String> entry : properties.entrySet()) {
            String key = "x-ms-meta-" + entry.getKey();
            String value = entry.getValue();
            if (!this.isPureASCII(value)) continue;
            try {
                if (!"hdi_permission".equalsIgnoreCase(entry.getKey())) {
                    value = AbfsBlobClient.encodeMetadataAttribute(value);
                }
            }
            catch (UnsupportedEncodingException e) {
                throw new InvalidAbfsRestOperationException(e);
            }
            metadataRequestHeaders.add(new AbfsHttpHeader(key, value));
        }
        return metadataRequestHeaders;
    }

    @VisibleForTesting
    public ListResponseData filterRenamePendingFiles(BlobListResultSchema listResultSchema, URI uri) throws IOException {
        ArrayList<VersionedFileStatus> fileStatuses = new ArrayList<VersionedFileStatus>();
        HashMap<Path, Integer> renamePendingJsonPaths = new HashMap<Path, Integer>();
        for (BlobListResultEntrySchema entry : listResultSchema.paths()) {
            if (this.isRenamePendingJsonPathEntry(entry)) {
                renamePendingJsonPaths.put(entry.path(), entry.contentLength().intValue());
                continue;
            }
            fileStatuses.add(this.getVersionedFileStatusFromEntry(entry, uri));
        }
        ListResponseData listResponseData = new ListResponseData();
        listResponseData.setFileStatusList(fileStatuses);
        listResponseData.setRenamePendingJsonPaths(renamePendingJsonPaths);
        listResponseData.setContinuationToken(listResultSchema.getNextMarker());
        return listResponseData;
    }

    private boolean isRenamePendingJsonPathEntry(BlobListResultEntrySchema entry) {
        String path = entry.path() != null ? entry.path().toUri().getPath() : null;
        return path != null && !entry.path().isRoot() && this.isAtomicRenameKey(path) && entry.isDirectory() == false && path.endsWith("-RenamePending.json");
    }

    private BlobListResultSchema getListResultSchemaFromPathStatus(String relativePath, AbfsRestOperation pathStatus) {
        BlobListResultSchema listResultSchema = new BlobListResultSchema();
        BlobListResultEntrySchema entrySchema = new BlobListResultEntrySchema();
        entrySchema.setUrl(pathStatus.getUrl().toString());
        entrySchema.setPath(new Path(relativePath));
        entrySchema.setName(relativePath.charAt(0) == '/' ? relativePath.substring(1) : relativePath);
        entrySchema.setIsDirectory(this.checkIsDir(pathStatus.getResult()));
        entrySchema.setContentLength(Long.parseLong(pathStatus.getResult().getResponseHeader("Content-Length")));
        entrySchema.setLastModifiedTime(pathStatus.getResult().getResponseHeader("Last-Modified"));
        entrySchema.setETag(AzureBlobFileSystemStore.extractEtagHeader(pathStatus.getResult()));
        if (!entrySchema.isDirectory().booleanValue()) {
            listResultSchema.paths().add(entrySchema);
        }
        return listResultSchema;
    }

    private static String encodeMetadataAttribute(String value) throws UnsupportedEncodingException {
        return value == null ? null : URLEncoder.encode(value, StandardCharsets.UTF_8.name());
    }

    private static String decodeMetadataAttribute(String encoded) throws UnsupportedEncodingException {
        return encoded == null ? null : URLDecoder.decode(encoded, StandardCharsets.UTF_8.name());
    }

    @VisibleForTesting
    public boolean isNonEmptyDirectory(String path, TracingContext tracingContext) throws AzureBlobFileSystemException {
        ListResponseData listResponseData;
        String continuationToken = null;
        ArrayList<VersionedFileStatus> fileStatusList = new ArrayList<VersionedFileStatus>();
        do {
            listResponseData = this.listPath(path, false, 1, continuationToken, tracingContext, null);
            fileStatusList.addAll(listResponseData.getFileStatusList());
        } while (StringUtils.isNotEmpty((CharSequence)(continuationToken = listResponseData.getContinuationToken())) && fileStatusList.isEmpty());
        return !fileStatusList.isEmpty();
    }

    public static String generateBlockListXml(String blockIdString) {
        StringBuilder stringBuilder = new StringBuilder();
        stringBuilder.append(String.format("<?xml version=\"1.0\" encoding=\"UTF-8\"?>%n", new Object[0]));
        stringBuilder.append(String.format("<BlockList>%n", new Object[0]));
        if (!blockIdString.isEmpty()) {
            String[] blockIds;
            for (String blockId : blockIds = blockIdString.split(",")) {
                stringBuilder.append(String.format("<Latest>%s</Latest>%n", blockId));
            }
        }
        stringBuilder.append(String.format("</BlockList>%n", new Object[0]));
        return stringBuilder.toString();
    }

    private boolean isExistingDirectory(String path, TracingContext tracingContext) throws AzureBlobFileSystemException {
        if (this.isNonEmptyDirectory(path, tracingContext)) {
            return true;
        }
        return this.isEmptyDirectory(path, tracingContext, true);
    }

    private boolean isEmptyDirectory(String path, TracingContext tracingContext, boolean isDirCheck) throws AzureBlobFileSystemException {
        AbfsRestOperation getPathStatusOp;
        block4: {
            getPathStatusOp = null;
            try {
                getPathStatusOp = this.getPathStatus(path, tracingContext, null, false);
            }
            catch (AbfsRestOperationException ex) {
                if (ex.getStatusCode() == 404) break block4;
                throw ex;
            }
        }
        if (getPathStatusOp != null) {
            boolean isDirectory = this.checkIsDir(getPathStatusOp.getResult());
            if (!isDirectory && isDirCheck) {
                throw new AbfsRestOperationException(409, AzureServiceErrorCode.PATH_CONFLICT.getErrorCode(), "The specified path, or an element of the path, exists and its resource type is invalid for this operation.", null);
            }
            return isDirectory;
        }
        return false;
    }

    private AbfsRestOperation createSuccessResponse(String path) throws AzureBlobFileSystemException {
        AbfsRestOperation successOp = this.getAbfsRestOperation(AbfsRestOperationType.PutBlob, "PUT", this.createRequestUrl(path, ""), this.createDefaultHeaders());
        successOp.hardSetResult(201);
        return successOp;
    }

    private AbfsRestOperation createDirectory(String path, AzureBlobFileSystemStore.Permissions permissions, boolean isAppendBlob, String eTag, ContextEncryptionAdapter contextEncryptionAdapter, TracingContext tracingContext) throws AzureBlobFileSystemException {
        if (!this.getIsNamespaceEnabled()) {
            try {
                if (this.isExistingDirectory(path, tracingContext)) {
                    return this.createSuccessResponse(path);
                }
            }
            catch (AzureBlobFileSystemException ex) {
                LOG.error("Path exists as file {} : {}", (Object)path, (Object)ex.getMessage());
                throw ex;
            }
            this.tryMarkerCreation(path, isAppendBlob, eTag, contextEncryptionAdapter, tracingContext);
        }
        return this.createPathRestOp(path, false, true, isAppendBlob, eTag, contextEncryptionAdapter, tracingContext);
    }

    private AbfsRestOperation createFile(String path, boolean overwrite, AzureBlobFileSystemStore.Permissions permissions, boolean isAppendBlob, String eTag, ContextEncryptionAdapter contextEncryptionAdapter, TracingContext tracingContext) throws AzureBlobFileSystemException {
        if (!this.getIsNamespaceEnabled()) {
            if (this.isNonEmptyDirectory(path, tracingContext)) {
                throw new AbfsRestOperationException(409, AzureServiceErrorCode.PATH_CONFLICT.getErrorCode(), "The specified path, or an element of the path, exists and its resource type is invalid for this operation.", null);
            }
            if (overwrite && this.isEmptyDirectory(path, tracingContext, false)) {
                throw new AbfsRestOperationException(409, AzureServiceErrorCode.PATH_CONFLICT.getErrorCode(), "The specified path, or an element of the path, exists and its resource type is invalid for this operation.", null);
            }
            this.tryMarkerCreation(path, isAppendBlob, eTag, contextEncryptionAdapter, tracingContext);
        }
        return this.createPathRestOp(path, true, overwrite, isAppendBlob, eTag, contextEncryptionAdapter, tracingContext);
    }

    @VisibleForTesting
    public List<Path> getMarkerPathsTobeCreated(Path path, TracingContext tracingContext) throws AzureBlobFileSystemException {
        ArrayList<Path> keysToCreateAsFolder = new ArrayList<Path>();
        this.findParentPathsForMarkerCreation(path, tracingContext, keysToCreateAsFolder);
        return keysToCreateAsFolder;
    }

    @VisibleForTesting
    public void tryMarkerCreation(String path, boolean isAppendBlob, String eTag, ContextEncryptionAdapter contextEncryptionAdapter, TracingContext tracingContext) throws AzureBlobFileSystemException {
        Path parentPath = new Path(path).getParent();
        if (parentPath != null && !parentPath.isRoot()) {
            List<Path> keysToCreateAsFolder = this.getMarkerPathsTobeCreated(parentPath, tracingContext);
            for (Path pathToCreate : keysToCreateAsFolder) {
                try {
                    this.createPathRestOp(pathToCreate.toUri().getPath(), false, false, isAppendBlob, eTag, contextEncryptionAdapter, tracingContext);
                }
                catch (AbfsRestOperationException e) {
                    LOG.debug("Swallow exception for failed marker creation");
                }
            }
        }
    }

    private void findParentPathsForMarkerCreation(Path path, TracingContext tracingContext, List<Path> keysToCreateAsFolder) throws AzureBlobFileSystemException {
        AbfsHttpOperation opResult = null;
        Path current = path;
        do {
            try {
                opResult = this.getPathStatus(current.toUri().getPath(), tracingContext, null, false).getResult();
            }
            catch (AbfsRestOperationException ex) {
                if (ex.getStatusCode() == 404) {
                    LOG.debug("No explicit directory/path found: {}", (Object)current);
                }
                LOG.debug("Exception occurred while getting path status: {}", (Object)current, (Object)ex);
                throw ex;
            }
            if (opResult != null) {
                if (this.checkIsDir(opResult)) {
                    return;
                }
                throw new AbfsRestOperationException(409, AzureServiceErrorCode.PATH_CONFLICT.getErrorCode(), "The specified path, or an element of the path, exists and its resource type is invalid for this operation.", null);
            }
            keysToCreateAsFolder.add(current);
            current = current.getParent();
        } while (current != null && !current.isRoot());
    }
}

