/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.azure.integration;

import java.io.Closeable;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.EnumSet;
import java.util.Iterator;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.StorageStatistics;
import org.apache.hadoop.fs.azure.AzureBlobStorageTestAccount;
import org.apache.hadoop.fs.azure.NativeAzureFileSystem;
import org.apache.hadoop.fs.azure.integration.AbstractAzureScaleTest;
import org.apache.hadoop.fs.azure.integration.AzureTestUtils;
import org.apache.hadoop.fs.contract.ContractTestUtils;
import org.apache.hadoop.io.IOUtils;
import org.assertj.core.api.AbstractBooleanAssert;
import org.assertj.core.api.AbstractLongAssert;
import org.assertj.core.api.Assumptions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.MethodOrderer;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestMethodOrder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@TestMethodOrder(value=MethodOrderer.Alphanumeric.class)
public class ITestAzureHugeFiles
extends AbstractAzureScaleTest {
    private static final Logger LOG = LoggerFactory.getLogger(ITestAzureHugeFiles.class);
    private Path scaleTestDir;
    private Path hugefile;
    private Path hugefileRenamed;
    private AzureBlobStorageTestAccount testAccountForCleanup;
    private static final int UPLOAD_BLOCKSIZE = 65536;
    private static final byte[] SOURCE_DATA = ContractTestUtils.dataset((int)65536, (int)0, (int)256);
    private Path testPath;

    @Override
    @BeforeEach
    public void setUp() throws Exception {
        super.setUp();
        this.testPath = this.path("ITestAzureHugeFiles");
        this.scaleTestDir = new Path(this.testPath, "scale");
        this.hugefile = new Path(this.scaleTestDir, "hugefile");
        this.hugefileRenamed = new Path(this.scaleTestDir, "hugefileRenamed");
    }

    @Override
    public void tearDown() throws Exception {
        this.testAccount = null;
        super.tearDown();
        if (this.testAccountForCleanup != null) {
            AzureTestUtils.cleanupTestAccount(this.testAccount);
        }
    }

    @Override
    protected AzureBlobStorageTestAccount createTestAccount() throws Exception {
        return AzureBlobStorageTestAccount.create("testazurehugefiles", EnumSet.of(AzureBlobStorageTestAccount.CreateOptions.CreateContainer), this.createConfiguration(), true);
    }

    protected void deleteTestDirInTeardown() throws IOException {
    }

    protected void deleteHugeFile() throws IOException {
        this.describe("Deleting %s", this.hugefile);
        ContractTestUtils.NanoTimer timer = new ContractTestUtils.NanoTimer();
        this.getFileSystem().delete(this.hugefile, false);
        timer.end("time to delete %s", new Object[]{this.hugefile});
    }

    protected void logTimePerIOP(String operation, ContractTestUtils.NanoTimer timer, long count) {
        LOG.info("Time per {}: {} nS", (Object)operation, (Object)ContractTestUtils.toHuman((long)(timer.duration() / count)));
    }

    FileStatus assumeHugeFileExists() throws IOException {
        ContractTestUtils.assertPathExists((FileSystem)this.getFileSystem(), (String)"huge file not created", (Path)this.hugefile);
        try {
            FileStatus status = this.getFileSystem().getFileStatus(this.hugefile);
            ((AbstractBooleanAssert)Assumptions.assumeThat((boolean)status.isFile()).as("Not a file: " + status, new Object[0])).isTrue();
            ((AbstractLongAssert)Assumptions.assumeThat((long)status.getLen()).as("File " + this.hugefile + " is empty", new Object[0])).isPositive();
            return status;
        }
        catch (FileNotFoundException e) {
            ContractTestUtils.skip((String)("huge file not created: " + this.hugefile));
            return null;
        }
    }

    private void logFSState() {
        StorageStatistics statistics = this.getFileSystem().getStorageStatistics();
        Iterator longStatistics = statistics.getLongStatistics();
        while (longStatistics.hasNext()) {
            StorageStatistics.LongStatistic next = (StorageStatistics.LongStatistic)longStatistics.next();
            LOG.info("{} = {}", (Object)next.getName(), (Object)next.getValue());
        }
    }

    @Test
    public void test_010_CreateHugeFile() throws IOException {
        long filesize = AzureTestUtils.getTestPropertyBytes(this.getConfiguration(), "fs.azure.scale.test.huge.filesize", "10M");
        long filesizeMB = filesize / 0x100000L;
        this.deleteHugeFile();
        this.describe("Creating file %s of size %d MB", this.hugefile, filesizeMB);
        ITestAzureHugeFiles.assertEquals((long)0L, (long)(filesize % 65536L), (String)("File size set in fs.azure.scale.test.huge.filesize = " + filesize + " is not a multiple of 65536"));
        byte[] data = SOURCE_DATA;
        long blocks = filesize / 65536L;
        long blocksPerMB = 16L;
        NativeAzureFileSystem fs = this.getFileSystem();
        ContractTestUtils.NanoTimer timer = new ContractTestUtils.NanoTimer();
        long blocksPer10MB = blocksPerMB * 10L;
        fs.mkdirs(this.hugefile.getParent());
        try (FSDataOutputStream out = fs.create(this.hugefile, true, 65536, null);){
            for (long block = 1L; block <= blocks; ++block) {
                out.write(data);
                long written = block * 65536L;
                if (block % blocksPer10MB != 0L && written != filesize) continue;
                long percentage = written * 100L / filesize;
                double elapsedTime = (double)timer.elapsedTime() / 1.0E9;
                double writtenMB = 1.0 * (double)written / 1048576.0;
                LOG.info(String.format("[%02d%%] Buffered %.2f MB out of %d MB; elapsedTime=%.2fs; write to buffer bandwidth=%.2f MB/s", percentage, writtenMB, filesizeMB, elapsedTime, writtenMB / elapsedTime));
            }
            LOG.info("Closing stream {}", (Object)out);
            ContractTestUtils.NanoTimer closeTimer = new ContractTestUtils.NanoTimer();
            out.close();
            closeTimer.end("time to close() output stream", new Object[0]);
        }
        timer.end("time to write %d MB in blocks of %d", new Object[]{filesizeMB, 65536});
        this.logFSState();
        ContractTestUtils.bandwidth((ContractTestUtils.NanoTimer)timer, (long)filesize);
        ContractTestUtils.assertPathExists((FileSystem)fs, (String)"Huge file", (Path)this.hugefile);
        FileStatus status = fs.getFileStatus(this.hugefile);
        ContractTestUtils.assertIsFile((Path)this.hugefile, (FileStatus)status);
        ITestAzureHugeFiles.assertEquals((long)filesize, (long)status.getLen(), (String)("File size in " + status));
    }

    @Test
    public void test_040_PositionedReadHugeFile() throws Throwable {
        this.assumeHugeFileExists();
        this.describe("Positioned reads of file %s", this.hugefile);
        NativeAzureFileSystem fs = this.getFileSystem();
        FileStatus status = fs.getFileStatus(this.hugefile);
        long filesize = status.getLen();
        int ops = 0;
        int bufferSize = 8192;
        byte[] buffer = new byte[8192];
        long eof = filesize - 1L;
        ContractTestUtils.NanoTimer timer = new ContractTestUtils.NanoTimer();
        try (FSDataInputStream in = this.openDataFile();){
            ContractTestUtils.NanoTimer readAtByte0 = new ContractTestUtils.NanoTimer();
            in.readFully(0L, buffer);
            readAtByte0.end("time to read data at start of file", new Object[0]);
            ++ops;
            ContractTestUtils.NanoTimer readAtEOF = new ContractTestUtils.NanoTimer();
            in.readFully(eof - 8192L, buffer);
            readAtEOF.end("time to read data at end of file", new Object[0]);
            ++ops;
            ContractTestUtils.NanoTimer readAtByte0Again = new ContractTestUtils.NanoTimer();
            in.readFully(0L, buffer);
            readAtByte0Again.end("time to read data at start of file again", new Object[0]);
            ++ops;
            LOG.info("Final stream state: {}", (Object)in);
        }
        long mb = Math.max(filesize / 0x100000L, 1L);
        this.logFSState();
        timer.end("time to performed positioned reads of %d MB ", new Object[]{mb});
        LOG.info("Time per positioned read = {} nS", (Object)ContractTestUtils.toHuman((long)timer.nanosPerOperation((long)ops)));
    }

    protected FSDataInputStream openDataFile() throws IOException {
        ContractTestUtils.NanoTimer openTimer = new ContractTestUtils.NanoTimer();
        FSDataInputStream inputStream = this.getFileSystem().open(this.hugefile, 65536);
        openTimer.end("open data file", new Object[0]);
        return inputStream;
    }

    public static double bandwidthInBytes(ContractTestUtils.NanoTimer timer, long bytes) {
        return (double)bytes * 1.0E9 / (double)timer.duration();
    }

    @Test
    public void test_050_readHugeFile() throws Throwable {
        this.assumeHugeFileExists();
        this.describe("Reading %s", this.hugefile);
        NativeAzureFileSystem fs = this.getFileSystem();
        FileStatus status = fs.getFileStatus(this.hugefile);
        long filesize = status.getLen();
        long blocks = filesize / 65536L;
        byte[] data = new byte[65536];
        ContractTestUtils.NanoTimer timer = new ContractTestUtils.NanoTimer();
        try (FSDataInputStream in = this.openDataFile();){
            for (long block = 0L; block < blocks; ++block) {
                in.readFully(data);
            }
            LOG.info("Final stream state: {}", (Object)in);
        }
        long mb = Math.max(filesize / 0x100000L, 1L);
        timer.end("time to read file of %d MB ", new Object[]{mb});
        LOG.info("Time per MB to read = {} nS", (Object)ContractTestUtils.toHuman((long)timer.nanosPerOperation(mb)));
        ContractTestUtils.bandwidth((ContractTestUtils.NanoTimer)timer, (long)filesize);
        this.logFSState();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void test_060_openAndReadWholeFileBlocks() throws Throwable {
        FileStatus status = this.assumeHugeFileExists();
        int blockSize = 0x100000;
        this.describe("Open the test file and read it in blocks of size %d", blockSize);
        long len = status.getLen();
        FSDataInputStream in = this.openDataFile();
        ContractTestUtils.NanoTimer timer2 = null;
        long blockCount = 0L;
        long totalToRead = 0L;
        int resetCount = 0;
        try {
            byte[] block = new byte[blockSize];
            timer2 = new ContractTestUtils.NanoTimer();
            long count = 0L;
            blockCount = len / (long)blockSize;
            totalToRead = blockCount * (long)blockSize;
            long minimumBandwidth = 131072L;
            int maxResetCount = 4;
            resetCount = 0;
            for (long i = 0L; i < blockCount; ++i) {
                int offset = 0;
                int remaining = blockSize;
                long blockId = i + 1L;
                ContractTestUtils.NanoTimer blockTimer = new ContractTestUtils.NanoTimer();
                int reads = 0;
                while (remaining > 0) {
                    ContractTestUtils.NanoTimer readTimer = new ContractTestUtils.NanoTimer();
                    int bytesRead = in.read(block, offset, remaining);
                    ++reads;
                    if (bytesRead == 1) break;
                    remaining -= bytesRead;
                    offset += bytesRead;
                    count += (long)bytesRead;
                    readTimer.end();
                    if (bytesRead != 0) {
                        LOG.debug("Bytes in read #{}: {} , block bytes: {}, remaining in block: {} duration={} nS; ns/byte: {}, bandwidth={} MB/s", new Object[]{reads, bytesRead, blockSize - remaining, remaining, readTimer.duration(), readTimer.nanosPerOperation((long)bytesRead), readTimer.bandwidthDescription((long)bytesRead)});
                        continue;
                    }
                    LOG.warn("0 bytes returned by read() operation #{}", (Object)reads);
                }
                blockTimer.end("Reading block %d in %d reads", new Object[]{blockId, reads});
                String bw = blockTimer.bandwidthDescription((long)blockSize);
                LOG.info("Bandwidth of block {}: {} MB/s: ", (Object)blockId, (Object)bw);
                if (!(ITestAzureHugeFiles.bandwidthInBytes(blockTimer, blockSize) < (double)minimumBandwidth)) continue;
                LOG.warn("Bandwidth {} too low on block {}: resetting connection", (Object)bw, (Object)blockId);
                ITestAzureHugeFiles.assertTrue((resetCount <= maxResetCount ? 1 : 0) != 0, (String)("Bandwidth of " + bw + " too low after " + resetCount + " attempts"));
                ++resetCount;
            }
        }
        finally {
            IOUtils.closeStream((Closeable)in);
        }
        timer2.end("Time to read %d bytes in %d blocks", new Object[]{totalToRead, blockCount});
        LOG.info("Overall Bandwidth {} MB/s; reset connections {}", (Object)timer2.bandwidth(totalToRead), (Object)resetCount);
    }

    @Test
    public void test_100_renameHugeFile() throws Throwable {
        this.assumeHugeFileExists();
        this.describe("renaming %s to %s", this.hugefile, this.hugefileRenamed);
        NativeAzureFileSystem fs = this.getFileSystem();
        FileStatus status = fs.getFileStatus(this.hugefile);
        long filesize = status.getLen();
        fs.delete(this.hugefileRenamed, false);
        ContractTestUtils.NanoTimer timer = new ContractTestUtils.NanoTimer();
        fs.rename(this.hugefile, this.hugefileRenamed);
        long mb = Math.max(filesize / 0x100000L, 1L);
        timer.end("time to rename file of %d MB", new Object[]{mb});
        LOG.info("Time per MB to rename = {} nS", (Object)ContractTestUtils.toHuman((long)timer.nanosPerOperation(mb)));
        ContractTestUtils.bandwidth((ContractTestUtils.NanoTimer)timer, (long)filesize);
        this.logFSState();
        FileStatus destFileStatus = fs.getFileStatus(this.hugefileRenamed);
        ITestAzureHugeFiles.assertEquals((long)filesize, (long)destFileStatus.getLen());
        ContractTestUtils.NanoTimer timer2 = new ContractTestUtils.NanoTimer();
        fs.rename(this.hugefileRenamed, this.hugefile);
        timer2.end("Renaming back", new Object[0]);
        LOG.info("Time per MB to rename = {} nS", (Object)ContractTestUtils.toHuman((long)timer2.nanosPerOperation(mb)));
        ContractTestUtils.bandwidth((ContractTestUtils.NanoTimer)timer2, (long)filesize);
    }

    @Test
    public void test_999_deleteHugeFiles() throws IOException {
        this.testAccountForCleanup = this.testAccount;
        this.deleteHugeFile();
        ContractTestUtils.NanoTimer timer2 = new ContractTestUtils.NanoTimer();
        NativeAzureFileSystem fs = this.getFileSystem();
        fs.delete(this.hugefileRenamed, false);
        timer2.end("time to delete %s", new Object[]{this.hugefileRenamed});
        ContractTestUtils.rm((FileSystem)fs, (Path)this.testPath, (boolean)true, (boolean)false);
        ContractTestUtils.assertPathDoesNotExist((FileSystem)fs, (String)"deleted huge file", (Path)this.testPath);
    }
}

