/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.s3a.impl;

import java.util.Locale;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.classification.VisibleForTesting;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.s3a.RemoteFileChangedException;
import org.apache.hadoop.fs.s3a.S3ObjectAttributes;
import org.apache.hadoop.fs.store.LogExactlyOnce;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.services.s3.model.CopyObjectRequest;
import software.amazon.awssdk.services.s3.model.CopyObjectResponse;
import software.amazon.awssdk.services.s3.model.GetObjectRequest;
import software.amazon.awssdk.services.s3.model.GetObjectResponse;
import software.amazon.awssdk.services.s3.model.HeadObjectRequest;
import software.amazon.awssdk.services.s3.model.HeadObjectResponse;

@InterfaceAudience.Private
@InterfaceStability.Unstable
public abstract class ChangeDetectionPolicy {
    private static final Logger LOG = LoggerFactory.getLogger(ChangeDetectionPolicy.class);
    @VisibleForTesting
    public static final String CHANGE_DETECTED = "change detected on client";
    private final Mode mode;
    private final boolean requireVersion;
    private final LogExactlyOnce logNoVersionSupport = new LogExactlyOnce(LOG);

    protected ChangeDetectionPolicy(Mode mode, boolean requireVersion) {
        this.mode = mode;
        this.requireVersion = requireVersion;
    }

    public Mode getMode() {
        return this.mode;
    }

    public abstract Source getSource();

    public boolean isRequireVersion() {
        return this.requireVersion;
    }

    public LogExactlyOnce getLogNoVersionSupport() {
        return this.logNoVersionSupport;
    }

    public static ChangeDetectionPolicy getPolicy(Configuration configuration) {
        Mode mode = Mode.fromConfiguration(configuration);
        Source source = Source.fromConfiguration(configuration);
        boolean requireVersion = configuration.getBoolean("fs.s3a.change.detection.version.required", true);
        return ChangeDetectionPolicy.createPolicy(mode, source, requireVersion);
    }

    @VisibleForTesting
    public static ChangeDetectionPolicy createPolicy(Mode mode, Source source, boolean requireVersion) {
        switch (source) {
            case ETag: {
                return new ETagChangeDetectionPolicy(mode, requireVersion);
            }
            case VersionId: {
                return new VersionIdChangeDetectionPolicy(mode, requireVersion);
            }
        }
        return new NoChangeDetection();
    }

    public String toString() {
        return "Policy " + this.getSource() + "/" + this.getMode();
    }

    public abstract String getRevisionId(HeadObjectResponse var1, String var2);

    public abstract String getRevisionId(GetObjectResponse var1, String var2);

    public abstract String getRevisionId(S3ObjectAttributes var1);

    public abstract String getRevisionId(CopyObjectResponse var1);

    public abstract void applyRevisionConstraint(GetObjectRequest.Builder var1, String var2);

    public abstract void applyRevisionConstraint(CopyObjectRequest.Builder var1, String var2);

    public abstract void applyRevisionConstraint(HeadObjectRequest.Builder var1, String var2);

    public ImmutablePair<Boolean, RemoteFileChangedException> onChangeDetected(String revisionId, String newRevisionId, String uri, long position, String operation, long timesAlreadyDetected) {
        Object positionText = position >= 0L ? " at " + position : "";
        switch (this.mode) {
            case None: {
                return new ImmutablePair((Object)false, null);
            }
            case Warn: {
                if (timesAlreadyDetected == 0L) {
                    LOG.warn(String.format("%s change detected on %s %s%s. Expected %s got %s", new Object[]{this.getSource(), operation, uri, positionText, revisionId, newRevisionId}));
                    return new ImmutablePair((Object)true, null);
                }
                return new ImmutablePair((Object)false, null);
            }
        }
        return new ImmutablePair((Object)true, (Object)new RemoteFileChangedException(uri, operation, String.format("%s change detected on client during %s%s. Expected %s got %s", new Object[]{this.getSource(), operation, positionText, revisionId, newRevisionId})));
    }

    public static enum Mode {
        Client("client"),
        Server("server"),
        Warn("warn"),
        None("none");

        private final String mode;

        private Mode(String mode) {
            this.mode = mode;
        }

        private static Mode fromString(String trimmed) {
            for (Mode value : Mode.values()) {
                if (!value.mode.equals(trimmed)) continue;
                return value;
            }
            LOG.warn("Unrecognized fs.s3a.change.detection.mode value: \"{}\"", (Object)trimmed);
            return Mode.fromString("server");
        }

        static Mode fromConfiguration(Configuration configuration) {
            String trimmed = configuration.get("fs.s3a.change.detection.mode", "server").trim().toLowerCase(Locale.ENGLISH);
            return Mode.fromString(trimmed);
        }
    }

    public static enum Source {
        ETag("etag"),
        VersionId("versionid"),
        None("none");

        private final String source;

        private Source(String source) {
            this.source = source;
        }

        private static Source fromString(String trimmed) {
            for (Source value : Source.values()) {
                if (!value.source.equals(trimmed)) continue;
                return value;
            }
            LOG.warn("Unrecognized fs.s3a.change.detection.source value: \"{}\"", (Object)trimmed);
            return Source.fromString("etag");
        }

        static Source fromConfiguration(Configuration configuration) {
            String trimmed = configuration.get("fs.s3a.change.detection.source", "etag").trim().toLowerCase(Locale.ENGLISH);
            return Source.fromString(trimmed);
        }
    }

    static class ETagChangeDetectionPolicy
    extends ChangeDetectionPolicy {
        ETagChangeDetectionPolicy(Mode mode, boolean requireVersion) {
            super(mode, requireVersion);
        }

        @Override
        public String getRevisionId(GetObjectResponse objectMetadata, String uri) {
            return objectMetadata.eTag();
        }

        @Override
        public String getRevisionId(HeadObjectResponse objectMetadata, String uri) {
            return objectMetadata.eTag();
        }

        @Override
        public String getRevisionId(S3ObjectAttributes s3Attributes) {
            return s3Attributes.getETag();
        }

        @Override
        public String getRevisionId(CopyObjectResponse copyObjectResponse) {
            return copyObjectResponse.copyObjectResult().eTag();
        }

        @Override
        public void applyRevisionConstraint(GetObjectRequest.Builder builder, String revisionId) {
            if (revisionId != null) {
                LOG.debug("Restricting get request to etag {}", (Object)revisionId);
                builder.ifMatch(revisionId);
            } else {
                LOG.debug("No etag revision ID to use as a constraint");
            }
        }

        @Override
        public void applyRevisionConstraint(CopyObjectRequest.Builder requestBuilder, String revisionId) {
            if (revisionId != null) {
                LOG.debug("Restricting copy request to etag {}", (Object)revisionId);
                requestBuilder.copySourceIfMatch(revisionId);
            } else {
                LOG.debug("No etag revision ID to use as a constraint");
            }
        }

        @Override
        public void applyRevisionConstraint(HeadObjectRequest.Builder requestBuilder, String revisionId) {
            LOG.debug("Unable to restrict HEAD request to etag; will check later");
        }

        @Override
        public Source getSource() {
            return Source.ETag;
        }

        @Override
        public String toString() {
            return "ETagChangeDetectionPolicy mode=" + this.getMode();
        }
    }

    static class VersionIdChangeDetectionPolicy
    extends ChangeDetectionPolicy {
        VersionIdChangeDetectionPolicy(Mode mode, boolean requireVersion) {
            super(mode, requireVersion);
        }

        @Override
        public String getRevisionId(HeadObjectResponse objectMetadata, String uri) {
            return this.logIfNull(objectMetadata.versionId(), uri);
        }

        @Override
        public String getRevisionId(GetObjectResponse getObjectResponse, String uri) {
            return this.logIfNull(getObjectResponse.versionId(), uri);
        }

        private String logIfNull(String versionId, String uri) {
            if (versionId == null) {
                this.getLogNoVersionSupport().warn("fs.s3a.change.detection.mode set to " + Source.VersionId + " but no versionId available while reading {}. Ensure your bucket has object versioning enabled. You may see inconsistent reads.", new Object[]{uri});
            }
            return versionId;
        }

        @Override
        public String getRevisionId(S3ObjectAttributes s3Attributes) {
            return s3Attributes.getVersionId();
        }

        @Override
        public String getRevisionId(CopyObjectResponse copyObjectResponse) {
            return copyObjectResponse.versionId();
        }

        @Override
        public void applyRevisionConstraint(GetObjectRequest.Builder builder, String revisionId) {
            if (revisionId != null) {
                LOG.debug("Restricting get request to version {}", (Object)revisionId);
                builder.versionId(revisionId);
            } else {
                LOG.debug("No version ID to use as a constraint");
            }
        }

        @Override
        public void applyRevisionConstraint(CopyObjectRequest.Builder requestBuilder, String revisionId) {
            if (revisionId != null) {
                LOG.debug("Restricting copy request to version {}", (Object)revisionId);
                requestBuilder.sourceVersionId(revisionId);
            } else {
                LOG.debug("No version ID to use as a constraint");
            }
        }

        @Override
        public void applyRevisionConstraint(HeadObjectRequest.Builder requestBuilder, String revisionId) {
            if (revisionId != null) {
                LOG.debug("Restricting metadata request to version {}", (Object)revisionId);
                requestBuilder.versionId(revisionId);
            } else {
                LOG.debug("No version ID to use as a constraint");
            }
        }

        @Override
        public Source getSource() {
            return Source.VersionId;
        }

        @Override
        public String toString() {
            return "VersionIdChangeDetectionPolicy mode=" + this.getMode();
        }
    }

    static class NoChangeDetection
    extends ChangeDetectionPolicy {
        NoChangeDetection() {
            super(Mode.None, false);
        }

        @Override
        public Source getSource() {
            return Source.None;
        }

        @Override
        public String getRevisionId(GetObjectResponse objectMetadata, String uri) {
            return null;
        }

        @Override
        public String getRevisionId(HeadObjectResponse objectMetadata, String uri) {
            return null;
        }

        @Override
        public String getRevisionId(S3ObjectAttributes s3ObjectAttributes) {
            return null;
        }

        @Override
        public String getRevisionId(CopyObjectResponse copyObjectResponse) {
            return null;
        }

        @Override
        public void applyRevisionConstraint(GetObjectRequest.Builder builder, String revisionId) {
        }

        @Override
        public void applyRevisionConstraint(CopyObjectRequest.Builder requestBuilder, String revisionId) {
        }

        @Override
        public void applyRevisionConstraint(HeadObjectRequest.Builder requestBuilder, String revisionId) {
        }

        @Override
        public String toString() {
            return "NoChangeDetection";
        }
    }
}

