/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.s3a.impl;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.hadoop.fs.s3a.Invoker;
import org.apache.hadoop.fs.s3a.RenameFailedException;
import org.apache.hadoop.fs.s3a.S3AFileStatus;
import org.apache.hadoop.fs.s3a.S3ALocatedFileStatus;
import org.apache.hadoop.fs.s3a.S3AReadOpContext;
import org.apache.hadoop.fs.s3a.S3AUtils;
import org.apache.hadoop.fs.s3a.S3ObjectAttributes;
import org.apache.hadoop.fs.s3a.Tristate;
import org.apache.hadoop.fs.s3a.impl.CallableSupplier;
import org.apache.hadoop.fs.s3a.impl.DirMarkerTracker;
import org.apache.hadoop.fs.s3a.impl.ExecutingStoreOperation;
import org.apache.hadoop.fs.s3a.impl.OperationCallbacks;
import org.apache.hadoop.fs.s3a.impl.StoreContext;
import org.apache.hadoop.fs.store.audit.AuditSpan;
import org.apache.hadoop.fs.store.audit.AuditingFunctions;
import org.apache.hadoop.util.DurationInfo;
import org.apache.hadoop.util.OperationDuration;
import org.apache.hadoop.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.core.exception.SdkException;
import software.amazon.awssdk.services.s3.model.ObjectIdentifier;

public class RenameOperation
extends ExecutingStoreOperation<Long> {
    private static final Logger LOG = LoggerFactory.getLogger(RenameOperation.class);
    private final Path sourcePath;
    private final String sourceKey;
    private final S3AFileStatus sourceStatus;
    private final Path destPath;
    private final String destKey;
    private final S3AFileStatus destStatus;
    private final OperationCallbacks callbacks;
    private final AtomicLong bytesCopied = new AtomicLong();
    private final int pageSize;
    private final List<CompletableFuture<Path>> activeCopies = new ArrayList<CompletableFuture<Path>>(10);
    private final List<ObjectIdentifier> keysToDelete = new ArrayList<ObjectIdentifier>();
    private final boolean dirOperationsPurgeUploads;
    private Optional<Long> uploadsAborted = Optional.empty();

    public RenameOperation(StoreContext storeContext, Path sourcePath, String sourceKey, S3AFileStatus sourceStatus, Path destPath, String destKey, S3AFileStatus destStatus, OperationCallbacks callbacks, int pageSize, boolean dirOperationsPurgeUploads) {
        super(storeContext);
        this.sourcePath = sourcePath;
        this.sourceKey = sourceKey;
        this.sourceStatus = sourceStatus;
        this.destPath = destPath;
        this.destKey = destKey;
        this.destStatus = destStatus;
        this.callbacks = callbacks;
        Preconditions.checkArgument((pageSize > 0 && pageSize <= 1000 ? 1 : 0) != 0, (String)"page size out of range: %s", (Object[])new Object[]{pageSize});
        this.pageSize = pageSize;
        this.dirOperationsPurgeUploads = dirOperationsPurgeUploads;
    }

    public Optional<Long> getUploadsAborted() {
        return this.uploadsAborted;
    }

    private void completeActiveCopies(String reason) throws IOException {
        LOG.debug("Waiting for {} active copies to complete: {}", (Object)this.activeCopies.size(), (Object)reason);
        CallableSupplier.waitForCompletion(this.activeCopies);
        this.activeCopies.clear();
    }

    private void queueToDelete(Path path, String key) {
        LOG.debug("Queueing to delete {}", (Object)path);
        this.keysToDelete.add((ObjectIdentifier)ObjectIdentifier.builder().key(key).build());
    }

    private void queueToDelete(List<DirMarkerTracker.Marker> markersToDelete) {
        markersToDelete.forEach(m -> this.queueToDelete(null, m.getKey()));
    }

    private void queueToDelete(DirMarkerTracker.Marker marker) {
        this.queueToDelete(marker.getPath(), marker.getKey());
    }

    private void completeActiveCopiesAndDeleteSources(String reason) throws IOException {
        this.completeActiveCopies(reason);
        this.removeSourceObjects(this.keysToDelete);
        this.keysToDelete.clear();
    }

    @Override
    public Long execute() throws IOException {
        this.executeOnlyOnce();
        Path destCreated = this.destPath;
        try {
            if (this.sourceStatus.isFile()) {
                destCreated = this.renameFileToDest();
            } else {
                this.recursiveDirectoryRename();
            }
        }
        catch (IOException | SdkException ex) {
            try {
                this.completeActiveCopies("failure handling");
            }
            catch (IOException e) {
                LOG.warn("While completing all active copies", (Throwable)e);
            }
            throw this.convertToIOException((Exception)ex);
        }
        this.callbacks.finishRename(this.sourcePath, destCreated);
        return this.bytesCopied.get();
    }

    protected Path renameFileToDest() throws IOException {
        StoreContext storeContext = this.getStoreContext();
        Path copyDestinationPath = this.destPath;
        String copyDestinationKey = this.destKey;
        S3ObjectAttributes sourceAttributes = this.callbacks.createObjectAttributes(this.sourceStatus);
        S3AReadOpContext readContext = this.callbacks.createReadContext(this.sourceStatus);
        if (this.destStatus != null && this.destStatus.isDirectory()) {
            String newDestKey = this.maybeAddTrailingSlash(this.destKey);
            String filename = this.sourceKey.substring(storeContext.pathToKey(this.sourcePath.getParent()).length() + 1);
            copyDestinationKey = newDestKey = newDestKey + filename;
            copyDestinationPath = storeContext.keyToPath(newDestKey);
        }
        LOG.debug("rename: renaming file {} to {}", (Object)this.sourcePath, (Object)copyDestinationPath);
        this.copySource(this.sourceKey, sourceAttributes, readContext, copyDestinationPath, copyDestinationKey);
        this.bytesCopied.addAndGet(this.sourceStatus.getLen());
        this.callbacks.deleteObjectAtPath(this.sourcePath, this.sourceKey, true);
        return copyDestinationPath;
    }

    protected void recursiveDirectoryRename() throws IOException {
        CompletableFuture<Long> abortUploads;
        StoreContext storeContext = this.getStoreContext();
        LOG.debug("rename: renaming directory {} to {}", (Object)this.sourcePath, (Object)this.destPath);
        String dstKey = this.maybeAddTrailingSlash(this.destKey);
        String srcKey = this.maybeAddTrailingSlash(this.sourceKey);
        if (dstKey.startsWith(srcKey)) {
            throw new RenameFailedException(srcKey, dstKey, "cannot rename a directory to a subdirectory of itself ");
        }
        if (this.dirOperationsPurgeUploads) {
            String key = srcKey;
            LOG.debug("All uploads under {} will be deleted", (Object)key);
            abortUploads = CallableSupplier.submit((Executor)this.getStoreContext().getExecutor(), () -> this.callbacks.abortMultipartUploadsUnderPrefix(key));
        } else {
            abortUploads = null;
        }
        if (this.destStatus != null && this.destStatus.isEmptyDirectory() == Tristate.TRUE) {
            LOG.debug("Deleting fake directory marker at destination {}", (Object)this.destStatus.getPath());
            this.callbacks.deleteObjectAtPath(this.destStatus.getPath(), dstKey, false);
        }
        Path parentPath = storeContext.keyToPath(srcKey);
        DirMarkerTracker dirMarkerTracker = new DirMarkerTracker(parentPath, false);
        RemoteIterator<S3ALocatedFileStatus> iterator = this.callbacks.listFilesAndDirectoryMarkers(parentPath, this.sourceStatus, true);
        while (iterator.hasNext()) {
            List<DirMarkerTracker.Marker> markersToDelete;
            S3ALocatedFileStatus child = (S3ALocatedFileStatus)((Object)iterator.next());
            LOG.debug("To rename {}", (Object)child);
            String k = storeContext.pathToKey(child.getPath());
            String key = child.isDirectory() && !k.endsWith("/") ? k + "/" : k;
            Path childSourcePath = storeContext.keyToPath(key);
            boolean isMarker = key.endsWith("/");
            if (isMarker) {
                markersToDelete = dirMarkerTracker.markerFound(childSourcePath, key, child);
            } else {
                markersToDelete = dirMarkerTracker.fileFound(childSourcePath, key, child);
                String newDestKey = dstKey + key.substring(srcKey.length());
                Path childDestPath = storeContext.keyToPath(newDestKey);
                this.queueToDelete(childSourcePath, key);
                CompletableFuture<Path> copy = this.initiateCopy(child, key, newDestKey, childDestPath);
                this.activeCopies.add(copy);
                this.bytesCopied.addAndGet(this.sourceStatus.getLen());
            }
            this.queueToDelete(markersToDelete);
            this.endOfLoopActions();
        }
        this.copyEmptyDirectoryMarkers(srcKey, dstKey, dirMarkerTracker);
        this.completeActiveCopiesAndDeleteSources("final copy and delete");
        this.uploadsAborted = CallableSupplier.waitForCompletionIgnoringExceptions(abortUploads);
    }

    private void endOfLoopActions() throws IOException {
        if (this.keysToDelete.size() == this.pageSize) {
            this.completeActiveCopiesAndDeleteSources("paged delete");
        } else if (this.activeCopies.size() == 10) {
            LOG.debug("Waiting for active copies to complete");
            this.completeActiveCopies("batch threshold reached");
        }
    }

    private OperationDuration copyEmptyDirectoryMarkers(String srcKey, String dstKey, DirMarkerTracker dirMarkerTracker) throws IOException {
        LOG.debug("Copying markers from {}", (Object)dirMarkerTracker);
        StoreContext storeContext = this.getStoreContext();
        Map<Path, DirMarkerTracker.Marker> leafMarkers = dirMarkerTracker.getLeafMarkers();
        Map<Path, DirMarkerTracker.Marker> surplus = dirMarkerTracker.getSurplusMarkers();
        DurationInfo duration = new DurationInfo(LOG, false, "copying %d leaf markers with %d surplus not copied", new Object[]{leafMarkers.size(), surplus.size()});
        for (DirMarkerTracker.Marker entry : leafMarkers.values()) {
            String key = entry.getKey();
            String newDestKey = dstKey + key.substring(srcKey.length());
            Path childDestPath = storeContext.keyToPath(newDestKey);
            LOG.debug("copying dir marker from {} to {}", (Object)key, (Object)newDestKey);
            this.activeCopies.add(this.initiateCopy(entry.getStatus(), key, newDestKey, childDestPath));
            this.queueToDelete(entry);
            this.endOfLoopActions();
        }
        duration.close();
        return duration;
    }

    protected CompletableFuture<Path> initiateCopy(S3ALocatedFileStatus source, String key, String newDestKey, Path childDestPath) {
        S3ObjectAttributes sourceAttributes = this.callbacks.createObjectAttributes(source.getPath(), source.getEtag(), source.getVersionId(), source.getLen());
        return CallableSupplier.submit((Executor)this.getStoreContext().getExecutor(), AuditingFunctions.callableWithinAuditSpan((AuditSpan)this.getAuditSpan(), () -> this.copySource(key, sourceAttributes, this.callbacks.createReadContext((FileStatus)source), childDestPath, newDestKey)));
    }

    private Path copySource(String srcKey, S3ObjectAttributes srcAttributes, S3AReadOpContext readContext, Path destination, String destinationKey) throws IOException {
        long len = srcAttributes.getLen();
        try (DurationInfo ignored = new DurationInfo(LOG, false, "Copy file from %s to %s (length=%d)", new Object[]{srcKey, destinationKey, len});){
            this.callbacks.copyFile(srcKey, destinationKey, srcAttributes, readContext);
        }
        return destination;
    }

    private void removeSourceObjects(List<ObjectIdentifier> keys) throws IOException {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Initiating delete operation for {} objects", (Object)keys.size());
            for (ObjectIdentifier objectIdentifier : keys) {
                LOG.debug(" {} {}", (Object)objectIdentifier.key(), (Object)(objectIdentifier.versionId() != null ? objectIdentifier.versionId() : ""));
            }
        }
        Invoker.once("rename " + this.sourcePath + " to " + this.destPath, this.sourcePath.toString(), () -> this.callbacks.removeKeys(keys, false));
    }

    private String maybeAddTrailingSlash(String key) {
        if (!key.isEmpty() && !key.endsWith("/")) {
            return key + '/';
        }
        return key;
    }

    protected IOException convertToIOException(Exception ex) {
        if (ex instanceof IOException) {
            return (IOException)ex;
        }
        if (ex instanceof SdkException) {
            return S3AUtils.translateException("rename " + this.sourcePath + " to " + this.destPath, this.sourcePath.toString(), (SdkException)((Object)ex));
        }
        return new IOException(ex);
    }
}

