/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.s3a.auth.delegation;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.IOException;
import java.net.URI;
import java.util.EnumSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.s3a.AWSCredentialProviderList;
import org.apache.hadoop.fs.s3a.S3AFileSystem;
import org.apache.hadoop.fs.s3a.Statistic;
import org.apache.hadoop.fs.s3a.auth.RoleModel;
import org.apache.hadoop.fs.s3a.auth.delegation.AWSPolicyProvider;
import org.apache.hadoop.fs.s3a.auth.delegation.AbstractDTService;
import org.apache.hadoop.fs.s3a.auth.delegation.AbstractDelegationTokenBinding;
import org.apache.hadoop.fs.s3a.auth.delegation.AbstractS3ATokenIdentifier;
import org.apache.hadoop.fs.s3a.auth.delegation.DelegationBindingInfo;
import org.apache.hadoop.fs.s3a.auth.delegation.DelegationTokenIOException;
import org.apache.hadoop.fs.s3a.auth.delegation.EncryptionSecrets;
import org.apache.hadoop.fs.s3a.auth.delegation.SessionTokenBinding;
import org.apache.hadoop.fs.s3a.statistics.DelegationTokenStatistics;
import org.apache.hadoop.fs.statistics.DurationTrackerFactory;
import org.apache.hadoop.fs.statistics.impl.IOStatisticsBinding;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.security.Credentials;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.security.token.DelegationTokenIssuer;
import org.apache.hadoop.security.token.Token;
import org.apache.hadoop.service.Service;
import org.apache.hadoop.service.ServiceOperations;
import org.apache.hadoop.util.DurationInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class S3ADelegationTokens
extends AbstractDTService {
    private static final Logger LOG = LoggerFactory.getLogger(S3ADelegationTokens.class);
    @VisibleForTesting
    static final String E_ALREADY_DEPLOYED = "S3A Delegation tokens has already been bound/deployed";
    public static final String E_DELEGATION_TOKENS_DISABLED = "Delegation tokens are not enabled";
    private final UserGroupInformation user;
    private final AtomicInteger creationCount = new AtomicInteger(0);
    private Text service;
    private Optional<Token<AbstractS3ATokenIdentifier>> boundDT = Optional.empty();
    private Optional<AbstractS3ATokenIdentifier> decodedIdentifier = Optional.empty();
    private AbstractDelegationTokenBinding tokenBinding;
    private Optional<DelegationBindingInfo> bindingInfo = Optional.empty();
    protected static final EnumSet<AWSPolicyProvider.AccessLevel> ACCESS_POLICY = EnumSet.of(AWSPolicyProvider.AccessLevel.READ, AWSPolicyProvider.AccessLevel.WRITE);
    private DelegationTokenStatistics stats;
    private String tokenBindingName = "";

    public S3ADelegationTokens() throws IOException {
        super("S3ADelegationTokens");
        this.user = UserGroupInformation.getCurrentUser();
    }

    @Override
    public void bindToFileSystem(URI uri, S3AFileSystem fs) throws IOException {
        super.bindToFileSystem(uri, fs);
        this.service = S3ADelegationTokens.getTokenService(this.getCanonicalUri());
        this.stats = fs.getInstrumentation().newDelegationTokenStatistics();
    }

    @Override
    protected void serviceInit(Configuration conf) throws Exception {
        super.serviceInit(conf);
        Preconditions.checkState((boolean)S3ADelegationTokens.hasDelegationTokenBinding(conf), (Object)E_DELEGATION_TOKENS_DISABLED);
        Class binding = conf.getClass("fs.s3a.delegation.token.binding", SessionTokenBinding.class, AbstractDelegationTokenBinding.class);
        this.tokenBinding = (AbstractDelegationTokenBinding)((Object)binding.newInstance());
        this.tokenBinding.bindToFileSystem(this.getCanonicalUri(), this.getFileSystem());
        this.tokenBinding.init(conf);
        this.tokenBindingName = this.tokenBinding.getKind().toString();
        LOG.debug("Filesystem {} is using delegation tokens of kind {}", (Object)this.getCanonicalUri(), (Object)this.tokenBindingName);
    }

    protected void serviceStart() throws Exception {
        super.serviceStart();
        this.tokenBinding.start();
        this.bindToAnyDelegationToken();
        LOG.debug("S3A Delegation support token {} with {}", (Object)this.identifierToString(), (Object)this.tokenBinding.getDescription());
    }

    private String identifierToString() {
        return this.decodedIdentifier.map(Objects::toString).orElse("(none)");
    }

    protected void serviceStop() throws Exception {
        LOG.debug("Stopping delegation tokens");
        try {
            super.serviceStop();
        }
        finally {
            ServiceOperations.stopQuietly((Logger)LOG, (Service)this.tokenBinding);
        }
    }

    private void deployUnbonded() throws IOException {
        this.requireServiceStarted();
        Preconditions.checkState((!this.isBoundToDT() ? 1 : 0) != 0, (Object)"Already Bound to a delegation token");
        LOG.debug("No delegation tokens present: using direct authentication");
        this.bindingInfo = Optional.of(this.tokenBinding.deploy(null));
    }

    private void bindToAnyDelegationToken() throws IOException {
        Preconditions.checkState((!this.bindingInfo.isPresent() ? 1 : 0) != 0, (Object)E_ALREADY_DEPLOYED);
        Token<AbstractS3ATokenIdentifier> token = this.selectTokenFromFSOwner();
        if (token != null) {
            this.bindToDelegationToken(token);
        } else {
            this.deployUnbonded();
        }
        if (this.getCredentialProviders().size() == 0) {
            throw new DelegationTokenIOException("No AWS credential providers created by Delegation Token Binding " + this.tokenBinding.getName());
        }
    }

    @VisibleForTesting
    void resetTokenBindingToDT(Token<AbstractS3ATokenIdentifier> token) throws IOException {
        this.bindingInfo = Optional.empty();
        this.bindToDelegationToken(token);
    }

    @VisibleForTesting
    public void bindToDelegationToken(Token<AbstractS3ATokenIdentifier> token) throws IOException {
        Preconditions.checkState((!this.bindingInfo.isPresent() ? 1 : 0) != 0, (Object)E_ALREADY_DEPLOYED);
        this.boundDT = Optional.of(token);
        AbstractS3ATokenIdentifier dti = this.extractIdentifier(token);
        LOG.info("Using delegation token {}", (Object)dti);
        this.decodedIdentifier = Optional.of(dti);
        try (DurationInfo ignored = new DurationInfo(LOG, true, "Creating Delegation Token", new Object[0]);){
            this.bindingInfo = Optional.of(this.tokenBinding.deploy(dti));
        }
    }

    public boolean isBoundToDT() {
        return this.boundDT.isPresent();
    }

    public Optional<Token<AbstractS3ATokenIdentifier>> getBoundDT() {
        return this.boundDT;
    }

    public TokenIssuingPolicy getTokenIssuingPolicy() {
        return this.isBoundToDT() ? TokenIssuingPolicy.ReturnExistingToken : this.tokenBinding.getTokenIssuingPolicy();
    }

    public Token<AbstractS3ATokenIdentifier> getBoundOrNewDT(EncryptionSecrets encryptionSecrets, Text renewer) throws IOException {
        LOG.debug("Delegation token requested");
        if (this.isBoundToDT()) {
            LOG.debug("Returning current token");
            return this.getBoundDT().get();
        }
        return this.createDelegationToken(encryptionSecrets, renewer);
    }

    public int getCreationCount() {
        return this.creationCount.get();
    }

    @VisibleForTesting
    public Token<AbstractS3ATokenIdentifier> createDelegationToken(EncryptionSecrets encryptionSecrets, Text renewer) throws IOException {
        this.requireServiceStarted();
        Preconditions.checkArgument((encryptionSecrets != null ? 1 : 0) != 0, (Object)"Null encryption secrets");
        List<RoleModel.Statement> statements = this.getFileSystem().listAWSPolicyRules(ACCESS_POLICY);
        Optional rolePolicy = statements.isEmpty() ? Optional.empty() : Optional.of(new RoleModel.Policy(statements));
        try (DurationInfo ignored = new DurationInfo(LOG, true, "Creating New Delegation Token", new Object[]{this.tokenBinding.getKind()});){
            Token token = (Token)IOStatisticsBinding.trackDuration((DurationTrackerFactory)this.stats, (String)Statistic.DELEGATION_TOKEN_ISSUED.getSymbol(), () -> this.tokenBinding.createDelegationToken(rolePolicy, encryptionSecrets, renewer));
            if (token != null) {
                token.setService(this.service);
                this.noteTokenCreated((Token<AbstractS3ATokenIdentifier>)token);
            }
            Token token2 = token;
            return token2;
        }
    }

    private void noteTokenCreated(Token<AbstractS3ATokenIdentifier> token) {
        LOG.info("Created S3A Delegation Token: {}", token);
        this.creationCount.incrementAndGet();
        this.stats.tokenIssued();
    }

    public DelegationTokenIssuer[] getAdditionalTokenIssuers() throws IOException {
        return null;
    }

    public AWSCredentialProviderList getCredentialProviders() throws IOException {
        return this.bindingInfo.map(DelegationBindingInfo::getCredentialProviders).orElseThrow(() -> new DelegationTokenIOException("Not yet bonded"));
    }

    public Optional<EncryptionSecrets> getEncryptionSecrets() {
        return this.decodedIdentifier.map(AbstractS3ATokenIdentifier::getEncryptionSecrets);
    }

    public Optional<AbstractS3ATokenIdentifier> getDecodedIdentifier() {
        return this.decodedIdentifier;
    }

    public Text getService() {
        return this.service;
    }

    public String getCanonicalServiceName() {
        return this.getCanonicalUri().toString();
    }

    @VisibleForTesting
    public Token<AbstractS3ATokenIdentifier> selectTokenFromFSOwner() throws IOException {
        return S3ADelegationTokens.lookupToken(this.user.getCredentials(), this.service, this.tokenBinding.getKind());
    }

    private static Text getTokenService(URI fsURI) {
        return S3ADelegationTokens.getTokenService(fsURI.toString());
    }

    public String toString() {
        StringBuilder sb = new StringBuilder("S3ADelegationTokens{");
        sb.append("canonicalServiceURI=").append(this.getCanonicalUri());
        sb.append("; owner=").append(this.user.getShortUserName());
        sb.append("; isBoundToDT=").append(this.isBoundToDT());
        sb.append("; token creation count=").append(this.getCreationCount());
        sb.append("; tokenManager=").append((Object)this.tokenBinding);
        sb.append("; token=").append(this.identifierToString());
        sb.append('}');
        return sb.toString();
    }

    public Text getTokenKind() {
        return this.tokenBinding.getKind();
    }

    @VisibleForTesting
    static Text getTokenService(String fsURI) {
        return new Text(fsURI);
    }

    public AbstractS3ATokenIdentifier extractIdentifier(Token<? extends AbstractS3ATokenIdentifier> token) throws IOException {
        AbstractS3ATokenIdentifier identifier;
        Preconditions.checkArgument((token != null ? 1 : 0) != 0, (Object)"null token");
        try {
            identifier = (AbstractS3ATokenIdentifier)token.decodeIdentifier();
        }
        catch (RuntimeException e) {
            Throwable cause = e.getCause();
            if (cause != null) {
                throw new DelegationTokenIOException("Decoding S3A token " + cause, cause);
            }
            throw e;
        }
        if (identifier == null) {
            throw new DelegationTokenIOException("Failed to unmarshall token for " + this.getCanonicalUri());
        }
        identifier.validate();
        return identifier;
    }

    public String getUserAgentField() {
        return this.tokenBinding.getUserAgentField();
    }

    @VisibleForTesting
    public static Token<AbstractS3ATokenIdentifier> lookupToken(Credentials credentials, Text service, Text kind) throws DelegationTokenIOException {
        LOG.debug("Looking for token for service {} in credentials", (Object)service);
        Token token = credentials.getToken(service);
        if (token != null) {
            Text tokenKind = token.getKind();
            LOG.debug("Found token of kind {}", (Object)tokenKind);
            if (kind.equals((Object)tokenKind)) {
                return token;
            }
            throw new DelegationTokenIOException("Token mismatch: expected token for " + service + " of type " + kind + " but got a token of type " + tokenKind);
        }
        LOG.debug("No token for {} found", (Object)service);
        return null;
    }

    public static Token<AbstractS3ATokenIdentifier> lookupToken(Credentials credentials, Text service) {
        return credentials.getToken(service);
    }

    public static Token<AbstractS3ATokenIdentifier> lookupS3ADelegationToken(Credentials credentials, URI uri) {
        return S3ADelegationTokens.lookupToken(credentials, S3ADelegationTokens.getTokenService(uri.toString()));
    }

    public static boolean hasDelegationTokenBinding(Configuration conf) {
        return StringUtils.isNotEmpty((CharSequence)conf.getTrimmed("fs.s3a.delegation.token.binding", ""));
    }

    public static enum TokenIssuingPolicy {
        ReturnExistingToken,
        NoTokensAvailable,
        RequestNewToken;

    }
}

