/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.util;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.nio.file.AccessDeniedException;
import java.nio.file.CopyOption;
import java.nio.file.FileSystemException;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Random;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;
import org.apache.flink.core.fs.FSDataOutputStream;
import org.apache.flink.core.fs.FileSystem;
import org.apache.flink.core.fs.Path;
import org.apache.flink.core.testutils.CheckedThread;
import org.apache.flink.testutils.junit.utils.TempDirUtils;
import org.apache.flink.util.AbstractID;
import org.apache.flink.util.FileUtils;
import org.apache.flink.util.OperatingSystem;
import org.apache.flink.util.StringUtils;
import org.assertj.core.api.AbstractBooleanAssert;
import org.assertj.core.api.AbstractThrowableAssert;
import org.assertj.core.api.Assertions;
import org.assertj.core.api.Assumptions;
import org.junit.jupiter.api.Tag;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.io.TempDir;

public class FileUtilsTest {
    @TempDir
    private java.nio.file.Path temporaryFolder;

    @Test
    void testReadAllBytes() throws Exception {
        File tempFile = TempDirUtils.newFolder((java.nio.file.Path)Paths.get(this.getClass().getResource("/").getPath(), new String[0]));
        int fileSize = 1024;
        String testFilePath = tempFile.toPath().resolve(this.getClass().getSimpleName() + "_1024.txt").toString();
        String expectedMD5 = FileUtilsTest.generateTestFile(testFilePath, 1024);
        byte[] data = FileUtils.readAllBytes((java.nio.file.Path)new File(testFilePath).toPath());
        Assertions.assertThat((String)FileUtilsTest.md5Hex(data)).isEqualTo(expectedMD5);
        expectedMD5 = FileUtilsTest.generateTestFile(testFilePath, 4096);
        data = FileUtils.readAllBytes((java.nio.file.Path)new File(testFilePath).toPath());
        Assertions.assertThat((String)FileUtilsTest.md5Hex(data)).isEqualTo(expectedMD5);
        expectedMD5 = FileUtilsTest.generateTestFile(testFilePath, 5120);
        data = FileUtils.readAllBytes((java.nio.file.Path)new File(testFilePath).toPath());
        Assertions.assertThat((String)FileUtilsTest.md5Hex(data)).isEqualTo(expectedMD5);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void testDeleteQuietly() throws Exception {
        FileUtils.deleteDirectoryQuietly(null);
        File doesNotExist = TempDirUtils.newFolder((java.nio.file.Path)this.temporaryFolder, (String[])new String[]{"abc"});
        FileUtils.deleteDirectoryQuietly((File)doesNotExist);
        File cannotDeleteParent = TempDirUtils.newFolder((java.nio.file.Path)this.temporaryFolder);
        File cannotDeleteChild = new File(cannotDeleteParent, "child");
        try {
            Assertions.assertThat((boolean)cannotDeleteChild.createNewFile()).isTrue();
            Assertions.assertThat((boolean)cannotDeleteParent.setWritable(false)).isTrue();
            Assertions.assertThat((boolean)cannotDeleteChild.setWritable(false)).isTrue();
            FileUtils.deleteDirectoryQuietly((File)cannotDeleteParent);
        }
        finally {
            cannotDeleteParent.setWritable(true);
            cannotDeleteChild.setWritable(true);
        }
    }

    @Test
    void testDeleteNonExistentDirectory() throws Exception {
        File doesNotExist = TempDirUtils.newFolder((java.nio.file.Path)this.temporaryFolder, (String[])new String[]{"abc"});
        FileUtils.deleteDirectory((File)doesNotExist);
    }

    @Tag(value="org.apache.flink.testutils.junit.FailsInGHAContainerWithRootUser")
    @Test
    void testDeleteProtectedDirectory() throws Exception {
        File cannotDeleteParent = TempDirUtils.newFolder((java.nio.file.Path)this.temporaryFolder);
        File cannotDeleteChild = new File(cannotDeleteParent, "child");
        try {
            Assumptions.assumeThat((boolean)cannotDeleteChild.createNewFile()).isTrue();
            Assumptions.assumeThat((boolean)cannotDeleteParent.setWritable(false)).isTrue();
            Assumptions.assumeThat((boolean)cannotDeleteChild.setWritable(false)).isTrue();
            Assertions.assertThatThrownBy(() -> FileUtils.deleteDirectory((File)cannotDeleteParent)).isInstanceOf(AccessDeniedException.class);
        }
        finally {
            cannotDeleteParent.setWritable(true);
            cannotDeleteChild.setWritable(true);
        }
    }

    @Test
    void testDeleteDirectoryWhichIsAFile() throws Exception {
        File file = TempDirUtils.newFile((java.nio.file.Path)this.temporaryFolder);
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> FileUtils.deleteDirectory((File)file)).withFailMessage("this should fail with an exception", new Object[0])).isInstanceOf(IOException.class);
    }

    @Test
    void testDeleteSymbolicLinkDirectory() throws Exception {
        File linkedDirectory = TempDirUtils.newFolder((java.nio.file.Path)this.temporaryFolder);
        File fileInLinkedDirectory = new File(linkedDirectory, "child");
        Assertions.assertThat((boolean)fileInLinkedDirectory.createNewFile()).isTrue();
        File symbolicLink = new File(this.temporaryFolder.toString(), "symLink");
        try {
            Files.createSymbolicLink(symbolicLink.toPath(), linkedDirectory.toPath(), new FileAttribute[0]);
        }
        catch (FileSystemException e) {
            ((AbstractBooleanAssert)Assumptions.assumeThat((boolean)OperatingSystem.isWindows()).withFailMessage("This test does not work properly under Windows", new Object[0])).isFalse();
            throw e;
        }
        FileUtils.deleteDirectory((File)symbolicLink);
        Assertions.assertThat((File)fileInLinkedDirectory).exists();
    }

    @Test
    void testDeleteDirectoryConcurrently() throws Exception {
        File parent = TempDirUtils.newFolder((java.nio.file.Path)this.temporaryFolder);
        FileUtilsTest.generateRandomDirs(parent, 20, 5, 3);
        Deleter t1 = new Deleter(parent);
        Deleter t2 = new Deleter(parent);
        Deleter t3 = new Deleter(parent);
        t1.start();
        t2.start();
        t3.start();
        t1.sync();
        t2.sync();
        t3.sync();
        Assertions.assertThat((File)parent).doesNotExist();
    }

    @Test
    void testCompressionOnAbsolutePath() throws IOException {
        java.nio.file.Path testDir = TempDirUtils.newFolder((java.nio.file.Path)this.temporaryFolder, (String[])new String[]{"compressDir"}).toPath();
        this.verifyDirectoryCompression(testDir, testDir);
    }

    @Test
    void testCompressionOnRelativePath() throws IOException {
        java.nio.file.Path testDir = TempDirUtils.newFolder((java.nio.file.Path)this.temporaryFolder, (String[])new String[]{"compressDir"}).toPath();
        java.nio.file.Path relativeCompressDir = Paths.get(new File("").getAbsolutePath(), new String[0]).relativize(testDir);
        this.verifyDirectoryCompression(testDir, relativeCompressDir);
    }

    @Test
    void testListFilesInPathWithoutAnyFileReturnEmptyList() throws IOException {
        java.nio.file.Path testDir = TempDirUtils.newFolder((java.nio.file.Path)this.temporaryFolder, (String[])new String[]{"_test_0"}).toPath();
        Assertions.assertThat((Collection)FileUtils.listFilesInDirectory((java.nio.file.Path)testDir, FileUtils::isJarFile)).isEmpty();
    }

    @Test
    void testListFilesInPath() throws IOException {
        java.nio.file.Path testDir = TempDirUtils.newFolder((java.nio.file.Path)this.temporaryFolder, (String[])new String[]{"_test_1"}).toPath();
        Collection<java.nio.file.Path> testFiles = FileUtilsTest.prepareTestFiles(testDir);
        Collection filesInDirectory = FileUtils.listFilesInDirectory((java.nio.file.Path)testDir, FileUtils::isJarFile);
        Assertions.assertThat((Collection)filesInDirectory).containsExactlyInAnyOrderElementsOf(testFiles);
    }

    @Test
    void testRelativizeOfAbsolutePath() throws IOException {
        java.nio.file.Path absolutePath = TempDirUtils.newFolder((java.nio.file.Path)this.temporaryFolder).toPath().toAbsolutePath();
        java.nio.file.Path rootPath = this.temporaryFolder.getRoot();
        java.nio.file.Path relativePath = FileUtils.relativizePath((java.nio.file.Path)rootPath, (java.nio.file.Path)absolutePath);
        Assertions.assertThat((java.nio.file.Path)relativePath).isRelative();
        Assertions.assertThat((java.nio.file.Path)absolutePath).isEqualTo((Object)rootPath.resolve(relativePath));
    }

    @Test
    void testRelativizeOfRelativePath() {
        java.nio.file.Path path = Paths.get("foobar", new String[0]);
        Assertions.assertThat((java.nio.file.Path)path).isRelative();
        java.nio.file.Path relativePath = FileUtils.relativizePath((java.nio.file.Path)this.temporaryFolder.getRoot(), (java.nio.file.Path)path);
        Assertions.assertThat((java.nio.file.Path)path).isEqualTo((Object)relativePath);
    }

    @Test
    void testAbsolutePathToURL() throws MalformedURLException {
        java.nio.file.Path absolutePath = this.temporaryFolder.getRoot().toAbsolutePath();
        URL absoluteURL = FileUtils.toURL((java.nio.file.Path)absolutePath);
        java.nio.file.Path transformedURL = Paths.get(absoluteURL.getPath(), new String[0]);
        Assertions.assertThat((java.nio.file.Path)transformedURL).isEqualTo((Object)absolutePath);
    }

    @Test
    void testRelativePathToURL() throws MalformedURLException {
        java.nio.file.Path relativePath = Paths.get("foobar", new String[0]);
        Assertions.assertThat((java.nio.file.Path)relativePath).isRelative();
        URL relativeURL = FileUtils.toURL((java.nio.file.Path)relativePath);
        java.nio.file.Path transformedPath = Paths.get(relativeURL.getPath(), new String[0]);
        Assertions.assertThat((java.nio.file.Path)transformedPath).isEqualTo((Object)relativePath);
    }

    @Test
    void testListDirFailsIfDirectoryDoesNotExist() {
        String fileName = "_does_not_exists_file";
        Assertions.assertThatThrownBy(() -> FileUtils.listFilesInDirectory((java.nio.file.Path)this.temporaryFolder.getRoot().resolve("_does_not_exists_file"), FileUtils::isJarFile)).isInstanceOf(IllegalArgumentException.class);
    }

    @Test
    void testListAFileFailsBecauseDirectoryIsExpected() throws IOException {
        String fileName = "a.jar";
        File file = TempDirUtils.newFile((java.nio.file.Path)this.temporaryFolder, (String)"a.jar");
        Assertions.assertThatThrownBy(() -> FileUtils.listFilesInDirectory((java.nio.file.Path)file.toPath(), FileUtils::isJarFile)).isInstanceOf(IllegalArgumentException.class);
    }

    @Test
    void testFollowSymbolicDirectoryLink() throws IOException {
        File directory = TempDirUtils.newFolder((java.nio.file.Path)this.temporaryFolder, (String[])new String[]{"a"});
        File file = new File(directory, "a.jar");
        Assertions.assertThat((boolean)file.createNewFile()).isTrue();
        File otherDirectory = TempDirUtils.newFolder((java.nio.file.Path)this.temporaryFolder);
        java.nio.file.Path linkPath = Paths.get(otherDirectory.getPath(), "a.lnk");
        Files.createSymbolicLink(linkPath, directory.toPath(), new FileAttribute[0]);
        Collection paths = FileUtils.listFilesInDirectory((java.nio.file.Path)linkPath, FileUtils::isJarFile);
        Assertions.assertThat((Collection)paths).containsExactlyInAnyOrder((Object[])new java.nio.file.Path[]{linkPath.resolve(file.getName())});
    }

    @Test
    void testGetTargetPathNotContainsSymbolicPath() throws IOException {
        java.nio.file.Path testPath = Paths.get("parent", "child");
        java.nio.file.Path targetPath = FileUtils.getTargetPathIfContainsSymbolicPath((java.nio.file.Path)testPath);
        Assertions.assertThat((java.nio.file.Path)targetPath).isEqualTo((Object)testPath);
    }

    @Test
    void testGetTargetPathContainsSymbolicPath() throws IOException {
        File linkedDir = TempDirUtils.newFolder((java.nio.file.Path)this.temporaryFolder, (String[])new String[]{"linked"});
        java.nio.file.Path symlink = Paths.get(this.temporaryFolder.toString(), "symlink");
        java.nio.file.Path dirInLinked = TempDirUtils.newFolder((java.nio.file.Path)linkedDir.toPath(), (String[])new String[]{"one", "two"}).toPath().toRealPath(new LinkOption[0]);
        Files.createSymbolicLink(symlink, linkedDir.toPath(), new FileAttribute[0]);
        java.nio.file.Path targetPath = FileUtils.getTargetPathIfContainsSymbolicPath((java.nio.file.Path)symlink.resolve("one").resolve("two"));
        Assertions.assertThat((java.nio.file.Path)targetPath).isEqualTo((Object)dirInLinked);
    }

    @Test
    void testGetTargetPathContainsMultipleSymbolicPath() throws IOException {
        File linked1Dir = TempDirUtils.newFolder((java.nio.file.Path)this.temporaryFolder, (String[])new String[]{"linked1"});
        java.nio.file.Path symlink1 = Paths.get(this.temporaryFolder.toString(), "symlink1");
        Files.createSymbolicLink(symlink1, linked1Dir.toPath(), new FileAttribute[0]);
        java.nio.file.Path symlink2 = Paths.get(symlink1.toString(), "symlink2");
        File linked2Dir = TempDirUtils.newFolder((java.nio.file.Path)this.temporaryFolder, (String[])new String[]{"linked2"});
        Files.createSymbolicLink(symlink2, linked2Dir.toPath(), new FileAttribute[0]);
        java.nio.file.Path dirInLinked2 = TempDirUtils.newFolder((java.nio.file.Path)linked2Dir.toPath(), (String[])new String[]{"one"}).toPath().toRealPath(new LinkOption[0]);
        java.nio.file.Path symlink3 = Paths.get(symlink1.toString(), "symlink3");
        Files.createSymbolicLink(symlink3, symlink2, new FileAttribute[0]);
        java.nio.file.Path targetPath = FileUtils.getTargetPathIfContainsSymbolicPath((java.nio.file.Path)symlink3.resolve("one"));
        Assertions.assertThat((java.nio.file.Path)targetPath).isEqualTo((Object)dirInLinked2);
    }

    @Test
    void testGetDirectorySize() throws Exception {
        File parent = TempDirUtils.newFolder((java.nio.file.Path)this.temporaryFolder);
        Assertions.assertThat((long)FileUtils.getDirectoryFilesSize((java.nio.file.Path)parent.toPath(), (FileVisitOption[])new FileVisitOption[0])).isZero();
        FileUtilsTest.generateRandomDirs(parent, 20, 5, 3);
        Assertions.assertThat((long)FileUtils.getDirectoryFilesSize((java.nio.file.Path)parent.toPath(), (FileVisitOption[])new FileVisitOption[0])).isEqualTo(3120L);
    }

    private static void assertDirEquals(java.nio.file.Path expected, java.nio.file.Path actual) throws IOException {
        Assertions.assertThat((boolean)Files.isDirectory(actual, new LinkOption[0])).isEqualTo(Files.isDirectory(expected, new LinkOption[0]));
        Assertions.assertThat((java.nio.file.Path)actual.getFileName()).isEqualTo((Object)expected.getFileName());
        if (Files.isDirectory(expected, new LinkOption[0])) {
            List actualContents;
            List expectedContents;
            try (Stream<java.nio.file.Path> files = Files.list(expected);){
                expectedContents = files.sorted(Comparator.comparing(java.nio.file.Path::toString)).collect(Collectors.toList());
            }
            try (Stream<java.nio.file.Path> files = Files.list(actual);){
                actualContents = files.sorted(Comparator.comparing(java.nio.file.Path::toString)).collect(Collectors.toList());
            }
            Assertions.assertThat(actualContents).hasSameSizeAs(expectedContents);
            for (int x = 0; x < expectedContents.size(); ++x) {
                FileUtilsTest.assertDirEquals((java.nio.file.Path)expectedContents.get(x), (java.nio.file.Path)actualContents.get(x));
            }
        } else {
            byte[] expectedBytes = Files.readAllBytes(expected);
            byte[] actualBytes = Files.readAllBytes(actual);
            Assertions.assertThat((byte[])actualBytes).isEqualTo((Object)expectedBytes);
        }
    }

    private static void generateRandomDirs(File dir, int numFiles, int numDirs, int depth) throws IOException {
        int i;
        for (i = 0; i < numFiles; ++i) {
            File file = new File(dir, new AbstractID().toString());
            try (FileOutputStream out = new FileOutputStream(file);){
                out.write(1);
                continue;
            }
        }
        if (depth > 0) {
            for (i = 0; i < numDirs; ++i) {
                File subdir = new File(dir, new AbstractID().toString());
                Assertions.assertThat((boolean)subdir.mkdir()).isTrue();
                FileUtilsTest.generateRandomDirs(subdir, numFiles, numDirs, depth - 1);
            }
        }
    }

    public static Collection<java.nio.file.Path> prepareTestFiles(java.nio.file.Path dir) throws IOException {
        java.nio.file.Path jobSubDir1 = Files.createDirectory(dir.resolve("_sub_dir1"), new FileAttribute[0]);
        java.nio.file.Path jobSubDir2 = Files.createDirectory(dir.resolve("_sub_dir2"), new FileAttribute[0]);
        java.nio.file.Path jarFile1 = Files.createFile(dir.resolve("file1.jar"), new FileAttribute[0]);
        java.nio.file.Path jarFile2 = Files.createFile(dir.resolve("file2.jar"), new FileAttribute[0]);
        java.nio.file.Path jarFile3 = Files.createFile(jobSubDir1.resolve("file3.jar"), new FileAttribute[0]);
        java.nio.file.Path jarFile4 = Files.createFile(jobSubDir2.resolve("file4.jar"), new FileAttribute[0]);
        ArrayList<java.nio.file.Path> jarFiles = new ArrayList<java.nio.file.Path>();
        Files.createFile(dir.resolve("file1.txt"), new FileAttribute[0]);
        Files.createFile(jobSubDir2.resolve("file2.txt"), new FileAttribute[0]);
        jarFiles.add(jarFile1);
        jarFiles.add(jarFile2);
        jarFiles.add(jarFile3);
        jarFiles.add(jarFile4);
        return jarFiles;
    }

    private void verifyDirectoryCompression(java.nio.file.Path testDir, java.nio.file.Path compressDir) throws IOException {
        String testFileContent = "Goethe - Faust: Der Tragoedie erster Teil\nProlog im Himmel.\nDer Herr. Die himmlischen Heerscharen. Nachher Mephistopheles. Die drei\nErzengel treten vor.\nRAPHAEL: Die Sonne toent, nach alter Weise, In Brudersphaeren Wettgesang,\nUnd ihre vorgeschriebne Reise Vollendet sie mit Donnergang. Ihr Anblick\ngibt den Engeln Staerke, Wenn keiner Sie ergruenden mag; die unbegreiflich\nhohen Werke Sind herrlich wie am ersten Tag.\nGABRIEL: Und schnell und unbegreiflich schnelle Dreht sich umher der Erde\nPracht; Es wechselt Paradieseshelle Mit tiefer, schauervoller Nacht. Es\nschaeumt das Meer in breiten Fluessen Am tiefen Grund der Felsen auf, Und\nFels und Meer wird fortgerissen Im ewig schnellem Sphaerenlauf.\nMICHAEL: Und Stuerme brausen um die Wette Vom Meer aufs Land, vom Land\naufs Meer, und bilden wuetend eine Kette Der tiefsten Wirkung rings umher.\nDa flammt ein blitzendes Verheeren Dem Pfade vor des Donnerschlags. Doch\ndeine Boten, Herr, verehren Das sanfte Wandeln deines Tags.";
        java.nio.file.Path extractDir = TempDirUtils.newFolder((java.nio.file.Path)this.temporaryFolder, (String[])new String[]{"extractDir"}).toPath();
        java.nio.file.Path originalDir = Paths.get("rootDir", new String[0]);
        java.nio.file.Path emptySubDir = originalDir.resolve("emptyDir");
        java.nio.file.Path fullSubDir = originalDir.resolve("fullDir");
        java.nio.file.Path file1 = originalDir.resolve("file1");
        java.nio.file.Path file2 = originalDir.resolve("file2");
        java.nio.file.Path file3 = fullSubDir.resolve("file3");
        Files.createDirectory(testDir.resolve(originalDir), new FileAttribute[0]);
        Files.createDirectory(testDir.resolve(emptySubDir), new FileAttribute[0]);
        Files.createDirectory(testDir.resolve(fullSubDir), new FileAttribute[0]);
        Files.copy(new ByteArrayInputStream("Goethe - Faust: Der Tragoedie erster Teil\nProlog im Himmel.\nDer Herr. Die himmlischen Heerscharen. Nachher Mephistopheles. Die drei\nErzengel treten vor.\nRAPHAEL: Die Sonne toent, nach alter Weise, In Brudersphaeren Wettgesang,\nUnd ihre vorgeschriebne Reise Vollendet sie mit Donnergang. Ihr Anblick\ngibt den Engeln Staerke, Wenn keiner Sie ergruenden mag; die unbegreiflich\nhohen Werke Sind herrlich wie am ersten Tag.\nGABRIEL: Und schnell und unbegreiflich schnelle Dreht sich umher der Erde\nPracht; Es wechselt Paradieseshelle Mit tiefer, schauervoller Nacht. Es\nschaeumt das Meer in breiten Fluessen Am tiefen Grund der Felsen auf, Und\nFels und Meer wird fortgerissen Im ewig schnellem Sphaerenlauf.\nMICHAEL: Und Stuerme brausen um die Wette Vom Meer aufs Land, vom Land\naufs Meer, und bilden wuetend eine Kette Der tiefsten Wirkung rings umher.\nDa flammt ein blitzendes Verheeren Dem Pfade vor des Donnerschlags. Doch\ndeine Boten, Herr, verehren Das sanfte Wandeln deines Tags.".getBytes(StandardCharsets.UTF_8)), testDir.resolve(file1), new CopyOption[0]);
        Files.createFile(testDir.resolve(file2), new FileAttribute[0]);
        Files.copy(new ByteArrayInputStream("Goethe - Faust: Der Tragoedie erster Teil\nProlog im Himmel.\nDer Herr. Die himmlischen Heerscharen. Nachher Mephistopheles. Die drei\nErzengel treten vor.\nRAPHAEL: Die Sonne toent, nach alter Weise, In Brudersphaeren Wettgesang,\nUnd ihre vorgeschriebne Reise Vollendet sie mit Donnergang. Ihr Anblick\ngibt den Engeln Staerke, Wenn keiner Sie ergruenden mag; die unbegreiflich\nhohen Werke Sind herrlich wie am ersten Tag.\nGABRIEL: Und schnell und unbegreiflich schnelle Dreht sich umher der Erde\nPracht; Es wechselt Paradieseshelle Mit tiefer, schauervoller Nacht. Es\nschaeumt das Meer in breiten Fluessen Am tiefen Grund der Felsen auf, Und\nFels und Meer wird fortgerissen Im ewig schnellem Sphaerenlauf.\nMICHAEL: Und Stuerme brausen um die Wette Vom Meer aufs Land, vom Land\naufs Meer, und bilden wuetend eine Kette Der tiefsten Wirkung rings umher.\nDa flammt ein blitzendes Verheeren Dem Pfade vor des Donnerschlags. Doch\ndeine Boten, Herr, verehren Das sanfte Wandeln deines Tags.".getBytes(StandardCharsets.UTF_8)), testDir.resolve(file3), new CopyOption[0]);
        Path zip = FileUtils.compressDirectory((Path)new Path(compressDir.resolve(originalDir).toString()), (Path)new Path(compressDir.resolve(originalDir) + ".zip"));
        FileUtils.expandDirectory((Path)zip, (Path)new Path(extractDir.toAbsolutePath().toString()));
        FileUtilsTest.assertDirEquals(compressDir.resolve(originalDir), extractDir.resolve(originalDir));
    }

    private Path writeZipAndFetchExpandedPath(String prefix, String path) throws IOException {
        String sourcePath = prefix + "src";
        String dstPath = prefix + "dst";
        java.nio.file.Path srcFolder = TempDirUtils.newFolder((java.nio.file.Path)this.temporaryFolder, (String[])new String[]{sourcePath}).toPath();
        java.nio.file.Path zippedFile = srcFolder.resolve("file.zip");
        ZipOutputStream out = new ZipOutputStream(Files.newOutputStream(zippedFile, new OpenOption[0]));
        ZipEntry e1 = new ZipEntry(path);
        out.putNextEntry(e1);
        out.close();
        java.nio.file.Path dstFolder = TempDirUtils.newFolder((java.nio.file.Path)this.temporaryFolder, (String[])new String[]{dstPath}).toPath();
        return FileUtils.expandDirectory((Path)new Path(zippedFile.toString()), (Path)new Path(dstFolder.toString()));
    }

    @Test
    void testExpandDirWithValidPaths() throws IOException {
        this.writeZipAndFetchExpandedPath("t0", "/level1/level2/");
        this.writeZipAndFetchExpandedPath("t1", "/level1/level2/file.txt");
        this.writeZipAndFetchExpandedPath("t2", "/level1/../level1/file.txt");
        this.writeZipAndFetchExpandedPath("t3", "/level1/level2/level3/../");
        Assertions.assertThatThrownBy(() -> this.writeZipAndFetchExpandedPath("t2", "/level1/level2/../../pass")).isInstanceOf(IOException.class);
    }

    @Test
    void testExpandDirWithForbiddenEscape() {
        Assertions.assertThatThrownBy(() -> this.writeZipAndFetchExpandedPath("t1", "/../../")).isInstanceOf(IOException.class);
        Assertions.assertThatThrownBy(() -> this.writeZipAndFetchExpandedPath("t2", "../")).isInstanceOf(IOException.class);
    }

    private static String generateTestFile(String outputFile, int length) throws IOException, NoSuchAlgorithmException {
        Path outputFilePath = new Path(outputFile);
        FileSystem fileSystem = outputFilePath.getFileSystem();
        try (FSDataOutputStream fsDataOutputStream = fileSystem.create(outputFilePath, FileSystem.WriteMode.OVERWRITE);){
            String string = FileUtilsTest.writeRandomContent((OutputStream)fsDataOutputStream, length);
            return string;
        }
    }

    private static String writeRandomContent(OutputStream out, int length) throws IOException, NoSuchAlgorithmException {
        MessageDigest messageDigest = MessageDigest.getInstance("MD5");
        Random random = new Random();
        int startChar = 32;
        int endChar = 127;
        for (int i = 0; i < length; ++i) {
            int rnd = random.nextInt(endChar - startChar);
            byte b = (byte)(startChar + rnd);
            out.write(b);
            messageDigest.update(b);
        }
        byte[] b = messageDigest.digest();
        return StringUtils.byteToHexString((byte[])b);
    }

    private static String md5Hex(byte[] data) throws NoSuchAlgorithmException {
        MessageDigest messageDigest = MessageDigest.getInstance("MD5");
        messageDigest.update(data);
        byte[] b = messageDigest.digest();
        return StringUtils.byteToHexString((byte[])b);
    }

    private static class Deleter
    extends CheckedThread {
        private final File target;

        Deleter(File target) {
            this.target = target;
        }

        public void go() throws Exception {
            FileUtils.deleteDirectory((File)this.target);
        }
    }
}

