/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.core.security;

import java.security.Permission;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;
import org.apache.flink.configuration.ClusterOptions;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.configuration.IllegalConfigurationException;
import org.apache.flink.core.security.FlinkSecurityManager;
import org.apache.flink.core.security.UserSystemExitException;
import org.apache.flink.core.testutils.CheckedThread;
import org.apache.flink.testutils.TestingUtils;
import org.apache.flink.testutils.executor.TestExecutorExtension;
import org.assertj.core.api.AbstractThrowableAssert;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.RegisterExtension;

class FlinkSecurityManagerTest {
    @RegisterExtension
    static final TestExecutorExtension<ScheduledExecutorService> EXECUTOR_RESOURCE = TestingUtils.defaultExecutorExtension();
    private static final int TEST_EXIT_CODE = 123;
    SecurityManager originalSecurityManager;

    FlinkSecurityManagerTest() {
    }

    @BeforeEach
    void setUp() {
        this.originalSecurityManager = System.getSecurityManager();
    }

    @AfterEach
    void tearDown() {
        System.setSecurityManager(this.originalSecurityManager);
    }

    @Test
    void testThrowUserExit() {
        Assertions.assertThatThrownBy(() -> {
            FlinkSecurityManager flinkSecurityManager = new FlinkSecurityManager(ClusterOptions.UserSystemExitMode.THROW, false);
            flinkSecurityManager.monitorUserSystemExit();
            flinkSecurityManager.checkExit(123);
        }).isInstanceOf(UserSystemExitException.class);
    }

    @Test
    void testToggleUserExit() {
        FlinkSecurityManager flinkSecurityManager = new FlinkSecurityManager(ClusterOptions.UserSystemExitMode.THROW, false);
        flinkSecurityManager.checkExit(123);
        flinkSecurityManager.monitorUserSystemExit();
        Assertions.assertThatThrownBy(() -> flinkSecurityManager.checkExit(123)).isInstanceOf(UserSystemExitException.class);
        flinkSecurityManager.unmonitorUserSystemExit();
        flinkSecurityManager.checkExit(123);
    }

    @Test
    void testPerThreadThrowUserExit() throws Exception {
        FlinkSecurityManager flinkSecurityManager = new FlinkSecurityManager(ClusterOptions.UserSystemExitMode.THROW, false);
        ExecutorService executorService = EXECUTOR_RESOURCE.getExecutor();
        CompletableFuture<Void> future = CompletableFuture.runAsync(() -> flinkSecurityManager.checkExit(123), executorService);
        future.get();
        flinkSecurityManager.monitorUserSystemExit();
        Assertions.assertThatThrownBy(() -> flinkSecurityManager.checkExit(123)).isInstanceOf(UserSystemExitException.class);
        future = CompletableFuture.runAsync(() -> flinkSecurityManager.checkExit(123), executorService);
        future.get();
    }

    @Test
    void testInheritedThrowUserExit() throws Exception {
        final FlinkSecurityManager flinkSecurityManager = new FlinkSecurityManager(ClusterOptions.UserSystemExitMode.THROW, false);
        flinkSecurityManager.monitorUserSystemExit();
        Assertions.assertThatThrownBy(() -> flinkSecurityManager.checkExit(123)).isInstanceOf(UserSystemExitException.class);
        CheckedThread thread = new CheckedThread(){

            public void go() {
                Assertions.assertThatThrownBy(() -> flinkSecurityManager.checkExit(123)).isInstanceOf(UserSystemExitException.class);
            }
        };
        thread.start();
        thread.sync();
    }

    @Test
    void testLogUserExit() {
        FlinkSecurityManager flinkSecurityManager = new FlinkSecurityManager(ClusterOptions.UserSystemExitMode.LOG, false);
        flinkSecurityManager.monitorUserSystemExit();
        flinkSecurityManager.checkExit(123);
    }

    @Test
    void testDisabledConfiguration() {
        Configuration configuration = new Configuration();
        FlinkSecurityManager flinkSecurityManager = FlinkSecurityManager.fromConfiguration((Configuration)configuration);
        Assertions.assertThat((Object)flinkSecurityManager).isNull();
        configuration.set(ClusterOptions.INTERCEPT_USER_SYSTEM_EXIT, (Object)ClusterOptions.UserSystemExitMode.DISABLED);
        flinkSecurityManager = FlinkSecurityManager.fromConfiguration((Configuration)configuration);
        Assertions.assertThat((Object)flinkSecurityManager).isNull();
        configuration.set(ClusterOptions.HALT_ON_FATAL_ERROR, (Object)false);
        flinkSecurityManager = FlinkSecurityManager.fromConfiguration((Configuration)configuration);
        Assertions.assertThat((Object)flinkSecurityManager).isNull();
    }

    @Test
    void testLogConfiguration() {
        Configuration configuration = new Configuration();
        configuration.set(ClusterOptions.INTERCEPT_USER_SYSTEM_EXIT, (Object)ClusterOptions.UserSystemExitMode.LOG);
        FlinkSecurityManager flinkSecurityManager = FlinkSecurityManager.fromConfiguration((Configuration)configuration);
        Assertions.assertThat((Object)flinkSecurityManager).isNotNull();
        Assertions.assertThat((boolean)flinkSecurityManager.userSystemExitMonitored()).isFalse();
        flinkSecurityManager.monitorUserSystemExit();
        Assertions.assertThat((boolean)flinkSecurityManager.userSystemExitMonitored()).isTrue();
        flinkSecurityManager.checkExit(123);
        flinkSecurityManager.unmonitorUserSystemExit();
        Assertions.assertThat((boolean)flinkSecurityManager.userSystemExitMonitored()).isFalse();
    }

    @Test
    void testThrowConfiguration() {
        Configuration configuration = new Configuration();
        configuration.set(ClusterOptions.INTERCEPT_USER_SYSTEM_EXIT, (Object)ClusterOptions.UserSystemExitMode.THROW);
        FlinkSecurityManager flinkSecurityManager = FlinkSecurityManager.fromConfiguration((Configuration)configuration);
        Assertions.assertThat((Object)flinkSecurityManager).isNotNull();
        Assertions.assertThat((boolean)flinkSecurityManager.userSystemExitMonitored()).isFalse();
        flinkSecurityManager.monitorUserSystemExit();
        Assertions.assertThat((boolean)flinkSecurityManager.userSystemExitMonitored()).isTrue();
        FlinkSecurityManager finalFlinkSecurityManager = flinkSecurityManager;
        Assertions.assertThatThrownBy(() -> finalFlinkSecurityManager.checkExit(123)).isInstanceOf(UserSystemExitException.class);
        flinkSecurityManager.unmonitorUserSystemExit();
        Assertions.assertThat((boolean)flinkSecurityManager.userSystemExitMonitored()).isFalse();
        flinkSecurityManager = new FlinkSecurityManager(ClusterOptions.UserSystemExitMode.DISABLED, false);
        flinkSecurityManager.monitorUserSystemExit();
        Assertions.assertThat((boolean)flinkSecurityManager.userSystemExitMonitored()).isTrue();
        flinkSecurityManager.checkExit(123);
    }

    @Test
    void testHaltConfiguration() {
        Configuration configuration = new Configuration();
        configuration.set(ClusterOptions.HALT_ON_FATAL_ERROR, (Object)true);
        FlinkSecurityManager flinkSecurityManager = FlinkSecurityManager.fromConfiguration((Configuration)configuration);
        Assertions.assertThat((Object)flinkSecurityManager).isNotNull();
    }

    @Test
    void testInvalidConfiguration() {
        Assertions.assertThatThrownBy(() -> {
            Configuration configuration = new Configuration();
            configuration.set(ClusterOptions.INTERCEPT_USER_SYSTEM_EXIT, null);
            FlinkSecurityManager.fromConfiguration((Configuration)configuration);
        }).isInstanceOf(NullPointerException.class);
    }

    @Test
    void testExistingSecurityManagerRespected() {
        SecurityManager originalSecurityManager = new SecurityManager(){

            @Override
            public void checkPermission(Permission perm) {
                throw new SecurityException("not allowed");
            }
        };
        FlinkSecurityManager flinkSecurityManager = new FlinkSecurityManager(ClusterOptions.UserSystemExitMode.DISABLED, false, originalSecurityManager);
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> flinkSecurityManager.checkExit(123)).isInstanceOf(SecurityException.class)).hasMessage("not allowed");
    }

    @Test
    void testRegistrationNotAllowedByExistingSecurityManager() {
        Configuration configuration = new Configuration();
        configuration.set(ClusterOptions.INTERCEPT_USER_SYSTEM_EXIT, (Object)ClusterOptions.UserSystemExitMode.THROW);
        System.setSecurityManager(new SecurityManager(){
            private boolean fired;

            @Override
            public void checkPermission(Permission perm) {
                if (!this.fired && perm.getName().equals("setSecurityManager")) {
                    try {
                        throw new SecurityException("not allowed");
                    }
                    catch (Throwable throwable) {
                        this.fired = true;
                        throw throwable;
                    }
                }
            }
        });
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> FlinkSecurityManager.setFromConfiguration((Configuration)configuration)).isInstanceOf(IllegalConfigurationException.class)).hasMessageContaining("Could not register security manager");
    }

    @Test
    void testMultiSecurityManagersWithSetFirstAndMonitored() {
        Configuration configuration = new Configuration();
        configuration.set(ClusterOptions.INTERCEPT_USER_SYSTEM_EXIT, (Object)ClusterOptions.UserSystemExitMode.THROW);
        configuration.set(ClusterOptions.HALT_ON_FATAL_ERROR, (Object)false);
        FlinkSecurityManager.setFromConfiguration((Configuration)configuration);
        TestExitSecurityManager newSecurityManager = new TestExitSecurityManager();
        System.setSecurityManager(newSecurityManager);
        FlinkSecurityManager.monitorUserSystemExitForCurrentThread();
        Assertions.assertThatThrownBy(() -> newSecurityManager.checkExit(123)).isInstanceOf(UserSystemExitException.class);
        Assertions.assertThat((Integer)newSecurityManager.getExitStatus()).isEqualTo(123);
    }

    @Test
    void testMultiSecurityManagersWithSetLastAndMonitored() {
        Configuration configuration = new Configuration();
        configuration.set(ClusterOptions.INTERCEPT_USER_SYSTEM_EXIT, (Object)ClusterOptions.UserSystemExitMode.THROW);
        configuration.set(ClusterOptions.HALT_ON_FATAL_ERROR, (Object)false);
        TestExitSecurityManager oldSecurityManager = new TestExitSecurityManager();
        System.setSecurityManager(oldSecurityManager);
        FlinkSecurityManager.setFromConfiguration((Configuration)configuration);
        FlinkSecurityManager.monitorUserSystemExitForCurrentThread();
        Assertions.assertThatThrownBy(() -> System.getSecurityManager().checkExit(123)).isInstanceOf(UserSystemExitException.class);
        Assertions.assertThat((Integer)oldSecurityManager.getExitStatus()).isNull();
    }

    @Test
    void testMultiSecurityManagersWithSetFirstAndUnmonitored() {
        Configuration configuration = new Configuration();
        configuration.set(ClusterOptions.INTERCEPT_USER_SYSTEM_EXIT, (Object)ClusterOptions.UserSystemExitMode.THROW);
        configuration.set(ClusterOptions.HALT_ON_FATAL_ERROR, (Object)false);
        FlinkSecurityManager.setFromConfiguration((Configuration)configuration);
        TestExitSecurityManager newSecurityManager = new TestExitSecurityManager();
        System.setSecurityManager(newSecurityManager);
        newSecurityManager.checkExit(123);
        Assertions.assertThat((Integer)newSecurityManager.getExitStatus()).isEqualTo(123);
    }

    @Test
    void testMultiSecurityManagersWithSetLastAndUnmonitored() {
        Configuration configuration = new Configuration();
        configuration.set(ClusterOptions.INTERCEPT_USER_SYSTEM_EXIT, (Object)ClusterOptions.UserSystemExitMode.THROW);
        configuration.set(ClusterOptions.HALT_ON_FATAL_ERROR, (Object)false);
        TestExitSecurityManager oldSecurityManager = new TestExitSecurityManager();
        System.setSecurityManager(oldSecurityManager);
        FlinkSecurityManager.setFromConfiguration((Configuration)configuration);
        System.getSecurityManager().checkExit(123);
        Assertions.assertThat((Integer)oldSecurityManager.getExitStatus()).isEqualTo(123);
    }

    private class TestExitSecurityManager
    extends SecurityManager {
        private final SecurityManager originalSecurityManager = System.getSecurityManager();
        private Integer exitStatus;

        @Override
        public void checkExit(int status) {
            this.exitStatus = status;
            if (this.originalSecurityManager != null) {
                this.originalSecurityManager.checkExit(status);
            }
        }

        @Override
        public void checkPermission(Permission perm) {
        }

        public Integer getExitStatus() {
            return this.exitStatus;
        }
    }
}

