/*
 * Decompiled with CFR 0.152.
 */
package org.apache.atlas.util;

import java.time.Clock;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import javax.inject.Inject;
import org.apache.atlas.model.instance.EntityMutationResponse;
import org.apache.atlas.repository.graphdb.AtlasGraph;
import org.apache.atlas.repository.store.graph.v2.AtlasGraphUtilsV2;
import org.apache.atlas.util.AtlasMetricsCounter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

@Component
public class AtlasMetricsUtil {
    private static final Logger LOG = LoggerFactory.getLogger(AtlasMetricsUtil.class);
    private static final long SEC_MS = 1000L;
    private static final long MIN_MS = 60000L;
    private static final long HOUR_MS = 3600000L;
    private static final long DAY_MS = 86400000L;
    private static final String STATUS_CONNECTED = "connected";
    private static final String STATUS_NOT_CONNECTED = "not-connected";
    private final AtlasGraph graph;
    private long serverStartTime = 0L;
    private long serverActiveTime = 0L;
    private final Map<String, TopicStats> topicStats = new HashMap<String, TopicStats>();
    private final AtlasMetricsCounter messagesProcessed = new AtlasMetricsCounter("messagesProcessed");
    private final AtlasMetricsCounter messagesFailed = new AtlasMetricsCounter("messagesFailed");
    private final AtlasMetricsCounter entityCreates = new AtlasMetricsCounter("entityCreates");
    private final AtlasMetricsCounter entityUpdates = new AtlasMetricsCounter("entityUpdates");
    private final AtlasMetricsCounter entityDeletes = new AtlasMetricsCounter("entityDeletes");

    @Inject
    public AtlasMetricsUtil(AtlasGraph graph) {
        this.graph = graph;
    }

    public void init(Clock clock) {
        this.messagesProcessed.init(clock);
        this.messagesFailed.init(clock);
        this.entityCreates.init(clock);
        this.entityUpdates.init(clock);
        this.entityDeletes.init(clock);
    }

    public void onServerStart() {
        this.serverStartTime = System.currentTimeMillis();
    }

    public void onServerActivation() {
        this.serverActiveTime = System.currentTimeMillis();
    }

    public void onNotificationProcessingComplete(String topicName, int partition, long msgOffset, NotificationStat stats) {
        TopicPartitionStat partitionStat;
        TopicStats topicStat;
        this.messagesProcessed.incrWithMeasure(stats.timeTakenMs);
        this.entityCreates.incrBy(stats.entityCreates);
        this.entityUpdates.incrBy(stats.entityUpdates);
        this.entityDeletes.incrBy(stats.entityDeletes);
        if (stats.isFailedMsg) {
            this.messagesFailed.incr();
        }
        if ((topicStat = this.topicStats.get(topicName)) == null) {
            topicStat = new TopicStats(topicName);
            this.topicStats.put(topicName, topicStat);
        }
        if ((partitionStat = topicStat.get(partition)) == null) {
            partitionStat = new TopicPartitionStat(topicName, partition, msgOffset, msgOffset);
            topicStat.set(partition, partitionStat);
        }
        partitionStat.setCurrentOffset(msgOffset + 1L);
        if (stats.isFailedMsg) {
            partitionStat.incrFailedMessageCount();
        }
        partitionStat.incrProcessedMessageCount();
        partitionStat.setLastMessageProcessedTime(this.messagesProcessed.getLastIncrTime().toEpochMilli());
    }

    public Map<String, Object> getStats() {
        HashMap<String, Object> ret = new HashMap<String, Object>();
        AtlasMetricsCounter.StatsReport messagesProcessed = this.messagesProcessed.report();
        AtlasMetricsCounter.StatsReport messagesFailed = this.messagesFailed.report();
        AtlasMetricsCounter.StatsReport entityCreates = this.entityCreates.report();
        AtlasMetricsCounter.StatsReport entityUpdates = this.entityUpdates.report();
        AtlasMetricsCounter.StatsReport entityDeletes = this.entityDeletes.report();
        ret.put("Server:startTimeStamp", this.serverStartTime);
        ret.put("Server:activeTimeStamp", this.serverActiveTime);
        ret.put("Server:upTime", this.millisToTimeDiff(System.currentTimeMillis() - this.serverStartTime));
        ret.put("Server:statusBackendStore", this.getBackendStoreStatus() ? STATUS_CONNECTED : STATUS_NOT_CONNECTED);
        ret.put("Server:statusIndexStore", this.getIndexStoreStatus() ? STATUS_CONNECTED : STATUS_NOT_CONNECTED);
        HashMap topicDetails = new HashMap();
        for (TopicStats tStat : this.topicStats.values()) {
            for (TopicPartitionStat tpStat : tStat.partitionStats.values()) {
                HashMap<String, Long> tpDetails = new HashMap<String, Long>();
                tpDetails.put("offsetStart", tpStat.startOffset);
                tpDetails.put("offsetCurrent", tpStat.currentOffset);
                tpDetails.put("failedMessageCount", tpStat.failedMessageCount);
                tpDetails.put("lastMessageProcessedTime", tpStat.lastMessageProcessedTime);
                tpDetails.put("processedMessageCount", tpStat.processedMessageCount);
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Setting failedMessageCount : {} and lastMessageProcessedTime : {} for topic {}-{}", new Object[]{tpStat.failedMessageCount, tpStat.lastMessageProcessedTime, tpStat.topicName, tpStat.partition});
                }
                topicDetails.put(tpStat.topicName + "-" + tpStat.partition, tpDetails);
            }
        }
        ret.put("Notification:topicDetails", topicDetails);
        ret.put("Notification:lastMessageProcessedTime", this.messagesProcessed.getLastIncrTime().toEpochMilli());
        ret.put("Notification:total", messagesProcessed.getCount(AtlasMetricsCounter.Period.ALL));
        ret.put("Notification:totalAvgTime", messagesProcessed.getMeasureAvg(AtlasMetricsCounter.Period.ALL));
        ret.put("Notification:totalFailed", messagesFailed.getCount(AtlasMetricsCounter.Period.ALL));
        ret.put("Notification:totalCreates", entityCreates.getCount(AtlasMetricsCounter.Period.ALL));
        ret.put("Notification:totalUpdates", entityUpdates.getCount(AtlasMetricsCounter.Period.ALL));
        ret.put("Notification:totalDeletes", entityDeletes.getCount(AtlasMetricsCounter.Period.ALL));
        ret.put("Notification:currentDayStartTime", messagesProcessed.getDayStartTimeMs());
        ret.put("Notification:currentDay", messagesProcessed.getCount(AtlasMetricsCounter.Period.CURR_DAY));
        ret.put("Notification:currentDayAvgTime", messagesProcessed.getMeasureAvg(AtlasMetricsCounter.Period.CURR_DAY));
        ret.put("Notification:currentDayFailed", messagesFailed.getCount(AtlasMetricsCounter.Period.CURR_DAY));
        ret.put("Notification:currentDayEntityCreates", entityCreates.getCount(AtlasMetricsCounter.Period.CURR_DAY));
        ret.put("Notification:currentDayEntityUpdates", entityUpdates.getCount(AtlasMetricsCounter.Period.CURR_DAY));
        ret.put("Notification:currentDayEntityDeletes", entityDeletes.getCount(AtlasMetricsCounter.Period.CURR_DAY));
        ret.put("Notification:currentHourStartTime", messagesProcessed.getHourStartTimeMs());
        ret.put("Notification:currentHour", messagesProcessed.getCount(AtlasMetricsCounter.Period.CURR_HOUR));
        ret.put("Notification:currentHourAvgTime", messagesProcessed.getMeasureAvg(AtlasMetricsCounter.Period.CURR_HOUR));
        ret.put("Notification:currentHourFailed", messagesFailed.getCount(AtlasMetricsCounter.Period.CURR_HOUR));
        ret.put("Notification:currentHourEntityCreates", entityCreates.getCount(AtlasMetricsCounter.Period.CURR_HOUR));
        ret.put("Notification:currentHourEntityUpdates", entityUpdates.getCount(AtlasMetricsCounter.Period.CURR_HOUR));
        ret.put("Notification:currentHourEntityDeletes", entityDeletes.getCount(AtlasMetricsCounter.Period.CURR_HOUR));
        ret.put("Notification:previousHour", messagesProcessed.getCount(AtlasMetricsCounter.Period.PREV_HOUR));
        ret.put("Notification:previousHourAvgTime", messagesProcessed.getMeasureAvg(AtlasMetricsCounter.Period.PREV_HOUR));
        ret.put("Notification:previousHourFailed", messagesFailed.getCount(AtlasMetricsCounter.Period.PREV_HOUR));
        ret.put("Notification:previousHourEntityCreates", entityCreates.getCount(AtlasMetricsCounter.Period.PREV_HOUR));
        ret.put("Notification:previousHourEntityUpdates", entityUpdates.getCount(AtlasMetricsCounter.Period.PREV_HOUR));
        ret.put("Notification:previousHourEntityDeletes", entityDeletes.getCount(AtlasMetricsCounter.Period.PREV_HOUR));
        ret.put("Notification:previousDay", messagesProcessed.getCount(AtlasMetricsCounter.Period.PREV_DAY));
        ret.put("Notification:previousDayAvgTime", messagesProcessed.getMeasureAvg(AtlasMetricsCounter.Period.PREV_DAY));
        ret.put("Notification:previousDayFailed", messagesFailed.getCount(AtlasMetricsCounter.Period.PREV_DAY));
        ret.put("Notification:previousDayEntityCreates", entityCreates.getCount(AtlasMetricsCounter.Period.PREV_DAY));
        ret.put("Notification:previousDayEntityUpdates", entityUpdates.getCount(AtlasMetricsCounter.Period.PREV_DAY));
        ret.put("Notification:previousDayEntityDeletes", entityDeletes.getCount(AtlasMetricsCounter.Period.PREV_DAY));
        return ret;
    }

    private boolean getBackendStoreStatus() {
        try {
            this.runWithTimeout(new Runnable(){

                @Override
                public void run() {
                    AtlasMetricsUtil.this.graph.query().has("__typeName", (Object)"__internal").vertices(1);
                    AtlasMetricsUtil.this.graphCommit();
                }
            }, 10L, TimeUnit.SECONDS);
        }
        catch (Exception e) {
            LOG.error(e.getMessage());
            this.graphRollback();
            return false;
        }
        return true;
    }

    private boolean getIndexStoreStatus() {
        final String query = AtlasGraphUtilsV2.getIndexSearchPrefix() + "\"" + "__typeName" + "\":(" + "__internal" + ")";
        try {
            this.runWithTimeout(new Runnable(){

                @Override
                public void run() {
                    AtlasMetricsUtil.this.graph.indexQuery("vertex_index", query).vertices(0, 1);
                    AtlasMetricsUtil.this.graphCommit();
                }
            }, 10L, TimeUnit.SECONDS);
        }
        catch (Exception e) {
            LOG.error(e.getMessage());
            this.graphRollback();
            return false;
        }
        return true;
    }

    public boolean isBackendStoreActive() {
        return this.getBackendStoreStatus();
    }

    public boolean isIndexStoreActive() {
        return this.getIndexStoreStatus();
    }

    private void runWithTimeout(final Runnable runnable, long timeout, TimeUnit timeUnit) throws Exception {
        this.runWithTimeout(new Callable<Object>(){

            @Override
            public Object call() {
                runnable.run();
                return null;
            }
        }, timeout, timeUnit);
    }

    private <T> T runWithTimeout(Callable<T> callable, long timeout, TimeUnit timeUnit) throws Exception {
        ExecutorService executor = Executors.newSingleThreadExecutor();
        Future<T> future = executor.submit(callable);
        executor.shutdown();
        try {
            return future.get(timeout, timeUnit);
        }
        catch (TimeoutException e) {
            future.cancel(true);
            throw e;
        }
        catch (ExecutionException e) {
            Throwable t = e.getCause();
            if (t instanceof Error) {
                throw (Error)t;
            }
            if (t instanceof Exception) {
                throw (Exception)t;
            }
            throw new IllegalStateException(t);
        }
    }

    private void graphCommit() {
        try {
            this.graph.commit();
        }
        catch (Exception ex) {
            LOG.warn("Graph transaction commit failed: {}; attempting to rollback graph transaction.", (Throwable)ex);
            this.graphRollback();
        }
    }

    private void graphRollback() {
        try {
            this.graph.rollback();
        }
        catch (Exception ex) {
            LOG.warn("Graph transaction rollback failed: {}", (Throwable)ex);
        }
    }

    private String millisToTimeDiff(long msDiff) {
        StringBuilder sb = new StringBuilder();
        long diffSeconds = msDiff / 1000L % 60L;
        long diffMinutes = msDiff / 60000L % 60L;
        long diffHours = msDiff / 3600000L % 24L;
        long diffDays = msDiff / 86400000L;
        if (diffDays > 0L) {
            sb.append(diffDays).append(" day ");
        }
        if (diffHours > 0L) {
            sb.append(diffHours).append(" hour ");
        }
        if (diffMinutes > 0L) {
            sb.append(diffMinutes).append(" min ");
        }
        if (diffSeconds > 0L) {
            sb.append(diffSeconds).append(" sec");
        }
        return sb.toString();
    }

    public static class NotificationStat {
        public boolean isFailedMsg = false;
        public long timeTakenMs = 0L;
        public int entityCreates = 0;
        public int entityUpdates = 0;
        public int entityDeletes = 0;

        public NotificationStat() {
        }

        public NotificationStat(boolean isFailedMsg, long timeTakenMs) {
            this.isFailedMsg = isFailedMsg;
            this.timeTakenMs = timeTakenMs;
        }

        public void updateStats(EntityMutationResponse response) {
            this.entityCreates += this.getSize(response.getCreatedEntities());
            this.entityUpdates += this.getSize(response.getUpdatedEntities());
            this.entityUpdates += this.getSize(response.getPartialUpdatedEntities());
            this.entityDeletes += this.getSize(response.getDeletedEntities());
        }

        private int getSize(Collection collection) {
            return collection != null ? collection.size() : 0;
        }
    }

    class TopicStats {
        private final String topicName;
        private final Map<Integer, TopicPartitionStat> partitionStats = new HashMap<Integer, TopicPartitionStat>();

        public TopicStats(String topicName) {
            this.topicName = topicName;
        }

        public String getTopicName() {
            return this.topicName;
        }

        public Map<Integer, TopicPartitionStat> getPartitionStats() {
            return this.partitionStats;
        }

        public TopicPartitionStat get(Integer partition) {
            return this.partitionStats.get(partition);
        }

        public void set(Integer partition, TopicPartitionStat partitionStat) {
            this.partitionStats.put(partition, partitionStat);
        }
    }

    class TopicPartitionStat {
        private final String topicName;
        private final int partition;
        private final long startOffset;
        private long currentOffset;
        private long lastMessageProcessedTime;
        private long failedMessageCount;
        private long processedMessageCount;

        public TopicPartitionStat(String topicName, int partition, long startOffset, long currentOffset) {
            this.topicName = topicName;
            this.partition = partition;
            this.startOffset = startOffset;
            this.currentOffset = currentOffset;
        }

        public String getTopicName() {
            return this.topicName;
        }

        public int getPartition() {
            return this.partition;
        }

        public long getStartOffset() {
            return this.startOffset;
        }

        public long getCurrentOffset() {
            return this.currentOffset;
        }

        public void setCurrentOffset(long currentOffset) {
            this.currentOffset = currentOffset;
        }

        public long getLastMessageProcessedTime() {
            return this.lastMessageProcessedTime;
        }

        public void setLastMessageProcessedTime(long lastMessageProcessedTime) {
            this.lastMessageProcessedTime = lastMessageProcessedTime;
        }

        public long getFailedMessageCount() {
            return this.failedMessageCount;
        }

        public void incrFailedMessageCount() {
            ++this.failedMessageCount;
        }

        public long getProcessedMessageCount() {
            return this.processedMessageCount;
        }

        public void incrProcessedMessageCount() {
            ++this.processedMessageCount;
        }
    }
}

