/*
 * Decompiled with CFR 0.152.
 */
package org.apache.atlas.repository.migration;

import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.InetAddress;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.zip.ZipFile;
import org.apache.atlas.ApplicationProperties;
import org.apache.atlas.AtlasException;
import org.apache.atlas.RequestContext;
import org.apache.atlas.exception.AtlasBaseException;
import org.apache.atlas.model.impexp.AtlasImportRequest;
import org.apache.atlas.model.migration.MigrationImportStatus;
import org.apache.atlas.repository.graph.AtlasGraphProvider;
import org.apache.atlas.repository.impexp.ImportService;
import org.apache.atlas.repository.migration.DataMigrationStatusService;
import org.apache.atlas.repository.migration.FileWatcher;
import org.apache.atlas.type.AtlasType;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.io.filefilter.WildcardFileFilter;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ZipFileMigrationImporter
implements Runnable {
    private static final Logger LOG = LoggerFactory.getLogger(ZipFileMigrationImporter.class);
    private static final String APPLICATION_PROPERTY_MIGRATION_NUMER_OF_WORKERS = "atlas.migration.mode.workers";
    private static final String APPLICATION_PROPERTY_MIGRATION_BATCH_SIZE = "atlas.migration.mode.batch.size";
    private static final String DEFAULT_NUMBER_OF_WORKERS = "4";
    private static final String DEFAULT_BATCH_SIZE = "100";
    private static final String ZIP_FILE_COMMENT_ENTITIES_COUNT = "entitiesCount";
    private static final String ZIP_FILE_COMMENT_TOTAL_COUNT = "total";
    private static final String FILE_EXTENSION_ZIP = ".zip";
    private static final String ENV_USER_NAME = "user.name";
    private static final String ARCHIVE_DIR = "archive";
    private final ImportService importService;
    private List<String> filesToImport;
    private DataMigrationStatusService dataMigrationStatusService;
    private MigrationImportStatus migrationImportStatus;
    private File archiveDir;

    public ZipFileMigrationImporter(ImportService importService, String fileName) {
        this.importService = importService;
        this.dataMigrationStatusService = new DataMigrationStatusService(AtlasGraphProvider.getGraphInstance());
        this.initialize(fileName);
    }

    private void initialize(String fileName) {
        this.filesToImport = this.getAllFilesToImport(fileName);
        if (CollectionUtils.isNotEmpty(this.filesToImport)) {
            this.createArchiveDirectory(fileName);
        }
    }

    @Override
    public void run() {
        for (String fileToImport : this.filesToImport) {
            try {
                this.detectFileToImport(fileToImport);
                int streamSize = this.getStreamSizeFromComment(fileToImport);
                this.migrationImportStatus = this.getCreateMigrationStatus(fileToImport, streamSize);
                this.performImport(fileToImport, streamSize, Long.toString(this.migrationImportStatus.getCurrentIndex()));
                this.dataMigrationStatusService.setStatus("DONE");
                this.moveZipFileToArchiveDir(fileToImport);
            }
            catch (IOException e) {
                LOG.error("Migration Import: IO Error!", (Throwable)e);
                this.dataMigrationStatusService.setStatus("FAIL");
            }
            catch (AtlasBaseException e) {
                LOG.error("Migration Import: Error!", (Throwable)e);
                this.dataMigrationStatusService.setStatus("FAIL");
            }
        }
    }

    private List<String> getAllFilesToImport(String fileName) {
        ArrayList<String> ret = new ArrayList<String>();
        File fileToImport = new File(fileName);
        if (fileToImport.exists() && fileToImport.isFile()) {
            LOG.info("Migration Import: zip file for import: " + fileToImport);
            ret.add(fileToImport.getAbsolutePath());
        } else {
            String dirPath = new File(fileToImport.getParent()).getAbsolutePath();
            File importDataDir = new File(dirPath);
            if (importDataDir.exists() && importDataDir.isDirectory()) {
                String fileNameWithWildcard = fileToImport.getName();
                WildcardFileFilter fileFilter = new WildcardFileFilter(fileNameWithWildcard);
                Object[] importFiles = importDataDir.listFiles((FileFilter)fileFilter);
                if (ArrayUtils.isNotEmpty((Object[])importFiles)) {
                    Arrays.sort(importFiles);
                    LOG.info("Migration Import: zip files for import: ");
                    for (Object importFile : importFiles) {
                        if (this.isValidImportFile((File)importFile)) {
                            LOG.info(((File)importFile).getName() + " with absolute path - " + ((File)importFile).getAbsolutePath());
                            ret.add(((File)importFile).getAbsolutePath());
                            continue;
                        }
                        LOG.warn("Ignoring " + ((File)importFile).getAbsolutePath() + " as it is not a file or does not end with extension " + FILE_EXTENSION_ZIP);
                    }
                } else {
                    LOG.warn("Migration Import: No files to import");
                }
            }
        }
        return ret;
    }

    private boolean isValidImportFile(File importFile) {
        return importFile.isFile() && StringUtils.endsWithIgnoreCase((String)importFile.getName(), (String)FILE_EXTENSION_ZIP);
    }

    private void createArchiveDirectory(String fileName) {
        File fileToImport = new File(fileName);
        String parentPath = new File(fileToImport.getParent()).getAbsolutePath();
        this.archiveDir = new File(parentPath + File.separator + ARCHIVE_DIR);
        if (this.archiveDir.exists() && !this.archiveDir.canWrite()) {
            LOG.warn("Migration Import: No write permission to archive directory {}", (Object)this.archiveDir.getAbsolutePath());
            this.archiveDir = null;
        } else if (!this.archiveDir.exists() && !this.archiveDir.getParentFile().canWrite()) {
            LOG.warn("Migration Import: No permission to create archive directory {}", (Object)this.archiveDir.getAbsolutePath());
            this.archiveDir = null;
        } else {
            this.archiveDir.mkdirs();
            LOG.info("Migration Import: archive directory for zip files: {}", (Object)this.archiveDir.getAbsolutePath());
        }
    }

    private void moveZipFileToArchiveDir(String srcFilePath) {
        if (this.archiveDir == null) {
            return;
        }
        File sourceFile = new File(srcFilePath);
        String newFile = this.archiveDir.getAbsolutePath() + File.separator + sourceFile.getName();
        if (!sourceFile.canWrite()) {
            LOG.warn("Migration Import: No permission to archive the zip file {}", (Object)sourceFile.getAbsolutePath());
            this.archiveDir = null;
        } else if (sourceFile.renameTo(new File(newFile))) {
            sourceFile.delete();
            LOG.info("Migration Import: Successfully archived the zip file: " + srcFilePath + " to " + this.archiveDir.getAbsolutePath());
        } else {
            LOG.warn("Migration Import: Failed to archive the zip file: " + srcFilePath);
        }
    }

    private MigrationImportStatus getCreateMigrationStatus(String fileName, int streamSize) {
        MigrationImportStatus status = null;
        try {
            status = new MigrationImportStatus(fileName, DigestUtils.md5Hex((InputStream)new FileInputStream(fileName)));
        }
        catch (IOException e) {
            LOG.error("Exception occurred while creating migration import", (Throwable)e);
        }
        status.setTotalCount((long)streamSize);
        MigrationImportStatus statusRetrieved = this.dataMigrationStatusService.getCreate(status);
        LOG.info("DataMigrationStatusService: Position: {}", (Object)statusRetrieved.getCurrentIndex());
        this.dataMigrationStatusService.setStatus("STARTED");
        return statusRetrieved;
    }

    private void detectFileToImport(String fileToImport) throws IOException {
        FileWatcher fileWatcher = new FileWatcher(fileToImport);
        fileWatcher.start();
    }

    private int getStreamSizeFromComment(String fileToImport) {
        int ret = 1;
        try {
            ZipFile zipFile = new ZipFile(fileToImport);
            String comment = zipFile.getComment();
            ret = this.processZipFileStreamSizeComment(comment);
            zipFile.close();
        }
        catch (IOException e) {
            LOG.error("Error opening ZIP file: {}", (Object)fileToImport, (Object)e);
        }
        return ret;
    }

    private int processZipFileStreamSizeComment(String comment) {
        if (StringUtils.isEmpty((String)comment)) {
            return 1;
        }
        Map map = (Map)AtlasType.fromJson((String)comment, Map.class);
        int entitiesCount = (Integer)map.get(ZIP_FILE_COMMENT_ENTITIES_COUNT);
        int totalCount = (Integer)map.get(ZIP_FILE_COMMENT_TOTAL_COUNT);
        LOG.info("ZipFileMigrationImporter: Zip file: Comment: streamSize: {}: total: {}", (Object)entitiesCount, (Object)totalCount);
        return entitiesCount;
    }

    private void performImport(String fileToImport, int streamSize, String startPosition) throws AtlasBaseException {
        try {
            LOG.info("Migration Import: {}: Starting at: {}...", (Object)fileToImport, (Object)startPosition);
            FileInputStream fs = new FileInputStream(fileToImport);
            RequestContext.get().setUser(this.getUserNameFromEnvironment(), null);
            this.importService.run(fs, this.getImportRequest(fileToImport, streamSize, startPosition), this.getUserNameFromEnvironment(), InetAddress.getLocalHost().getHostName(), InetAddress.getLocalHost().getHostAddress());
        }
        catch (Exception ex) {
            LOG.error("Migration Import: Error loading zip for migration!", (Throwable)ex);
            throw new AtlasBaseException((Throwable)ex);
        }
        finally {
            LOG.info("Migration Import: {}: Done!", (Object)fileToImport);
        }
    }

    private String getUserNameFromEnvironment() {
        return System.getProperty(ENV_USER_NAME);
    }

    private AtlasImportRequest getImportRequest(String fileToImport, int streamSize, String position) throws AtlasException {
        AtlasImportRequest request = new AtlasImportRequest();
        request.setOption("migrationFileName", fileToImport);
        request.setSizeOption(streamSize);
        request.setOption("migration", "true");
        request.setOption("numWorkers", this.getPropertyValue(APPLICATION_PROPERTY_MIGRATION_NUMER_OF_WORKERS, DEFAULT_NUMBER_OF_WORKERS));
        request.setOption("batchSize", this.getPropertyValue(APPLICATION_PROPERTY_MIGRATION_BATCH_SIZE, DEFAULT_BATCH_SIZE));
        request.setOption("startPosition", StringUtils.isEmpty((String)position) ? "0" : position);
        return request;
    }

    private String getPropertyValue(String property, String defaultValue) throws AtlasException {
        return ApplicationProperties.get().getString(property, defaultValue);
    }
}

