/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ambari.view.utils.hdfs;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Strings;
import java.io.IOException;
import java.io.StringWriter;
import java.io.Writer;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import org.apache.ambari.view.ViewContext;
import org.apache.ambari.view.utils.hdfs.AuthConfigurationBuilder;
import org.apache.ambari.view.utils.hdfs.HdfsApiException;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.conf.Configuration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConfigurationBuilder {
    protected static final Logger LOG = LoggerFactory.getLogger(ConfigurationBuilder.class);
    public static final String CORE_SITE = "core-site";
    public static final String HDFS_SITE = "hdfs-site";
    public static final String DEFAULT_FS_INSTANCE_PROPERTY = "webhdfs.url";
    public static final String DEFAULT_FS_CLUSTER_PROPERTY = "fs.defaultFS";
    public static final String NAMESERVICES_INSTANCE_PROPERTY = "webhdfs.nameservices";
    public static final String NAMESERVICES_CLUSTER_PROPERTY = "dfs.nameservices";
    public static final String HA_NAMENODES_INSTANCE_PROPERTY = "webhdfs.ha.namenodes.list";
    public static final String HA_NAMENODES_CLUSTER_PROPERTY = "dfs.ha.namenodes.%s";
    public static final String NAMENODE_RPC_NN_INSTANCE_PROPERTY = "webhdfs.ha.namenode.rpc-address.list";
    public static final String NAMENODE_RPC_NN_CLUSTER_PROPERTY = "dfs.namenode.rpc-address.%s.%s";
    public static final String NAMENODE_HTTP_NN_INSTANCE_PROPERTY = "webhdfs.ha.namenode.http-address.list";
    public static final String NAMENODE_HTTP_NN_CLUSTER_PROPERTY = "dfs.namenode.http-address.%s.%s";
    public static final String NAMENODE_HTTPS_NN_INSTANCE_PROPERTY = "webhdfs.ha.namenode.https-address.list";
    public static final String NAMENODE_HTTPS_NN_CLUSTER_PROPERTY = "dfs.namenode.https-address.%s.%s";
    public static final String FAILOVER_PROXY_PROVIDER_INSTANCE_PROPERTY = "webhdfs.client.failover.proxy.provider";
    public static final String FAILOVER_PROXY_PROVIDER_CLUSTER_PROPERTY = "dfs.client.failover.proxy.provider.%s";
    public static final String UMASK_CLUSTER_PROPERTY = "fs.permissions.umask-mode";
    public static final String UMASK_INSTANCE_PROPERTY = "hdfs.umask-mode";
    public static final String DFS_WEBHDFS_ENABLED = "dfs.webhdfs.enabled";
    public static final String DFS_HTTP_POLICY = "dfs.http.policy";
    public static final String DFS_HTTP_POLICY_HTTPS_ONLY = "HTTPS_ONLY";
    public static final String DFS_NAMENODE_HTTP_ADDERSS = "dfs.namenode.http-address";
    public static final String DFS_NAMENODE_HTTPS_ADDERSS = "dfs.namenode.https-address";
    protected Configuration conf = new Configuration();
    private ViewContext context;
    private AuthConfigurationBuilder authParamsBuilder;
    private Map<String, String> authParams;
    private URI defaultFsUri;
    private Map<String, String> customProperties;

    public ConfigurationBuilder(ViewContext context) {
        this.context = context;
        this.authParamsBuilder = new AuthConfigurationBuilder(context);
    }

    public ConfigurationBuilder(ViewContext context, Map<String, String> customProperties) {
        this.context = context;
        this.authParamsBuilder = new AuthConfigurationBuilder(context);
        this.customProperties = customProperties;
    }

    private void parseProperties() throws HdfsApiException {
        String encryptionKeyProviderUri;
        String defaultFS = this.getDefaultFS(this.context);
        try {
            if (this.isHAEnabled(defaultFS)) {
                this.copyHAProperties(defaultFS);
                LOG.info("HA HDFS cluster found.");
            } else if (defaultFS.startsWith("webhdfs://") && !ConfigurationBuilder.hasPort(defaultFS)) {
                defaultFS = ConfigurationBuilder.addPortIfMissing(defaultFS);
            }
            this.defaultFsUri = new URI(defaultFS);
        }
        catch (URISyntaxException e) {
            throw new HdfsApiException("HDFS060 Invalid webhdfs.url='" + defaultFS + "' URI", e);
        }
        this.conf.set(DEFAULT_FS_CLUSTER_PROPERTY, defaultFS);
        LOG.info(String.format("HdfsApi configured to connect to defaultFS='%s'", defaultFS));
        if (this.context.getCluster() != null && (encryptionKeyProviderUri = this.getEncryptionKeyProviderUri()) != null) {
            this.conf.set("dfs.encryption.key.provider.uri", encryptionKeyProviderUri);
        }
    }

    protected String getEncryptionKeyProviderUri() {
        String encryptionKeyProviderUri = this.getProperty(HDFS_SITE, "dfs.encryption.key.provider.uri");
        return encryptionKeyProviderUri;
    }

    private String getDefaultFS(ViewContext context) throws HdfsApiException {
        String defaultFS = this.getProperty(CORE_SITE, DEFAULT_FS_CLUSTER_PROPERTY, DEFAULT_FS_INSTANCE_PROPERTY);
        if (defaultFS == null || defaultFS.isEmpty()) {
            throw new HdfsApiException("HDFS070 fs.defaultFS is not configured");
        }
        defaultFS = ConfigurationBuilder.addProtocolIfMissing(defaultFS);
        if (context.getCluster() != null) {
            try {
                URI fsUri = new URI(defaultFS);
                String protocol = fsUri.getScheme();
                String hostWithPort = defaultFS.substring(protocol.length() + 3);
                Boolean webHdfsEnabled = Boolean.valueOf(this.getProperty(HDFS_SITE, DFS_WEBHDFS_ENABLED));
                Boolean isHttps = DFS_HTTP_POLICY_HTTPS_ONLY.equals(this.getProperty(HDFS_SITE, DFS_HTTP_POLICY));
                boolean isHA = this.isHAEnabled(defaultFS);
                if (webHdfsEnabled.booleanValue() && isHttps.booleanValue() && "hdfs".equals(protocol)) {
                    protocol = "swebhdfs";
                    String httpAddr = this.getProperty(HDFS_SITE, DFS_NAMENODE_HTTPS_ADDERSS);
                    if (!isHA && httpAddr != null) {
                        hostWithPort = httpAddr;
                    }
                } else if (webHdfsEnabled.booleanValue() && "hdfs".equals(protocol)) {
                    protocol = "webhdfs";
                    String httpsAddr = this.getProperty(HDFS_SITE, DFS_NAMENODE_HTTP_ADDERSS);
                    if (!isHA) {
                        hostWithPort = httpsAddr;
                    }
                }
                return protocol + "://" + hostWithPort;
            }
            catch (URISyntaxException e) {
                throw new HdfsApiException("Invalid URI format." + e.getMessage(), e);
            }
        }
        return defaultFS;
    }

    private String getProperty(String type, String key, String instanceProperty) {
        String value = this.context.getCluster() != null ? this.getProperty(type, key) : this.getViewProperty(instanceProperty);
        return value;
    }

    private String getViewProperty(String instanceProperty) {
        return (String)this.context.getProperties().get(instanceProperty);
    }

    private String getProperty(String type, String key) {
        if (this.context.getCluster() != null) {
            return this.context.getCluster().getConfigurationValue(type, key);
        }
        return null;
    }

    private void copyPropertyIfExists(String type, String key) {
        if (this.context.getCluster() != null) {
            String value = this.context.getCluster().getConfigurationValue(type, key);
            if (value != null) {
                this.conf.set(key, value);
                LOG.debug("set " + key + " = " + value);
            } else {
                LOG.debug("No such property " + type + "/" + key);
            }
        } else {
            LOG.debug("No such property " + type + "/" + key);
        }
    }

    private void copyPropertiesBySite(String type) {
        if (this.context.getCluster() != null) {
            Map configs = this.context.getCluster().getConfigByType(type);
            LOG.debug("configs from core-site : {}", (Object)configs);
            this.copyProperties(configs);
        } else {
            LOG.error("Cannot find cluster.");
        }
    }

    private void copyProperties(Map<String, String> configs) {
        if (null != configs) {
            for (Map.Entry<String, String> entry : configs.entrySet()) {
                String key = entry.getKey();
                String value = entry.getValue();
                this.conf.set(key, value);
            }
        } else {
            LOG.error("configs were null.");
        }
    }

    @VisibleForTesting
    void copyHAProperties(String defaultFS) throws URISyntaxException, HdfsApiException {
        URI uri = new URI(defaultFS);
        String nameservice = uri.getHost();
        this.copyClusterProperty(NAMESERVICES_CLUSTER_PROPERTY, NAMESERVICES_INSTANCE_PROPERTY);
        String namenodeIDs = this.copyClusterProperty(String.format(HA_NAMENODES_CLUSTER_PROPERTY, nameservice), HA_NAMENODES_INSTANCE_PROPERTY);
        String[] namenodes = namenodeIDs.split(",");
        ArrayList<String> rpcAddresses = new ArrayList<String>(namenodes.length);
        ArrayList<String> httpAddresses = new ArrayList<String>(namenodes.length);
        ArrayList<String> httpsAddresses = new ArrayList<String>(namenodes.length);
        for (String namenode : namenodes) {
            String httpsAddress;
            String httpAddress;
            String rpcAddress = this.getProperty(HDFS_SITE, String.format(NAMENODE_RPC_NN_CLUSTER_PROPERTY, nameservice, namenode));
            if (!Strings.isNullOrEmpty((String)rpcAddress)) {
                rpcAddresses.add(rpcAddress);
            }
            if (!Strings.isNullOrEmpty((String)(httpAddress = this.getProperty(HDFS_SITE, String.format(NAMENODE_HTTP_NN_CLUSTER_PROPERTY, nameservice, namenode))))) {
                httpAddresses.add(httpAddress);
            }
            if (Strings.isNullOrEmpty((String)(httpsAddress = this.getProperty(HDFS_SITE, String.format(NAMENODE_HTTPS_NN_CLUSTER_PROPERTY, nameservice, namenode))))) continue;
            httpsAddresses.add(httpsAddress);
        }
        this.addAddresses(rpcAddresses, NAMENODE_RPC_NN_INSTANCE_PROPERTY);
        this.addAddresses(httpAddresses, NAMENODE_HTTP_NN_INSTANCE_PROPERTY);
        this.addAddresses(httpsAddresses, NAMENODE_HTTPS_NN_INSTANCE_PROPERTY);
        for (int i = 0; i < namenodes.length; ++i) {
            this.conf.set(String.format(NAMENODE_RPC_NN_CLUSTER_PROPERTY, nameservice, namenodes[i]), (String)rpcAddresses.get(i));
            this.conf.set(String.format(NAMENODE_HTTP_NN_CLUSTER_PROPERTY, nameservice, namenodes[i]), (String)httpAddresses.get(i));
            this.conf.set(String.format(NAMENODE_HTTPS_NN_CLUSTER_PROPERTY, nameservice, namenodes[i]), (String)httpsAddresses.get(i));
        }
        this.copyClusterProperty(String.format(FAILOVER_PROXY_PROVIDER_CLUSTER_PROPERTY, nameservice), FAILOVER_PROXY_PROVIDER_INSTANCE_PROPERTY);
    }

    private void addAddresses(List<String> addressList, String propertyName) {
        if (addressList.isEmpty()) {
            String addressString = this.getViewProperty(propertyName);
            LOG.debug("value of {} in view is : {}", (Object)propertyName, (Object)addressString);
            if (!Strings.isNullOrEmpty((String)addressString)) {
                addressList.addAll(Arrays.asList(addressString.split(",")));
            }
        }
    }

    private String copyClusterProperty(String propertyName, String instancePropertyName) {
        String value = this.getProperty(HDFS_SITE, propertyName, instancePropertyName);
        if (!StringUtils.isEmpty((CharSequence)value)) {
            this.conf.set(propertyName, value);
        }
        LOG.debug("set " + propertyName + " = " + value);
        return value;
    }

    private boolean isHAEnabled(String defaultFS) throws URISyntaxException {
        URI uri = new URI(defaultFS);
        String nameservice = uri.getHost();
        String namenodeIDs = this.getProperty(HDFS_SITE, String.format(HA_NAMENODES_CLUSTER_PROPERTY, nameservice), HA_NAMENODES_INSTANCE_PROPERTY);
        LOG.debug("namenodeIDs " + namenodeIDs);
        return !StringUtils.isEmpty((CharSequence)namenodeIDs);
    }

    private static boolean hasPort(String url) throws URISyntaxException {
        URI uri = new URI(url);
        return uri.getPort() != -1;
    }

    protected static String addPortIfMissing(String defaultFs) throws URISyntaxException {
        if (!ConfigurationBuilder.hasPort(defaultFs)) {
            defaultFs = defaultFs + ":50070";
        }
        return defaultFs;
    }

    protected static String addProtocolIfMissing(String defaultFs) {
        if (!defaultFs.matches("^[^:]+://.*$")) {
            defaultFs = "webhdfs://" + defaultFs;
        }
        return defaultFs;
    }

    public void setAuthParams(Map<String, String> authParams) {
        String auth = authParams.get("auth");
        if (auth != null) {
            this.conf.set("hadoop.security.authentication", auth);
        }
    }

    public Configuration buildConfig() throws HdfsApiException {
        this.copyPropertiesBySite(CORE_SITE);
        this.copyPropertiesBySite(HDFS_SITE);
        this.parseProperties();
        this.setAuthParams(this.buildAuthenticationConfig());
        String umask = this.getViewProperty(UMASK_INSTANCE_PROPERTY);
        if (umask != null && !umask.isEmpty()) {
            this.conf.set(UMASK_CLUSTER_PROPERTY, umask);
        }
        if (null != this.customProperties) {
            this.copyProperties(this.customProperties);
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("final conf : {}", (Object)this.printConf());
        }
        return this.conf;
    }

    private String printConf() {
        try {
            StringWriter stringWriter = new StringWriter();
            this.conf.writeXml((Writer)stringWriter);
            stringWriter.close();
            return stringWriter.toString().replace("\n", "");
        }
        catch (IOException e) {
            LOG.error("error while converting conf to xml : ", (Throwable)e);
            return "";
        }
    }

    public Map<String, String> buildAuthenticationConfig() throws HdfsApiException {
        if (this.authParams == null) {
            this.authParams = this.authParamsBuilder.build();
        }
        return this.authParams;
    }
}

