/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ambari.view.utils.hdfs;

import com.google.common.base.Strings;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.security.PrivilegedExceptionAction;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import org.apache.ambari.view.utils.hdfs.ConfigurationBuilder;
import org.apache.ambari.view.utils.hdfs.DirListInfo;
import org.apache.ambari.view.utils.hdfs.DirStatus;
import org.apache.ambari.view.utils.hdfs.HdfsApiException;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.fs.FsStatus;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.Trash;
import org.apache.hadoop.fs.TrashPolicy;
import org.apache.hadoop.fs.permission.FsAction;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.hdfs.protocol.HdfsFileStatus;
import org.apache.hadoop.security.UserGroupInformation;
import org.json.simple.JSONArray;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HdfsApi {
    private static final Logger LOG = LoggerFactory.getLogger(HdfsApi.class);
    public static String KeyIsErasureCoded = "isErasureCoded";
    public static String KeyIsEncrypted = "isEncrypted";
    public static String KeyErasureCodingPolicyName = "erasureCodingPolicyName";
    private final Configuration conf;
    private Map<String, String> authParams;
    private FileSystem fs;
    private UserGroupInformation ugi;

    public HdfsApi(ConfigurationBuilder configurationBuilder, String username) throws IOException, InterruptedException, HdfsApiException {
        this.authParams = configurationBuilder.buildAuthenticationConfig();
        this.conf = configurationBuilder.buildConfig();
        UserGroupInformation.setConfiguration((Configuration)this.conf);
        this.ugi = UserGroupInformation.createProxyUser((String)username, (UserGroupInformation)this.getProxyUser());
        this.fs = this.execute(new PrivilegedExceptionAction<FileSystem>(){

            @Override
            public FileSystem run() throws IOException {
                return FileSystem.get((Configuration)HdfsApi.this.conf);
            }
        });
    }

    HdfsApi(Configuration configuration, FileSystem fs, UserGroupInformation ugi) throws IOException, InterruptedException, HdfsApiException {
        this.conf = null != configuration ? configuration : new Configuration();
        UserGroupInformation.setConfiguration((Configuration)this.conf);
        this.ugi = null != ugi ? ugi : UserGroupInformation.getCurrentUser();
        this.fs = null != fs ? fs : this.execute(new PrivilegedExceptionAction<FileSystem>(){

            @Override
            public FileSystem run() throws IOException {
                return FileSystem.get((Configuration)HdfsApi.this.conf);
            }
        });
    }

    private UserGroupInformation getProxyUser() throws IOException {
        UserGroupInformation proxyuser = this.authParams.containsKey("proxyuser") ? UserGroupInformation.createRemoteUser((String)this.authParams.get("proxyuser")) : UserGroupInformation.getCurrentUser();
        proxyuser.setAuthenticationMethod(this.getAuthenticationMethod());
        return proxyuser;
    }

    private UserGroupInformation.AuthenticationMethod getAuthenticationMethod() {
        UserGroupInformation.AuthenticationMethod authMethod;
        if (this.authParams.containsKey("auth")) {
            String authName = this.authParams.get("auth");
            authMethod = UserGroupInformation.AuthenticationMethod.valueOf((String)authName.toUpperCase());
        } else {
            authMethod = UserGroupInformation.AuthenticationMethod.SIMPLE;
        }
        return authMethod;
    }

    public FileStatus[] listdir(final String path) throws FileNotFoundException, IOException, InterruptedException {
        return this.execute(new PrivilegedExceptionAction<FileStatus[]>(){

            @Override
            public FileStatus[] run() throws FileNotFoundException, Exception {
                return HdfsApi.this.fs.listStatus(new Path(path));
            }
        });
    }

    public DirStatus listdir(String path, String nameFilter, int maxAllowedSize) throws FileNotFoundException, IOException, InterruptedException {
        FileStatus[] fileStatuses = this.listdir(path);
        return this.filterAndTruncateDirStatus(nameFilter, maxAllowedSize, fileStatuses);
    }

    public DirStatus filterAndTruncateDirStatus(String nameFilter, int maxAllowedSize, FileStatus[] fileStatuses) {
        if (null == fileStatuses) {
            return new DirStatus(null, new DirListInfo(0, false, 0, nameFilter));
        }
        int originalSize = fileStatuses.length;
        boolean truncated = false;
        if (!Strings.isNullOrEmpty((String)nameFilter)) {
            LinkedList<FileStatus> filteredList = new LinkedList<FileStatus>();
            for (FileStatus fileStatus : fileStatuses) {
                if (maxAllowedSize >= 0 && maxAllowedSize <= filteredList.size()) {
                    truncated = true;
                    break;
                }
                if (!fileStatus.getPath().getName().contains(nameFilter)) continue;
                filteredList.add(fileStatus);
            }
            fileStatuses = filteredList.toArray(new FileStatus[0]);
        }
        if (maxAllowedSize >= 0 && fileStatuses.length > maxAllowedSize) {
            truncated = true;
            fileStatuses = Arrays.copyOf(fileStatuses, maxAllowedSize);
        }
        int finalSize = fileStatuses.length;
        return new DirStatus(fileStatuses, new DirListInfo(originalSize, truncated, finalSize, nameFilter));
    }

    public FileStatus getFileStatus(final String path) throws IOException, FileNotFoundException, InterruptedException {
        return this.execute(new PrivilegedExceptionAction<FileStatus>(){

            @Override
            public FileStatus run() throws FileNotFoundException, IOException {
                return HdfsApi.this.fs.getFileStatus(new Path(path));
            }
        });
    }

    public boolean mkdir(final String path) throws IOException, InterruptedException {
        return this.execute(new PrivilegedExceptionAction<Boolean>(){

            @Override
            public Boolean run() throws Exception {
                return HdfsApi.this.fs.mkdirs(new Path(path));
            }
        });
    }

    public boolean rename(final String src, final String dst) throws IOException, InterruptedException {
        return this.execute(new PrivilegedExceptionAction<Boolean>(){

            @Override
            public Boolean run() throws Exception {
                return HdfsApi.this.fs.rename(new Path(src), new Path(dst));
            }
        });
    }

    public boolean trashEnabled() throws Exception {
        return this.execute(new PrivilegedExceptionAction<Boolean>(){

            @Override
            public Boolean run() throws IOException {
                Trash tr = new Trash(HdfsApi.this.fs, HdfsApi.this.conf);
                return tr.isEnabled();
            }
        });
    }

    public Path getHomeDir() throws Exception {
        return this.execute(new PrivilegedExceptionAction<Path>(){

            @Override
            public Path run() throws IOException {
                return HdfsApi.this.fs.getHomeDirectory();
            }
        });
    }

    public synchronized FsStatus getStatus() throws Exception {
        return this.execute(new PrivilegedExceptionAction<FsStatus>(){

            @Override
            public FsStatus run() throws IOException {
                return HdfsApi.this.fs.getStatus();
            }
        });
    }

    public Path getTrashDir() throws Exception {
        return this.execute(new PrivilegedExceptionAction<Path>(){

            @Override
            public Path run() throws IOException {
                TrashPolicy trashPolicy = TrashPolicy.getInstance((Configuration)HdfsApi.this.conf, (FileSystem)HdfsApi.this.fs, (Path)HdfsApi.this.fs.getHomeDirectory());
                return trashPolicy.getCurrentTrashDir().getParent();
            }
        });
    }

    public String getTrashDirPath() throws Exception {
        Path trashDir = this.getTrashDir();
        return trashDir.toUri().getRawPath();
    }

    public String getTrashDirPath(String filePath) throws Exception {
        String trashDirPath = this.getTrashDirPath();
        Path path = new Path(filePath);
        trashDirPath = trashDirPath + "/" + path.getName();
        return trashDirPath;
    }

    public Void emptyTrash() throws Exception {
        return this.execute(new PrivilegedExceptionAction<Void>(){

            @Override
            public Void run() throws IOException {
                Trash tr = new Trash(HdfsApi.this.fs, HdfsApi.this.conf);
                tr.expunge();
                return null;
            }
        });
    }

    public boolean moveToTrash(final String path) throws IOException, InterruptedException {
        return this.execute(new PrivilegedExceptionAction<Boolean>(){

            @Override
            public Boolean run() throws Exception {
                return Trash.moveToAppropriateTrash((FileSystem)HdfsApi.this.fs, (Path)new Path(path), (Configuration)HdfsApi.this.conf);
            }
        });
    }

    public boolean delete(final String path, final boolean recursive) throws IOException, InterruptedException {
        return this.execute(new PrivilegedExceptionAction<Boolean>(){

            @Override
            public Boolean run() throws Exception {
                return HdfsApi.this.fs.delete(new Path(path), recursive);
            }
        });
    }

    public FSDataOutputStream create(final String path, final boolean overwrite) throws IOException, InterruptedException {
        return this.execute(new PrivilegedExceptionAction<FSDataOutputStream>(){

            @Override
            public FSDataOutputStream run() throws Exception {
                return HdfsApi.this.fs.create(new Path(path), overwrite);
            }
        });
    }

    public FSDataInputStream open(final String path) throws IOException, InterruptedException {
        return this.execute(new PrivilegedExceptionAction<FSDataInputStream>(){

            @Override
            public FSDataInputStream run() throws Exception {
                return HdfsApi.this.fs.open(new Path(path));
            }
        });
    }

    public boolean chmod(final String path, final String permissions) throws IOException, InterruptedException {
        return this.execute(new PrivilegedExceptionAction<Boolean>(){

            @Override
            public Boolean run() throws Exception {
                try {
                    HdfsApi.this.fs.setPermission(new Path(path), FsPermission.valueOf((String)permissions));
                }
                catch (Exception ex) {
                    return false;
                }
                return true;
            }
        });
    }

    public void copy(final String src, final String dest) throws IOException, InterruptedException, HdfsApiException {
        boolean result = this.execute(new PrivilegedExceptionAction<Boolean>(){

            @Override
            public Boolean run() throws Exception {
                return FileUtil.copy((FileSystem)HdfsApi.this.fs, (Path)new Path(src), (FileSystem)HdfsApi.this.fs, (Path)new Path(dest), (boolean)false, (boolean)false, (Configuration)HdfsApi.this.conf);
            }
        });
        if (!result) {
            throw new HdfsApiException("HDFS010 Can't copy source file from \" + src + \" to \" + dest");
        }
    }

    public boolean exists(final String newFilePath) throws IOException, InterruptedException {
        return this.execute(new PrivilegedExceptionAction<Boolean>(){

            @Override
            public Boolean run() throws Exception {
                return HdfsApi.this.fs.exists(new Path(newFilePath));
            }
        });
    }

    public <T> T execute(PrivilegedExceptionAction<T> action) throws IOException, InterruptedException {
        return this.execute(action, false);
    }

    public <T> T execute(PrivilegedExceptionAction<T> action, boolean alwaysRetry) throws IOException, InterruptedException {
        Object result = null;
        int tryNumber = 0;
        boolean succeeded = false;
        do {
            ++tryNumber;
            try {
                result = this.ugi.doAs(action);
                succeeded = true;
            }
            catch (IOException ex) {
                if (!Strings.isNullOrEmpty((String)ex.getMessage()) && !ex.getMessage().contains("Cannot obtain block length for")) {
                    throw ex;
                }
                if (tryNumber >= 3) {
                    throw ex;
                }
                LOG.info("HDFS threw 'IOException: Cannot obtain block length' exception. Retrying... Try #" + (tryNumber + 1));
                LOG.error("Retrying: " + ex.getMessage(), (Throwable)ex);
                Thread.sleep(1000L);
            }
        } while (!succeeded);
        return (T)result;
    }

    private static String permissionToString(FsPermission p) {
        return p == null ? "default" : "-" + p.getUserAction().SYMBOL + p.getGroupAction().SYMBOL + p.getOtherAction().SYMBOL;
    }

    public Map<String, Object> fileStatusToJSON(FileStatus status) {
        HdfsFileStatus hdfsFileStatus;
        LinkedHashMap<String, Object> json = new LinkedHashMap<String, Object>();
        json.put("path", Path.getPathWithoutSchemeAndAuthority((Path)status.getPath()).toString());
        json.put("isDirectory", status.isDirectory());
        json.put("len", status.getLen());
        json.put("owner", status.getOwner());
        json.put("group", status.getGroup());
        json.put("permission", HdfsApi.permissionToString(status.getPermission()));
        json.put("accessTime", status.getAccessTime());
        json.put("modificationTime", status.getModificationTime());
        json.put("blockSize", status.getBlockSize());
        json.put("replication", status.getReplication());
        json.put("readAccess", HdfsApi.checkAccessPermissions(status, FsAction.READ, this.ugi));
        json.put("writeAccess", HdfsApi.checkAccessPermissions(status, FsAction.WRITE, this.ugi));
        json.put("executeAccess", HdfsApi.checkAccessPermissions(status, FsAction.EXECUTE, this.ugi));
        json.put(KeyIsErasureCoded, status.isErasureCoded());
        json.put(KeyIsEncrypted, status.isEncrypted());
        if (status instanceof HdfsFileStatus && null != (hdfsFileStatus = (HdfsFileStatus)status).getErasureCodingPolicy()) {
            json.put(KeyErasureCodingPolicyName, hdfsFileStatus.getErasureCodingPolicy().getName());
        }
        return json;
    }

    public JSONArray fileStatusToJSON(FileStatus[] status) {
        JSONArray json = new JSONArray();
        if (status != null) {
            for (FileStatus s : status) {
                json.add(this.fileStatusToJSON(s));
            }
        }
        return json;
    }

    public static boolean checkAccessPermissions(FileStatus stat, FsAction mode, UserGroupInformation ugi) {
        FsPermission perm = stat.getPermission();
        String user = ugi.getShortUserName();
        List<String> groups = Arrays.asList(ugi.getGroupNames());
        return user.equals(stat.getOwner()) ? perm.getUserAction().implies(mode) : (groups.contains(stat.getGroup()) ? perm.getGroupAction().implies(mode) : perm.getOtherAction().implies(mode));
    }
}

