/*
 * Decompiled with CFR 0.152.
 */
package io.airlift.compress;

import com.google.common.base.Preconditions;
import com.google.common.primitives.Bytes;
import com.google.inject.Inject;
import io.airlift.compress.Compressor;
import io.airlift.compress.Decompressor;
import io.airlift.compress.MalformedInputException;
import io.airlift.compress.TestingModule;
import io.airlift.compress.benchmark.DataSet;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.Buffer;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Random;
import java.util.concurrent.ThreadLocalRandom;
import org.assertj.core.api.Assertions;
import org.testng.Assert;
import org.testng.SkipException;
import org.testng.annotations.DataProvider;
import org.testng.annotations.Guice;
import org.testng.annotations.Test;

@Guice(modules={TestingModule.class})
public abstract class AbstractTestCompression {
    private List<DataSet> testCases;

    protected abstract Compressor getCompressor();

    protected abstract Decompressor getDecompressor();

    protected abstract Compressor getVerifyCompressor();

    protected abstract Decompressor getVerifyDecompressor();

    protected boolean isByteBufferSupported() {
        return true;
    }

    @Inject
    public void setup(List<DataSet> dataSets) {
        this.testCases = new ArrayList<DataSet>();
        this.testCases.add(new DataSet("nothing", new byte[0]));
        this.testCases.add(new DataSet("short literal", "hello world!".getBytes(StandardCharsets.UTF_8)));
        this.testCases.add(new DataSet("small copy", "XXXXabcdabcdABCDABCDwxyzwzyz123".getBytes(StandardCharsets.UTF_8)));
        this.testCases.add(new DataSet("long copy", "XXXXabcdefgh abcdefgh abcdefgh abcdefgh abcdefgh abcdefgh ABC".getBytes(StandardCharsets.UTF_8)));
        byte[] data = new byte[256];
        for (int i = 0; i < data.length; ++i) {
            data[i] = (byte)i;
        }
        this.testCases.add(new DataSet("long literal", data));
        this.testCases.addAll(dataSets);
    }

    @Test(dataProvider="data")
    public void testDecompress(DataSet dataSet) throws Exception {
        byte[] uncompressedOriginal = dataSet.getUncompressed();
        byte[] compressed = this.prepareCompressedData(uncompressedOriginal);
        byte[] uncompressed = new byte[uncompressedOriginal.length];
        Decompressor decompressor = this.getDecompressor();
        int uncompressedSize = decompressor.decompress(compressed, 0, compressed.length, uncompressed, 0, uncompressed.length);
        AbstractTestCompression.assertByteArraysEqual(uncompressed, 0, uncompressedSize, uncompressedOriginal, 0, uncompressedOriginal.length);
    }

    @Test(dataProvider="data")
    public void testDecompressWithOutputPadding(DataSet dataSet) {
        int padding = 1021;
        byte[] uncompressedOriginal = dataSet.getUncompressed();
        byte[] compressed = this.prepareCompressedData(uncompressedOriginal);
        byte[] uncompressed = new byte[uncompressedOriginal.length + 2 * padding];
        Decompressor decompressor = this.getDecompressor();
        int uncompressedSize = decompressor.decompress(compressed, 0, compressed.length, uncompressed, padding, uncompressedOriginal.length + padding);
        AbstractTestCompression.assertByteArraysEqual(uncompressed, padding, uncompressedSize, uncompressedOriginal, 0, uncompressedOriginal.length);
    }

    @Test(dataProvider="data")
    public void testDecompressionBufferOverrun(DataSet dataSet) {
        byte[] uncompressedOriginal = dataSet.getUncompressed();
        byte[] compressed = this.prepareCompressedData(uncompressedOriginal);
        byte[] padding = new byte[100];
        ThreadLocalRandom.current().nextBytes(padding);
        byte[] uncompressed = Bytes.concat((byte[][])new byte[][]{new byte[uncompressedOriginal.length], padding});
        Decompressor decompressor = this.getDecompressor();
        int uncompressedSize = decompressor.decompress(compressed, 0, compressed.length, uncompressed, 0, uncompressedOriginal.length);
        AbstractTestCompression.assertByteArraysEqual(uncompressed, 0, uncompressedSize, uncompressedOriginal, 0, uncompressedOriginal.length);
        AbstractTestCompression.assertByteArraysEqual(padding, 0, padding.length, uncompressed, uncompressed.length - padding.length, padding.length);
    }

    @Test
    public void testDecompressInputBoundsChecks() {
        byte[] data = new byte[1024];
        new Random(1234L).nextBytes(data);
        Compressor compressor = this.getCompressor();
        byte[] compressed = new byte[compressor.maxCompressedLength(data.length)];
        int compressedLength = compressor.compress(data, 0, data.length, compressed, 0, compressed.length);
        Decompressor decompressor = this.getDecompressor();
        Assertions.assertThatThrownBy(() -> decompressor.decompress(null, 0, compressedLength, data, 0, data.length)).isInstanceOf(NullPointerException.class);
        byte[] compressedChoppedOff = Arrays.copyOf(compressed, compressedLength - 1);
        Throwable throwable = Assertions.catchThrowable(() -> decompressor.decompress(compressedChoppedOff, 0, compressedLength, data, 0, data.length));
        if (!(throwable instanceof UncheckedIOException)) {
            Assertions.assertThat((Throwable)throwable).hasMessageMatching(".*must not be greater than size.*|Invalid offset or length.*");
        }
        byte[] compressedWithPadding = new byte[10 + compressedLength - 1];
        System.arraycopy(compressed, 0, compressedWithPadding, 10, compressedLength - 1);
        throwable = Assertions.catchThrowable(() -> decompressor.decompress(compressedWithPadding, 10, compressedLength, data, 0, data.length));
        if (!(throwable instanceof UncheckedIOException)) {
            Assertions.assertThat((Throwable)throwable).hasMessageMatching(".*must not be greater than size.*|Invalid offset or length.*");
        }
    }

    @Test
    public void testDecompressOutputBoundsChecks() {
        byte[] data = new byte[1024];
        new Random(1234L).nextBytes(data);
        Compressor compressor = this.getCompressor();
        byte[] compressed = new byte[compressor.maxCompressedLength(data.length)];
        int compressedLength = compressor.compress(data, 0, data.length, compressed, 0, compressed.length);
        byte[] input = Arrays.copyOf(compressed, compressedLength);
        Decompressor decompressor = this.getDecompressor();
        Assertions.assertThatThrownBy(() -> decompressor.decompress(input, 0, input.length, null, 0, data.length)).isInstanceOf(NullPointerException.class);
        Assertions.assertThatThrownBy(() -> decompressor.decompress(input, 0, input.length, new byte[1], 0, 1)).hasMessageMatching("All input was not consumed|attempt to write.* outside of destination buffer.*|Malformed input.*|Uncompressed length 1024 must be less than 1|Output buffer too small.*");
        Throwable throwable = Assertions.catchThrowable(() -> decompressor.decompress(input, 0, input.length, new byte[1], 0, data.length));
        if (!(throwable instanceof IndexOutOfBoundsException)) {
            Assertions.assertThat((Throwable)throwable).hasMessageMatching(".*must not be greater than size.*|Invalid offset or length.*");
        }
        throwable = Assertions.catchThrowable(() -> decompressor.decompress(input, 0, input.length, new byte[data.length - 1], 0, data.length));
        if (!(throwable instanceof IndexOutOfBoundsException)) {
            Assertions.assertThat((Throwable)throwable).hasMessageMatching(".*must not be greater than size.*|Invalid offset or length.*");
        }
    }

    @Test(dataProvider="data")
    public void testDecompressByteBufferHeapToHeap(DataSet dataSet) throws Exception {
        if (!this.isByteBufferSupported()) {
            throw new SkipException("ByteBuffer not supported");
        }
        byte[] uncompressedOriginal = dataSet.getUncompressed();
        ByteBuffer compressed = ByteBuffer.wrap(this.prepareCompressedData(uncompressedOriginal));
        ByteBuffer uncompressed = ByteBuffer.allocate(uncompressedOriginal.length);
        this.getDecompressor().decompress(compressed, uncompressed);
        ((Buffer)uncompressed).flip();
        AbstractTestCompression.assertByteBufferEqual(ByteBuffer.wrap(uncompressedOriginal), uncompressed);
    }

    @Test(dataProvider="data")
    public void testDecompressByteBufferHeapToDirect(DataSet dataSet) throws Exception {
        if (!this.isByteBufferSupported()) {
            throw new SkipException("ByteBuffer not supported");
        }
        byte[] uncompressedOriginal = dataSet.getUncompressed();
        ByteBuffer compressed = ByteBuffer.wrap(this.prepareCompressedData(uncompressedOriginal));
        ByteBuffer uncompressed = ByteBuffer.allocateDirect(uncompressedOriginal.length);
        this.getDecompressor().decompress(compressed, uncompressed);
        ((Buffer)uncompressed).flip();
        AbstractTestCompression.assertByteBufferEqual(ByteBuffer.wrap(uncompressedOriginal), uncompressed);
    }

    @Test(dataProvider="data")
    public void testDecompressByteBufferDirectToHeap(DataSet dataSet) throws Exception {
        if (!this.isByteBufferSupported()) {
            throw new SkipException("ByteBuffer not supported");
        }
        byte[] uncompressedOriginal = dataSet.getUncompressed();
        ByteBuffer compressed = AbstractTestCompression.toDirectBuffer(this.prepareCompressedData(uncompressedOriginal));
        ByteBuffer uncompressed = ByteBuffer.allocate(uncompressedOriginal.length);
        this.getDecompressor().decompress(compressed, uncompressed);
        ((Buffer)uncompressed).flip();
        AbstractTestCompression.assertByteBufferEqual(ByteBuffer.wrap(uncompressedOriginal), uncompressed);
    }

    @Test(dataProvider="data")
    public void testDecompressByteBufferDirectToDirect(DataSet dataSet) throws Exception {
        if (!this.isByteBufferSupported()) {
            throw new SkipException("ByteBuffer not supported");
        }
        byte[] uncompressedOriginal = dataSet.getUncompressed();
        ByteBuffer compressed = AbstractTestCompression.toDirectBuffer(this.prepareCompressedData(uncompressedOriginal));
        ByteBuffer uncompressed = ByteBuffer.allocateDirect(uncompressedOriginal.length);
        this.getDecompressor().decompress(compressed, uncompressed);
        ((Buffer)uncompressed).flip();
        AbstractTestCompression.assertByteBufferEqual(ByteBuffer.wrap(uncompressedOriginal), uncompressed);
    }

    @Test(dataProvider="data")
    public void testCompress(DataSet testCase) throws Exception {
        Compressor compressor = this.getCompressor();
        byte[] originalUncompressed = testCase.getUncompressed();
        byte[] compressed = new byte[compressor.maxCompressedLength(originalUncompressed.length)];
        if (originalUncompressed.length > 1) {
            byte[] output = new byte[compressor.maxCompressedLength(originalUncompressed.length - 1)];
            compressor.compress(originalUncompressed, 1, originalUncompressed.length - 1, output, 0, output.length);
        }
        int compressedLength = compressor.compress(originalUncompressed, 0, originalUncompressed.length, compressed, 0, compressed.length);
        this.verifyCompressedData(originalUncompressed, compressed, compressedLength);
    }

    @Test
    public void testCompressInputBoundsChecks() {
        Compressor compressor = this.getCompressor();
        int declaredInputLength = 1024;
        int maxCompressedLength = compressor.maxCompressedLength(1024);
        byte[] output = new byte[maxCompressedLength];
        Assertions.assertThatThrownBy(() -> compressor.compress(null, 0, declaredInputLength, output, 0, output.length)).isInstanceOf(NullPointerException.class);
        Throwable throwable = Assertions.catchThrowable(() -> compressor.compress(new byte[1], 0, declaredInputLength, output, 0, output.length));
        if (!(throwable instanceof IndexOutOfBoundsException)) {
            Assertions.assertThat((Throwable)throwable).hasMessageMatching(".*must not be greater than size.*|Invalid offset or length.*");
        }
        throwable = Assertions.catchThrowable(() -> compressor.compress(new byte[declaredInputLength - 1], 0, declaredInputLength, output, 0, output.length));
        if (!(throwable instanceof IndexOutOfBoundsException)) {
            Assertions.assertThat((Throwable)throwable).hasMessageMatching(".*must not be greater than size.*|Invalid offset or length.*");
        }
        throwable = Assertions.catchThrowable(() -> compressor.compress(new byte[declaredInputLength + 10], 11, declaredInputLength, output, 0, output.length));
        if (!(throwable instanceof IndexOutOfBoundsException)) {
            Assertions.assertThat((Throwable)throwable).hasMessageMatching(".*must not be greater than size.*|Invalid offset or length.*");
        }
    }

    @Test
    public void testCompressOutputBoundsChecks() {
        Compressor compressor = this.getCompressor();
        int minCompressionOverhead = compressor.maxCompressedLength(0);
        byte[] input = new byte[minCompressionOverhead * 4 + 1024];
        new Random(1234L).nextBytes(input);
        int maxCompressedLength = compressor.maxCompressedLength(input.length);
        Assertions.assertThatThrownBy(() -> compressor.compress(input, 0, input.length, null, 0, maxCompressedLength)).isInstanceOf(NullPointerException.class);
        Assertions.assertThatThrownBy(() -> compressor.compress(input, 0, input.length, new byte[1], 0, 1)).hasMessageMatching(".*must not be greater than size.*|Invalid offset or length.*|Max output length must be larger than .*|Output buffer must be at least.*|Output buffer too small");
        Throwable throwable = Assertions.catchThrowable(() -> compressor.compress(input, 0, input.length, new byte[1], 0, maxCompressedLength));
        if (!(throwable instanceof ArrayIndexOutOfBoundsException)) {
            Assertions.assertThat((Throwable)throwable).hasMessageMatching(".*must not be greater than size.*|Invalid offset or length.*");
        }
        throwable = Assertions.catchThrowable(() -> compressor.compress(input, 0, input.length, new byte[minCompressionOverhead * 2], 0, maxCompressedLength));
        if (!(throwable instanceof ArrayIndexOutOfBoundsException)) {
            Assertions.assertThat((Throwable)throwable).hasMessageMatching(".*must not be greater than size.*|Invalid offset or length.*");
        }
    }

    @Test(dataProvider="data")
    public void testCompressByteBufferHeapToHeap(DataSet dataSet) throws Exception {
        if (!this.isByteBufferSupported()) {
            throw new SkipException("ByteBuffer not supported");
        }
        byte[] uncompressedOriginal = dataSet.getUncompressed();
        Compressor compressor = this.getCompressor();
        this.verifyCompressByteBuffer(compressor, ByteBuffer.wrap(uncompressedOriginal), ByteBuffer.allocate(compressor.maxCompressedLength(uncompressedOriginal.length)));
    }

    @Test(dataProvider="data")
    public void testCompressByteBufferHeapToDirect(DataSet dataSet) throws Exception {
        if (!this.isByteBufferSupported()) {
            throw new SkipException("ByteBuffer not supported");
        }
        byte[] uncompressedOriginal = dataSet.getUncompressed();
        Compressor compressor = this.getCompressor();
        this.verifyCompressByteBuffer(compressor, ByteBuffer.wrap(uncompressedOriginal), ByteBuffer.allocateDirect(compressor.maxCompressedLength(uncompressedOriginal.length)));
    }

    @Test(dataProvider="data")
    public void testCompressByteBufferDirectToHeap(DataSet dataSet) throws Exception {
        if (!this.isByteBufferSupported()) {
            throw new SkipException("ByteBuffer not supported");
        }
        byte[] uncompressedOriginal = dataSet.getUncompressed();
        Compressor compressor = this.getCompressor();
        this.verifyCompressByteBuffer(compressor, AbstractTestCompression.toDirectBuffer(uncompressedOriginal), ByteBuffer.allocate(compressor.maxCompressedLength(uncompressedOriginal.length)));
    }

    @Test(dataProvider="data")
    public void testCompressByteBufferDirectToDirect(DataSet dataSet) throws Exception {
        if (!this.isByteBufferSupported()) {
            throw new SkipException("ByteBuffer not supported");
        }
        byte[] uncompressedOriginal = dataSet.getUncompressed();
        Compressor compressor = this.getCompressor();
        this.verifyCompressByteBuffer(compressor, AbstractTestCompression.toDirectBuffer(uncompressedOriginal), ByteBuffer.allocateDirect(compressor.maxCompressedLength(uncompressedOriginal.length)));
    }

    private void verifyCompressByteBuffer(Compressor compressor, ByteBuffer expected, ByteBuffer compressed) {
        if (expected.remaining() > 1) {
            ByteBuffer duplicate = expected.duplicate();
            duplicate.get();
            compressor.compress(duplicate, ByteBuffer.allocate(compressed.remaining()));
        }
        compressor.compress(expected.duplicate(), compressed);
        ((Buffer)compressed).flip();
        ByteBuffer uncompressed = ByteBuffer.allocate(expected.remaining());
        this.getDecompressor().decompress(compressed, uncompressed);
        ((Buffer)uncompressed).flip();
        AbstractTestCompression.assertByteBufferEqual(expected.duplicate(), uncompressed);
    }

    private void verifyCompressedData(byte[] originalUncompressed, byte[] compressed, int compressedLength) {
        byte[] uncompressed = new byte[originalUncompressed.length];
        int uncompressedSize = this.getVerifyDecompressor().decompress(compressed, 0, compressedLength, uncompressed, 0, uncompressed.length);
        AbstractTestCompression.assertByteArraysEqual(uncompressed, 0, uncompressedSize, originalUncompressed, 0, originalUncompressed.length);
    }

    @Test
    public void testRoundTripSmallLiteral() throws Exception {
        byte[] data = new byte[256];
        for (int i = 0; i < data.length; ++i) {
            data[i] = (byte)i;
        }
        Compressor compressor = this.getCompressor();
        byte[] compressed = new byte[compressor.maxCompressedLength(data.length)];
        byte[] uncompressed = new byte[data.length];
        for (int i = 1; i < data.length; ++i) {
            try {
                int written = compressor.compress(data, 0, i, compressed, 0, compressed.length);
                int decompressedSize = this.getDecompressor().decompress(compressed, 0, written, uncompressed, 0, uncompressed.length);
                AbstractTestCompression.assertByteArraysEqual(data, 0, i, uncompressed, 0, decompressedSize);
                Assert.assertEquals((int)decompressedSize, (int)i);
                continue;
            }
            catch (MalformedInputException e) {
                throw new RuntimeException("Failed with " + i + " bytes of input", e);
            }
        }
    }

    @DataProvider(name="data")
    public Object[][] getTestCases() throws IOException {
        Object[][] result = new Object[this.testCases.size()][];
        for (int i = 0; i < this.testCases.size(); ++i) {
            result[i] = new Object[]{this.testCases.get(i)};
        }
        return result;
    }

    public static void assertByteArraysEqual(byte[] left, int leftOffset, int leftLength, byte[] right, int rightOffset, int rightLength) {
        Preconditions.checkPositionIndexes((int)leftOffset, (int)(leftOffset + leftLength), (int)left.length);
        Preconditions.checkPositionIndexes((int)rightOffset, (int)(rightOffset + rightLength), (int)right.length);
        for (int i = 0; i < Math.min(leftLength, rightLength); ++i) {
            if (left[leftOffset + i] == right[rightOffset + i]) continue;
            Assert.fail((String)String.format("Byte arrays differ at position %s: 0x%02X vs 0x%02X", i, left[leftOffset + i], right[rightOffset + i]));
        }
        Assert.assertEquals((int)leftLength, (int)rightLength, (String)String.format("Array lengths differ: %s vs %s", leftLength, rightLength));
    }

    private static void assertByteBufferEqual(ByteBuffer left, ByteBuffer right) {
        ByteBuffer leftBuffer = left;
        ByteBuffer rightBuffer = right;
        int leftPosition = leftBuffer.position();
        int rightPosition = rightBuffer.position();
        for (int i = 0; i < Math.min(leftBuffer.remaining(), rightBuffer.remaining()); ++i) {
            if (left.get(leftPosition + i) == right.get(rightPosition + i)) continue;
            Assert.fail((String)String.format("Byte buffers differ at position %s: 0x%02X vs 0x%02X", i, left.get(leftPosition + i), right.get(rightPosition + i)));
        }
        Assert.assertEquals((int)leftBuffer.remaining(), (int)rightBuffer.remaining(), (String)String.format("Buffer lengths differ: %s vs %s", leftBuffer.remaining(), leftBuffer.remaining()));
    }

    private static ByteBuffer toDirectBuffer(byte[] data) {
        ByteBuffer direct = ByteBuffer.allocateDirect(data.length);
        direct.put(data);
        ((Buffer)direct).flip();
        return direct;
    }

    private byte[] prepareCompressedData(byte[] uncompressed) {
        Compressor compressor = this.getVerifyCompressor();
        byte[] compressed = new byte[compressor.maxCompressedLength(uncompressed.length)];
        int compressedLength = compressor.compress(uncompressed, 0, uncompressed.length, compressed, 0, compressed.length);
        return Arrays.copyOf(compressed, compressedLength);
    }
}

