/*
 * Decompiled with CFR 0.152.
 */
package id.onyx.obdp.server.topology;

import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AsyncCallableService<T>
implements Callable<T> {
    private static final Logger LOG = LoggerFactory.getLogger(AsyncCallableService.class);
    private final ScheduledExecutorService executorService;
    private final Callable<T> task;
    private final String taskName;
    private final long timeout;
    private final long retryDelay;
    private final Consumer<Throwable> onError;

    public AsyncCallableService(Callable<T> task, long timeout, long retryDelay, String taskName, Consumer<Throwable> onError) {
        this(task, timeout, retryDelay, taskName, Executors.newScheduledThreadPool(1), onError);
    }

    public AsyncCallableService(Callable<T> task, long timeout, long retryDelay, String taskName, ScheduledExecutorService executorService, Consumer<Throwable> onError) {
        Preconditions.checkArgument((retryDelay > 0L ? 1 : 0) != 0, (Object)"retryDelay should be positive");
        this.task = task;
        this.executorService = executorService;
        this.timeout = timeout;
        this.retryDelay = retryDelay;
        this.taskName = taskName;
        this.onError = onError;
    }

    @Override
    public T call() throws Exception {
        long startTime = System.currentTimeMillis();
        long timeLeft = this.timeout;
        Future<T> future = this.executorService.submit(this.task);
        LOG.info("Task {} execution started at {}", (Object)this.taskName, (Object)startTime);
        Throwable lastError = null;
        while (true) {
            try {
                LOG.debug("Task {} waiting for result at most {} ms", (Object)this.taskName, (Object)timeLeft);
                T taskResult = future.get(timeLeft, TimeUnit.MILLISECONDS);
                LOG.info("Task {} successfully completed with result: {}", (Object)this.taskName, taskResult);
                return taskResult;
            }
            catch (TimeoutException e) {
                LOG.debug("Task {} timeout", (Object)this.taskName);
                if (lastError == null) {
                    lastError = e;
                }
                timeLeft = 0L;
            }
            catch (ExecutionException e) {
                Throwable cause = Throwables.getRootCause((Throwable)e);
                if (!(cause instanceof RetryTaskSilently)) {
                    LOG.info(String.format("Task %s exception during execution", this.taskName), cause);
                }
                lastError = cause;
                timeLeft = this.timeout - (System.currentTimeMillis() - startTime) - this.retryDelay;
            }
            if (timeLeft <= 0L) {
                this.attemptToCancel(future);
                LOG.warn("Task {} timeout exceeded, no more retries", (Object)this.taskName);
                this.onError.accept(lastError);
                return null;
            }
            LOG.debug("Task {} retrying execution in {} milliseconds", (Object)this.taskName, (Object)this.retryDelay);
            future = this.executorService.schedule(this.task, this.retryDelay, TimeUnit.MILLISECONDS);
        }
    }

    private void attemptToCancel(Future<?> future) {
        LOG.debug("Task {} timeout exceeded, cancelling", (Object)this.taskName);
        if (!future.isDone() && future.cancel(true)) {
            LOG.debug("Task {} cancelled", (Object)this.taskName);
        } else {
            LOG.debug("Task {} already done", (Object)this.taskName);
        }
    }

    public static class RetryTaskSilently
    extends RuntimeException {
        public RetryTaskSilently() {
        }

        public RetryTaskSilently(String message) {
            super(message);
        }
    }
}

