/*
 * Decompiled with CFR 0.152.
 */
package id.onyx.obdp.server.state.services;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.MapperFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.Sets;
import com.google.common.util.concurrent.AbstractService;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;
import com.google.gson.stream.JsonReader;
import com.google.inject.Inject;
import id.onyx.obdp.server.OBDPService;
import id.onyx.obdp.server.configuration.Configuration;
import id.onyx.obdp.server.controller.jmx.JMXMetricHolder;
import id.onyx.obdp.server.controller.utilities.ScalingThreadPoolExecutor;
import id.onyx.obdp.server.controller.utilities.StreamProvider;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.lang.reflect.Type;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@OBDPService
public class MetricsRetrievalService
extends AbstractService {
    private static final Logger LOG = LoggerFactory.getLogger(MetricsRetrievalService.class);
    private static final int EXCEPTION_CACHE_TIMEOUT_MINUTES = 20;
    private static final Cache<String, Throwable> s_exceptionCache = CacheBuilder.newBuilder().expireAfterWrite(20L, TimeUnit.MINUTES).build();
    @Inject
    private Configuration m_configuration;
    @Inject
    private Gson m_gson;
    private Cache<String, JMXMetricHolder> m_jmxCache;
    private Cache<String, Map<String, String>> m_restCache;
    private ThreadPoolExecutor m_threadPoolExecutor;
    private final ObjectReader m_jmxObjectReader;
    private final Set<String> m_queuedUrls = Sets.newConcurrentHashSet();
    private Cache<String, String> m_ttlUrlCache;
    private int m_queueMaximumSize;

    public MetricsRetrievalService() {
        ObjectMapper jmxObjectMapper = new ObjectMapper();
        jmxObjectMapper.configure(MapperFeature.USE_ANNOTATIONS, false);
        jmxObjectMapper.configure(JsonParser.Feature.ALLOW_NON_NUMERIC_NUMBERS, true);
        this.m_jmxObjectReader = jmxObjectMapper.reader(JMXMetricHolder.class);
    }

    protected void doStart() {
        int jmxCacheExpirationMinutes = this.m_configuration.getMetricsServiceCacheTimeout();
        this.m_jmxCache = CacheBuilder.newBuilder().expireAfterWrite((long)jmxCacheExpirationMinutes, TimeUnit.MINUTES).build();
        this.m_restCache = CacheBuilder.newBuilder().expireAfterWrite((long)jmxCacheExpirationMinutes, TimeUnit.MINUTES).build();
        int ttlSeconds = this.m_configuration.getMetricsServiceRequestTTL();
        boolean ttlCacheEnabled = this.m_configuration.isMetricsServiceRequestTTLCacheEnabled();
        if (ttlCacheEnabled) {
            this.m_ttlUrlCache = CacheBuilder.newBuilder().expireAfterWrite((long)ttlSeconds, TimeUnit.SECONDS).build();
        }
        int corePoolSize = this.m_configuration.getMetricsServiceThreadPoolCoreSize();
        int maxPoolSize = this.m_configuration.getMetricsServiceThreadPoolMaxSize();
        this.m_queueMaximumSize = this.m_configuration.getMetricsServiceWorkerQueueSize();
        int threadPriority = this.m_configuration.getMetricsServiceThreadPriority();
        this.m_threadPoolExecutor = new ScalingThreadPoolExecutor(corePoolSize, maxPoolSize, 30L, TimeUnit.SECONDS, this.m_queueMaximumSize);
        this.m_threadPoolExecutor.allowCoreThreadTimeOut(true);
        this.m_threadPoolExecutor.setRejectedExecutionHandler(new ThreadPoolExecutor.DiscardOldestPolicy());
        ThreadFactory threadFactory = new ThreadFactoryBuilder().setDaemon(true).setNameFormat("obdp-metrics-retrieval-service-thread-%d").setPriority(threadPriority).setUncaughtExceptionHandler((Thread.UncaughtExceptionHandler)new MetricRunnableExceptionHandler()).build();
        this.m_threadPoolExecutor.setThreadFactory(threadFactory);
        LOG.info("Initializing the Metrics Retrieval Service with core={}, max={}, workerQueue={}, threadPriority={}", new Object[]{corePoolSize, maxPoolSize, this.m_queueMaximumSize, threadPriority});
        if (ttlCacheEnabled) {
            LOG.info("Metrics Retrieval Service request TTL cache is enabled and set to {} seconds", (Object)ttlSeconds);
        }
        this.notifyStarted();
    }

    public void setThreadPoolExecutor(ThreadPoolExecutor threadPoolExecutor) {
        this.m_threadPoolExecutor = threadPoolExecutor;
    }

    protected void doStop() {
        this.m_jmxCache.invalidateAll();
        this.m_restCache.invalidateAll();
        if (null != this.m_ttlUrlCache) {
            this.m_ttlUrlCache.invalidateAll();
        }
        this.m_queuedUrls.clear();
        this.m_threadPoolExecutor.shutdownNow();
        this.notifyStopped();
    }

    public void submitRequest(MetricSourceType type, StreamProvider streamProvider, String url) {
        if (this.m_queuedUrls.contains(url)) {
            return;
        }
        if (null != this.m_ttlUrlCache && null != this.m_ttlUrlCache.getIfPresent((Object)url)) {
            return;
        }
        BlockingQueue<Runnable> queue = this.m_threadPoolExecutor.getQueue();
        int queueSize = queue.size();
        if ((double)queueSize > Math.floor(0.9f * (float)this.m_queueMaximumSize)) {
            LOG.warn("The worker queue contains {} work items and is at {}% of capacity", (Object)queueSize, (Object)Float.valueOf((float)queueSize / (float)this.m_queueMaximumSize * 100.0f));
        }
        this.m_queuedUrls.add(url);
        MetricRunnable runnable = null;
        switch (type) {
            case JMX: {
                runnable = new JMXRunnable(this.m_jmxCache, this.m_queuedUrls, this.m_ttlUrlCache, this.m_jmxObjectReader, streamProvider, url);
                break;
            }
            case REST: {
                runnable = new RESTRunnable(this.m_restCache, this.m_queuedUrls, this.m_ttlUrlCache, this.m_gson, streamProvider, url);
                break;
            }
            default: {
                LOG.warn("Unable to retrieve metrics for the unknown type {}", (Object)type);
            }
        }
        if (null != runnable) {
            this.m_threadPoolExecutor.execute(runnable);
        }
    }

    public JMXMetricHolder getCachedJMXMetric(String jmxUrl) {
        return (JMXMetricHolder)this.m_jmxCache.getIfPresent((Object)jmxUrl);
    }

    public Map<String, String> getCachedRESTMetric(String restUrl) {
        return (Map)this.m_restCache.getIfPresent((Object)restUrl);
    }

    private static final class MetricRunnableExceptionHandler
    implements Thread.UncaughtExceptionHandler {
        private MetricRunnableExceptionHandler() {
        }

        @Override
        public void uncaughtException(Thread t, Throwable e) {
            LOG.error("Asynchronous metric retrieval encountered an exception with thread {}", (Object)t, (Object)e);
        }
    }

    public static enum MetricSourceType {
        JMX,
        REST;

    }

    private static final class JMXRunnable
    extends MetricRunnable {
        private final ObjectReader m_jmxObjectReader;
        private final Cache<String, JMXMetricHolder> m_cache;

        private JMXRunnable(Cache<String, JMXMetricHolder> cache, Set<String> queuedUrls, Cache<String, String> ttlUrlCache, ObjectReader jmxObjectReader, StreamProvider streamProvider, String jmxUrl) {
            super(streamProvider, jmxUrl, queuedUrls, ttlUrlCache);
            this.m_cache = cache;
            this.m_jmxObjectReader = jmxObjectReader;
        }

        @Override
        protected void removeCachedMetricsForCurrentURL() {
            this.m_cache.invalidate((Object)this.m_url);
        }

        @Override
        protected void processInputStreamAndCacheResult(InputStream inputStream) throws Exception {
            JMXMetricHolder jmxMetricHolder = (JMXMetricHolder)this.m_jmxObjectReader.readValue(inputStream);
            this.m_cache.put((Object)this.m_url, (Object)jmxMetricHolder);
        }
    }

    private static final class RESTRunnable
    extends MetricRunnable {
        private final Gson m_gson;
        private final Cache<String, Map<String, String>> m_cache;

        private RESTRunnable(Cache<String, Map<String, String>> cache, Set<String> queuedUrls, Cache<String, String> ttlUrlCache, Gson gson, StreamProvider streamProvider, String restUrl) {
            super(streamProvider, restUrl, queuedUrls, ttlUrlCache);
            this.m_cache = cache;
            this.m_gson = gson;
        }

        @Override
        protected void removeCachedMetricsForCurrentURL() {
            this.m_cache.invalidate((Object)this.m_url);
        }

        @Override
        protected void processInputStreamAndCacheResult(InputStream inputStream) throws Exception {
            Type type = new TypeToken<Map<Object, Object>>(){}.getType();
            JsonReader jsonReader = new JsonReader((Reader)new BufferedReader(new InputStreamReader(inputStream)));
            Map jsonMap = (Map)this.m_gson.fromJson(jsonReader, type);
            this.m_cache.put((Object)this.m_url, (Object)jsonMap);
        }
    }

    private static abstract class MetricRunnable
    implements Runnable {
        protected final StreamProvider m_streamProvider;
        protected final String m_url;
        private final Set<String> m_queuedUrls;
        private final Cache<String, String> m_ttlUrlCache;

        private MetricRunnable(StreamProvider streamProvider, String url, Set<String> queuedUrls, Cache<String, String> ttlUrlCache) {
            this.m_streamProvider = streamProvider;
            this.m_url = url;
            this.m_queuedUrls = queuedUrls;
            this.m_ttlUrlCache = ttlUrlCache;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         * Loose catch block
         */
        @Override
        public final void run() {
            InputStream inputStream;
            block8: {
                long startTime = 0L;
                long endTime = 0L;
                boolean isDebugEnabled = LOG.isDebugEnabled();
                if (isDebugEnabled) {
                    startTime = System.currentTimeMillis();
                }
                inputStream = null;
                try {
                    if (isDebugEnabled) {
                        endTime = System.currentTimeMillis();
                        LOG.debug("Loading metric JSON from {} took {}ms", (Object)this.m_url, (Object)(endTime - startTime));
                    }
                    inputStream = this.m_streamProvider.readFrom(this.m_url);
                    this.processInputStreamAndCacheResult(inputStream);
                    if (null == this.m_ttlUrlCache) break block8;
                    this.m_ttlUrlCache.put((Object)this.m_url, (Object)this.m_url);
                }
                catch (IOException exception) {
                    LOG.debug("Removing cached values for url {}", (Object)this.m_url);
                    this.removeCachedMetricsForCurrentURL();
                    this.logException(exception, this.m_url);
                    IOUtils.closeQuietly((InputStream)inputStream);
                    this.m_queuedUrls.remove(this.m_url);
                }
                catch (Exception exception2) {
                    this.logException(exception2, this.m_url);
                    {
                        catch (Throwable throwable) {
                            IOUtils.closeQuietly(inputStream);
                            this.m_queuedUrls.remove(this.m_url);
                            throw throwable;
                        }
                    }
                    IOUtils.closeQuietly((InputStream)inputStream);
                    this.m_queuedUrls.remove(this.m_url);
                }
            }
            IOUtils.closeQuietly((InputStream)inputStream);
            this.m_queuedUrls.remove(this.m_url);
        }

        protected abstract void removeCachedMetricsForCurrentURL();

        protected abstract void processInputStreamAndCacheResult(InputStream var1) throws Exception;

        final void logException(Throwable throwable, String url) {
            String cacheKey = this.buildCacheKey(throwable, url);
            if (null == s_exceptionCache.getIfPresent((Object)cacheKey)) {
                s_exceptionCache.put((Object)cacheKey, (Object)throwable);
                LOG.error("Unable to retrieve metrics from {}. Subsequent failures will be suppressed from the log for {} minutes.", new Object[]{url, 20, throwable});
            }
        }

        private String buildCacheKey(Throwable throwable, String url) {
            if (null == throwable || null == url) {
                return "";
            }
            String throwableName = throwable.getClass().getSimpleName();
            return throwableName + "-" + url;
        }
    }
}

