/*
 * Decompiled with CFR 0.152.
 */
package id.onyx.obdp.server.stack;

import com.google.common.collect.Lists;
import com.google.common.reflect.TypeToken;
import id.onyx.obdp.server.OBDPException;
import id.onyx.obdp.server.orm.entities.RepositoryVersionEntity;
import id.onyx.obdp.server.stack.ClassifyNameNodeException;
import id.onyx.obdp.server.stack.HostsType;
import id.onyx.obdp.server.stack.NameService;
import id.onyx.obdp.server.stack.upgrade.Direction;
import id.onyx.obdp.server.stack.upgrade.ExecuteHostType;
import id.onyx.obdp.server.stack.upgrade.orchestrate.UpgradeContext;
import id.onyx.obdp.server.state.Cluster;
import id.onyx.obdp.server.state.ConfigHelper;
import id.onyx.obdp.server.state.Host;
import id.onyx.obdp.server.state.MaintenanceState;
import id.onyx.obdp.server.state.ServiceComponent;
import id.onyx.obdp.server.state.ServiceComponentHost;
import id.onyx.obdp.server.state.UpgradeState;
import id.onyx.obdp.server.utils.HTTPUtils;
import id.onyx.obdp.server.utils.HostAndPort;
import id.onyx.obdp.server.utils.StageUtils;
import java.lang.reflect.Type;
import java.net.MalformedURLException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MasterHostResolver {
    private static final Logger LOG = LoggerFactory.getLogger(MasterHostResolver.class);
    private final UpgradeContext m_upgradeContext;
    private final Cluster m_cluster;
    private final ConfigHelper m_configHelper;

    public MasterHostResolver(Cluster cluster, ConfigHelper configHelper, UpgradeContext upgradeContext) {
        this.m_configHelper = configHelper;
        this.m_upgradeContext = upgradeContext;
        this.m_cluster = cluster;
    }

    public Cluster getCluster() {
        return this.m_cluster;
    }

    public HostsType getMasterAndHosts(String serviceName, String componentName) {
        if (serviceName == null || componentName == null) {
            return null;
        }
        LinkedHashSet<String> componentHosts = new LinkedHashSet<String>(this.m_cluster.getHosts(serviceName, componentName));
        if (componentHosts.isEmpty()) {
            return null;
        }
        HostsType hostsType = HostsType.normal(componentHosts);
        try {
            switch (Service.fromString(serviceName)) {
                case HDFS: {
                    if (!componentName.equalsIgnoreCase("NAMENODE") || componentHosts.size() < 2) break;
                    try {
                        hostsType = HostsType.federated(this.nameSpaces(componentHosts), componentHosts);
                    }
                    catch (ClassifyNameNodeException | IllegalArgumentException e) {
                        if (componentHosts.size() == 2) {
                            hostsType = HostsType.guessHighAvailability(componentHosts);
                            LOG.warn("Could not determine the active/standby states from NameNodes {}. Using {} as active and {} as standbys.", new Object[]{componentHosts, hostsType.getMasters(), hostsType.getSecondaries()});
                            break;
                        }
                        LOG.warn("Could not determine the active/standby states of federated NameNode from NameNodes {}.", componentHosts);
                    }
                    break;
                }
                case YARN: {
                    if (!componentName.equalsIgnoreCase("RESOURCEMANAGER")) break;
                    hostsType = this.resolveResourceManagers(this.getCluster(), componentHosts);
                    break;
                }
                case HBASE: {
                    if (!componentName.equalsIgnoreCase("HBASE_MASTER")) break;
                    hostsType = this.resolveHBaseMasters(this.getCluster(), componentHosts);
                    break;
                }
            }
        }
        catch (Exception err) {
            LOG.error("Unable to get master and hosts for Component " + componentName + ". Error: " + err.getMessage(), (Throwable)err);
        }
        return this.filterHosts(hostsType, serviceName, componentName);
    }

    public static Collection<Host> getCandidateHosts(Cluster cluster, ExecuteHostType executeHostType, String serviceName, String componentName) {
        Collection candidates = cluster.getHosts();
        if (StringUtils.isNotBlank((String)serviceName) && StringUtils.isNotBlank((String)componentName)) {
            List<ServiceComponentHost> schs = cluster.getServiceComponentHosts(serviceName, componentName);
            candidates = schs.stream().map(sch -> sch.getHost()).collect(Collectors.toList());
        }
        if (candidates.isEmpty()) {
            return candidates;
        }
        ArrayList winners = Lists.newArrayList();
        switch (executeHostType) {
            case ALL: {
                winners.addAll(candidates);
                break;
            }
            case FIRST: {
                winners.add(candidates.iterator().next());
                break;
            }
            case MASTER: {
                winners.add(candidates.iterator().next());
                break;
            }
            case ANY: {
                winners.add(candidates.iterator().next());
            }
        }
        return winners;
    }

    private HostsType filterHosts(HostsType hostsType, String service, String component) {
        try {
            id.onyx.obdp.server.state.Service svc = this.m_cluster.getService(service);
            ServiceComponent sc = svc.getServiceComponent(component);
            ArrayList<ServiceComponentHost> unhealthyHosts = new ArrayList<ServiceComponentHost>();
            LinkedHashSet<String> upgradeHosts = new LinkedHashSet<String>();
            for (String hostName : hostsType.getHosts()) {
                ServiceComponentHost sch = sc.getServiceComponentHost(hostName);
                Host host = sch.getHost();
                MaintenanceState maintenanceState = host.getMaintenanceState(sch.getClusterId());
                if (maintenanceState != MaintenanceState.OFF) {
                    unhealthyHosts.add(sch);
                    continue;
                }
                if (sch.getUpgradeState() == UpgradeState.FAILED) {
                    upgradeHosts.add(hostName);
                    continue;
                }
                if (this.m_upgradeContext.getDirection() == Direction.UPGRADE) {
                    RepositoryVersionEntity targetRepositoryVersion = this.m_upgradeContext.getRepositoryVersion();
                    if (StringUtils.equals((String)targetRepositoryVersion.getVersion(), (String)sch.getVersion())) continue;
                    upgradeHosts.add(hostName);
                    continue;
                }
                RepositoryVersionEntity downgradeToRepositoryVersion = this.m_upgradeContext.getTargetRepositoryVersion(service);
                String downgradeToVersion = downgradeToRepositoryVersion.getVersion();
                if (StringUtils.equals((String)downgradeToVersion, (String)sch.getVersion())) continue;
                upgradeHosts.add(hostName);
            }
            hostsType.unhealthy = unhealthyHosts;
            hostsType.setHosts(upgradeHosts);
            return hostsType;
        }
        catch (OBDPException e) {
            LOG.warn("Could not determine host components to upgrade. Defaulting to saved hosts.", (Throwable)e);
            return hostsType;
        }
    }

    public boolean isNameNodeHA() throws OBDPException {
        Map<String, id.onyx.obdp.server.state.Service> services = this.m_cluster.getServices();
        if (services != null && services.containsKey("HDFS")) {
            Set<String> secondaryNameNodeHosts = this.m_cluster.getHosts("HDFS", "SECONDARY_NAMENODE");
            Set<String> nameNodeHosts = this.m_cluster.getHosts("HDFS", "NAMENODE");
            if (secondaryNameNodeHosts.size() == 1 && nameNodeHosts.size() == 1) {
                return false;
            }
            if (nameNodeHosts.size() > 1) {
                return true;
            }
            throw new OBDPException("Unable to determine if cluster has NameNode HA.");
        }
        return false;
    }

    private List<HostsType.HighAvailabilityHosts> nameSpaces(Set<String> componentHosts) {
        return NameService.fromConfig(this.m_configHelper, this.getCluster()).stream().map(each -> this.findMasterAndSecondaries((NameService)each, componentHosts)).collect(Collectors.toList());
    }

    public String getValueFromDesiredConfigurations(String configType, String propertyName) {
        return this.m_configHelper.getValueFromDesiredConfigurations(this.m_cluster, configType, propertyName);
    }

    private HostsType.HighAvailabilityHosts findMasterAndSecondaries(NameService nameService, Set<String> componentHosts) throws ClassifyNameNodeException {
        String master = null;
        ArrayList<String> secondaries = new ArrayList<String>();
        for (NameService.NameNode nameNode : nameService.getNameNodes()) {
            MasterHostResolver.checkForDualNetworkCards(componentHosts, nameNode);
            String state = this.queryJmxBeanValue(nameNode.getHost(), nameNode.getPort(), "Hadoop:service=NameNode,name=NameNodeStatus", "State", true, nameNode.isEncrypted());
            if (Status.ACTIVE.toString().equalsIgnoreCase(state)) {
                master = nameNode.getHost();
                continue;
            }
            if (Status.STANDBY.toString().equalsIgnoreCase(state)) {
                secondaries.add(nameNode.getHost());
                continue;
            }
            LOG.error(String.format("Could not retrieve state for NameNode %s from property %s by querying JMX.", nameNode.getHost(), nameNode.getPropertyName()));
        }
        if (MasterHostResolver.masterAndSecondariesAreFound(componentHosts, master, secondaries)) {
            return new HostsType.HighAvailabilityHosts(master, secondaries);
        }
        throw new ClassifyNameNodeException(nameService);
    }

    private static void checkForDualNetworkCards(Set<String> componentHosts, NameService.NameNode nameNode) {
        if (!componentHosts.contains(nameNode.getHost())) {
            LOG.error(MessageFormat.format("Hadoop NameNode HA configuration {0} contains host {1} that does not exist in the NameNode hosts list {3}", nameNode.getPropertyName(), nameNode.getHost(), componentHosts.toString()));
        }
    }

    private static boolean masterAndSecondariesAreFound(Set<String> componentHosts, String master, List<String> secondaries) {
        return master != null && secondaries.size() + 1 == componentHosts.size() && !secondaries.contains(master);
    }

    private HostAndPort parseHostPort(Cluster cluster, String propertyName, String configType) throws MalformedURLException {
        String propertyValue = this.m_configHelper.getValueFromDesiredConfigurations(cluster, configType, propertyName);
        HostAndPort hp = HTTPUtils.getHostAndPortFromProperty(propertyValue);
        if (hp == null) {
            throw new MalformedURLException("Could not parse host and port from " + propertyValue);
        }
        return hp;
    }

    private HostsType resolveResourceManagers(Cluster cluster, Set<String> hosts) throws MalformedURLException {
        String master = null;
        LinkedHashSet<String> orderedHosts = new LinkedHashSet<String>(hosts);
        HostAndPort hp = this.parseHostPort(cluster, "yarn.resourcemanager.webapp.address", "yarn-site");
        for (String hostname : hosts) {
            String value = this.queryJmxBeanValue(hostname, hp.port, "Hadoop:service=ResourceManager,name=RMNMInfo", "modelerType", true);
            if (null == value) continue;
            if (master != null) {
                master = hostname.toLowerCase();
            }
            orderedHosts.remove(hostname.toLowerCase());
            orderedHosts.add(hostname.toLowerCase());
        }
        return HostsType.from(master, null, orderedHosts);
    }

    private HostsType resolveHBaseMasters(Cluster cluster, Set<String> hosts) throws OBDPException {
        String master = null;
        String secondary = null;
        String hbaseMasterInfoPortProperty = "hbase.master.info.port";
        String hbaseMasterInfoPortValue = this.m_configHelper.getValueFromDesiredConfigurations(cluster, "hbase-site", hbaseMasterInfoPortProperty);
        if (hbaseMasterInfoPortValue == null || hbaseMasterInfoPortValue.isEmpty()) {
            throw new OBDPException("Could not find property " + hbaseMasterInfoPortProperty);
        }
        int hbaseMasterInfoPort = Integer.parseInt(hbaseMasterInfoPortValue);
        for (String hostname : hosts) {
            String value = this.queryJmxBeanValue(hostname, hbaseMasterInfoPort, "Hadoop:service=HBase,name=Master,sub=Server", "tag.isActiveMaster", false);
            if (null == value) continue;
            Boolean bool = Boolean.valueOf(value);
            if (bool.booleanValue()) {
                master = hostname.toLowerCase();
                continue;
            }
            secondary = hostname.toLowerCase();
        }
        return HostsType.from(master, secondary, new LinkedHashSet<String>(hosts));
    }

    protected String queryJmxBeanValue(String hostname, int port, String beanName, String attributeName, boolean asQuery) {
        return this.queryJmxBeanValue(hostname, port, beanName, attributeName, asQuery, false);
    }

    protected String queryJmxBeanValue(String hostname, int port, String beanName, String attributeName, boolean asQuery, boolean encrypted) {
        String protocol = encrypted ? "https://" : "http://";
        String endPoint = protocol + (asQuery ? String.format("%s:%s/jmx?qry=%s", hostname, port, beanName) : String.format("%s:%s/jmx?get=%s::%s", hostname, port, beanName, attributeName));
        String response = HTTPUtils.requestURL(endPoint);
        if (null == response || response.isEmpty()) {
            return null;
        }
        Type type = new TypeToken<Map<String, ArrayList<HashMap<String, String>>>>(){}.getType();
        try {
            Map jmxBeans = (Map)StageUtils.getGson().fromJson(response, type);
            return (String)((HashMap)((ArrayList)jmxBeans.get("beans")).get(0)).get(attributeName);
        }
        catch (Exception e) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Could not load JMX from {}/{} from {}", new Object[]{beanName, attributeName, hostname, e});
            } else {
                LOG.debug("Could not load JMX from {}/{} from {}", new Object[]{beanName, attributeName, hostname});
            }
            return null;
        }
    }

    public static enum Service {
        HDFS,
        HBASE,
        YARN,
        OTHER;


        public static Service fromString(String serviceName) {
            try {
                return Service.valueOf(serviceName.toUpperCase());
            }
            catch (Exception ignore) {
                return OTHER;
            }
        }
    }

    protected static enum Status {
        ACTIVE,
        STANDBY;

    }
}

