/*
 * Decompiled with CFR 0.152.
 */
package id.onyx.obdp.server.security.encryption;

import com.google.inject.Inject;
import com.google.inject.Singleton;
import id.onyx.obdp.server.OBDPException;
import id.onyx.obdp.server.configuration.Configuration;
import id.onyx.obdp.server.security.SecurePasswordHelper;
import id.onyx.obdp.server.security.credential.Credential;
import id.onyx.obdp.server.security.encryption.CredentialStore;
import id.onyx.obdp.server.security.encryption.CredentialStoreService;
import id.onyx.obdp.server.security.encryption.CredentialStoreType;
import id.onyx.obdp.server.security.encryption.FileBasedCredentialStore;
import id.onyx.obdp.server.security.encryption.InMemoryCredentialStore;
import id.onyx.obdp.server.security.encryption.MasterKeyService;
import id.onyx.obdp.server.security.encryption.MasterKeyServiceImpl;
import java.io.File;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class CredentialStoreServiceImpl
implements CredentialStoreService {
    private static final Logger LOG = LoggerFactory.getLogger(CredentialStoreServiceImpl.class);
    private SecurePasswordHelper securePasswordHelper;
    private FileBasedCredentialStore persistedCredentialStore = null;
    private InMemoryCredentialStore temporaryCredentialStore = null;

    @Inject
    public CredentialStoreServiceImpl(Configuration configuration, SecurePasswordHelper securePasswordHelper) {
        this.securePasswordHelper = securePasswordHelper;
        if (configuration != null) {
            try {
                this.initializeTemporaryCredentialStore(configuration.getTemporaryKeyStoreRetentionMinutes(), TimeUnit.MINUTES, configuration.isActivelyPurgeTemporaryKeyStore());
                LOG.info("Initialized the temporary credential store. KeyStore entries will be retained for {} minutes and {} be actively purged", (Object)configuration.getTemporaryKeyStoreRetentionMinutes(), (Object)(configuration.isActivelyPurgeTemporaryKeyStore() ? "will" : "will not"));
            }
            catch (OBDPException e) {
                LOG.error("Failed to initialize the temporary credential store.  Storage of temporary credentials will fail.", (Throwable)e);
            }
            MasterKeyServiceImpl masterKeyService = new MasterKeyServiceImpl(configuration);
            if (masterKeyService.isMasterKeyInitialized()) {
                try {
                    this.initializePersistedCredentialStore(configuration.getMasterKeyStoreLocation(), masterKeyService);
                    LOG.info("Initialized the persistent credential store. Using KeyStore file at {}", (Object)this.persistedCredentialStore.getKeyStorePath().getAbsolutePath());
                }
                catch (OBDPException e) {
                    LOG.error("Failed to initialize the persistent credential store.  Storage of persisted credentials will fail.", (Throwable)e);
                }
            }
        }
    }

    public synchronized void initializeTemporaryCredentialStore(long retentionDuration, TimeUnit units, boolean activelyPurge) throws OBDPException {
        if (this.isInitialized(CredentialStoreType.TEMPORARY)) {
            throw new OBDPException("This temporary CredentialStore has already been initialized");
        }
        this.temporaryCredentialStore = new InMemoryCredentialStore(retentionDuration, units, activelyPurge);
        this.temporaryCredentialStore.setMasterKeyService(new MasterKeyServiceImpl(this.securePasswordHelper.createSecurePassword()));
    }

    public synchronized void initializePersistedCredentialStore(File credentialStoreLocation, MasterKeyService masterKeyService) throws OBDPException {
        if (this.isInitialized(CredentialStoreType.PERSISTED)) {
            throw new OBDPException("This persisted CredentialStore has already been initialized");
        }
        this.persistedCredentialStore = new FileBasedCredentialStore(credentialStoreLocation);
        this.persistedCredentialStore.setMasterKeyService(masterKeyService);
    }

    @Override
    public void setCredential(String clusterName, String alias, Credential credential, CredentialStoreType credentialStoreType) throws OBDPException {
        this.validateInitialized(credentialStoreType);
        this.removeCredential(clusterName, alias);
        this.getCredentialStore(credentialStoreType).addCredential(CredentialStoreServiceImpl.canonicalizeAlias(clusterName, alias), credential);
    }

    @Override
    public Credential getCredential(String clusterName, String alias) throws OBDPException {
        Credential credential = this.getCredential(clusterName, alias, CredentialStoreType.TEMPORARY);
        if (credential == null) {
            credential = this.getCredential(clusterName, alias, CredentialStoreType.PERSISTED);
        }
        return credential;
    }

    @Override
    public Credential getCredential(String clusterName, String alias, CredentialStoreType credentialStoreType) throws OBDPException {
        return this.isInitialized(credentialStoreType) ? this.getCredentialStore(credentialStoreType).getCredential(CredentialStoreServiceImpl.canonicalizeAlias(clusterName, alias)) : null;
    }

    @Override
    public void removeCredential(String clusterName, String alias) throws OBDPException {
        this.removeCredential(clusterName, alias, CredentialStoreType.PERSISTED);
        this.removeCredential(clusterName, alias, CredentialStoreType.TEMPORARY);
    }

    @Override
    public void removeCredential(String clusterName, String alias, CredentialStoreType credentialStoreType) throws OBDPException {
        if (this.isInitialized(credentialStoreType)) {
            this.getCredentialStore(credentialStoreType).removeCredential(CredentialStoreServiceImpl.canonicalizeAlias(clusterName, alias));
        }
    }

    @Override
    public boolean containsCredential(String clusterName, String alias) throws OBDPException {
        return this.containsCredential(clusterName, alias, CredentialStoreType.TEMPORARY) || this.containsCredential(clusterName, alias, CredentialStoreType.PERSISTED);
    }

    @Override
    public boolean containsCredential(String clusterName, String alias, CredentialStoreType credentialStoreType) throws OBDPException {
        return this.isInitialized(credentialStoreType) && this.getCredentialStore(credentialStoreType).containsCredential(CredentialStoreServiceImpl.canonicalizeAlias(clusterName, alias));
    }

    @Override
    public CredentialStoreType getCredentialStoreType(String clusterName, String alias) throws OBDPException {
        if (this.containsCredential(clusterName, alias, CredentialStoreType.TEMPORARY)) {
            return CredentialStoreType.TEMPORARY;
        }
        if (this.containsCredential(clusterName, alias, CredentialStoreType.PERSISTED)) {
            return CredentialStoreType.PERSISTED;
        }
        throw new OBDPException("The alias was not found in either the persisted or temporary credential stores");
    }

    @Override
    public Map<String, CredentialStoreType> listCredentials(String clusterName) throws OBDPException {
        if (!this.isInitialized()) {
            throw new OBDPException("This CredentialStoreService has not yet been initialized");
        }
        Set<String> persistedAliases = this.isInitialized(CredentialStoreType.PERSISTED) ? this.persistedCredentialStore.listCredentials() : null;
        Set<String> temporaryAliases = this.isInitialized(CredentialStoreType.TEMPORARY) ? this.temporaryCredentialStore.listCredentials() : null;
        HashMap<String, CredentialStoreType> map = new HashMap<String, CredentialStoreType>();
        if (persistedAliases != null) {
            for (String alias : persistedAliases) {
                if (!this.isAliasRequested(clusterName, alias)) continue;
                map.put(CredentialStoreServiceImpl.decanonicalizeAlias(clusterName, alias), CredentialStoreType.PERSISTED);
            }
        }
        if (temporaryAliases != null) {
            for (String alias : temporaryAliases) {
                if (!this.isAliasRequested(clusterName, alias)) continue;
                map.put(CredentialStoreServiceImpl.decanonicalizeAlias(clusterName, alias), CredentialStoreType.TEMPORARY);
            }
        }
        return map;
    }

    @Override
    public synchronized boolean isInitialized() {
        return this.isInitialized(CredentialStoreType.PERSISTED) || this.isInitialized(CredentialStoreType.TEMPORARY);
    }

    @Override
    public synchronized boolean isInitialized(CredentialStoreType credentialStoreType) {
        if (CredentialStoreType.PERSISTED == credentialStoreType) {
            return this.persistedCredentialStore != null;
        }
        if (CredentialStoreType.TEMPORARY == credentialStoreType) {
            return this.temporaryCredentialStore != null;
        }
        throw new IllegalArgumentException("Invalid or unexpected credential store type specified");
    }

    public static String canonicalizeAlias(String clusterName, String alias) {
        Object canonicaizedAlias;
        if (clusterName == null || clusterName.isEmpty() || alias == null || alias.isEmpty()) {
            canonicaizedAlias = alias;
        } else {
            String prefix = CredentialStoreServiceImpl.createAliasPrefix(clusterName);
            canonicaizedAlias = alias.toLowerCase().startsWith(prefix) ? alias : prefix + alias;
        }
        return canonicaizedAlias == null ? null : ((String)canonicaizedAlias).toLowerCase();
    }

    public static String decanonicalizeAlias(String clusterName, String canonicaizedAlias) {
        if (clusterName == null || clusterName.isEmpty() || canonicaizedAlias == null || canonicaizedAlias.isEmpty()) {
            return canonicaizedAlias;
        }
        String prefix = CredentialStoreServiceImpl.createAliasPrefix(clusterName);
        if (canonicaizedAlias.startsWith(prefix)) {
            return canonicaizedAlias.substring(prefix.length());
        }
        return canonicaizedAlias;
    }

    private static String createAliasPrefix(String clusterName) {
        return ("cluster." + clusterName + ".").toLowerCase();
    }

    private boolean isAliasRequested(String clusterName, String canonicalizedAlias) {
        return clusterName == null || canonicalizedAlias.toLowerCase().startsWith(CredentialStoreServiceImpl.createAliasPrefix(clusterName));
    }

    private CredentialStore getCredentialStore(CredentialStoreType credentialStoreType) {
        if (CredentialStoreType.PERSISTED == credentialStoreType) {
            return this.persistedCredentialStore;
        }
        if (CredentialStoreType.TEMPORARY == credentialStoreType) {
            return this.temporaryCredentialStore;
        }
        throw new IllegalArgumentException("Invalid or unexpected credential store type specified");
    }

    private void validateInitialized(CredentialStoreType credentialStoreType) throws OBDPException {
        if (!this.isInitialized(credentialStoreType)) {
            throw new OBDPException(String.format("The %s CredentialStore for this CredentialStoreService has not yet been initialized", credentialStoreType.name().toLowerCase()));
        }
    }
}

