/*
 * Decompiled with CFR 0.152.
 */
package id.onyx.obdp.server.security.authentication.jwt;

import com.nimbusds.jose.JOSEException;
import com.nimbusds.jose.JWSObject;
import com.nimbusds.jose.JWSVerifier;
import com.nimbusds.jose.crypto.RSASSAVerifier;
import com.nimbusds.jwt.SignedJWT;
import id.onyx.obdp.server.configuration.OBDPServerConfigurationKey;
import id.onyx.obdp.server.security.authentication.AmbariAuthenticationException;
import id.onyx.obdp.server.security.authentication.OBDPAuthenticationEventHandler;
import id.onyx.obdp.server.security.authentication.OBDPAuthenticationFilter;
import id.onyx.obdp.server.security.authentication.OBDPUserAuthentication;
import id.onyx.obdp.server.security.authentication.jwt.JwtAuthenticationProperties;
import id.onyx.obdp.server.security.authentication.jwt.JwtAuthenticationPropertiesProvider;
import id.onyx.obdp.server.security.authentication.jwt.JwtAuthenticationToken;
import id.onyx.obdp.server.security.authentication.jwt.OBDPJwtAuthenticationProvider;
import jakarta.servlet.FilterChain;
import jakarta.servlet.FilterConfig;
import jakarta.servlet.ServletException;
import jakarta.servlet.ServletRequest;
import jakarta.servlet.ServletResponse;
import jakarta.servlet.http.Cookie;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.security.interfaces.RSAPublicKey;
import java.text.ParseException;
import java.util.Date;
import java.util.List;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.annotation.Order;
import org.springframework.security.authentication.AnonymousAuthenticationToken;
import org.springframework.security.authentication.AuthenticationProvider;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.web.AuthenticationEntryPoint;
import org.springframework.stereotype.Component;

@Component
@Order(value=1)
public class OBDPJwtAuthenticationFilter
implements OBDPAuthenticationFilter {
    private static final Logger LOG = LoggerFactory.getLogger(OBDPJwtAuthenticationFilter.class);
    private final OBDPAuthenticationEventHandler eventHandler;
    private final AuthenticationEntryPoint ambariEntryPoint;
    private final AuthenticationProvider authenticationProvider;
    private final JwtAuthenticationPropertiesProvider propertiesProvider;

    OBDPJwtAuthenticationFilter(AuthenticationEntryPoint ambariEntryPoint, JwtAuthenticationPropertiesProvider propertiesProvider, OBDPJwtAuthenticationProvider authenticationProvider, OBDPAuthenticationEventHandler eventHandler) {
        if (eventHandler == null) {
            throw new IllegalArgumentException("The OBDPAuthenticationEventHandler must not be null");
        }
        this.ambariEntryPoint = ambariEntryPoint;
        this.eventHandler = eventHandler;
        this.propertiesProvider = propertiesProvider;
        this.authenticationProvider = authenticationProvider;
    }

    @Override
    public boolean shouldApply(HttpServletRequest httpServletRequest) {
        boolean shouldApply = false;
        JwtAuthenticationProperties jwtProperties = (JwtAuthenticationProperties)this.propertiesProvider.get();
        if (jwtProperties != null && jwtProperties.isEnabledForAmbari()) {
            String serializedJWT = this.getJWTFromCookie(httpServletRequest);
            shouldApply = serializedJWT != null && this.isAuthenticationRequired(serializedJWT);
        }
        return shouldApply;
    }

    @Override
    public boolean shouldIncrementFailureCount() {
        return false;
    }

    public void init(FilterConfig filterConfig) {
    }

    public void doFilter(ServletRequest servletRequest, ServletResponse servletResponse, FilterChain chain) throws IOException, ServletException {
        this.eventHandler.beforeAttemptAuthentication(this, servletRequest, servletResponse);
        JwtAuthenticationProperties jwtProperties = (JwtAuthenticationProperties)this.propertiesProvider.get();
        if (jwtProperties == null || !jwtProperties.isEnabledForAmbari()) {
            chain.doFilter(servletRequest, servletResponse);
            return;
        }
        HttpServletRequest httpServletRequest = (HttpServletRequest)servletRequest;
        HttpServletResponse httpServletResponse = (HttpServletResponse)servletResponse;
        try {
            String serializedJWT = this.getJWTFromCookie(httpServletRequest);
            if (serializedJWT != null && this.isAuthenticationRequired(serializedJWT)) {
                try {
                    SignedJWT jwtToken = SignedJWT.parse((String)serializedJWT);
                    boolean valid = this.validateToken(jwtToken);
                    if (!valid) {
                        throw new BadCredentialsException("Invalid JWT token");
                    }
                    String userName = jwtToken.getJWTClaimsSet().getSubject();
                    Authentication authentication = this.authenticationProvider.authenticate((Authentication)new JwtAuthenticationToken(userName, serializedJWT, null));
                    SecurityContextHolder.getContext().setAuthentication(authentication);
                    this.eventHandler.onSuccessfulAuthentication(this, httpServletRequest, httpServletResponse, authentication);
                }
                catch (ParseException e) {
                    LOG.warn("Unable to parse the JWT token", (Throwable)e);
                    throw new BadCredentialsException("Unable to parse the JWT token - " + e.getLocalizedMessage());
                }
            } else {
                LOG.trace("No JWT cookie found, do nothing");
            }
            chain.doFilter(servletRequest, servletResponse);
        }
        catch (AuthenticationException e) {
            LOG.warn("JWT authentication failed - {}", (Object)e.getLocalizedMessage());
            SecurityContextHolder.clearContext();
            AmbariAuthenticationException cause = e instanceof AmbariAuthenticationException ? (AmbariAuthenticationException)e : new AmbariAuthenticationException(null, e.getMessage(), false, e);
            this.eventHandler.onUnsuccessfulAuthentication(this, httpServletRequest, httpServletResponse, cause);
            this.ambariEntryPoint.commence(httpServletRequest, httpServletResponse, e);
        }
    }

    public void destroy() {
    }

    private boolean isAuthenticationRequired(String token) {
        Authentication existingAuth = SecurityContextHolder.getContext().getAuthentication();
        if (existingAuth == null || !existingAuth.isAuthenticated()) {
            return true;
        }
        if (existingAuth instanceof OBDPUserAuthentication && !StringUtils.equals((String)token, (String)((String)existingAuth.getCredentials()))) {
            return true;
        }
        return existingAuth instanceof AnonymousAuthenticationToken;
    }

    String getJWTFromCookie(HttpServletRequest req) {
        String serializedJWT = null;
        Cookie[] cookies = req.getCookies();
        if (cookies != null) {
            String jwtCookieName;
            JwtAuthenticationProperties jwtProperties = (JwtAuthenticationProperties)this.propertiesProvider.get();
            String string = jwtCookieName = jwtProperties == null ? null : jwtProperties.getCookieName();
            if (StringUtils.isEmpty((String)jwtCookieName)) {
                jwtCookieName = OBDPServerConfigurationKey.SSO_JWT_COOKIE_NAME.getDefaultValue();
            }
            for (Cookie cookie : cookies) {
                if (!jwtCookieName.equals(cookie.getName())) continue;
                LOG.info("{} cookie has been found and is being processed", (Object)jwtCookieName);
                serializedJWT = cookie.getValue();
                break;
            }
        }
        return serializedJWT;
    }

    private boolean validateToken(SignedJWT jwtToken) {
        boolean expValid;
        boolean audValid;
        boolean sigValid = this.validateSignature(jwtToken);
        if (!sigValid) {
            LOG.warn("Signature could not be verified");
        }
        if (!(audValid = this.validateAudiences(jwtToken))) {
            LOG.warn("Audience validation failed.");
        }
        if (!(expValid = this.validateExpiration(jwtToken))) {
            LOG.info("Expiration validation failed.");
        }
        return sigValid && audValid && expValid;
    }

    boolean validateSignature(SignedJWT jwtToken) {
        boolean valid = false;
        if (JWSObject.State.SIGNED == jwtToken.getState()) {
            LOG.debug("JWT token is in a SIGNED state");
            if (jwtToken.getSignature() != null) {
                RSAPublicKey publicKey;
                LOG.debug("JWT token signature is not null");
                JwtAuthenticationProperties jwtProperties = (JwtAuthenticationProperties)this.propertiesProvider.get();
                RSAPublicKey rSAPublicKey = publicKey = jwtProperties == null ? null : jwtProperties.getPublicKey();
                if (publicKey == null) {
                    LOG.warn("SSO server public key has not be set, validation of the JWT token cannot be performed.");
                } else {
                    try {
                        RSASSAVerifier verifier = new RSASSAVerifier(publicKey);
                        if (jwtToken.verify((JWSVerifier)verifier)) {
                            valid = true;
                            LOG.debug("JWT token has been successfully verified");
                        } else {
                            LOG.warn("JWT signature verification failed.");
                        }
                    }
                    catch (JOSEException je) {
                        LOG.warn("Error while validating signature", (Throwable)je);
                    }
                }
            }
        }
        return valid;
    }

    boolean validateAudiences(SignedJWT jwtToken) {
        boolean valid = false;
        try {
            List<String> audiences;
            List tokenAudienceList = jwtToken.getJWTClaimsSet().getAudience();
            JwtAuthenticationProperties jwtProperties = (JwtAuthenticationProperties)this.propertiesProvider.get();
            List<String> list = audiences = jwtProperties == null ? null : jwtProperties.getAudiences();
            if (audiences == null) {
                valid = true;
            } else {
                if (tokenAudienceList == null) {
                    LOG.warn("JWT token has no audiences, validation failed.");
                    return false;
                }
                LOG.info("Audience List: {}", audiences);
                for (String aud : tokenAudienceList) {
                    LOG.info("Found audience: {}", (Object)aud);
                    if (!audiences.contains(aud)) continue;
                    LOG.debug("JWT token audience has been successfully validated");
                    valid = true;
                    break;
                }
                if (!valid) {
                    LOG.warn("JWT audience validation failed.");
                }
            }
        }
        catch (ParseException pe) {
            LOG.warn("Unable to parse the JWT token.", (Throwable)pe);
        }
        return valid;
    }

    boolean validateExpiration(SignedJWT jwtToken) {
        boolean valid = false;
        try {
            Date expires = jwtToken.getJWTClaimsSet().getExpirationTime();
            if (expires == null || new Date().before(expires)) {
                LOG.debug("JWT token expiration date has been successfully validated");
                valid = true;
            } else {
                LOG.warn("JWT expiration date validation failed.");
            }
        }
        catch (ParseException pe) {
            LOG.warn("JWT expiration date validation failed.", (Throwable)pe);
        }
        return valid;
    }
}

