/*
 * Decompiled with CFR 0.152.
 */
package id.onyx.obdp.server.checks;

import com.google.common.base.Splitter;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Multimap;
import com.google.inject.Inject;
import com.google.inject.Injector;
import com.google.inject.Provider;
import com.google.inject.persist.Transactional;
import id.onyx.obdp.server.OBDPException;
import id.onyx.obdp.server.api.services.OBDPMetaInfo;
import id.onyx.obdp.server.checks.DatabaseConsistencyCheckResult;
import id.onyx.obdp.server.configuration.Configuration;
import id.onyx.obdp.server.orm.DBAccessor;
import id.onyx.obdp.server.orm.dao.AlertDefinitionDAO;
import id.onyx.obdp.server.orm.dao.ClusterDAO;
import id.onyx.obdp.server.orm.dao.ExecutionCommandDAO;
import id.onyx.obdp.server.orm.dao.HostComponentDesiredStateDAO;
import id.onyx.obdp.server.orm.dao.HostComponentStateDAO;
import id.onyx.obdp.server.orm.dao.HostRoleCommandDAO;
import id.onyx.obdp.server.orm.dao.MetainfoDAO;
import id.onyx.obdp.server.orm.dao.StageDAO;
import id.onyx.obdp.server.orm.entities.AlertDefinitionEntity;
import id.onyx.obdp.server.orm.entities.ClusterConfigEntity;
import id.onyx.obdp.server.orm.entities.HostComponentDesiredStateEntity;
import id.onyx.obdp.server.orm.entities.HostComponentStateEntity;
import id.onyx.obdp.server.orm.entities.MetainfoEntity;
import id.onyx.obdp.server.state.ClientConfigFileDefinition;
import id.onyx.obdp.server.state.Cluster;
import id.onyx.obdp.server.state.Clusters;
import id.onyx.obdp.server.state.ComponentInfo;
import id.onyx.obdp.server.state.Host;
import id.onyx.obdp.server.state.Service;
import id.onyx.obdp.server.state.ServiceInfo;
import id.onyx.obdp.server.state.State;
import id.onyx.obdp.server.state.UpgradeState;
import id.onyx.obdp.server.state.configgroup.ConfigGroup;
import id.onyx.obdp.server.utils.VersionUtils;
import jakarta.persistence.EntityManager;
import jakarta.persistence.TypedQuery;
import java.io.File;
import java.io.IOException;
import java.lang.invoke.CallSite;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Scanner;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DatabaseConsistencyCheckHelper {
    private static final Logger LOG = LoggerFactory.getLogger(DatabaseConsistencyCheckHelper.class);
    @Inject
    private static Injector injector;
    private static MetainfoDAO metainfoDAO;
    private static AlertDefinitionDAO alertDefinitionDAO;
    private static Connection connection;
    private static OBDPMetaInfo obdpMetaInfo;
    private static DBAccessor dbAccessor;
    private static HostRoleCommandDAO hostRoleCommandDAO;
    private static ExecutionCommandDAO executionCommandDAO;
    private static StageDAO stageDAO;
    private static DatabaseConsistencyCheckResult checkResult;
    public static final String GET_CONFIGS_SELECTED_MORE_THAN_ONCE_QUERY = "select c.cluster_name, cc.type_name from clusterconfig cc join clusters c on cc.cluster_id=c.cluster_id group by c.cluster_name, cc.type_name having sum(cc.selected) > 1";

    public static DatabaseConsistencyCheckResult getLastCheckResult() {
        return checkResult;
    }

    public static void resetCheckResult() {
        checkResult = DatabaseConsistencyCheckResult.DB_CHECK_SUCCESS;
    }

    private static void setCheckResult(DatabaseConsistencyCheckResult newResult) {
        if (newResult.ordinal() > checkResult.ordinal()) {
            checkResult = newResult;
        }
    }

    private static void warning(String messageTemplate, Object ... messageParams) {
        LOG.warn(messageTemplate, messageParams);
        DatabaseConsistencyCheckHelper.setCheckResult(DatabaseConsistencyCheckResult.DB_CHECK_WARNING);
    }

    private static void error(String messageTemplate, Object ... messageParams) {
        LOG.error(messageTemplate, messageParams);
        DatabaseConsistencyCheckHelper.setCheckResult(DatabaseConsistencyCheckResult.DB_CHECK_ERROR);
    }

    public static void setInjector(Injector injector) {
        DatabaseConsistencyCheckHelper.injector = injector;
        DatabaseConsistencyCheckHelper.closeConnection();
        connection = null;
        metainfoDAO = null;
        alertDefinitionDAO = null;
        obdpMetaInfo = null;
        dbAccessor = null;
    }

    public static void setConnection(Connection connection) {
        DatabaseConsistencyCheckHelper.connection = connection;
    }

    public static void closeConnection() {
        try {
            if (connection != null) {
                connection.close();
            }
        }
        catch (SQLException e) {
            LOG.error("Exception occurred during connection close procedure: ", (Throwable)e);
        }
    }

    public static DatabaseConsistencyCheckResult runAllDBChecks(boolean fixIssues) throws Throwable {
        LOG.info("******************************* Check database started *******************************");
        try {
            if (fixIssues) {
                DatabaseConsistencyCheckHelper.fixHostComponentStatesCountEqualsHostComponentsDesiredStates();
                DatabaseConsistencyCheckHelper.fixClusterConfigsNotMappedToAnyService();
                DatabaseConsistencyCheckHelper.fixConfigGroupServiceNames();
                DatabaseConsistencyCheckHelper.fixConfigGroupHostMappings();
                DatabaseConsistencyCheckHelper.fixConfigGroupsForDeletedServices();
                DatabaseConsistencyCheckHelper.fixConfigsSelectedMoreThanOnce();
                DatabaseConsistencyCheckHelper.fixAlertsForDeletedServices();
            }
            DatabaseConsistencyCheckHelper.checkSchemaName();
            DatabaseConsistencyCheckHelper.checkMySQLEngine();
            DatabaseConsistencyCheckHelper.checkForConfigsNotMappedToService();
            DatabaseConsistencyCheckHelper.checkForConfigsSelectedMoreThanOnce();
            DatabaseConsistencyCheckHelper.checkForHostsWithoutState();
            DatabaseConsistencyCheckHelper.checkHostComponentStates();
            DatabaseConsistencyCheckHelper.checkServiceConfigs();
            DatabaseConsistencyCheckHelper.checkForLargeTables();
            DatabaseConsistencyCheckHelper.checkConfigGroupsHasServiceName();
            DatabaseConsistencyCheckHelper.checkConfigGroupHostMapping(true);
            DatabaseConsistencyCheckHelper.checkConfigGroupsForDeletedServices(true);
            DatabaseConsistencyCheckHelper.checkForStalealertdefs();
            LOG.info("******************************* Check database completed *******************************");
            return checkResult;
        }
        catch (Throwable ex) {
            LOG.error("An error occurred during database consistency check.", ex);
            throw ex;
        }
    }

    public static void checkDBVersionCompatible() throws OBDPException {
        Configuration conf;
        File versionFile;
        LOG.info("Checking DB store version");
        if (metainfoDAO == null) {
            metainfoDAO = (MetainfoDAO)injector.getInstance(MetainfoDAO.class);
        }
        if (alertDefinitionDAO == null) {
            alertDefinitionDAO = (AlertDefinitionDAO)injector.getInstance(AlertDefinitionDAO.class);
        }
        MetainfoEntity schemaVersionEntity = metainfoDAO.findByKey("version");
        String schemaVersion = null;
        if (schemaVersionEntity != null) {
            schemaVersion = schemaVersionEntity.getMetainfoValue();
        }
        if (!(versionFile = new File((conf = (Configuration)injector.getInstance(Configuration.class)).getServerVersionFilePath())).exists()) {
            throw new OBDPException("Server version file does not exist.");
        }
        String serverVersion = null;
        try (Scanner scanner = new Scanner(versionFile);){
            serverVersion = scanner.useDelimiter("\\Z").next();
        }
        catch (IOException ioe) {
            throw new OBDPException("Unable to read server version file.");
        }
        if (schemaVersionEntity == null || VersionUtils.compareVersions((String)schemaVersion, (String)serverVersion, (int)3) != 0) {
            String error = "Current database store version is not compatible with current server version, serverVersion=" + serverVersion + ", schemaVersion=" + schemaVersion;
            LOG.error(error);
            throw new OBDPException(error);
        }
        LOG.info("DB store version is compatible");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    static void checkForLargeTables() {
        LOG.info("Checking for tables with large physical size");
        DatabaseConsistencyCheckHelper.ensureConnection();
        if (dbAccessor.getDbType() == DBAccessor.DbType.H2) {
            return;
        }
        DBAccessor.DbType dbType = dbAccessor.getDbType();
        String schemaName = dbAccessor.getDbSchema();
        String GET_TABLE_SIZE_IN_BYTES_POSTGRESQL = "SELECT pg_total_relation_size('%s') \"Table Size\"";
        String GET_TABLE_SIZE_IN_BYTES_MYSQL = "SELECT (data_length + index_length) \"Table Size\" FROM information_schema.TABLES WHERE table_schema = \"" + schemaName + "\" AND table_name =\"%s\"";
        String GET_TABLE_SIZE_IN_BYTES_ORACLE = "SELECT bytes \"Table Size\" FROM user_segments WHERE segment_type='TABLE' AND segment_name='%s'";
        String GET_ROW_COUNT_QUERY = "SELECT COUNT(*) FROM %s";
        HashMap<DBAccessor.DbType, Object> tableSizeQueryMap = new HashMap<DBAccessor.DbType, Object>();
        tableSizeQueryMap.put(DBAccessor.DbType.POSTGRES, GET_TABLE_SIZE_IN_BYTES_POSTGRESQL);
        tableSizeQueryMap.put(DBAccessor.DbType.MYSQL, GET_TABLE_SIZE_IN_BYTES_MYSQL);
        tableSizeQueryMap.put(DBAccessor.DbType.ORACLE, GET_TABLE_SIZE_IN_BYTES_ORACLE);
        List<String> tablesToCheck = Arrays.asList("host_role_command", "execution_command", "stage", "request", "alert_history");
        double TABLE_SIZE_LIMIT_MB = 3000.0;
        int TABLE_ROW_COUNT_LIMIT = 3000000;
        String findTableSizeQuery = (String)tableSizeQueryMap.get((Object)dbType);
        if (dbType == DBAccessor.DbType.ORACLE) {
            for (int i = 0; i < tablesToCheck.size(); ++i) {
                tablesToCheck.set(i, tablesToCheck.get(i).toUpperCase());
            }
        }
        for (String tableName : tablesToCheck) {
            ResultSet rs = null;
            Statement statement = null;
            Double tableSizeInMB = null;
            Long tableSizeInBytes = null;
            int tableRowCount = -1;
            try {
                statement = connection.createStatement(1005, 1008);
                rs = statement.executeQuery(String.format(findTableSizeQuery, tableName));
                if (rs != null) {
                    while (rs.next()) {
                        tableSizeInBytes = rs.getLong(1);
                        if (tableSizeInBytes == null) continue;
                        tableSizeInMB = (double)tableSizeInBytes.longValue() / 1024.0 / 1024.0;
                    }
                }
                if (tableSizeInMB != null && tableSizeInMB > 3000.0) {
                    DatabaseConsistencyCheckHelper.warning("The database table {} is currently {} MB (limit is {}) and may impact performance. It is recommended that you reduce its size by executing \"obdp-server db-purge-history\".", tableName, tableSizeInMB, 3000.0);
                    continue;
                }
                if (tableSizeInMB != null && tableSizeInMB < 3000.0) {
                    LOG.info(String.format("The database table %s is currently %.3f MB and is within normal limits (%.3f)", tableName, tableSizeInMB, 3000.0));
                    continue;
                }
                throw new Exception();
            }
            catch (Exception e) {
                LOG.error(String.format("Failed to get %s table size from database, will check row count: ", tableName), (Throwable)e);
                try {
                    rs = statement.executeQuery(String.format(GET_ROW_COUNT_QUERY, tableName));
                    if (rs != null) {
                        while (rs.next()) {
                            tableRowCount = rs.getInt(1);
                        }
                    }
                    if (tableRowCount > 3000000) {
                        DatabaseConsistencyCheckHelper.warning("The database table {} currently has {} rows (limit is {}) and may impact performance. It is recommended that you reduce its size by executing \"obdp-server db-purge-history\".", tableName, tableRowCount, 3000000);
                        continue;
                    }
                    if (tableRowCount != -1 && tableRowCount < 3000000) {
                        LOG.info(String.format("The database table %s currently has %d rows and is within normal limits (%d)", tableName, tableRowCount, 3000000));
                        continue;
                    }
                    DatabaseConsistencyCheckHelper.warning("Unable to get size for table {}!", tableName);
                }
                catch (SQLException ex) {
                    DatabaseConsistencyCheckHelper.warning(String.format("Failed to get %s row count: ", tableName), e);
                }
            }
            finally {
                if (rs != null) {
                    try {
                        rs.close();
                    }
                    catch (SQLException e) {
                        LOG.error("Exception occurred during result set closing procedure: ", (Throwable)e);
                    }
                }
                if (statement == null) continue;
                try {
                    statement.close();
                }
                catch (SQLException e) {
                    LOG.error("Exception occurred during statement closing procedure: ", (Throwable)e);
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    static void checkForConfigsSelectedMoreThanOnce() {
        LOG.info("Checking for more than 1 configuration of the same type being enabled.");
        String GET_CONFIGS_SELECTED_MORE_THAN_ONCE_QUERY = "select c.cluster_name, cc.type_name from clusterconfig cc join clusters c on cc.cluster_id=c.cluster_id group by c.cluster_name, cc.type_name having sum(selected) > 1";
        HashMultimap clusterConfigTypeMap = HashMultimap.create();
        ResultSet rs = null;
        Statement statement = null;
        DatabaseConsistencyCheckHelper.ensureConnection();
        try {
            statement = connection.createStatement(1005, 1008);
            rs = statement.executeQuery(GET_CONFIGS_SELECTED_MORE_THAN_ONCE_QUERY);
            if (rs != null) {
                while (rs.next()) {
                    clusterConfigTypeMap.put((Object)rs.getString("cluster_name"), (Object)rs.getString("type_name"));
                }
                for (String clusterName : clusterConfigTypeMap.keySet()) {
                    DatabaseConsistencyCheckHelper.error("You have config(s), in cluster {}, that is(are) selected more than once in clusterconfig table: {}", clusterName, StringUtils.join((Collection)clusterConfigTypeMap.get((Object)clusterName), (String)","));
                }
            }
        }
        catch (SQLException e) {
            DatabaseConsistencyCheckHelper.warning("Exception occurred during check for config selected more than once procedure: ", e);
        }
        finally {
            if (rs != null) {
                try {
                    rs.close();
                }
                catch (SQLException e) {
                    LOG.error("Exception occurred during result set closing procedure: ", (Throwable)e);
                }
            }
            if (statement != null) {
                try {
                    statement.close();
                }
                catch (SQLException e) {
                    LOG.error("Exception occurred during statement closing procedure: ", (Throwable)e);
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    static void checkForHostsWithoutState() {
        LOG.info("Checking for hosts without state");
        String GET_HOSTS_WITHOUT_STATUS_QUERY = "select host_name from hosts where host_id not in (select host_id from hoststate)";
        HashSet<String> hostsWithoutStatus = new HashSet<String>();
        ResultSet rs = null;
        Statement statement = null;
        DatabaseConsistencyCheckHelper.ensureConnection();
        try {
            statement = connection.createStatement(1005, 1008);
            rs = statement.executeQuery(GET_HOSTS_WITHOUT_STATUS_QUERY);
            if (rs != null) {
                while (rs.next()) {
                    hostsWithoutStatus.add(rs.getString("host_name"));
                }
                if (!hostsWithoutStatus.isEmpty()) {
                    DatabaseConsistencyCheckHelper.warning("You have host(s) without state (in hoststate table): " + StringUtils.join(hostsWithoutStatus, (String)","), new Object[0]);
                }
            }
        }
        catch (SQLException e) {
            DatabaseConsistencyCheckHelper.warning("Exception occurred during check for host without state procedure: ", e);
        }
        finally {
            if (rs != null) {
                try {
                    rs.close();
                }
                catch (SQLException e) {
                    LOG.error("Exception occurred during result set closing procedure: ", (Throwable)e);
                }
            }
            if (statement != null) {
                try {
                    statement.close();
                }
                catch (SQLException e) {
                    LOG.error("Exception occurred during statement closing procedure: ", (Throwable)e);
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static int runQuery(Statement statement, String query) {
        ResultSet rs = null;
        int result = 0;
        try {
            rs = statement.executeQuery(query);
            if (rs != null) {
                while (rs.next()) {
                    result = rs.getInt(1);
                }
            }
        }
        catch (SQLException e) {
            DatabaseConsistencyCheckHelper.warning("Exception occurred during topology request tables check: ", e);
        }
        finally {
            if (rs != null) {
                try {
                    rs.close();
                }
                catch (SQLException e) {
                    LOG.error("Exception occurred during result set closing procedure: ", (Throwable)e);
                }
            }
        }
        return result;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    static void checkHostComponentStates() {
        LOG.info("Checking host component states count equals host component desired states count");
        String GET_HOST_COMPONENT_STATE_COUNT_QUERY = "select count(*) from hostcomponentstate";
        String GET_HOST_COMPONENT_DESIRED_STATE_COUNT_QUERY = "select count(*) from hostcomponentdesiredstate";
        String GET_MERGED_TABLE_ROW_COUNT_QUERY = "select count(*) FROM hostcomponentstate hcs JOIN hostcomponentdesiredstate hcds ON hcs.service_name=hcds.service_name AND hcs.component_name=hcds.component_name AND hcs.host_id=hcds.host_id";
        String GET_HOST_COMPONENT_STATE_DUPLICATES_QUERY = "select component_name, host_id from hostcomponentstate group by component_name, host_id having count(component_name) > 1";
        int hostComponentStateCount = 0;
        int hostComponentDesiredStateCount = 0;
        HashMap<String, String> hostComponentStateDuplicates = new HashMap<String, String>();
        int mergedCount = 0;
        ResultSet rs = null;
        Statement statement = null;
        DatabaseConsistencyCheckHelper.ensureConnection();
        try {
            statement = connection.createStatement(1005, 1008);
            rs = statement.executeQuery(GET_HOST_COMPONENT_STATE_COUNT_QUERY);
            if (rs != null) {
                while (rs.next()) {
                    hostComponentStateCount = rs.getInt(1);
                }
            }
            if ((rs = statement.executeQuery(GET_HOST_COMPONENT_DESIRED_STATE_COUNT_QUERY)) != null) {
                while (rs.next()) {
                    hostComponentDesiredStateCount = rs.getInt(1);
                }
            }
            if ((rs = statement.executeQuery(GET_MERGED_TABLE_ROW_COUNT_QUERY)) != null) {
                while (rs.next()) {
                    mergedCount = rs.getInt(1);
                }
            }
            if (hostComponentStateCount != hostComponentDesiredStateCount || hostComponentStateCount != mergedCount) {
                DatabaseConsistencyCheckHelper.warning("Your host component states (hostcomponentstate table) count not equals host component desired states (hostcomponentdesiredstate table) count!", new Object[0]);
            }
            if ((rs = statement.executeQuery(GET_HOST_COMPONENT_STATE_DUPLICATES_QUERY)) != null) {
                while (rs.next()) {
                    hostComponentStateDuplicates.put(rs.getString("component_name"), rs.getString("host_id"));
                }
            }
            for (Map.Entry component : hostComponentStateDuplicates.entrySet()) {
                DatabaseConsistencyCheckHelper.warning("Component {} on host with id {}, has more than one host component state (hostcomponentstate table)!", component.getKey(), component.getValue());
            }
        }
        catch (SQLException e) {
            DatabaseConsistencyCheckHelper.warning("Exception occurred during check for same count of host component states and host component desired states: ", e);
        }
        finally {
            if (rs != null) {
                try {
                    rs.close();
                }
                catch (SQLException e) {
                    LOG.error("Exception occurred during result set closing procedure: ", (Throwable)e);
                }
            }
            if (statement != null) {
                try {
                    statement.close();
                }
                catch (SQLException e) {
                    LOG.error("Exception occurred during statement closing procedure: ", (Throwable)e);
                }
            }
        }
    }

    @Transactional
    static void fixClusterConfigsNotMappedToAnyService() {
        LOG.info("Checking for configs not mapped to any Service");
        ClusterDAO clusterDAO = (ClusterDAO)injector.getInstance(ClusterDAO.class);
        List<ClusterConfigEntity> notMappedClusterConfigs = DatabaseConsistencyCheckHelper.getNotMappedClusterConfigsToService();
        for (ClusterConfigEntity clusterConfigEntity : notMappedClusterConfigs) {
            if (clusterConfigEntity.isUnmapped()) continue;
            ArrayList<String> types = new ArrayList<String>();
            String type = clusterConfigEntity.getType();
            types.add(type);
            LOG.error("Removing config that is not mapped to any service", (Object)clusterConfigEntity);
            clusterDAO.removeConfig(clusterConfigEntity);
        }
    }

    private static List<ClusterConfigEntity> getNotMappedClusterConfigsToService() {
        Provider entityManagerProvider = injector.getProvider(EntityManager.class);
        EntityManager entityManager = (EntityManager)entityManagerProvider.get();
        String queryName = "ClusterConfigEntity.findNotMappedClusterConfigsToService";
        TypedQuery query = entityManager.createNamedQuery(queryName, ClusterConfigEntity.class);
        return query.getResultList();
    }

    static void checkForConfigsNotMappedToService() {
        LOG.info("Checking for configs that are not mapped to any service");
        List<ClusterConfigEntity> notMappedClasterConfigs = DatabaseConsistencyCheckHelper.getNotMappedClusterConfigsToService();
        HashSet<CallSite> nonMappedConfigs = new HashSet<CallSite>();
        for (ClusterConfigEntity clusterConfigEntity : notMappedClasterConfigs) {
            if (clusterConfigEntity.isUnmapped()) continue;
            nonMappedConfigs.add((CallSite)((Object)(clusterConfigEntity.getType() + "-" + clusterConfigEntity.getTag())));
        }
        if (!nonMappedConfigs.isEmpty()) {
            DatabaseConsistencyCheckHelper.warning("You have config(s): {} that is(are) not mapped (in serviceconfigmapping table) to any service!", StringUtils.join(nonMappedConfigs, (String)","));
        }
    }

    @Transactional
    static void fixHostComponentStatesCountEqualsHostComponentsDesiredStates() {
        Object stateEntity;
        LOG.info("Checking that there are the same number of actual and desired host components");
        HostComponentStateDAO hostComponentStateDAO = (HostComponentStateDAO)injector.getInstance(HostComponentStateDAO.class);
        HostComponentDesiredStateDAO hostComponentDesiredStateDAO = (HostComponentDesiredStateDAO)injector.getInstance(HostComponentDesiredStateDAO.class);
        List<HostComponentDesiredStateEntity> hostComponentDesiredStates = hostComponentDesiredStateDAO.findAll();
        List<HostComponentStateEntity> hostComponentStates = hostComponentStateDAO.findAll();
        HashSet<HostComponentDesiredStateEntity> missedHostComponentDesiredStates = new HashSet<HostComponentDesiredStateEntity>();
        missedHostComponentDesiredStates.addAll(hostComponentDesiredStates);
        HashSet<HostComponentStateEntity> missedHostComponentStates = new HashSet<HostComponentStateEntity>();
        missedHostComponentStates.addAll(hostComponentStates);
        Iterator stateIterator = missedHostComponentStates.iterator();
        block0: while (stateIterator.hasNext()) {
            HostComponentStateEntity hostComponentStateEntity = (HostComponentStateEntity)stateIterator.next();
            Iterator desiredStateIterator = missedHostComponentDesiredStates.iterator();
            while (desiredStateIterator.hasNext()) {
                HostComponentDesiredStateEntity hostComponentDesiredStateEntity = (HostComponentDesiredStateEntity)desiredStateIterator.next();
                if (!hostComponentStateEntity.getComponentName().equals(hostComponentDesiredStateEntity.getComponentName()) || !hostComponentStateEntity.getServiceName().equals(hostComponentDesiredStateEntity.getServiceName()) || !hostComponentStateEntity.getHostId().equals(hostComponentDesiredStateEntity.getHostId())) continue;
                desiredStateIterator.remove();
                stateIterator.remove();
                continue block0;
            }
        }
        for (HostComponentDesiredStateEntity hostComponentDesiredStateEntity : missedHostComponentDesiredStates) {
            stateEntity = new HostComponentStateEntity();
            ((HostComponentStateEntity)stateEntity).setClusterId(hostComponentDesiredStateEntity.getClusterId());
            ((HostComponentStateEntity)stateEntity).setComponentName(hostComponentDesiredStateEntity.getComponentName());
            ((HostComponentStateEntity)stateEntity).setServiceName(hostComponentDesiredStateEntity.getServiceName());
            ((HostComponentStateEntity)stateEntity).setVersion(State.UNKNOWN.toString());
            ((HostComponentStateEntity)stateEntity).setHostEntity(hostComponentDesiredStateEntity.getHostEntity());
            ((HostComponentStateEntity)stateEntity).setCurrentState(State.UNKNOWN);
            ((HostComponentStateEntity)stateEntity).setUpgradeState(UpgradeState.NONE);
            ((HostComponentStateEntity)stateEntity).setServiceComponentDesiredStateEntity(hostComponentDesiredStateEntity.getServiceComponentDesiredStateEntity());
            LOG.error("Trying to add missing record in hostcomponentstate: {}", stateEntity);
            hostComponentStateDAO.create((HostComponentStateEntity)stateEntity);
        }
        for (HostComponentStateEntity missedHostComponentState : missedHostComponentStates) {
            stateEntity = new HostComponentDesiredStateEntity();
            ((HostComponentDesiredStateEntity)stateEntity).setClusterId(missedHostComponentState.getClusterId());
            ((HostComponentDesiredStateEntity)stateEntity).setComponentName(missedHostComponentState.getComponentName());
            ((HostComponentDesiredStateEntity)stateEntity).setServiceName(missedHostComponentState.getServiceName());
            ((HostComponentDesiredStateEntity)stateEntity).setHostEntity(missedHostComponentState.getHostEntity());
            ((HostComponentDesiredStateEntity)stateEntity).setDesiredState(State.UNKNOWN);
            ((HostComponentDesiredStateEntity)stateEntity).setServiceComponentDesiredStateEntity(missedHostComponentState.getServiceComponentDesiredStateEntity());
            LOG.error("Trying to add missing record in hostcomponentdesiredstate: {}", stateEntity);
            hostComponentDesiredStateDAO.create((HostComponentDesiredStateEntity)stateEntity);
        }
    }

    static void checkSchemaName() {
        Configuration conf = (Configuration)injector.getInstance(Configuration.class);
        if (conf.getDatabaseType() == Configuration.DatabaseType.POSTGRES) {
            LOG.info("Ensuring that the schema set for Postgres is correct");
            DatabaseConsistencyCheckHelper.ensureConnection();
            try (ResultSet schemaRs = connection.getMetaData().getSchemas();
                 ResultSet searchPathRs = connection.createStatement().executeQuery("show search_path");
                 ResultSet ambariTablesRs = connection.createStatement().executeQuery("select table_schema from information_schema.tables where table_name = 'hostcomponentdesiredstate'");){
                String firstSearchPathItem;
                ArrayList<Object> searchPathResultColumn;
                boolean ambariSchemaExists = DatabaseConsistencyCheckHelper.getResultSetColumn(schemaRs, "TABLE_SCHEM").contains(conf.getDatabaseSchema());
                if (!ambariSchemaExists) {
                    DatabaseConsistencyCheckHelper.warning("The schema [{}] defined for OBDP from obdp.properties has not been found in the database. Storing OBDP tables under a different schema can lead to problems.", conf.getDatabaseSchema());
                }
                ImmutableList searchPath = (searchPathResultColumn = DatabaseConsistencyCheckHelper.getResultSetColumn(searchPathRs, "search_path")).isEmpty() ? ImmutableList.of() : ImmutableList.copyOf((Iterable)Splitter.on((String)",").trimResults().split((CharSequence)String.valueOf(searchPathResultColumn.get(0))));
                String string = firstSearchPathItem = searchPath.isEmpty() ? null : (String)searchPath.get(0);
                if (!Objects.equals(firstSearchPathItem, conf.getDatabaseSchema())) {
                    DatabaseConsistencyCheckHelper.warning("The schema [{}] defined for OBDP in obdp.properties is not first on the search path: {}. This can lead to problems.", conf.getDatabaseSchema(), searchPath);
                }
                ArrayList<Object> schemasWithAmbariTables = DatabaseConsistencyCheckHelper.getResultSetColumn(ambariTablesRs, "table_schema");
                if (ambariSchemaExists && !schemasWithAmbariTables.contains(conf.getDatabaseSchema())) {
                    DatabaseConsistencyCheckHelper.warning("The schema [{}] defined for OBDP from obdp.properties does not contain the OBDP tables. Storing OBDP tables under a different schema can lead to problems.", conf.getDatabaseSchema());
                }
                if (schemasWithAmbariTables.size() > 1) {
                    DatabaseConsistencyCheckHelper.warning("Multiple schemas contain the OBDP tables: {}. This can lead to problems.", schemasWithAmbariTables);
                }
            }
            catch (SQLException e) {
                DatabaseConsistencyCheckHelper.warning("Exception occurred during checking db schema name: ", e);
            }
        }
    }

    private static ArrayList<Object> getResultSetColumn(@Nullable ResultSet rs, String columnName) throws SQLException {
        ArrayList<Object> values = new ArrayList<Object>();
        if (null != rs) {
            while (rs.next()) {
                values.add(rs.getObject(columnName));
            }
        }
        return values;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    static void checkMySQLEngine() {
        Configuration conf = (Configuration)injector.getInstance(Configuration.class);
        if (conf.getDatabaseType() != Configuration.DatabaseType.MYSQL) {
            return;
        }
        LOG.info("Checking to ensure that the MySQL DB engine type is set to InnoDB");
        DatabaseConsistencyCheckHelper.ensureConnection();
        String GET_INNODB_ENGINE_SUPPORT = "select TABLE_NAME, ENGINE from information_schema.tables where TABLE_SCHEMA = '%s' and LOWER(ENGINE) != 'innodb';";
        ResultSet rs = null;
        try {
            Statement statement = connection.createStatement(1005, 1008);
            rs = statement.executeQuery(String.format(GET_INNODB_ENGINE_SUPPORT, conf.getDatabaseSchema()));
            if (rs != null) {
                ArrayList<String> tablesInfo = new ArrayList<String>();
                while (rs.next()) {
                    tablesInfo.add(rs.getString("TABLE_NAME"));
                }
                if (!tablesInfo.isEmpty()) {
                    DatabaseConsistencyCheckHelper.warning("Found tables with engine type that is not InnoDB : {}", tablesInfo);
                }
            }
        }
        catch (SQLException e) {
            DatabaseConsistencyCheckHelper.warning("Exception occurred during checking MySQL engine to be innodb: ", e);
        }
        finally {
            if (rs != null) {
                try {
                    rs.close();
                }
                catch (SQLException e) {
                    LOG.error("Exception occurred during result set closing procedure: ", (Throwable)e);
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    static Map<String, String> checkForStalealertdefs() {
        Configuration conf = (Configuration)injector.getInstance(Configuration.class);
        HashMap<String, String> alertInfo = new HashMap<String, String>();
        LOG.info("Checking to ensure there is no stale alert definitions");
        DatabaseConsistencyCheckHelper.ensureConnection();
        String STALE_ALERT_DEFINITIONS = "select definition_name, service_name from alert_definition where service_name not in (select service_name from clusterservices) and service_name not in ('OBDP')";
        ResultSet rs = null;
        try {
            Statement statement = connection.createStatement(1005, 1008);
            rs = statement.executeQuery(STALE_ALERT_DEFINITIONS);
            if (rs != null) {
                while (rs.next()) {
                    alertInfo.put(rs.getString("definition_name"), rs.getString("service_name"));
                }
                if (!alertInfo.isEmpty()) {
                    Object alertInfoStr = "";
                    for (Map.Entry entry : alertInfo.entrySet()) {
                        alertInfoStr = (String)entry.getKey() + "(" + (String)entry.getValue() + ")";
                    }
                    DatabaseConsistencyCheckHelper.warning("You have Alerts that are not mapped with any services : {}.Run --auto-fix-database to fix this automatically. Please backup OBDP Server database before running --auto-fix-database.", alertInfoStr);
                }
            }
        }
        catch (SQLException e) {
            DatabaseConsistencyCheckHelper.warning("Exception occurred during checking for stale alert definitions: ", e);
        }
        finally {
            if (rs != null) {
                try {
                    rs.close();
                }
                catch (SQLException e) {
                    LOG.error("Exception occurred during  checking for stale alert definitions: ", (Throwable)e);
                }
            }
        }
        return alertInfo;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Transactional
    static void fixConfigsSelectedMoreThanOnce() {
        LOG.info("Fix configs selected more than once");
        ClusterDAO clusterDAO = (ClusterDAO)injector.getInstance(ClusterDAO.class);
        Clusters clusters = (Clusters)injector.getInstance(Clusters.class);
        HashMultimap clusterConfigTypeMap = HashMultimap.create();
        ResultSet rs = null;
        Statement statement = null;
        DatabaseConsistencyCheckHelper.ensureConnection();
        try {
            statement = connection.createStatement(1005, 1008);
            rs = statement.executeQuery(GET_CONFIGS_SELECTED_MORE_THAN_ONCE_QUERY);
            if (rs != null) {
                while (rs.next()) {
                    clusterConfigTypeMap.put((Object)rs.getString("cluster_name"), (Object)rs.getString("type_name"));
                }
            }
        }
        catch (SQLException e) {
            DatabaseConsistencyCheckHelper.warning("Exception occurred during check for config selected more than once procedure: ", e);
        }
        finally {
            if (rs != null) {
                try {
                    rs.close();
                }
                catch (SQLException e) {
                    LOG.error("Exception occurred during result set closing procedure: ", (Throwable)e);
                }
            }
            if (statement != null) {
                try {
                    statement.close();
                }
                catch (SQLException e) {
                    LOG.error("Exception occurred during statement closing procedure: ", (Throwable)e);
                }
            }
        }
        for (String clusterName : clusterConfigTypeMap.keySet()) {
            Cluster cluster = null;
            try {
                cluster = clusters.getCluster(clusterName);
                Collection typesWithMultipleSelectedConfigs = clusterConfigTypeMap.get((Object)clusterName);
                for (String type : typesWithMultipleSelectedConfigs) {
                    List<ClusterConfigEntity> enabledConfigsByType = DatabaseConsistencyCheckHelper.getEnabledConfigsByType(cluster.getClusterId(), type);
                    ClusterConfigEntity latestConfig = enabledConfigsByType.get(0);
                    for (ClusterConfigEntity entity : enabledConfigsByType) {
                        entity.setSelected(false);
                        if (latestConfig.getSelectedTimestamp() < entity.getSelectedTimestamp()) {
                            latestConfig = entity;
                        }
                        clusterDAO.merge(entity, true);
                    }
                    latestConfig.setSelected(true);
                    clusterDAO.merge(latestConfig, true);
                }
            }
            catch (OBDPException e) {
                DatabaseConsistencyCheckHelper.warning("Exception occurred during fix for config selected more than once procedure: ", new Object[]{e});
            }
        }
    }

    private static List<ClusterConfigEntity> getEnabledConfigsByType(long clusterId, String type) {
        Provider entityManagerProvider = injector.getProvider(EntityManager.class);
        EntityManager entityManager = (EntityManager)entityManagerProvider.get();
        TypedQuery query = entityManager.createNamedQuery("ClusterConfigEntity.findEnabledConfigByType", ClusterConfigEntity.class);
        query.setParameter("clusterId", (Object)clusterId);
        query.setParameter("type", (Object)type);
        return query.getResultList();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    static void checkServiceConfigs() {
        LOG.info("Checking services and their configs");
        String GET_SERVICES_WITHOUT_CONFIGS_QUERY = "select c.cluster_name, service_name from clusterservices cs join clusters c on cs.cluster_id=c.cluster_id where service_name not in (select service_name from serviceconfig sc where sc.cluster_id=cs.cluster_id and sc.service_name=cs.service_name and sc.group_id is null)";
        String GET_SERVICE_CONFIG_WITHOUT_MAPPING_QUERY = "select c.cluster_name, sc.service_name, sc.version from serviceconfig sc join clusters c on sc.cluster_id=c.cluster_id where service_config_id not in (select service_config_id from serviceconfigmapping) and group_id is null";
        String GET_STACK_NAME_VERSION_QUERY = "select c.cluster_name, s.stack_name, s.stack_version from clusters c join stack s on c.desired_stack_id = s.stack_id";
        String GET_SERVICES_WITH_CONFIGS_QUERY = "select c.cluster_name, cs.service_name, cc.type_name, sc.version from clusterservices cs join serviceconfig sc on cs.service_name=sc.service_name and cs.cluster_id=sc.cluster_id join serviceconfigmapping scm on sc.service_config_id=scm.service_config_id join clusterconfig cc on scm.config_id=cc.config_id and sc.cluster_id=cc.cluster_id join clusters c on cc.cluster_id=c.cluster_id and sc.stack_id=c.desired_stack_id where sc.group_id is null and sc.service_config_id=(select max(service_config_id) from serviceconfig sc2 where sc2.service_name=sc.service_name and sc2.cluster_id=sc.cluster_id) group by c.cluster_name, cs.service_name, cc.type_name, sc.version";
        String GET_NOT_SELECTED_SERVICE_CONFIGS_QUERY = "select c.cluster_name, cs.service_name, cc.type_name from clusterservices cs join serviceconfig sc on cs.service_name=sc.service_name and cs.cluster_id=sc.cluster_id join serviceconfigmapping scm on sc.service_config_id=scm.service_config_id join clusterconfig cc on scm.config_id=cc.config_id and cc.cluster_id=sc.cluster_id join clusters c on cc.cluster_id=c.cluster_id where sc.group_id is null and sc.service_config_id = (select max(service_config_id) from serviceconfig sc2 where sc2.service_name=sc.service_name and sc2.cluster_id=sc.cluster_id) group by c.cluster_name, cs.service_name, cc.type_name having sum(cc.selected) < 1";
        HashMultimap clusterServiceMap = HashMultimap.create();
        HashMap clusterStackInfo = new HashMap();
        HashMap<String, HashMultimap> clusterServiceVersionMap = new HashMap<String, HashMultimap>();
        HashMap<String, HashMultimap> clusterServiceConfigType = new HashMap<String, HashMultimap>();
        ResultSet rs = null;
        Statement statement = null;
        DatabaseConsistencyCheckHelper.ensureConnection();
        LOG.info("Getting OBDP metainfo instance");
        if (obdpMetaInfo == null) {
            obdpMetaInfo = (OBDPMetaInfo)injector.getInstance(OBDPMetaInfo.class);
        }
        try {
            String clusterName;
            String configType;
            Object serviceName;
            LOG.info("Executing query 'GET_SERVICES_WITHOUT_CONFIGS'");
            statement = connection.createStatement(1005, 1008);
            rs = statement.executeQuery(GET_SERVICES_WITHOUT_CONFIGS_QUERY);
            if (rs != null) {
                while (rs.next()) {
                    clusterServiceMap.put((Object)rs.getString("cluster_name"), (Object)rs.getString("service_name"));
                }
                for (String clusterName2 : clusterServiceMap.keySet()) {
                    DatabaseConsistencyCheckHelper.warning("Service(s): {}, from cluster {} has no config(s) in serviceconfig table!", StringUtils.join((Collection)clusterServiceMap.get((Object)clusterName2), (String)","), clusterName2);
                }
            }
            LOG.info("Executing query 'GET_SERVICE_CONFIG_WITHOUT_MAPPING'");
            rs = statement.executeQuery(GET_SERVICE_CONFIG_WITHOUT_MAPPING_QUERY);
            if (rs != null) {
                String serviceName2 = null;
                String version = null;
                String clusterName3 = null;
                while (rs.next()) {
                    HashMultimap serviceVersion;
                    serviceName2 = rs.getString("service_name");
                    clusterName3 = rs.getString("cluster_name");
                    version = rs.getString("version");
                    if (clusterServiceVersionMap.get(clusterName3) != null) {
                        serviceVersion = (Multimap)clusterServiceVersionMap.get(clusterName3);
                        serviceVersion.put((Object)serviceName2, (Object)version);
                        continue;
                    }
                    serviceVersion = HashMultimap.create();
                    serviceVersion.put((Object)serviceName2, (Object)version);
                    clusterServiceVersionMap.put(clusterName3, serviceVersion);
                }
                for (String clName : clusterServiceVersionMap.keySet()) {
                    Multimap serviceVersion = (Multimap)clusterServiceVersionMap.get(clName);
                    for (String servName : serviceVersion.keySet()) {
                        DatabaseConsistencyCheckHelper.warning("In cluster {}, service config mapping is unavailable (in table serviceconfigmapping) for service {} with version(s) {}! ", clName, servName, StringUtils.join((Collection)serviceVersion.get((Object)servName), (String)","));
                    }
                }
            }
            LOG.info("Getting stack info from database");
            rs = statement.executeQuery(GET_STACK_NAME_VERSION_QUERY);
            if (rs != null) {
                while (rs.next()) {
                    HashMap<String, String> stackInfoMap = new HashMap<String, String>();
                    stackInfoMap.put(rs.getString("stack_name"), rs.getString("stack_version"));
                    clusterStackInfo.put(rs.getString("cluster_name"), stackInfoMap);
                }
            }
            HashSet<Object> serviceNames = new HashSet<Object>();
            HashMap<String, Map<Integer, HashMultimap>> dbClusterServiceVersionConfigs = new HashMap<String, Map<Integer, HashMultimap>>();
            HashMultimap stackServiceConfigs = HashMultimap.create();
            LOG.info("Executing query 'GET_SERVICES_WITH_CONFIGS'");
            rs = statement.executeQuery(GET_SERVICES_WITH_CONFIGS_QUERY);
            if (rs != null) {
                serviceName = null;
                configType = null;
                clusterName = null;
                Integer serviceVersion = null;
                while (rs.next()) {
                    HashMultimap dbServiceConfigs;
                    Map<Integer, HashMultimap> dbServiceVersionConfigs;
                    clusterName = rs.getString("cluster_name");
                    serviceName = rs.getString("service_name");
                    configType = rs.getString("type_name");
                    serviceVersion = rs.getInt("version");
                    serviceNames.add(serviceName);
                    if (dbClusterServiceVersionConfigs.get(clusterName) != null) {
                        dbServiceVersionConfigs = (Map)dbClusterServiceVersionConfigs.get(clusterName);
                        if (dbServiceVersionConfigs.get(serviceVersion) != null) {
                            ((Multimap)dbServiceVersionConfigs.get(serviceVersion)).put(serviceName, (Object)configType);
                            continue;
                        }
                        dbServiceConfigs = HashMultimap.create();
                        dbServiceConfigs.put(serviceName, (Object)configType);
                        dbServiceVersionConfigs.put(serviceVersion, dbServiceConfigs);
                        continue;
                    }
                    dbServiceVersionConfigs = new HashMap();
                    dbServiceConfigs = HashMultimap.create();
                    dbServiceConfigs.put(serviceName, (Object)configType);
                    dbServiceVersionConfigs.put(serviceVersion, dbServiceConfigs);
                    dbClusterServiceVersionConfigs.put(clusterName, dbServiceVersionConfigs);
                }
            }
            LOG.info("Comparing service configs from stack with configs that we got from db");
            for (Map.Entry clusterStackInfoEntry : clusterStackInfo.entrySet()) {
                clusterName = (String)clusterStackInfoEntry.getKey();
                Map stackInfo = (Map)clusterStackInfoEntry.getValue();
                String stackName = (String)stackInfo.keySet().iterator().next();
                String stackVersion = (String)stackInfo.get(stackName);
                LOG.info("Getting services from metainfo");
                Map<String, ServiceInfo> serviceInfoMap = obdpMetaInfo.getServices(stackName, stackVersion);
                for (String string : serviceNames) {
                    LOG.info("Processing {}-{} / {}", new Object[]{stackName, stackVersion, string});
                    ServiceInfo serviceInfo = serviceInfoMap.get(string);
                    if (serviceInfo != null) {
                        Set<String> configTypes = serviceInfo.getConfigTypeAttributes().keySet();
                        for (String configType2 : configTypes) {
                            stackServiceConfigs.put((Object)string, (Object)configType2);
                        }
                        continue;
                    }
                    DatabaseConsistencyCheckHelper.warning("Service {} is not available for stack {} in cluster {}", string, stackName + "-" + stackVersion, clusterName);
                }
                LOG.info("Comparing required service configs from stack with mapped service configs from db");
                Map dbServiceVersionConfigs = (Map)dbClusterServiceVersionConfigs.get(clusterName);
                if (dbServiceVersionConfigs == null) continue;
                for (Integer serviceVersion : dbServiceVersionConfigs.keySet()) {
                    Multimap dbServiceConfigs = (Multimap)dbServiceVersionConfigs.get(serviceVersion);
                    if (dbServiceConfigs == null) continue;
                    for (String serviceName4 : dbServiceConfigs.keySet()) {
                        ServiceInfo serviceInfo = serviceInfoMap.get(serviceName4);
                        Collection serviceConfigsFromStack = stackServiceConfigs.get((Object)serviceName4);
                        Collection serviceConfigsFromDB = dbServiceConfigs.get((Object)serviceName4);
                        if (serviceConfigsFromDB == null || serviceConfigsFromStack == null) continue;
                        serviceConfigsFromStack.removeAll(serviceConfigsFromDB);
                        if (serviceInfo != null && serviceInfo.getComponents() != null) {
                            for (ComponentInfo componentInfo : serviceInfo.getComponents()) {
                                if (componentInfo.getClientConfigFiles() == null) continue;
                                for (ClientConfigFileDefinition clientConfigFileDefinition : componentInfo.getClientConfigFiles()) {
                                    if (!clientConfigFileDefinition.isOptional()) continue;
                                    serviceConfigsFromStack.remove(clientConfigFileDefinition.getDictionaryName());
                                }
                            }
                        }
                        if (!dbServiceConfigs.containsKey((Object)"RANGER")) {
                            DatabaseConsistencyCheckHelper.removeStringsByRegexp(serviceConfigsFromStack, "^ranger-" + serviceName4.toLowerCase() + "-*");
                        }
                        if (serviceConfigsFromStack.isEmpty()) continue;
                        DatabaseConsistencyCheckHelper.warning("Required config(s): {} is(are) not available for service {} with service config version {} in cluster {}", StringUtils.join((Collection)serviceConfigsFromStack, (String)","), serviceName4, Integer.toString(serviceVersion), clusterName);
                    }
                }
            }
            LOG.info("Getting services which has mapped configs which are not selected in clusterconfig");
            rs = statement.executeQuery(GET_NOT_SELECTED_SERVICE_CONFIGS_QUERY);
            if (rs != null) {
                serviceName = null;
                configType = null;
                clusterName = null;
                while (rs.next()) {
                    HashMultimap serviceConfigs;
                    clusterName = rs.getString("cluster_name");
                    serviceName = rs.getString("service_name");
                    configType = rs.getString("type_name");
                    if (clusterServiceConfigType.get(clusterName) != null) {
                        serviceConfigs = (Multimap)clusterServiceConfigType.get(clusterName);
                        serviceConfigs.put(serviceName, (Object)configType);
                        continue;
                    }
                    serviceConfigs = HashMultimap.create();
                    serviceConfigs.put(serviceName, (Object)configType);
                    clusterServiceConfigType.put(clusterName, serviceConfigs);
                }
            }
            for (String clusterName4 : clusterServiceConfigType.keySet()) {
                Multimap serviceConfig = (Multimap)clusterServiceConfigType.get(clusterName4);
                for (String serviceName5 : serviceConfig.keySet()) {
                    DatabaseConsistencyCheckHelper.warning("You have non selected configs: {} for service {} from cluster {}!", StringUtils.join((Collection)serviceConfig.get((Object)serviceName5), (String)","), serviceName5, clusterName4);
                }
            }
        }
        catch (OBDPException | SQLException e) {
            DatabaseConsistencyCheckHelper.warning("Exception occurred during complex service check procedure: ", e);
        }
        finally {
            if (rs != null) {
                try {
                    rs.close();
                }
                catch (SQLException e) {
                    LOG.error("Exception occurred during result set closing procedure: ", (Throwable)e);
                }
            }
            if (statement != null) {
                try {
                    statement.close();
                }
                catch (SQLException e) {
                    LOG.error("Exception occurred during statement closing procedure: ", (Throwable)e);
                }
            }
        }
    }

    static Map<Long, ConfigGroup> collectConfigGroupsWithoutServiceName() {
        HashMap<Long, ConfigGroup> configGroupMap = new HashMap<Long, ConfigGroup>();
        Clusters clusters = (Clusters)injector.getInstance(Clusters.class);
        Map<String, Cluster> clusterMap = clusters.getClusters();
        if (MapUtils.isEmpty(clusterMap)) {
            return configGroupMap;
        }
        for (Cluster cluster : clusterMap.values()) {
            Map<Long, ConfigGroup> configGroups = cluster.getConfigGroups();
            if (MapUtils.isEmpty(configGroups)) continue;
            for (ConfigGroup configGroup : configGroups.values()) {
                if (!StringUtils.isEmpty((String)configGroup.getServiceName())) continue;
                configGroupMap.put(configGroup.getId(), configGroup);
            }
        }
        return configGroupMap;
    }

    static void checkConfigGroupsHasServiceName() {
        Map<Long, ConfigGroup> configGroupMap = DatabaseConsistencyCheckHelper.collectConfigGroupsWithoutServiceName();
        if (MapUtils.isEmpty(configGroupMap)) {
            return;
        }
        String message = String.join((CharSequence)" ), ( ", configGroupMap.values().stream().map(ConfigGroup::getName).collect(Collectors.toList()));
        DatabaseConsistencyCheckHelper.warning("You have config groups present in the database with no service name, [(ConfigGroup) => ( {} )]. Run --auto-fix-database to fix this automatically. Please backup OBDP Server database before running --auto-fix-database.", message);
    }

    @Transactional
    static void fixConfigGroupServiceNames() {
        Map<Long, ConfigGroup> configGroupMap = DatabaseConsistencyCheckHelper.collectConfigGroupsWithoutServiceName();
        if (MapUtils.isEmpty(configGroupMap)) {
            return;
        }
        Clusters clusters = (Clusters)injector.getInstance(Clusters.class);
        for (Map.Entry<Long, ConfigGroup> configGroupEntry : configGroupMap.entrySet()) {
            ConfigGroup configGroup = configGroupEntry.getValue();
            try {
                Cluster cluster = clusters.getCluster(configGroup.getClusterName());
                Map<String, Service> serviceMap = cluster.getServices();
                if (serviceMap.containsKey(configGroup.getTag())) {
                    LOG.info("Setting service name of config group {} with id {} to {}", new Object[]{configGroup.getName(), configGroupEntry.getKey(), configGroup.getTag()});
                    configGroup.setServiceName(configGroup.getTag());
                    continue;
                }
                LOG.info("Config group {} with id {} contains a tag {} which is not a service name in the cluster {}", new Object[]{configGroup.getName(), configGroupEntry.getKey(), configGroup.getTag(), cluster.getClusterName()});
            }
            catch (OBDPException oBDPException) {}
        }
    }

    static Map<Long, Set<Long>> checkConfigGroupHostMapping(boolean warnIfFound) {
        LOG.info("Checking config group host mappings");
        HashMap<Long, Set<Long>> nonMappedHostIds = new HashMap<Long, Set<Long>>();
        Clusters clusters = (Clusters)injector.getInstance(Clusters.class);
        Map<String, Cluster> clusterMap = clusters.getClusters();
        StringBuilder output = new StringBuilder("[(ConfigGroup, Service, HostCount) => ");
        if (!MapUtils.isEmpty(clusterMap)) {
            for (Cluster cluster : clusterMap.values()) {
                Map<Long, ConfigGroup> configGroups = cluster.getConfigGroups();
                Map<String, Host> clusterHosts = clusters.getHostsForCluster(cluster.getClusterName());
                if (MapUtils.isEmpty(configGroups) || MapUtils.isEmpty(clusterHosts)) continue;
                for (ConfigGroup configGroup : configGroups.values()) {
                    Map<Long, Host> hosts = configGroup.getHosts();
                    boolean addToOutput = false;
                    HashSet<String> hostnames = new HashSet<String>();
                    if (!MapUtils.isEmpty(hosts)) {
                        for (Host host : hosts.values()) {
                            if (clusterHosts.containsKey(host.getHostName())) continue;
                            Set hostIds = nonMappedHostIds.computeIfAbsent(configGroup.getId(), configGroupId -> new HashSet());
                            hostIds.add(host.getHostId());
                            hostnames.add(host.getHostName());
                            addToOutput = true;
                        }
                    }
                    if (!addToOutput) continue;
                    output.append("( ");
                    output.append(configGroup.getName());
                    output.append(", ");
                    output.append(configGroup.getTag());
                    output.append(", ");
                    output.append(hostnames);
                    output.append(" ), ");
                }
            }
        }
        if (!MapUtils.isEmpty(nonMappedHostIds) && warnIfFound) {
            output.replace(output.lastIndexOf(","), output.length(), "]");
            DatabaseConsistencyCheckHelper.warning("You have config group host mappings with hosts that are no longer associated with the cluster, {}. Run --auto-fix-database to fix this automatically. Alternatively, you can remove this mapping from the UI. Please backup OBDP Server database before running --auto-fix-database.", output.toString());
        }
        return nonMappedHostIds;
    }

    static Map<Long, ConfigGroup> checkConfigGroupsForDeletedServices(boolean warnIfFound) {
        HashMap<Long, ConfigGroup> configGroupMap = new HashMap<Long, ConfigGroup>();
        Clusters clusters = (Clusters)injector.getInstance(Clusters.class);
        Map<String, Cluster> clusterMap = clusters.getClusters();
        StringBuilder output = new StringBuilder("[(ConfigGroup, Service) => ");
        if (!MapUtils.isEmpty(clusterMap)) {
            for (Cluster cluster : clusterMap.values()) {
                Map<Long, ConfigGroup> configGroups = cluster.getConfigGroups();
                Map<String, Service> services = cluster.getServices();
                if (MapUtils.isEmpty(configGroups)) continue;
                for (ConfigGroup configGroup : configGroups.values()) {
                    if (services.containsKey(configGroup.getServiceName())) continue;
                    configGroupMap.put(configGroup.getId(), configGroup);
                    output.append("( ");
                    output.append(configGroup.getName());
                    output.append(", ");
                    output.append(configGroup.getServiceName());
                    output.append(" ), ");
                }
            }
        }
        if (warnIfFound && !configGroupMap.isEmpty()) {
            output.replace(output.lastIndexOf(","), output.length(), "]");
            DatabaseConsistencyCheckHelper.warning("You have config groups present in the database with no corresponding service found, {}. Run --auto-fix-database to fix this automatically. Please backup OBDP Server database before running --auto-fix-database.", output.toString());
        }
        return configGroupMap;
    }

    @Transactional
    static void fixConfigGroupsForDeletedServices() {
        Map<Long, ConfigGroup> configGroupMap = DatabaseConsistencyCheckHelper.checkConfigGroupsForDeletedServices(false);
        Clusters clusters = (Clusters)injector.getInstance(Clusters.class);
        if (!MapUtils.isEmpty(configGroupMap)) {
            for (Map.Entry<Long, ConfigGroup> configGroupEntry : configGroupMap.entrySet()) {
                Long id = configGroupEntry.getKey();
                ConfigGroup configGroup = configGroupEntry.getValue();
                if (!StringUtils.isEmpty((String)configGroup.getServiceName())) {
                    LOG.info("Deleting config group {} with id {} for deleted service {}", new Object[]{configGroup.getName(), id, configGroup.getServiceName()});
                    try {
                        Cluster cluster = clusters.getCluster(configGroup.getClusterName());
                        cluster.deleteConfigGroup(id);
                    }
                    catch (OBDPException oBDPException) {}
                    continue;
                }
                DatabaseConsistencyCheckHelper.warning("The config group {} with id {} can not be fixed automatically because service name is missing.", configGroup.getName(), id);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Transactional
    static void fixAlertsForDeletedServices() {
        Configuration conf = (Configuration)injector.getInstance(Configuration.class);
        LOG.info("fixAlertsForDeletedServices stale alert definitions for deleted services");
        DatabaseConsistencyCheckHelper.ensureConnection();
        String SELECT_STALE_ALERT_DEFINITIONS = "select definition_id from alert_definition where service_name not in (select service_name from clusterservices) and service_name not in ('OBDP')";
        boolean recordsCount = false;
        Statement statement = null;
        ResultSet rs = null;
        ArrayList<Integer> alertIds = new ArrayList<Integer>();
        try {
            statement = connection.createStatement(1005, 1008);
            rs = statement.executeQuery(SELECT_STALE_ALERT_DEFINITIONS);
            while (rs.next()) {
                alertIds.add(rs.getInt("definition_id"));
            }
        }
        catch (SQLException e) {
            DatabaseConsistencyCheckHelper.warning("Exception occurred during fixing stale alert definitions: ", e);
        }
        finally {
            if (statement != null) {
                try {
                    statement.close();
                }
                catch (SQLException e) {
                    LOG.error("Exception occurred during fixing stale alert definitions: ", (Throwable)e);
                }
            }
            if (rs != null) {
                try {
                    rs.close();
                }
                catch (SQLException e) {
                    LOG.error("Exception occurred during fixing stale alert definitions: ", (Throwable)e);
                }
            }
        }
        for (Integer alertId : alertIds) {
            AlertDefinitionEntity entity = alertDefinitionDAO.findById(alertId.intValue());
            alertDefinitionDAO.remove(entity);
        }
        DatabaseConsistencyCheckHelper.warning("fixAlertsForDeletedServices - {}  Stale alerts were deleted", alertIds.size());
    }

    @Transactional
    static void fixConfigGroupHostMappings() {
        Map<Long, Set<Long>> nonMappedHostIds = DatabaseConsistencyCheckHelper.checkConfigGroupHostMapping(false);
        Clusters clusters = (Clusters)injector.getInstance(Clusters.class);
        if (!MapUtils.isEmpty(nonMappedHostIds)) {
            LOG.info("Fixing {} config groups with inconsistent host mappings", (Object)nonMappedHostIds.size());
            for (Map.Entry<Long, Set<Long>> nonMappedHostEntry : nonMappedHostIds.entrySet()) {
                if (MapUtils.isEmpty(clusters.getClusters())) continue;
                for (Cluster cluster : clusters.getClusters().values()) {
                    Map<Long, ConfigGroup> configGroups = cluster.getConfigGroups();
                    if (MapUtils.isEmpty(configGroups)) continue;
                    ConfigGroup configGroup = configGroups.get(nonMappedHostEntry.getKey());
                    if (configGroup != null) {
                        for (Long hostId : nonMappedHostEntry.getValue()) {
                            try {
                                configGroup.removeHost(hostId);
                            }
                            catch (OBDPException e) {
                                LOG.warn("Unable to fix inconsistency by removing host mapping for config group: {}, service: {}, hostId = {}", new Object[]{configGroup.getName(), configGroup.getTag(), hostId});
                            }
                        }
                        continue;
                    }
                    LOG.warn("Unable to find config group with id = {}", (Object)nonMappedHostEntry.getKey());
                }
            }
        }
    }

    private static void ensureConnection() {
        if (connection == null) {
            if (dbAccessor == null) {
                dbAccessor = (DBAccessor)injector.getInstance(DBAccessor.class);
            }
            connection = dbAccessor.getConnection();
        }
    }

    private static void removeStringsByRegexp(Collection<String> stringItems, String regexp) {
        Pattern pattern = Pattern.compile(regexp);
        Iterator<String> iterator = stringItems.iterator();
        while (iterator.hasNext()) {
            String stringItem = iterator.next();
            Matcher matcher = pattern.matcher(stringItem);
            if (!matcher.find()) continue;
            iterator.remove();
        }
    }

    static {
        checkResult = DatabaseConsistencyCheckResult.DB_CHECK_SUCCESS;
    }
}

