/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.hadoop.gcsio;

import com.google.api.client.googleapis.services.AbstractGoogleClientRequest;
import com.google.api.client.http.HttpHeaders;
import com.google.api.client.http.HttpResponse;
import com.google.api.client.util.BackOff;
import com.google.api.client.util.BackOffUtils;
import com.google.api.client.util.ExponentialBackOff;
import com.google.api.client.util.Sleeper;
import com.google.api.services.storage.Storage;
import com.google.api.services.storage.model.StorageObject;
import com.google.cloud.hadoop.gcsio.FileAccessPatternManager;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageExceptions;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageItemInfo;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageReadOptions;
import com.google.cloud.hadoop.gcsio.StorageRequestFactory;
import com.google.cloud.hadoop.gcsio.StorageResourceId;
import com.google.cloud.hadoop.util.ApiErrorExtractor;
import com.google.cloud.hadoop.util.ClientRequestHelper;
import com.google.cloud.hadoop.util.GoogleCloudStorageEventBus;
import com.google.cloud.hadoop.util.ResilientOperation;
import com.google.cloud.hadoop.util.RetryDeterminer;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.base.Suppliers;
import com.google.common.flogger.GoogleLogger;
import com.google.errorprone.annotations.CanIgnoreReturnValue;
import java.io.ByteArrayInputStream;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigInteger;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.SeekableByteChannel;
import java.util.function.Supplier;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;

public class GoogleCloudStorageReadChannel
implements SeekableByteChannel {
    private static final GoogleLogger logger = GoogleLogger.forEnclosingClass();
    @VisibleForTesting
    static final int SKIP_BUFFER_SIZE = 8192;
    private static final String GZIP_ENCODING = "gzip";
    private StorageResourceId resourceId;
    private final StorageRequestFactory storageRequestFactory;
    @VisibleForTesting
    ReadableByteChannel contentChannel;
    private boolean channelIsOpen = true;
    protected long currentPosition = 0L;
    protected long contentChannelPosition = -1L;
    private long size = -1L;
    private long contentChannelEnd = -1L;
    private int maxRetries = 10;
    private final ApiErrorExtractor errorExtractor;
    private final ClientRequestHelper<StorageObject> clientRequestHelper;
    private final GoogleCloudStorageReadOptions readOptions;
    private final FileAccessPatternManager fileAccessManager;
    private Sleeper sleeper = Sleeper.DEFAULT;
    private Supplier<BackOff> readBackOff = Suppliers.memoize(this::createBackOff);
    private byte[] skipBuffer = null;
    private boolean gzipEncoded = false;
    private byte[] footerContent;
    @VisibleForTesting
    protected boolean metadataInitialized = false;

    public GoogleCloudStorageReadChannel(Storage gcs, StorageResourceId resourceId, ApiErrorExtractor errorExtractor, ClientRequestHelper<StorageObject> requestHelper, @Nonnull GoogleCloudStorageReadOptions readOptions) throws IOException {
        this.storageRequestFactory = new StorageRequestFactory(gcs);
        this.clientRequestHelper = requestHelper;
        this.errorExtractor = errorExtractor;
        this.readOptions = readOptions;
        this.resourceId = resourceId;
        this.fileAccessManager = new FileAccessPatternManager(resourceId, readOptions);
        GoogleCloudStorageItemInfo info = this.getInitialMetadata();
        if (info != null || readOptions.isFastFailOnNotFoundEnabled()) {
            this.initMetadata(info == null ? this.fetchInitialMetadata() : info);
        }
    }

    @VisibleForTesting
    void setSleeper(Sleeper sleeper) {
        Preconditions.checkArgument((sleeper != null ? 1 : 0) != 0, (Object)"sleeper must not be null!");
        this.sleeper = sleeper;
    }

    void setReadBackOff(BackOff backOff) {
        this.readBackOff = Suppliers.ofInstance((Object)((BackOff)Preconditions.checkNotNull((Object)backOff, (Object)"backOff could not be null")));
    }

    @VisibleForTesting
    ExponentialBackOff createBackOff() {
        return new ExponentialBackOff.Builder().setInitialIntervalMillis(Math.toIntExact(this.readOptions.getBackoffInitialInterval().toMillis())).setRandomizationFactor(this.readOptions.getBackoffRandomizationFactor()).setMultiplier(this.readOptions.getBackoffMultiplier()).setMaxIntervalMillis(Math.toIntExact(this.readOptions.getBackoffMaxInterval().toMillis())).setMaxElapsedTimeMillis(Math.toIntExact(this.readOptions.getBackoffMaxElapsedTime().toMillis())).build();
    }

    @Nullable
    protected GoogleCloudStorageItemInfo getInitialMetadata() throws IOException {
        return null;
    }

    private GoogleCloudStorageItemInfo fetchInitialMetadata() throws IOException {
        StorageObject object;
        try {
            Storage.Objects.Get getObject = this.createMetadataRequest().setFields("contentEncoding,generation,size");
            object = (StorageObject)ResilientOperation.retry(() -> ((Storage.Objects.Get)getObject).execute(), (BackOff)this.readBackOff.get(), (RetryDeterminer)RetryDeterminer.SOCKET_ERRORS, IOException.class, (Sleeper)this.sleeper);
        }
        catch (IOException e) {
            GoogleCloudStorageEventBus.postOnException();
            throw this.errorExtractor.itemNotFound(e) ? GoogleCloudStorageExceptions.createFileNotFoundException(this.resourceId, e) : new IOException("Error reading " + String.valueOf(this.resourceId), e);
        }
        catch (InterruptedException e) {
            GoogleCloudStorageEventBus.postOnException();
            Thread.currentThread().interrupt();
            throw new IOException("Thread interrupt received.", e);
        }
        return GoogleCloudStorageItemInfo.createObject(this.resourceId, 0L, 0L, ((BigInteger)Preconditions.checkNotNull((Object)object.getSize(), (String)"size can not be null for '%s'", (Object)this.resourceId)).longValue(), null, object.getContentEncoding(), null, (Long)Preconditions.checkNotNull((Object)object.getGeneration(), (String)"generation can not be null for '%s'", (Object)this.resourceId), 0L, null);
    }

    public void setMaxRetries(int maxRetries) {
        this.maxRetries = maxRetries;
    }

    @Override
    public int read(ByteBuffer buffer) throws IOException {
        boolean isEndOfStream;
        this.throwIfNotOpen();
        if (buffer.remaining() == 0) {
            return 0;
        }
        ((GoogleLogger.Api)logger.atFiner()).log("Reading %d bytes at %d position from '%s'", (Object)buffer.remaining(), (Object)this.currentPosition, (Object)this.resourceId);
        if (this.currentPosition == this.size) {
            return -1;
        }
        int totalBytesRead = 0;
        int retriesAttempted = 0;
        do {
            int remainingBeforeRead = buffer.remaining();
            this.performLazySeek(remainingBeforeRead);
            Preconditions.checkState((this.contentChannelPosition == this.currentPosition ? 1 : 0) != 0, (String)"contentChannelPosition (%s) should be equal to currentPosition (%s) after lazy seek", (long)this.contentChannelPosition, (long)this.currentPosition);
            try {
                int numBytesRead = this.contentChannel.read(buffer);
                if (numBytesRead == 0) {
                    throw new IOException(String.format("Read 0 bytes without blocking from object: '%s'", this.resourceId));
                }
                if (numBytesRead < 0) {
                    if (this.gzipEncoded) {
                        this.size = this.currentPosition;
                        this.contentChannelEnd = this.currentPosition;
                    }
                    if (this.currentPosition != this.contentChannelEnd && this.currentPosition != this.size) {
                        GoogleCloudStorageEventBus.postOnException();
                        throw new IOException(String.format("Received end of stream result before all the file data has been received; totalBytesRead: %d, currentPosition: %d, contentChannelEnd %d, size: %d, object: '%s'", totalBytesRead, this.currentPosition, this.contentChannelEnd, this.size, this.resourceId));
                    }
                    if (this.contentChannelEnd == this.size || this.currentPosition != this.contentChannelEnd) break;
                    this.closeContentChannel();
                }
                if (numBytesRead > 0) {
                    totalBytesRead += numBytesRead;
                    this.currentPosition += (long)numBytesRead;
                    this.contentChannelPosition += (long)numBytesRead;
                    Preconditions.checkState((this.contentChannelPosition == this.currentPosition ? 1 : 0) != 0, (String)"contentChannelPosition (%s) should be equal to currentPosition (%s) after successful read", (long)this.contentChannelPosition, (long)this.currentPosition);
                }
                if (retriesAttempted != 0) {
                    ((GoogleLogger.Api)logger.atInfo()).log("Success after %d retries on reading '%s'", retriesAttempted, (Object)this.resourceId);
                }
                retriesAttempted = 0;
            }
            catch (IOException ioe) {
                ((GoogleLogger.Api)logger.atFine()).log("Closing contentChannel after %s exception for '%s'.", (Object)ioe.getMessage(), (Object)this.resourceId);
                this.closeContentChannel();
                if (buffer.remaining() != remainingBeforeRead) {
                    int partialRead = remainingBeforeRead - buffer.remaining();
                    ((GoogleLogger.Api)logger.atInfo()).log("Despite exception, had partial read of %d bytes from '%s'; resetting retry count.", partialRead, (Object)this.resourceId);
                    retriesAttempted = 0;
                    totalBytesRead += partialRead;
                    this.currentPosition += (long)partialRead;
                }
                if (retriesAttempted == this.maxRetries) {
                    GoogleCloudStorageEventBus.postOnException();
                    ((GoogleLogger.Api)logger.atSevere()).log("Throwing exception after reaching max read retries (%d) for '%s'.", this.maxRetries, (Object)this.resourceId);
                    throw ioe;
                }
                if (retriesAttempted == 0) {
                    this.readBackOff.get().reset();
                }
                ((GoogleLogger.Api)((GoogleLogger.Api)logger.atWarning()).withCause((Throwable)ioe)).log("Failed read retry #%d/%d for '%s'. Sleeping...", (Object)(++retriesAttempted), (Object)this.maxRetries, (Object)this.resourceId);
                try {
                    boolean backOffSuccessful = BackOffUtils.next((Sleeper)this.sleeper, (BackOff)this.readBackOff.get());
                    if (!backOffSuccessful) {
                        GoogleCloudStorageEventBus.postOnException();
                        ((GoogleLogger.Api)logger.atSevere()).log("BackOff returned false; maximum total elapsed time exhausted. Giving up after %d/%d retries for '%s'", (Object)retriesAttempted, (Object)this.maxRetries, (Object)this.resourceId);
                        throw ioe;
                    }
                }
                catch (InterruptedException ie) {
                    GoogleCloudStorageEventBus.postOnException();
                    Thread.currentThread().interrupt();
                    ((GoogleLogger.Api)logger.atSevere()).log("Interrupted while sleeping before retry. Giving up after %d/%d retries for '%s'", (Object)retriesAttempted, (Object)this.maxRetries, (Object)this.resourceId);
                    ioe.addSuppressed(ie);
                    throw ioe;
                }
                ((GoogleLogger.Api)logger.atInfo()).log("Done sleeping before retry #%d/%d for '%s'", (Object)retriesAttempted, (Object)this.maxRetries, (Object)this.resourceId);
            }
            catch (RuntimeException r) {
                GoogleCloudStorageEventBus.postOnException();
                this.closeContentChannel();
                throw r;
            }
        } while (buffer.remaining() > 0 && this.currentPosition < this.size);
        boolean bl = isEndOfStream = totalBytesRead == 0;
        if (isEndOfStream) {
            if (this.currentPosition != this.size) {
                GoogleCloudStorageEventBus.postOnException();
                throw new IOException(String.format("Failed to read any data before all the file data has been received; currentPosition: %d, size: %d, object '%s'", this.currentPosition, this.size, this.resourceId));
            }
            return -1;
        }
        return totalBytesRead;
    }

    @Override
    public SeekableByteChannel truncate(long size) {
        throw new UnsupportedOperationException("Cannot mutate read-only channel");
    }

    @Override
    public int write(ByteBuffer src) {
        throw new UnsupportedOperationException("Cannot mutate read-only channel");
    }

    @Override
    public boolean isOpen() {
        return this.channelIsOpen;
    }

    protected void closeContentChannel() {
        if (this.contentChannel != null) {
            ((GoogleLogger.Api)logger.atFiner()).log("Closing internal contentChannel for '%s'", (Object)this.resourceId);
            try {
                this.contentChannel.close();
            }
            catch (Exception e) {
                GoogleCloudStorageEventBus.postOnException();
                ((GoogleLogger.Api)((GoogleLogger.Api)logger.atFine()).withCause((Throwable)e)).log("Got an exception on contentChannel.close() for '%s'; ignoring it.", (Object)this.resourceId);
            }
            finally {
                this.contentChannel = null;
                this.fileAccessManager.updateLastServedIndex(this.contentChannelPosition);
                this.resetContentChannel();
            }
        }
    }

    private void resetContentChannel() {
        Preconditions.checkState((this.contentChannel == null ? 1 : 0) != 0, (String)"contentChannel should be null for '%s'", (Object)this.resourceId);
        this.contentChannelPosition = -1L;
        this.contentChannelEnd = -1L;
    }

    @Override
    public void close() {
        if (!this.channelIsOpen) {
            ((GoogleLogger.Api)logger.atFiner()).log("Ignoring close: channel for '%s' is not open.", (Object)this.resourceId);
            return;
        }
        ((GoogleLogger.Api)logger.atFiner()).log("Closing channel for '%s'", (Object)this.resourceId);
        this.channelIsOpen = false;
        this.closeContentChannel();
    }

    @Override
    public long position() throws IOException {
        this.throwIfNotOpen();
        return this.currentPosition;
    }

    @Override
    @CanIgnoreReturnValue
    public SeekableByteChannel position(long newPosition) throws IOException {
        this.throwIfNotOpen();
        if (newPosition == this.currentPosition) {
            return this;
        }
        this.validatePosition(newPosition);
        ((GoogleLogger.Api)logger.atFiner()).log("Seek from %s to %s position for '%s'", (Object)this.currentPosition, (Object)newPosition, (Object)this.resourceId);
        this.currentPosition = newPosition;
        return this;
    }

    private void skipInPlace(long seekDistance) {
        try {
            this.contentChannelPosition += this.skip(this.contentChannel, seekDistance);
        }
        catch (IOException e) {
            GoogleCloudStorageEventBus.postOnException();
            ((GoogleLogger.Api)((GoogleLogger.Api)logger.atInfo()).withCause((Throwable)e)).log("Got an IO exception on contentChannel.read(), a lazy-seek will be pending for '%s'", (Object)this.resourceId);
            this.closeContentChannel();
        }
        Preconditions.checkState((this.contentChannel == null || this.contentChannelPosition == this.currentPosition ? 1 : 0) != 0, (String)"contentChannelPosition (%s) should be equal to currentPosition (%s) after successful in-place skip", (long)this.contentChannelPosition, (long)this.currentPosition);
    }

    private long skip(@Nonnull ReadableByteChannel channel, long bytesToSkip) throws IOException {
        if (this.skipBuffer == null) {
            this.skipBuffer = new byte[8192];
        }
        long totalBytesSkipped = 0L;
        while (bytesToSkip > 0L) {
            int bufferSize = Math.toIntExact(Math.min((long)this.skipBuffer.length, bytesToSkip));
            int bytesRead = channel.read(ByteBuffer.wrap(this.skipBuffer, 0, bufferSize));
            if (bytesRead < 0) {
                throw new EOFException(String.format("Unexpected end of stream trying to skip %d bytes to seek to position %d, size: %d for '%s'", bytesToSkip, this.currentPosition, this.size, this.resourceId));
            }
            bytesToSkip -= (long)bytesRead;
            totalBytesSkipped += (long)bytesRead;
        }
        return totalBytesSkipped;
    }

    @Override
    public long size() throws IOException {
        this.throwIfNotOpen();
        if (!this.metadataInitialized) {
            this.initMetadata(this.fetchInitialMetadata());
        }
        return this.size;
    }

    @VisibleForTesting
    long generation() {
        return this.resourceId.getGenerationId();
    }

    protected void setSize(long size) {
        this.size = size;
    }

    private void checkEncodingAndAccess() {
        Preconditions.checkState((!this.gzipEncoded || !this.fileAccessManager.shouldAdaptToRandomAccess() ? 1 : 0) != 0, (String)"gzipEncoded and randomAccess should not be true at the same time for '%s'", (Object)this.resourceId);
    }

    protected void validatePosition(long position) throws IOException {
        if (position < 0L) {
            GoogleCloudStorageEventBus.postOnException();
            throw new EOFException(String.format("Invalid seek offset: position value (%d) must be >= 0 for '%s'", position, this.resourceId));
        }
        if (this.size >= 0L && position >= this.size) {
            GoogleCloudStorageEventBus.postOnException();
            throw new EOFException(String.format("Invalid seek offset: position value (%d) must be between 0 and %d for '%s'", position, this.size, this.resourceId));
        }
    }

    @VisibleForTesting
    void performLazySeek(long bytesToRead) throws IOException {
        this.throwIfNotOpen();
        if (this.currentPosition == this.contentChannelPosition && this.contentChannel != null) {
            return;
        }
        ((GoogleLogger.Api)logger.atFiner()).log("Performing lazySeek from %s to %s position with %s bytesToRead for '%s'", (Object)this.contentChannelPosition, (Object)this.currentPosition, (Object)bytesToRead, (Object)this.resourceId);
        long oldPosition = this.contentChannelPosition;
        long seekDistance = this.currentPosition - this.contentChannelPosition;
        if (this.contentChannel != null && seekDistance > 0L && (this.gzipEncoded || seekDistance <= this.readOptions.getInplaceSeekLimit()) && this.currentPosition < this.contentChannelEnd) {
            ((GoogleLogger.Api)logger.atFiner()).log("Seeking forward %d bytes (inplaceSeekLimit: %d) in-place to position %d for '%s'", (Object)seekDistance, (Object)this.readOptions.getInplaceSeekLimit(), (Object)this.currentPosition, (Object)this.resourceId);
            this.skipInPlace(seekDistance);
        } else {
            this.closeContentChannel();
        }
        if (this.contentChannel == null) {
            this.openContentChannel(bytesToRead);
        }
    }

    private void openContentChannel(long bytesToRead) throws IOException {
        InputStream objectContentStream;
        Preconditions.checkState((this.contentChannel == null ? 1 : 0) != 0, (Object)"contentChannel should be null, before opening new");
        if (this.footerContent != null && this.currentPosition >= this.size - (long)this.footerContent.length) {
            objectContentStream = this.openFooterStream();
        } else {
            if (!this.gzipEncoded) {
                this.fileAccessManager.updateAccessPattern(this.currentPosition);
            }
            objectContentStream = this.openStream(bytesToRead);
        }
        this.contentChannel = Channels.newChannel(objectContentStream);
        Preconditions.checkState((this.contentChannelPosition == this.currentPosition ? 1 : 0) != 0, (String)"contentChannelPosition (%s) should be equal to currentPosition (%s) for '%s'", (Object)this.contentChannelPosition, (Object)this.currentPosition, (Object)this.resourceId);
    }

    private void initMetadata(GoogleCloudStorageItemInfo info) throws IOException {
        this.initMetadata(info.getContentEncoding(), info.getSize(), info.getContentGeneration());
    }

    @VisibleForTesting
    protected void initMetadata(HttpHeaders headers) throws IOException {
        Preconditions.checkState((!this.metadataInitialized ? 1 : 0) != 0, (String)"Cannot initialize metadata, it already initialized for '%s'", (Object)this.resourceId);
        String generationString = headers.getFirstHeaderStringValue("x-goog-generation");
        if (generationString == null) {
            GoogleCloudStorageEventBus.postOnException();
            throw new IOException(String.format("Failed to retrieve generation for '%s'", this.resourceId));
        }
        long generation = Long.parseLong(generationString);
        String range = headers.getContentRange();
        long sizeFromMetadata = range == null ? headers.getContentLength() : Long.parseLong(range.substring(range.lastIndexOf(47) + 1));
        this.initMetadata(headers.getContentEncoding(), sizeFromMetadata, generation);
    }

    @VisibleForTesting
    protected void initMetadata(@Nullable String encoding, long sizeFromMetadata, long generation) throws IOException {
        Preconditions.checkState((!this.metadataInitialized ? 1 : 0) != 0, (String)"Cannot initialize metadata, it already initialized for '%s'", (Object)this.resourceId);
        Preconditions.checkState((generation != -1L ? 1 : 0) != 0, (String)"Generation parameter of %s is invalid for resourceId of '%s'", (long)generation, (Object)this.resourceId);
        this.gzipEncoded = Strings.nullToEmpty((String)encoding).contains(GZIP_ENCODING);
        if (this.gzipEncoded) {
            this.fileAccessManager.overrideAccessPattern(false);
        }
        if (this.gzipEncoded && !this.readOptions.isGzipEncodingSupportEnabled()) {
            GoogleCloudStorageEventBus.postOnException();
            throw new IOException("Cannot read GZIP encoded files - content encoding support is disabled.");
        }
        this.size = this.gzipEncoded ? Long.MAX_VALUE : sizeFromMetadata;
        this.checkEncodingAndAccess();
        if (this.resourceId.hasGenerationId() && this.resourceId.getGenerationId() > 0L) {
            Preconditions.checkState((this.resourceId.getGenerationId() == generation ? 1 : 0) != 0, (String)"Provided generation (%s) should be equal to fetched generation (%s) for '%s'", (Object)this.resourceId.getGenerationId(), (Object)generation, (Object)this.resourceId);
        } else {
            this.resourceId = new StorageResourceId(this.resourceId.getBucketName(), this.resourceId.getObjectName(), generation);
        }
        this.metadataInitialized = true;
        ((GoogleLogger.Api)logger.atFiner()).log("Initialized metadata (gzipEncoded=%s, size=%s, randomAccess=%s, generation=%s) for '%s'", (Object)this.gzipEncoded, (Object)this.size, (Object)this.fileAccessManager.shouldAdaptToRandomAccess(), (Object)this.resourceId.getGenerationId(), (Object)this.resourceId);
    }

    private void cacheFooter(HttpResponse response) throws IOException {
        Preconditions.checkState((this.size > 0L ? 1 : 0) != 0, (String)"size should be greater than 0 for '%s'", (Object)this.resourceId);
        int footerSize = Math.toIntExact(response.getHeaders().getContentLength());
        this.footerContent = new byte[footerSize];
        try (InputStream footerStream = response.getContent();){
            int totalBytesRead = 0;
            int bytesRead = 0;
            while ((bytesRead = footerStream.read(this.footerContent, totalBytesRead += bytesRead, footerSize - totalBytesRead)) >= 0 && totalBytesRead < footerSize) {
            }
            Preconditions.checkState((footerStream.read() < 0 ? 1 : 0) != 0, (String)"footerStream should be empty after reading %s bytes from %s bytes for '%s'", (Object)totalBytesRead, (Object)footerSize, (Object)this.resourceId);
            Preconditions.checkState((totalBytesRead == footerSize ? 1 : 0) != 0, (String)"totalBytesRead (%s) should equal footerSize (%s) for '%s'", (Object)totalBytesRead, (Object)footerSize, (Object)this.resourceId);
        }
        catch (IOException e) {
            GoogleCloudStorageEventBus.postOnException();
            this.footerContent = null;
            throw e;
        }
        ((GoogleLogger.Api)logger.atFiner()).log("Prefetched %s bytes footer for '%s'", this.footerContent.length, (Object)this.resourceId);
    }

    private InputStream openFooterStream() {
        this.contentChannelPosition = this.currentPosition;
        int offset = Math.toIntExact(this.currentPosition - (this.size - (long)this.footerContent.length));
        int length = this.footerContent.length - offset;
        ((GoogleLogger.Api)logger.atFiner()).log("Opened stream (prefetched footer) from %d position for '%s'", this.currentPosition, (Object)this.resourceId);
        return new ByteArrayInputStream(this.footerContent, offset, length);
    }

    protected InputStream openStream(long bytesToRead) throws IOException {
        HttpResponse response;
        String rangeHeader;
        Preconditions.checkArgument((bytesToRead > 0L ? 1 : 0) != 0, (String)"bytesToRead should be greater than 0, but was %s", (long)bytesToRead);
        Preconditions.checkState((this.contentChannel == null && this.contentChannelEnd < 0L ? 1 : 0) != 0, (String)"contentChannel and contentChannelEnd should be not initialized yet for '%s'", (Object)this.resourceId);
        if (this.size == 0L) {
            return new ByteArrayInputStream(new byte[0]);
        }
        if (this.readOptions.isReadExactRequestedBytesEnabled() && !this.gzipEncoded) {
            this.contentChannelPosition = this.currentPosition;
            this.contentChannelEnd = this.contentChannelPosition + bytesToRead;
            rangeHeader = "bytes=" + this.contentChannelPosition + "-" + (this.contentChannelEnd - 1L);
        } else if (!this.metadataInitialized) {
            this.contentChannelPosition = this.getContentChannelPositionForFirstRead(bytesToRead);
            rangeHeader = "bytes=" + this.contentChannelPosition + "-";
            if (this.readOptions.getFadvise() == GoogleCloudStorageReadOptions.Fadvise.RANDOM || this.readOptions.getFadvise() == GoogleCloudStorageReadOptions.Fadvise.AUTO_RANDOM) {
                long maxBytesToRead = Math.max(this.readOptions.getMinRangeRequestSize(), bytesToRead);
                rangeHeader = rangeHeader + (this.contentChannelPosition + maxBytesToRead - 1L);
            }
        } else if (this.gzipEncoded) {
            rangeHeader = null;
            this.contentChannelPosition = 0L;
            this.contentChannelEnd = this.size;
        } else {
            this.contentChannelPosition = this.readOptions.getFadvise() != GoogleCloudStorageReadOptions.Fadvise.SEQUENTIAL && this.isFooterRead() ? Math.max(0L, this.size - this.readOptions.getMinRangeRequestSize()) : this.currentPosition;
            long rangeSize = this.size - this.contentChannelPosition;
            if (this.fileAccessManager.shouldAdaptToRandomAccess()) {
                long randomRangeSize = Math.max(bytesToRead, this.readOptions.getMinRangeRequestSize());
                rangeSize = Math.min(randomRangeSize, rangeSize);
            } else if (this.readOptions.getFadvise() == GoogleCloudStorageReadOptions.Fadvise.AUTO_RANDOM) {
                rangeSize = Math.min(rangeSize, this.readOptions.getBlockSize());
            }
            this.contentChannelEnd = this.contentChannelPosition + rangeSize;
            if (this.footerContent != null) {
                this.contentChannelEnd = Math.min(this.contentChannelEnd, this.size - (long)this.footerContent.length);
            }
            Preconditions.checkState((this.currentPosition < this.contentChannelEnd ? 1 : 0) != 0, (String)"currentPosition (%s) should be less than contentChannelEnd (%s) for '%s'", (Object)this.currentPosition, (Object)this.contentChannelEnd, (Object)this.resourceId);
            Preconditions.checkState((this.contentChannelPosition <= this.currentPosition ? 1 : 0) != 0, (String)"contentChannelPosition (%s) should be less or equal to currentPosition (%s) for '%s'", (Object)this.contentChannelPosition, (Object)this.currentPosition, (Object)this.resourceId);
            rangeHeader = "bytes=" + this.contentChannelPosition + "-";
            if (this.fileAccessManager.shouldAdaptToRandomAccess() || !this.fileAccessManager.shouldAdaptToRandomAccess() && this.readOptions.getFadvise() == GoogleCloudStorageReadOptions.Fadvise.AUTO_RANDOM || this.contentChannelEnd != this.size) {
                rangeHeader = rangeHeader + (this.contentChannelEnd - 1L);
            }
        }
        Preconditions.checkState((!this.metadataInitialized || this.contentChannelEnd > 0L ? 1 : 0) != 0, (String)"contentChannelEnd should be initialized already for '%s'", (Object)this.resourceId);
        Storage.Objects.Get getObject = this.createDataRequest(rangeHeader);
        try {
            response = getObject.executeMedia();
        }
        catch (IOException e) {
            if (!this.metadataInitialized && this.errorExtractor.rangeNotSatisfiable(e) && this.currentPosition == 0L) {
                GoogleCloudStorageEventBus.postOnException();
                ((GoogleLogger.Api)logger.atInfo()).log("Got 'range not satisfiable' for reading '%s' at position 0; assuming empty.", (Object)this.resourceId);
                this.size = 0L;
                return new ByteArrayInputStream(new byte[0]);
            }
            response = this.handleExecuteMediaException(e);
        }
        if (!this.metadataInitialized) {
            this.initMetadata(response.getHeaders());
            Preconditions.checkState((boolean)this.metadataInitialized, (String)"metadata should be initialized already for '%s'", (Object)this.resourceId);
            if (this.size == 0L) {
                this.resetContentChannel();
                return new ByteArrayInputStream(new byte[0]);
            }
            if (this.gzipEncoded) {
                if (this.currentPosition == 0L) {
                    this.contentChannelEnd = this.size;
                } else {
                    this.resetContentChannel();
                    return this.openStream(bytesToRead);
                }
            }
        }
        if (this.contentChannelEnd < 0L) {
            String contentRange = response.getHeaders().getContentRange();
            if (contentRange != null) {
                String contentEnd = contentRange.substring(contentRange.lastIndexOf(45) + 1, contentRange.lastIndexOf(47));
                this.contentChannelEnd = Long.parseLong(contentEnd) + 1L;
            } else {
                this.contentChannelEnd = response.getHeaders().getContentLength();
            }
        }
        Preconditions.checkState((this.contentChannelEnd > 0L ? 1 : 0) != 0, (String)"contentChannelEnd should be initialized already for '%s'", (Object)this.resourceId);
        if (!this.gzipEncoded && this.readOptions.getFadvise() != GoogleCloudStorageReadOptions.Fadvise.SEQUENTIAL && this.contentChannelEnd == this.size && this.contentChannelEnd - this.contentChannelPosition <= this.readOptions.getMinRangeRequestSize()) {
            for (int retriesCount = 0; retriesCount < this.maxRetries; ++retriesCount) {
                try {
                    this.cacheFooter(response);
                    if (retriesCount == 0) break;
                    ((GoogleLogger.Api)logger.atInfo()).log("Successfully cached footer after %d retries for '%s'", retriesCount, (Object)this.resourceId);
                    break;
                }
                catch (IOException footerException) {
                    GoogleCloudStorageEventBus.postOnException();
                    ((GoogleLogger.Api)((GoogleLogger.Api)logger.atInfo()).withCause((Throwable)footerException)).log("Failed to prefetch footer (retry #%d/%d) for '%s'", (Object)(retriesCount + 1), (Object)this.maxRetries, (Object)this.resourceId);
                    if (retriesCount == 0) {
                        this.readBackOff.get().reset();
                    }
                    if (retriesCount == this.maxRetries) {
                        this.resetContentChannel();
                        throw footerException;
                    }
                    try {
                        response = getObject.executeMedia();
                    }
                    catch (IOException e) {
                        response = this.handleExecuteMediaException(e);
                    }
                    continue;
                }
            }
            Preconditions.checkState((this.footerContent != null ? 1 : 0) != 0, (String)"footerContent should not be null after successful footer prefetch for '%s'", (Object)this.resourceId);
            this.resetContentChannel();
            return this.openFooterStream();
        }
        try {
            InputStream contentStream = response.getContent();
            ((GoogleLogger.Api)logger.atFiner()).log("Opened stream from %d position with %s range and %d bytesToRead for '%s'", (Object)this.currentPosition, (Object)rangeHeader, (Object)bytesToRead, (Object)this.resourceId);
            if (this.contentChannelPosition < this.currentPosition) {
                long bytesToSkip = this.currentPosition - this.contentChannelPosition;
                ((GoogleLogger.Api)logger.atFiner()).log("Skipping %d bytes from %d to %d position for '%s'", (Object)bytesToSkip, (Object)this.contentChannelPosition, (Object)this.currentPosition, (Object)this.resourceId);
                this.contentChannelPosition += this.skip(Channels.newChannel(contentStream), bytesToSkip);
            }
            Preconditions.checkState((this.contentChannelPosition == this.currentPosition ? 1 : 0) != 0, (String)"contentChannelPosition (%s) should be equal to currentPosition (%s) for '%s'", (Object)this.contentChannelPosition, (Object)this.currentPosition, (Object)this.resourceId);
            return contentStream;
        }
        catch (IOException e) {
            GoogleCloudStorageEventBus.postOnException();
            try {
                response.disconnect();
            }
            catch (IOException closeException) {
                e.addSuppressed(closeException);
            }
            throw e;
        }
    }

    private boolean isFooterRead() {
        return this.size - this.currentPosition <= this.readOptions.getMinRangeRequestSize();
    }

    private long getContentChannelPositionForFirstRead(long bytesToRead) {
        if (this.readOptions.getFadvise() == GoogleCloudStorageReadOptions.Fadvise.SEQUENTIAL || bytesToRead >= this.readOptions.getMinRangeRequestSize()) {
            return this.currentPosition;
        }
        if (bytesToRead <= this.readOptions.getMinRangeRequestSize() / 2L) {
            return Math.max(0L, this.currentPosition - this.readOptions.getMinRangeRequestSize() / 2L);
        }
        return Math.max(0L, this.currentPosition - (this.readOptions.getMinRangeRequestSize() - bytesToRead));
    }

    private HttpResponse handleExecuteMediaException(IOException e) throws IOException {
        if (this.errorExtractor.itemNotFound(e)) {
            GoogleCloudStorageEventBus.postOnException();
            throw GoogleCloudStorageExceptions.createFileNotFoundException(this.resourceId, e);
        }
        String msg = String.format("Error reading '%s' at position %d", this.resourceId, this.currentPosition);
        if (this.errorExtractor.rangeNotSatisfiable(e)) {
            GoogleCloudStorageEventBus.postOnException();
            throw (EOFException)new EOFException(msg).initCause(e);
        }
        throw new IOException(msg, e);
    }

    private Storage.Objects.Get createDataRequest(String rangeHeader) throws IOException {
        Storage.Objects.Get dataRequest = this.createDataRequest();
        HttpHeaders requestHeaders = this.clientRequestHelper.getRequestHeaders((AbstractGoogleClientRequest)dataRequest);
        requestHeaders.setAcceptEncoding(GZIP_ENCODING);
        requestHeaders.setRange(rangeHeader);
        return dataRequest;
    }

    protected Storage.Objects.Get createDataRequest() throws IOException {
        Preconditions.checkState((!this.metadataInitialized || this.resourceId.hasGenerationId() ? 1 : 0) != 0, (String)"Generation should always be included for resource '%s'", (Object)this.resourceId);
        StorageRequestFactory.ObjectsGetData getData = this.storageRequestFactory.objectsGetData(this.resourceId.getBucketName(), this.resourceId.getObjectName());
        if (this.resourceId.hasGenerationId() && this.resourceId.getGenerationId() > 0L) {
            getData.setGeneration(this.resourceId.getGenerationId());
        }
        return getData;
    }

    protected Storage.Objects.Get createMetadataRequest() throws IOException {
        Preconditions.checkState((!this.metadataInitialized || this.resourceId.hasGenerationId() ? 1 : 0) != 0, (String)"Generation should always be included for resource '%s'", (Object)this.resourceId);
        StorageRequestFactory.ObjectsGetMetadata getMetadata = this.storageRequestFactory.objectsGetMetadata(this.resourceId.getBucketName(), this.resourceId.getObjectName());
        if (this.resourceId.hasGenerationId() && this.resourceId.getGenerationId() > 0L) {
            getMetadata.setGeneration(this.resourceId.getGenerationId());
        }
        return getMetadata;
    }

    private void throwIfNotOpen() throws IOException {
        if (!this.isOpen()) {
            GoogleCloudStorageEventBus.postOnException();
            throw new ClosedChannelException();
        }
    }

    @VisibleForTesting
    boolean randomAccessStatus() {
        return this.fileAccessManager.shouldAdaptToRandomAccess();
    }
}

