/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.hadoop.gcsio.integration;

import com.google.auth.Credentials;
import com.google.cloud.hadoop.gcsio.CreateObjectOptions;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorage;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageClientImpl;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageItemInfo;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageOptions;
import com.google.cloud.hadoop.gcsio.ListObjectOptions;
import com.google.cloud.hadoop.gcsio.StorageResourceId;
import com.google.cloud.hadoop.gcsio.integration.GoogleCloudStorageTestHelper;
import com.google.cloud.hadoop.util.AsyncWriteChannelOptions;
import com.google.cloud.storage.StorageException;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.io.Files;
import com.google.common.truth.Truth;
import com.google.common.util.concurrent.MoreExecutors;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.WritableByteChannel;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.stream.Collectors;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;

public class GoogleCloudStorageClientImplIntegrationTest {
    private static final GoogleCloudStorageTestHelper.TestBucketHelper BUCKET_HELPER = new GoogleCloudStorageTestHelper.TestBucketHelper("dataproc-gcs-client-impl");
    private static final String TEST_BUCKET = BUCKET_HELPER.getUniqueBucketPrefix();
    private static final String TEMP_DIR_PATH = Files.createTempDir().getAbsolutePath();
    private static final String GCS_WRITE_TMP_DIR = String.format("%s/%s", TEMP_DIR_PATH, "gcs-write-dir");
    private static final String GCS_WRITE_TMP_DIR_1 = String.format("%s/%s", TEMP_DIR_PATH, "gcs-write-dir-1");
    private static final int ONE_MiB = 0x100000;
    private static GoogleCloudStorage helperGcs;
    private static final int partFileCount = 2;
    private static final int bufferCapacity = 0x200000;
    private static final AsyncWriteChannelOptions pcuDefaultOptions;
    private static ImmutableSet<String> tempDirs;
    private static ImmutableSet<Path> tempDirsPath;
    @Rule
    public TestName name = new TestName();
    private GoogleCloudStorage gcs;

    @BeforeClass
    public static void before() throws IOException {
        helperGcs = GoogleCloudStorageTestHelper.createGcsClientImpl();
        helperGcs.createBucket(TEST_BUCKET);
    }

    @AfterClass
    public static void after() throws IOException {
        try {
            BUCKET_HELPER.cleanup(helperGcs);
        }
        finally {
            helperGcs.close();
        }
    }

    @Before
    public void setUp() {
        System.setProperty("java.io.tmpdir", GCS_WRITE_TMP_DIR);
    }

    @After
    public void cleanUp() {
        ImmutableSet tempDirs = ImmutableSet.of((Object)GCS_WRITE_TMP_DIR_1, (Object)GCS_WRITE_TMP_DIR);
        Iterator iterator = tempDirs.stream().iterator();
        while (iterator.hasNext()) {
            String filePath = (String)iterator.next();
            File directory = new File(filePath);
            if (directory.listFiles() == null) continue;
            for (File file : new File(filePath).listFiles()) {
                file.delete();
            }
        }
        if (this.gcs != null) {
            this.gcs.close();
        }
    }

    @Test
    public void writeToDiskDisabled() throws IOException {
        GoogleCloudStorageOptions storageOptions = GoogleCloudStorageTestHelper.getStandardOptionBuilder().setWriteChannelOptions(AsyncWriteChannelOptions.builder().setUploadType(AsyncWriteChannelOptions.UploadType.CHUNK_UPLOAD).build()).build();
        this.gcs = this.getGCSImpl(storageOptions);
        StorageResourceId resourceId = new StorageResourceId(TEST_BUCKET, this.name.getMethodName());
        this.writeAndVerifyTemporaryFiles(resourceId, 0);
    }

    @Test
    public void writeToDefaultPathThenUploadEnabled() throws IOException {
        GoogleCloudStorageOptions storageOptions = GoogleCloudStorageTestHelper.getStandardOptionBuilder().setWriteChannelOptions(AsyncWriteChannelOptions.builder().setUploadType(AsyncWriteChannelOptions.UploadType.WRITE_TO_DISK_THEN_UPLOAD).build()).build();
        this.gcs = this.getGCSImpl(storageOptions);
        StorageResourceId resourceId = new StorageResourceId(TEST_BUCKET, this.name.getMethodName());
        this.writeAndVerifyTemporaryFiles(resourceId, 1);
    }

    @Test
    public void writeToPathThenUploadEnabled() throws IOException {
        GoogleCloudStorageOptions storageOptions = GoogleCloudStorageTestHelper.getStandardOptionBuilder().setWriteChannelOptions(AsyncWriteChannelOptions.builder().setUploadType(AsyncWriteChannelOptions.UploadType.WRITE_TO_DISK_THEN_UPLOAD).setTemporaryPaths(tempDirs).build()).build();
        this.gcs = this.getGCSImpl(storageOptions);
        StorageResourceId resourceId = new StorageResourceId(TEST_BUCKET, this.name.getMethodName());
        this.writeAndVerifyTemporaryFiles(resourceId, 1);
    }

    @Test
    public void uploadViaJournalingThrowsIfTempDirNotProvided() {
        GoogleCloudStorageOptions storageOptions = GoogleCloudStorageTestHelper.getStandardOptionBuilder().setWriteChannelOptions(AsyncWriteChannelOptions.builder().setUploadType(AsyncWriteChannelOptions.UploadType.JOURNALING).build()).build();
        Assert.assertThrows(IllegalArgumentException.class, () -> this.getGCSImpl(storageOptions));
    }

    @Test
    public void uploadViaJournaling() throws IOException {
        GoogleCloudStorageOptions storageOptions = GoogleCloudStorageTestHelper.getStandardOptionBuilder().setWriteChannelOptions(AsyncWriteChannelOptions.builder().setTemporaryPaths(tempDirs).setUploadType(AsyncWriteChannelOptions.UploadType.JOURNALING).build()).build();
        this.gcs = this.getGCSImpl(storageOptions);
        StorageResourceId resourceId = new StorageResourceId(TEST_BUCKET, this.name.getMethodName());
        this.writeAndVerifyTemporaryFiles(resourceId, 1);
    }

    @Test
    public void uploadViaPCUVerifyPartFileCleanup() throws IOException, InterruptedException {
        String partFilePrefix = this.name.getMethodName();
        GoogleCloudStorageOptions storageOptions = GoogleCloudStorageTestHelper.getStandardOptionBuilder().setWriteChannelOptions(pcuDefaultOptions.toBuilder().setPartFileNamePrefix(partFilePrefix).build()).build();
        this.gcs = this.getGCSImpl(storageOptions);
        StorageResourceId resourceId = new StorageResourceId(TEST_BUCKET, this.name.getMethodName());
        this.writeAndVerifyPartFiles(0x200000, resourceId, 0, partFilePrefix);
    }

    @Test
    public void uploadViaPCUVerifyPartFileNotCleanedUp() throws IOException {
        String partFilePrefix = this.name.getMethodName();
        GoogleCloudStorageOptions storageOptions = GoogleCloudStorageTestHelper.getStandardOptionBuilder().setWriteChannelOptions(pcuDefaultOptions.toBuilder().setPartFileNamePrefix(partFilePrefix).setPartFileCleanupType(AsyncWriteChannelOptions.PartFileCleanupType.NEVER).build()).build();
        this.gcs = this.getGCSImpl(storageOptions);
        StorageResourceId resourceId = new StorageResourceId(TEST_BUCKET, this.name.getMethodName());
        this.writeAndVerifyPartFiles(0x200000, resourceId, 2, partFilePrefix);
    }

    @Test
    public void uploadViaPCUComposeFileMissingFailure() throws IOException {
        String partFilePrefix = this.name.getMethodName();
        GoogleCloudStorageOptions storageOptions = GoogleCloudStorageTestHelper.getStandardOptionBuilder().setWriteChannelOptions(pcuDefaultOptions.toBuilder().setPartFileNamePrefix(partFilePrefix).build()).build();
        this.gcs = this.getGCSImpl(storageOptions);
        StorageResourceId resourceId = new StorageResourceId(TEST_BUCKET, this.name.getMethodName());
        byte[] bytesToWrite = new byte[0x400000];
        GoogleCloudStorageTestHelper.fillBytes(bytesToWrite);
        WritableByteChannel writeChannel = this.gcs.create(resourceId);
        writeChannel.write(ByteBuffer.wrap(bytesToWrite));
        List<GoogleCloudStorageItemInfo> partFiles = this.getPartFiles(partFilePrefix);
        StorageResourceId partFileToBeDeleted = partFiles.get(0).getResourceId();
        this.gcs.deleteObjects((List)ImmutableList.of((Object)partFileToBeDeleted));
        Exception e = (Exception)Assert.assertThrows(IOException.class, writeChannel::close);
        this.verifyPartFileNotFound(e, partFileToBeDeleted.getObjectName());
        partFiles = this.getPartFiles(partFilePrefix);
        Truth.assertThat((Integer)partFiles.size()).isEqualTo((Object)0);
    }

    @Test
    public void uploadViaPCUComposeMissingObjectVersion() throws IOException {
        String partFilePrefix = this.name.getMethodName();
        GoogleCloudStorageOptions storageOptions = GoogleCloudStorageTestHelper.getStandardOptionBuilder().setWriteChannelOptions(pcuDefaultOptions.toBuilder().setPartFileNamePrefix(partFilePrefix).setPartFileCleanupType(AsyncWriteChannelOptions.PartFileCleanupType.ON_SUCCESS).build()).build();
        this.gcs = this.getGCSImpl(storageOptions);
        StorageResourceId resourceId = new StorageResourceId(TEST_BUCKET, this.name.getMethodName());
        byte[] bytesToWrite = new byte[0x400000];
        GoogleCloudStorageTestHelper.fillBytes(bytesToWrite);
        WritableByteChannel writeChannel = this.gcs.create(resourceId);
        writeChannel.write(ByteBuffer.wrap(bytesToWrite));
        List<GoogleCloudStorageItemInfo> partFiles = this.getPartFiles(partFilePrefix);
        GoogleCloudStorageItemInfo itemInfoBeforeModification = partFiles.get(0);
        this.gcs.create(itemInfoBeforeModification.getResourceId(), CreateObjectOptions.DEFAULT_OVERWRITE).close();
        GoogleCloudStorageItemInfo itemInfoAfterModification = this.gcs.getItemInfo(itemInfoBeforeModification.getResourceId());
        List<GoogleCloudStorageItemInfo> updatedFiles = this.getPartFiles(partFilePrefix);
        Truth.assertThat((Boolean)updatedFiles.stream().anyMatch(itemInfo -> itemInfo.getObjectName().equals(itemInfoAfterModification.getObjectName()) && itemInfo.getContentGeneration() != itemInfoBeforeModification.getContentGeneration())).isTrue();
        Exception e = (Exception)Assert.assertThrows(IOException.class, writeChannel::close);
        this.verifyPartFileNotFound(e, itemInfoBeforeModification.getObjectName());
        partFiles = this.getPartFiles(partFilePrefix);
        Truth.assertThat((Integer)partFiles.size()).isEqualTo((Object)2);
    }

    @Test
    public void uploadViaPCUInvalidPartFileNamePrefix() throws IOException {
        String partFilePrefix = "\n";
        GoogleCloudStorageOptions storageOptions = GoogleCloudStorageTestHelper.getStandardOptionBuilder().setWriteChannelOptions(pcuDefaultOptions.toBuilder().setPartFileNamePrefix(partFilePrefix).setPartFileCleanupType(AsyncWriteChannelOptions.PartFileCleanupType.NEVER).build()).build();
        this.gcs = this.getGCSImpl(storageOptions);
        StorageResourceId resourceId = new StorageResourceId(TEST_BUCKET, this.name.getMethodName());
        byte[] bytesToWrite = new byte[0x400000];
        GoogleCloudStorageTestHelper.fillBytes(bytesToWrite);
        WritableByteChannel writeChannel = this.gcs.create(resourceId);
        Exception e = (Exception)Assert.assertThrows(StorageException.class, () -> writeChannel.write(ByteBuffer.wrap(bytesToWrite)));
        this.verifyPartFileInvalidArgument(e);
    }

    @Test
    public void uploadViaPCUPartFileCleanupOnSuccess() throws IOException, InterruptedException {
        String partFilePrefix = this.name.getMethodName();
        GoogleCloudStorageOptions storageOptions = GoogleCloudStorageTestHelper.getStandardOptionBuilder().setWriteChannelOptions(pcuDefaultOptions.toBuilder().setPartFileNamePrefix(partFilePrefix).setPartFileCleanupType(AsyncWriteChannelOptions.PartFileCleanupType.ON_SUCCESS).build()).build();
        this.gcs = this.getGCSImpl(storageOptions);
        StorageResourceId resourceId = new StorageResourceId(TEST_BUCKET, this.name.getMethodName());
        this.writeAndVerifyPartFiles(0x200000, resourceId, 0, partFilePrefix);
    }

    private void verifyPartFileNotFound(Throwable throwable, String partFileName) {
        StorageException exception = this.getStorageException(throwable);
        Truth.assertThat((String)exception.getMessage()).contains((CharSequence)partFileName);
        Truth.assertThat((Integer)exception.getCode()).isEqualTo((Object)404);
    }

    private void verifyPartFileInvalidArgument(Throwable throwable) {
        StorageException exception = this.getStorageException(throwable);
        Truth.assertThat((String)exception.getMessage()).contains((CharSequence)"INVALID_ARGUMENT");
        Truth.assertThat((Integer)exception.getCode()).isEqualTo((Object)400);
    }

    private StorageException getStorageException(Throwable throwable) {
        for (Throwable cause = throwable; cause != null; cause = cause.getCause()) {
            if (!(cause instanceof StorageException)) continue;
            return (StorageException)cause;
        }
        return null;
    }

    private List<GoogleCloudStorageItemInfo> getPartFiles(String prefix) throws IOException {
        List itemInfos = this.gcs.listObjectInfo(TEST_BUCKET, prefix, ListObjectOptions.builder().setDelimiter(null).build());
        return itemInfos.stream().filter(x -> x.getObjectName().endsWith(".part")).collect(Collectors.toList());
    }

    private void writeAndVerifyPartFiles(int bufferCapacity, StorageResourceId resourceId, int expectedPartFileCountAfterCleanup, String partFilePrefix) throws IOException {
        byte[] bytesToWrite = new byte[2 * bufferCapacity];
        GoogleCloudStorageTestHelper.fillBytes(bytesToWrite);
        WritableByteChannel writeChannel = this.gcs.create(resourceId);
        writeChannel.write(ByteBuffer.wrap(bytesToWrite));
        writeChannel.close();
        List<GoogleCloudStorageItemInfo> partFiles = this.getPartFiles(partFilePrefix);
        Truth.assertThat((Long)partFiles.stream().count()).isEqualTo((Object)expectedPartFileCountAfterCleanup);
        this.verifyFileContent(resourceId, bytesToWrite);
    }

    private void writeAndVerifyTemporaryFiles(StorageResourceId resourceId, int expectedTemporaryFileCount) throws IOException {
        byte[] bytesToWrite = new byte[0x300000];
        GoogleCloudStorageTestHelper.fillBytes(bytesToWrite);
        this.verifyTemporaryFileCount(tempDirsPath, 0);
        WritableByteChannel writeChannel = this.gcs.create(resourceId);
        writeChannel.write(ByteBuffer.wrap(bytesToWrite));
        this.verifyTemporaryFileCount(tempDirsPath, expectedTemporaryFileCount);
        writeChannel.close();
        this.verifyTemporaryFileCount(tempDirsPath, 0);
    }

    private GoogleCloudStorage getGCSImpl(GoogleCloudStorageOptions storageOptions) throws IOException {
        Credentials credentials = GoogleCloudStorageTestHelper.getCredentials();
        return GoogleCloudStorageClientImpl.builder().setOptions(storageOptions).setCredentials(credentials).setPCUExecutorService((ExecutorService)MoreExecutors.newDirectExecutorService()).build();
    }

    private void verifyTemporaryFileCount(ImmutableSet<Path> paths, int expectedCount) {
        Iterator iterator = paths.stream().iterator();
        int fileCount = 0;
        while (iterator.hasNext()) {
            Path path = (Path)iterator.next();
            File directory = path.toFile();
            fileCount += this.getFileCount(directory);
        }
        Truth.assertThat((Integer)fileCount).isEqualTo((Object)expectedCount);
    }

    private void verifyFileContent(StorageResourceId resourceId, byte[] bytesWritten) throws IOException {
        GoogleCloudStorageItemInfo fileInfo = this.gcs.getItemInfo(resourceId);
        Truth.assertThat((Boolean)fileInfo.exists()).isTrue();
        GoogleCloudStorageTestHelper.assertObjectContent(this.gcs, resourceId, bytesWritten);
    }

    private int getFileCount(File file) {
        File[] files = file.listFiles();
        if (files == null) {
            return 0;
        }
        int count = 0;
        for (File f : files) {
            if (f.isDirectory()) {
                count += this.getFileCount(f);
                continue;
            }
            ++count;
        }
        return count;
    }

    static {
        pcuDefaultOptions = AsyncWriteChannelOptions.builder().setUploadType(AsyncWriteChannelOptions.UploadType.PARALLEL_COMPOSITE_UPLOAD).setPartFileCleanupType(AsyncWriteChannelOptions.PartFileCleanupType.ALWAYS).setPCUBufferCount(2).setPCUBufferCapacity(0x200000).build();
        tempDirs = ImmutableSet.of((Object)GCS_WRITE_TMP_DIR_1, (Object)GCS_WRITE_TMP_DIR);
        tempDirsPath = (ImmutableSet)tempDirs.stream().map(x -> Paths.get(x, new String[0])).collect(ImmutableSet.toImmutableSet());
    }
}

