/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.hadoop.gcsio.integration;

import com.google.cloud.hadoop.gcsio.GoogleCloudStorage;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageImpl;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageItemInfo;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageReadOptions;
import com.google.cloud.hadoop.gcsio.StorageResourceId;
import com.google.cloud.hadoop.gcsio.integration.GoogleCloudStorageTestHelper;
import com.google.cloud.hadoop.util.AsyncWriteChannelOptions;
import com.google.common.truth.Truth;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.SeekableByteChannel;
import java.util.Arrays;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.Test;

public class GoogleCloudStorageGrpcIntegrationTest {
    private static final String BUCKET_NAME_PREFIX = "gcs-grpc-team-dataproc-it";
    private static final GoogleCloudStorageTestHelper.TestBucketHelper BUCKET_HELPER = new GoogleCloudStorageTestHelper.TestBucketHelper("gcs-grpc-team-dataproc-it");
    private static final String BUCKET_NAME = BUCKET_HELPER.getUniqueBucketName("shared");

    private static GoogleCloudStorage createGoogleCloudStorage() throws IOException {
        return new GoogleCloudStorageImpl(GoogleCloudStorageTestHelper.getStandardOptionBuilder().setGrpcEnabled(true).build(), GoogleCloudStorageTestHelper.getCredential());
    }

    private static GoogleCloudStorage createGoogleCloudStorage(AsyncWriteChannelOptions asyncWriteChannelOptions) throws IOException {
        return new GoogleCloudStorageImpl(GoogleCloudStorageTestHelper.getStandardOptionBuilder().setWriteChannelOptions(asyncWriteChannelOptions).setGrpcEnabled(true).build(), GoogleCloudStorageTestHelper.getCredential());
    }

    @BeforeClass
    public static void createBuckets() throws IOException {
        GoogleCloudStorage rawStorage = GoogleCloudStorageGrpcIntegrationTest.createGoogleCloudStorage();
        rawStorage.createBucket(BUCKET_NAME);
    }

    @AfterClass
    public static void cleanupBuckets() throws IOException {
        GoogleCloudStorage rawStorage = GoogleCloudStorageGrpcIntegrationTest.createGoogleCloudStorage();
        BUCKET_HELPER.cleanup(rawStorage);
    }

    @Test
    public void testCreateObject() throws IOException {
        GoogleCloudStorage rawStorage = GoogleCloudStorageGrpcIntegrationTest.createGoogleCloudStorage();
        StorageResourceId objectToCreate = new StorageResourceId(BUCKET_NAME, "testCreateObject_Object");
        byte[] objectBytes = GoogleCloudStorageTestHelper.writeObject(rawStorage, objectToCreate, 512);
        GoogleCloudStorageTestHelper.assertObjectContent(rawStorage, objectToCreate, objectBytes);
    }

    @Test
    public void testCreateExistingObject() throws IOException {
        GoogleCloudStorage rawStorage = GoogleCloudStorageGrpcIntegrationTest.createGoogleCloudStorage();
        StorageResourceId objectToCreate = new StorageResourceId(BUCKET_NAME, "testCreateExistingObject_Object");
        GoogleCloudStorageTestHelper.writeObject(rawStorage, objectToCreate, 128);
        GoogleCloudStorageItemInfo createdItemInfo = rawStorage.getItemInfo(objectToCreate);
        Truth.assertThat((Boolean)createdItemInfo.exists()).isTrue();
        Truth.assertThat((Long)createdItemInfo.getSize()).isEqualTo((Object)128);
        byte[] overwriteBytesToWrite = GoogleCloudStorageTestHelper.writeObject(rawStorage, objectToCreate, 256);
        GoogleCloudStorageItemInfo overwrittenItemInfo = rawStorage.getItemInfo(objectToCreate);
        Truth.assertThat((Boolean)overwrittenItemInfo.exists()).isTrue();
        Truth.assertThat((Long)overwrittenItemInfo.getSize()).isEqualTo((Object)256);
        GoogleCloudStorageTestHelper.assertObjectContent(rawStorage, objectToCreate, overwriteBytesToWrite);
    }

    @Test
    public void testCreateEmptyObject() throws IOException {
        GoogleCloudStorage rawStorage = GoogleCloudStorageGrpcIntegrationTest.createGoogleCloudStorage();
        StorageResourceId objectToCreate = new StorageResourceId(BUCKET_NAME, "testCreateEmptyObject_Object");
        rawStorage.createEmptyObject(objectToCreate);
        GoogleCloudStorageItemInfo itemInfo = rawStorage.getItemInfo(objectToCreate);
        Truth.assertThat((Boolean)itemInfo.exists()).isTrue();
        Truth.assertThat((Long)itemInfo.getSize()).isEqualTo((Object)0);
    }

    @Test
    public void testCreateInvalidObject() throws IOException {
        GoogleCloudStorage rawStorage = GoogleCloudStorageGrpcIntegrationTest.createGoogleCloudStorage();
        StorageResourceId objectToCreate = new StorageResourceId(BUCKET_NAME, "testCreateInvalidObject_InvalidObject\n");
        Assert.assertThrows(IOException.class, () -> GoogleCloudStorageTestHelper.writeObject(rawStorage, objectToCreate, 10));
    }

    @Test
    public void testOpen() throws IOException {
        GoogleCloudStorage rawStorage = GoogleCloudStorageGrpcIntegrationTest.createGoogleCloudStorage();
        StorageResourceId objectToCreate = new StorageResourceId(BUCKET_NAME, "testOpen_Object");
        byte[] objectBytes = GoogleCloudStorageTestHelper.writeObject(rawStorage, objectToCreate, 100);
        GoogleCloudStorageTestHelper.assertObjectContent(rawStorage, objectToCreate, objectBytes);
    }

    @Test
    public void testOpenNonExistentItem() throws IOException {
        GoogleCloudStorage rawStorage = GoogleCloudStorageGrpcIntegrationTest.createGoogleCloudStorage();
        Throwable throwable = Assert.assertThrows(IOException.class, () -> rawStorage.open(new StorageResourceId(BUCKET_NAME, "testOpenNonExistentItem_Object")));
        Truth.assertThat((Throwable)throwable).hasCauseThat().hasMessageThat().contains((CharSequence)"Item not found");
    }

    @Test
    public void testOpenEmptyObject() throws IOException {
        GoogleCloudStorage rawStorage = GoogleCloudStorageGrpcIntegrationTest.createGoogleCloudStorage();
        StorageResourceId resourceId = new StorageResourceId(BUCKET_NAME, "testOpenEmptyObject_Object");
        rawStorage.createEmptyObject(resourceId);
        GoogleCloudStorageTestHelper.assertObjectContent(rawStorage, resourceId, new byte[0]);
    }

    @Test
    public void testOpenLargeObject() throws IOException {
        GoogleCloudStorage rawStorage = GoogleCloudStorageGrpcIntegrationTest.createGoogleCloudStorage();
        StorageResourceId resourceId = new StorageResourceId(BUCKET_NAME, "testOpenLargeObject_Object");
        int partitionsCount = 50;
        byte[] partition = GoogleCloudStorageTestHelper.writeObject(rawStorage, resourceId, 0xA00000, partitionsCount);
        GoogleCloudStorageTestHelper.assertObjectContent(rawStorage, resourceId, partition, partitionsCount);
    }

    @Test
    public void testOpenObjectWithChecksum() throws IOException {
        AsyncWriteChannelOptions asyncWriteChannelOptions = AsyncWriteChannelOptions.builder().setGrpcChecksumsEnabled(true).build();
        GoogleCloudStorage rawStorage = GoogleCloudStorageGrpcIntegrationTest.createGoogleCloudStorage(asyncWriteChannelOptions);
        StorageResourceId objectToCreate = new StorageResourceId(BUCKET_NAME, "testOpenObjectWithChecksum_Object");
        byte[] objectBytes = GoogleCloudStorageTestHelper.writeObject(rawStorage, objectToCreate, 100);
        GoogleCloudStorageReadOptions readOptions = GoogleCloudStorageReadOptions.builder().setGrpcChecksumsEnabled(true).build();
        GoogleCloudStorageTestHelper.assertObjectContent(rawStorage, objectToCreate, readOptions, objectBytes);
    }

    @Test
    public void testOpenObjectWithSeek() throws IOException {
        AsyncWriteChannelOptions asyncWriteChannelOptions = AsyncWriteChannelOptions.builder().setGrpcChecksumsEnabled(true).build();
        GoogleCloudStorage rawStorage = GoogleCloudStorageGrpcIntegrationTest.createGoogleCloudStorage(asyncWriteChannelOptions);
        StorageResourceId objectToCreate = new StorageResourceId(BUCKET_NAME, "testOpenObjectWithSeek_Object");
        byte[] objectBytes = GoogleCloudStorageTestHelper.writeObject(rawStorage, objectToCreate, 100);
        int offset = 10;
        byte[] trimmedObjectBytes = Arrays.copyOfRange(objectBytes, offset, objectBytes.length);
        GoogleCloudStorageReadOptions readOptions = GoogleCloudStorageReadOptions.builder().setGrpcChecksumsEnabled(true).build();
        GoogleCloudStorageTestHelper.assertObjectContent(rawStorage, objectToCreate, readOptions, trimmedObjectBytes, 1, offset);
    }

    @Test
    public void testOpenObjectWithSeekOverBounds() throws IOException {
        AsyncWriteChannelOptions asyncWriteChannelOptions = AsyncWriteChannelOptions.builder().setGrpcChecksumsEnabled(true).build();
        GoogleCloudStorage rawStorage = GoogleCloudStorageGrpcIntegrationTest.createGoogleCloudStorage(asyncWriteChannelOptions);
        StorageResourceId objectToCreate = new StorageResourceId(BUCKET_NAME, "testOpenObjectWithSeekOverBounds_Object");
        byte[] objectBytes = GoogleCloudStorageTestHelper.writeObject(rawStorage, objectToCreate, 0x400000);
        int offset = 0x300000;
        byte[] trimmedObjectBytes = Arrays.copyOfRange(objectBytes, offset, objectBytes.length);
        GoogleCloudStorageReadOptions readOptions = GoogleCloudStorageReadOptions.builder().setGrpcChecksumsEnabled(true).build();
        GoogleCloudStorageTestHelper.assertObjectContent(rawStorage, objectToCreate, readOptions, trimmedObjectBytes, 1, offset);
    }

    @Test
    public void testOpenObjectWithSeekLimits() throws IOException {
        AsyncWriteChannelOptions asyncWriteChannelOptions = AsyncWriteChannelOptions.builder().setGrpcChecksumsEnabled(true).build();
        GoogleCloudStorage rawStorage = GoogleCloudStorageGrpcIntegrationTest.createGoogleCloudStorage(asyncWriteChannelOptions);
        StorageResourceId objectToCreate = new StorageResourceId(BUCKET_NAME, "testOpenObjectWithSeekOverBounds_Object");
        byte[] objectBytes = GoogleCloudStorageTestHelper.writeObject(rawStorage, objectToCreate, 1024);
        int offset = 100;
        byte[] trimmedObjectBytes = Arrays.copyOfRange(objectBytes, offset, objectBytes.length);
        GoogleCloudStorageReadOptions readOptions = GoogleCloudStorageReadOptions.builder().setInplaceSeekLimit(50L).setGrpcChecksumsEnabled(true).build();
        GoogleCloudStorageTestHelper.assertObjectContent(rawStorage, objectToCreate, readOptions, trimmedObjectBytes, 1, offset);
    }

    @Test
    public void testPartialRead() throws IOException {
        GoogleCloudStorage rawStorage = GoogleCloudStorageGrpcIntegrationTest.createGoogleCloudStorage();
        int segmentSize = 553;
        int segmentCount = 5;
        StorageResourceId resourceId = new StorageResourceId(BUCKET_NAME, "testReadPartialObjects_Object");
        byte[] data = GoogleCloudStorageTestHelper.writeObject(rawStorage, resourceId, segmentCount * segmentSize);
        byte[][] readSegments = new byte[segmentCount][segmentSize];
        try (SeekableByteChannel readChannel = rawStorage.open(resourceId);){
            for (int i = 0; i < segmentCount; ++i) {
                ByteBuffer segmentBuffer = ByteBuffer.wrap(readSegments[i]);
                int bytesRead = readChannel.read(segmentBuffer);
                Truth.assertThat((Integer)bytesRead).isEqualTo((Object)segmentSize);
                byte[] expectedSegment = Arrays.copyOfRange(data, i * segmentSize, i * segmentSize + segmentSize);
                Truth.assertWithMessage((String)"Unexpected segment data read.").that(readSegments[i]).isEqualTo((Object)expectedSegment);
            }
        }
    }

    @Test
    public void testChannelClosedException() throws IOException {
        GoogleCloudStorage rawStorage = GoogleCloudStorageGrpcIntegrationTest.createGoogleCloudStorage();
        int totalBytes = 1200;
        StorageResourceId resourceId = new StorageResourceId(BUCKET_NAME, "testChannelClosedException_Object");
        GoogleCloudStorageTestHelper.writeObject(rawStorage, resourceId, totalBytes);
        byte[] readArray = new byte[totalBytes];
        SeekableByteChannel readableByteChannel = rawStorage.open(resourceId);
        ByteBuffer readBuffer = ByteBuffer.wrap(readArray);
        readBuffer.limit(5);
        readableByteChannel.read(readBuffer);
        Truth.assertThat((Long)readableByteChannel.position()).isEqualTo((Object)readBuffer.position());
        readableByteChannel.close();
        readBuffer.clear();
        Assert.assertThrows(ClosedChannelException.class, () -> readableByteChannel.read(readBuffer));
    }
}

